#include "cashboxstatus.h"

#include "formatutils.h"

CashboxStatus::CashboxStatus() noexcept
    : modelData_()
    , regData_()
    , dev_()
    , fs_()
{

}

CashboxStatus::CashboxStatus(const CashboxStatus &other) noexcept
    : modelData_(other.modelData_)
    , regData_(other.regData_)
    , dev_(other.dev_)
    , fs_(other.fs_)
{

}

CashboxStatus::CashboxStatus(CashboxStatus &&other) noexcept
    : modelData_(other.modelData_)
    , regData_(other.regData_)
    , dev_(other.dev_)
    , fs_(other.fs_)

{
}

CashboxStatus::CashboxStatus(const QVariantMap &map) noexcept
    : modelData_()
    , regData_()
    , dev_()
    , fs_()

{
    parseMap(map);
}

CashboxStatus::~CashboxStatus()
{

}


bool CashboxStatus::isFiscalized() const
{
    return fs_.isFiscalized() && fs_.cashboxSerial().trimmed() == modelData_.serial().trimmed();
}



bool CashboxStatus::needFiscalization() const
{
    return !isFiscalized();
}

bool CashboxStatus::cycleIsOpen() const
{
    return fs_.cycleIsOpen();
}

bool CashboxStatus::cycleIsOver() const
{
    return fs_.cycleIsOver();
}


ModelData CashboxStatus::modelData() const
{
    return modelData_;
}

void CashboxStatus::setModelData(const ModelData &modelData)
{
    modelData_ = modelData;
}

RegData CashboxStatus::regData() const
{
    return regData_;
}

void CashboxStatus::setRegData(const RegData &regData)
{
    regData_ = regData;
}


DeviceStatus CashboxStatus::dev() const
{
    return dev_;
}

void CashboxStatus::setDev(const DeviceStatus &dev)
{
    dev_ = dev;
}

FsFullStatus CashboxStatus::fs() const
{
    return fs_;
}

void CashboxStatus::setFs(const FsFullStatus &newFs)
{
    fs_ = newFs;
}





QVariantMap CashboxStatus::toMap() const
{
    QVariantMap res;
    res.insert("model", modelData_.toMap());
    res.insert("reg", isFiscalized() ? QVariant(regData_.toMap()) : QVariant());
    res.insert("dev", dev_.toMap());
    res.insert("fs", fs_.toMap());
    res.insert("dt", DT2STR_(QDateTime::currentDateTime()));
    return res;
}

QVariantMap CashboxStatus::toExternalMap() const
{
    QVariantMap res;
    res.insert("dt", DT2STR_(QDateTime::currentDateTime()));
    res.insert("model", modelData_.toExternalMap());
    res.insert("reg", isFiscalized()  ? QVariant(regData_.toExternalMap()) : QVariant());
    res.insert("dev", dev_.toExternalMap());
    res.insert("fs", fs_.toExternalMap());
    return res;
}

void CashboxStatus::parseMap(const QVariantMap &map)
{
    modelData_.parseMap(map["model"].toMap());
    regData_.parseMap(map["reg"].toMap());
    dev_.parseMap(map["dev"].toMap());
    fs_.parseMap(map["fs"].toMap());
}

QString CashboxStatus::toHtml() const
{
    QStringList res;
    res << "<html><head>"
        << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">"
        << "</head><body>" ;
    res.append(modelData_.toHtml().split("\n"));
    if(fs_.isFiscalized()) res.append(regData_.toHtml().split("\n"));
    res.append(fs_.toHtml());
    res.append(dev_.toHtml());
    res	<< "</body></html>";
//    lmWarning() << loglist(res);
    return res.join("\n");

}

CashboxStatus &CashboxStatus::operator =(const CashboxStatus &other) noexcept
{
    modelData_ = other.modelData_;
    regData_ = other.regData_;
    dev_ = other.dev_;
    fs_ = other.fs_;
    return *this;
}

CashboxStatus &CashboxStatus::operator =(CashboxStatus &&other)noexcept
{
    modelData_ = other.modelData_;
    regData_ = other.regData_;
    dev_ = other.dev_;
    fs_ = other.fs_;
    return *this;

}

bool CashboxStatus::operator ==(const CashboxStatus &other) const
{
    return modelData_ == other.modelData_ &&
           regData_ == other.regData_ &&
           dev_ == other.dev_ &&
           fs_ == other.fs_;
}

bool CashboxStatus::operator !=(const CashboxStatus &other) const
{
    return !(*this == other);
}
