#include "cashier.h"

const Cashier::Permissions Cashier::SU_PERMS (
        Cashier::Permission::Receipts |
        Cashier::Permission::ReturnReceipts |
        Cashier::Permission::Corrections |
        Cashier::Permission::ReturnCorrections |
        Cashier::Permission::ReturnReceipts |
        Cashier::Permission::CycleOpen |
        Cashier::Permission::ReturnReceipts |

        Cashier::Permission::OfflineNotifications |
        Cashier::Permission::CycleClose |
        Cashier::Permission::Reports |
        Cashier::Permission::Settings|
        Cashier::Permission::Fiscalization |
        Cashier::Permission::WriteSerial);

Cashier::Cashier() noexcept
    : id_(-1)
    , serverId_(0)
    , login_()
    , name_()
    , password_()
    , inn_()
    , permissions_{}
{

}

Cashier::Cashier(const QVariantMap &map) noexcept
    : id_(-1)
    , serverId_(0)
    , login_()
    , name_()
    , password_()
    , inn_()
    , permissions_(Permission::Receipts)
{
    setMap(map);
}

Cashier::Cashier(const Cashier &other) noexcept
    : id_(other.id_)
    , serverId_(other.serverId_)
    , login_(other.login_)
    , name_(other.name_)
    , password_(other.password_)
    , inn_(other.inn_)
    , permissions_(other.permissions_)
{

}

Cashier::Cashier(Cashier &&other) noexcept
    : id_(other.id_)
    , serverId_(other.serverId_)
    , login_()
    , name_()
    , password_()
    , inn_()
    , permissions_(other.permissions_)
{
    login_.swap(other.login_);
    name_.swap(other.name_);
    password_.swap(other.password_);
    inn_.swap(other.inn_);
}

Cashier::~Cashier()
{

}

bool Cashier::isValid() const
{
    return id_ >= 1  && !login_.isEmpty() && !password_.isEmpty() && isActive();
}

bool Cashier::isSuperUser() const
{
    return id_ == SUPER_USER_ID || login_ == "su";
}

bool Cashier::fromServer() const
{
    return serverId_ > 0;
}


qint32 Cashier::id() const
{
    return id_;
}

void Cashier::setId(const qint32 &id)
{
    id_ = id;
}

qint64 Cashier::serverId() const
{
    return serverId_;
}

void Cashier::setServerId(const qint64 &serverId)
{
    serverId_ = serverId;
}

QString Cashier::login() const
{
    return login_;
}

void Cashier::setLogin(const QString &login)
{
    login_ = login;
}

QString Cashier::name() const
{
    return name_;
}

void Cashier::setName(const QString &name)
{
    name_ = name;
}

QString Cashier::password() const
{
    return password_;
}

void Cashier::setPassword(const QString &password)
{
    password_ = password;
}

const QString &Cashier::inn() const
{
    return inn_;
}

void Cashier::setInn(const QString &newInn)
{
    inn_ = newInn;
}

Cashier::Permissions Cashier::permissions() const
{
    if(isSuperUser()) return SU_PERMS;
    return permissions_;
}

bool Cashier::hasPermission(Permission perm) const
{
    return permissions().testFlag(perm);
}

void Cashier::setPermissions(Permissions perms)
{
    permissions_ = perms;
}

void Cashier::setPermission(Cashier::Permission p, bool value)
{
    permissions_.setFlag(p, value);
}

bool Cashier::hasOneReceiptPermission() const
{
    return permissions_.testFlag(Permission::Receipts) ||
            permissions_.testFlag(Permission::ReturnReceipts);
}

bool Cashier::hasOneCorrectionPermission() const
{
    return permissions_.testFlag(Permission::Corrections) ||
            permissions_.testFlag(Permission::ReturnCorrections);
}

bool Cashier::isActive() const
{
    return static_cast<quint32>(permissions_) > 0u;
}

QVariantMap Cashier::toMap() const
{
    QVariantMap res;
    res.insert("id", id_);
    res.insert("serverId", QString::number(serverId_));
    res.insert("login", login_);
    if(!name_.isEmpty()) res.insert("name", name_);
    res.insert("password", password_);
    if(!inn_.isEmpty())res.insert("inn", inn_);
    res.insert("permissions", static_cast<quint32>(permissions_));
    return res;
}

QVariantMap Cashier::toExternalMap(bool withPassword) const
{
    QVariantMap res = toMap();
    if(!withPassword)res.remove("password");
    QVariantMap p = {
        {"Receipts"            , permissions_.testFlag(Permission::Receipts            )},
        {"Reports"             , permissions_.testFlag(Permission::Reports             )},
        {"Settings"            , permissions_.testFlag(Permission::Settings            )},
        {"Fiscalization"       , permissions_.testFlag(Permission::Fiscalization       )},
        {"ReturnReceipts"      , permissions_.testFlag(Permission::ReturnReceipts      )},
        {"WriteSerial"         , permissions_.testFlag(Permission::WriteSerial         )},
        {"CycleOpen"           , permissions_.testFlag(Permission::CycleOpen           )},
        {"CycleClose"          , permissions_.testFlag(Permission::CycleClose          )},
        {"Corrections"         , permissions_.testFlag(Permission::Corrections         )},
        {"ReturnCorrections"   , permissions_.testFlag(Permission::ReturnCorrections   )},
        {"OfflineNotifications", permissions_.testFlag(Permission::OfflineNotifications)},
        };
    res["permissions"] = p;
    return res;
}

void Cashier::setMap(const QVariantMap &map)
{
    id_ = map.contains("id") ? map["id"].toInt() : -1;
    serverId_ = map["serverId"].toInt();
    login_ = map["login"].toString();
    name_ = map["name"].toString();
    password_ = map["password"].toString();
    inn_ = map["inn"].toString().trimmed();
    permissions_ = Permissions();
    if(map.contains("permissions"))
    {
        QVariant p = map["permissions"];
        if(p.type() != QVariant::Map) permissions_ = Permissions(map["permissions"].toUInt());
        else
        {
            QVariantMap m = p.toMap();
            permissions_.setFlag(Permission::Receipts            ,
                                 m.contains("Receipts"            ) &&
                                     m["Receipts"            ].toBool());
            permissions_.setFlag(Permission::Reports             ,
                                 m.contains("Reports"             ) &&
                                     m["Reports"             ].toBool());
            permissions_.setFlag(Permission::Settings            ,
                                 m.contains("Settings"            ) &&
                                     m["Settings"            ].toBool());
            permissions_.setFlag(Permission::Fiscalization       ,
                                 m.contains("Fiscalization"       ) &&
                                     m["Fiscalization"       ].toBool());
            permissions_.setFlag(Permission::ReturnReceipts      ,
                                 m.contains("ReturnReceipts"      ) &&
                                     m["ReturnReceipts"      ].toBool());
            permissions_.setFlag(Permission::WriteSerial         ,
                                 m.contains("WriteSerial"         ) &&
                                     m["WriteSerial"         ].toBool());
            permissions_.setFlag(Permission::CycleOpen           ,
                                 m.contains("CycleOpen"           ) &&
                                     m["CycleOpen"           ].toBool());
            permissions_.setFlag(Permission::CycleClose          ,
                                 m.contains("CycleClose"          ) &&
                                     m["CycleClose"          ].toBool());
            permissions_.setFlag(Permission::Corrections         ,
                                 m.contains("Corrections"         ) &&
                                     m["Corrections"         ].toBool());
            permissions_.setFlag(Permission::ReturnCorrections   ,
                                 m.contains("ReturnCorrections"   ) &&
                                     m["ReturnCorrections"   ].toBool());
            permissions_.setFlag(Permission::OfflineNotifications,
                                 m.contains("OfflineNotifications") &&
                                     m["OfflineNotifications"].toBool());
        }
    }
}

QList<Cashier> Cashier::parseCashiers(const QVariantMap &map)
{
    QList<Cashier> res;
    if(!map.contains("cashiers")) return res;
    QVariantList list = map["cashiers"].toList();
    for(const QVariant &v :list)
    {
        res << Cashier(v.toMap());
    }
    return res;
}

Cashier &Cashier::operator=(const Cashier &other) noexcept
{
   id_ = other.id_;
   serverId_ = other.serverId_;
   login_ = other.login_;
   name_ = other.name_;
   password_ = other.password_;
   inn_ = other.inn_;
   permissions_ = other.permissions_;
   return *this;
}

Cashier &Cashier::operator=(Cashier &&other)noexcept
{
    id_ = other.id_;
    serverId_ = other.serverId_;
    login_.swap(other.login_);
    name_.swap(other.name_);
    password_.swap(other.password_);
    inn_.swap(other.inn_);
    permissions_ = other.permissions_;
    return *this;
}

bool Cashier::operator==(const Cashier &other) const noexcept
{
    return id_ == other.id_ &&
            serverId_ == other.serverId_ &&
            login_ == other.login_ &&
            name_ == other.name_ &&
            password_ == other.password_ &&
            inn_ == other.inn_  &&
            permissions_ == other.permissions_ ;
}

bool Cashier::operator!=(const Cashier &other) const noexcept
{
    return !(*this == other);
}

bool Cashier::operator <(const Cashier &other) const noexcept
{
    return  id_ < other.id_ ||
            (id_ == other.id_ && login_ < other.login_);
}

