#ifndef CASHIER_H
#define CASHIER_H

#include "crc.h"

#include <QVariantMap>
#include <QList>

class Cashier
{
public:
    static const quint8 SUPER_USER_ID          = 1;
    static const quint8 MAX_CASHIERS_ID        = 20;
    enum class Permission: quint32
    {
        Receipts                = 0x00000001u,
        Reports                 = 0x00000002u,
        Settings                = 0x00000004u,
        Fiscalization           = 0x00000008u,
        ReturnReceipts          = 0x00000010u,
        WriteSerial             = 0x00000020u,
        CycleOpen               = 0x00000040u,
        CycleClose              = 0x00000080u,
        Corrections             = 0x00000100u,
        ReturnCorrections       = 0x00000200u,
        OfflineNotifications    = 0x00000400u,
    };

    Q_DECLARE_FLAGS(Permissions, Permission)
    static const Permissions SU_PERMS;

    Cashier()noexcept;
    explicit Cashier(const QVariantMap &map)noexcept;
    Cashier(const Cashier &other)noexcept;
    Cashier(Cashier &&other)noexcept;
    ~Cashier();

    bool isValid() const;
    bool isSuperUser() const;
    bool fromServer() const;

    qint32 id() const;
    void setId(const qint32 &id);

    qint64 serverId() const;
    void setServerId(const qint64 &serverId);

    QString login() const;
    void setLogin(const QString &login);

    QString name() const;
    void setName(const QString &name);

    QString password() const;
    void setPassword(const QString &password);

    const QString &inn() const;
    void setInn(const QString &newInn);

    Permissions permissions() const;
    bool hasPermission(Permission perm) const;
    void setPermissions(Permissions perms);
    void setPermission(Permission p, bool value);
    bool hasOneReceiptPermission() const;
    bool hasOneCorrectionPermission() const;

    bool isActive() const;

    QVariantMap toMap() const;
    QVariantMap toExternalMap(bool withPassword) const;
    void setMap(const QVariantMap &map);

    static QList<Cashier> parseCashiers(const QVariantMap &map);

    Cashier &operator= (const Cashier &other)noexcept;
    Cashier &operator= (Cashier &&other)noexcept;
    bool operator== (const Cashier &other) const noexcept;
    bool operator!= (const Cashier &other) const noexcept;

    bool operator < (const Cashier &other) const noexcept;




private:
    qint32 id_;
    qint64 serverId_;
    QString login_;
    QString name_;
    QString password_;
    QString inn_;
    Permissions permissions_;
};

typedef QList<Cashier> Cashiers;

#ifdef Q_DECL_CONST_FUNCTION
Q_DECL_CONST_FUNCTION inline quint32 qHash(const Cashier &key, uint seed = 0) Q_DECL_NOTHROW {
    return Crc32()(QByteArray::number(key.id()) + key.name().toUtf8() + key.login().toUtf8()) ^ static_cast<quint8>(seed); }
#else
inline quint32 qHash(const Cashier &key, uint seed = 0) Q_DECL_NOTHROW {
    return Crc32()(QByteArray::number(key.id()) + key.name().toUtf8()+ key.login().toUtf8()) ^ static_cast<quint8>(seed); }
#endif


Q_DECLARE_OPERATORS_FOR_FLAGS(Cashier::Permissions)
Q_DECL_CONSTEXPR inline QFlags<Cashier::Permission> operator~(Cashier::Permission f) Q_DECL_NOTHROW
{ return QFlags<Cashier::Permission>(~static_cast<quint32>(f)) ;}

typedef QMap<qint32, Cashier>  ActiveCashiers;
typedef QList<Cashier>  ActiveCashiersList;




Q_DECLARE_METATYPE(Cashier)

#endif // CASHIER_H



