#include "coreapiresult.h"

CoreApiResult::CoreApiResult() noexcept
    : code_(CoreApiConst::ErrorCode::UnknownError)
    , uid_()
    , descr_()
    , data_()
{

}

CoreApiResult::CoreApiResult(const CoreApiConst::ErrorCode &code) noexcept
    : code_(code)
    , uid_()
    , descr_()
    , data_()
{

}

CoreApiResult::CoreApiResult(const CoreApiConst::ErrorCode &code, const QString &msg) noexcept
    : code_(code)
    , uid_()
    , descr_(msg)
    , data_()
{

}

CoreApiResult::CoreApiResult(const CoreApiConst::ErrorCode &code, const QString &msg, const QVariantMap &data) noexcept
    : code_(code)
    , uid_()
    , descr_(msg)
    , data_(data)
{

}

CoreApiResult::CoreApiResult(const QVariantMap &data) noexcept
    : code_(data.contains("result") ?
                static_cast<CoreApiConst::ErrorCode>(data["result"].toUInt()) :
    CoreApiConst::ErrorCode::UnknownError)
    , uid_(data["uid"].toString())
    , descr_(data["descr"].toString())
    , data_(data["data"].toMap())
{
}

CoreApiResult::CoreApiResult(const CoreApiResult &other) noexcept
    : code_(other.code_)
    , uid_(other.uid_)
    , descr_(other.descr_)
    , data_(other.data_)
{

}

CoreApiResult::CoreApiResult(CoreApiResult &&other) noexcept
    : code_(other.code_)
    , uid_()
    , descr_()
    , data_()
{
    uid_.swap(other.uid_);
    descr_.swap(other.descr_);
    data_.swap(other.data_);
}

CoreApiResult::~CoreApiResult()
{

}

bool CoreApiResult::isOk() const
{
    return code_ == CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode CoreApiResult::code() const
{
    return code_;
}

CoreApiResult &CoreApiResult::setCode(CoreApiConst::ErrorCode code)
{
    code_ = code;
    return *this;
}

QString CoreApiResult::descr() const
{
    return descr_.isEmpty() && code_ != CoreApiConst::ErrorCode::Ok ?
                CoreApiConst::defaultErrorMsg(code_) : descr_;
}

CoreApiResult &CoreApiResult::setDescr(const QString &msg)
{
    descr_ = msg;
    return *this;
}

QVariantMap CoreApiResult::data() const
{
    return data_;
}

CoreApiResult &CoreApiResult::setData(const QVariantMap &data)
{
    data_ = data;
    return *this;
}

QString CoreApiResult::uid() const
{
    return uid_;
}

CoreApiResult &CoreApiResult::setUid(const QString &uid)
{
    uid_ = uid;
    return *this;
}

QVariantMap CoreApiResult::toMap() const
{
    QVariantMap res;
    res.insert("result", static_cast<qint32>(code_));
    if(!descr_.isEmpty())res.insert("descr", descr_);
    else if(code_ != CoreApiConst::ErrorCode::Ok)
    {
        res.insert("descr", CoreApiConst::defaultErrorMsg(
                       static_cast<CoreApiConst::ErrorCode>(code_)));
    }
    if(!data_.isEmpty()) res.insert("data", data_);
    res.insert("uid", uid_);
    return res;
}

CoreApiResult &CoreApiResult::operator =(const CoreApiResult &other) noexcept
{
    code_ = other.code_;
    uid_ = other.uid_;
    descr_ = other.descr_;
    data_ = other.data_;
    return *this;
}

CoreApiResult &CoreApiResult::operator =(CoreApiResult &&other)noexcept
{
    code_ = other.code_;
    uid_.swap(other.uid_);
    descr_.swap(other.descr_);
    data_.swap(other.data_);
    return *this;
}

bool CoreApiResult::operator ==(const CoreApiResult &other) const noexcept
{
    return code_ == other.code_ &&
            uid_ == other.uid_ &&
            descr_ == other.descr_ &&
            data_ == other.data_ ;
}

bool CoreApiResult::operator !=(const CoreApiResult &other) const noexcept
{
    return !(*this == other);
}
