#include "coretransaction.h"
#include "cashboxbuildconfig.h"
#include "cashier.h"
#include "formatutils.h"

#include <random>
#include <QDataStream>
#include <QCryptographicHash>



CoreTransaction::CoreTransaction() noexcept
    : operation_(Operation::INVALID)
    , uid_()
    , externalId_()
    , clientId_()
    , params_()
    , cashierLogin_()
    , cashierPassword_()
    , formatToExternal_(false)
    , clientDateTime_()
{

}

CoreTransaction::CoreTransaction(const CoreTransaction &other) noexcept
    : operation_(other.operation_)
    , uid_(other.uid_)
    , externalId_(other.externalId_)
    , clientId_(other.clientId_)
    , params_(other.params_)
    , cashierLogin_(other.cashierLogin_)
    , cashierPassword_(other.cashierPassword_)
    , formatToExternal_(other.formatToExternal_)
    , clientDateTime_(other.clientDateTime_)
{

}

CoreTransaction::CoreTransaction(CoreTransaction &&other)noexcept
    : operation_(other.operation_)
    , uid_()
    , externalId_()
    , clientId_()
    , params_()
    , cashierLogin_()
    , cashierPassword_()
    , formatToExternal_(other.formatToExternal_)
    , clientDateTime_()


{
    uid_.swap(other.uid_);
    externalId_.swap(other.externalId_);
    clientId_.swap(other.clientId_);
    params_.swap(other.params_);
    cashierLogin_.swap(other.cashierLogin_);
    cashierPassword_.swap(other.cashierPassword_);
    clientDateTime_.swap(other.clientDateTime_);
}

CoreTransaction::CoreTransaction(CoreApiConst::RemoteCommand cmd, const QVariantMap &data)noexcept
    : operation_(cmd)
    , uid_()
    , externalId_()
    , clientId_()
    , params_()
    , cashierLogin_()
    , cashierPassword_()
    , formatToExternal_(false)
    , clientDateTime_()
{
    parseMap(data);
}

CoreTransaction::~CoreTransaction()
{

}

bool CoreTransaction::isValid(CoreApiConst::ErrorCode &error, QString *message, bool checkUid) const
{
    if(cashierLogin_.isEmpty() || (cashierPassword_.length() < CoreApiConst::MIN_PASSWORD_LENGTH))
    {
        error = CoreApiConst::ErrorCode::InvalidTransactionCashier;
        if(message) *message = QObject::tr("Кассир с таким логином или паролем не существует");
//        lmWarning() << cashierLogin_ << cashierLogin_.length() << cashierLogin_.isEmpty()<< logtab
//                    << cashierPassword_ << cashierPassword_.length() << CoreApiConst::MIN_PASSWORD_LENGTH<< logtab
//                    << logvariant(toMap());
        return false;
    }
    if(uid_.trimmed().isEmpty() && checkUid)
    {
        error = CoreApiConst::ErrorCode::EmptyTransactionUid;
        if(message) *message = QObject::tr("Не задан уникальный ид транзакции");
        return false;
    }
    if(operation_ <= CoreApiConst::RemoteCommand::INVALID || operation_ > CoreApiConst::RemoteCommand::MAX_VALID_OPERATION)
    {
        error = CoreApiConst::ErrorCode::InvalidTransactionType;
        if(message) *message = QObject::tr("Не задана операция");
        return false;
    }
    return true;
}

CoreTransaction::Operation CoreTransaction::operation() const
{
    return operation_;
}

void CoreTransaction::setOperation(const Operation &operation)
{
    operation_ = operation;
}

QString CoreTransaction::uid() const
{
    return uid_;
}

void CoreTransaction::setUid(const QString &uid)
{
    uid_ = uid;
}

QString CoreTransaction::genUid()
{
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds << static_cast<qint32>(operation_);
    ds.writeRawData(cashierLogin_.toUtf8().constData(), cashierLogin_.toUtf8().size());
    ds.writeRawData(cashierPassword_.toUtf8().constData(), cashierPassword_.toUtf8().size());
    ds << QDateTime::currentDateTime().currentMSecsSinceEpoch();

    std::random_device rd;
    std::uniform_int_distribution<quint32> rdd(0, 0x7FFFFFFFu);
    ds << rdd(rd) << rdd(rd);
    buf = QCryptographicHash::hash(buf, QCryptographicHash::Sha1);
    uid_ = QString::fromLatin1(buf.toBase64());
    return uid_;
}

QString CoreTransaction::externalId() const
{
    return externalId_.isEmpty() ? uid_ : externalId_;
}

void CoreTransaction::setExternalId(const QString &externalId)
{
    externalId_ = externalId;
}


const QString &CoreTransaction::clientId() const
{
    return clientId_;
}

void CoreTransaction::setClientId(const QString &newClientId)
{
    clientId_ = newClientId;
}

QVariantMap CoreTransaction::params() const
{
    return params_;
}

void CoreTransaction::setParams(const QVariantMap &params)
{
    params_ = params;
}

QString CoreTransaction::cashierLogin() const
{
    return cashierLogin_;
}

void CoreTransaction::setCashierLogin(const QString &login)
{
    cashierLogin_ = login;
}




QString CoreTransaction::cashierPassword() const
{
    return cashierPassword_;
}

void CoreTransaction::setCashierPassword(const QString &cashierPassword)
{
    cashierPassword_ = cashierPassword;
}

bool CoreTransaction::formatToExternal() const
{
    return formatToExternal_;
}

void CoreTransaction::setFormatToExternal(bool newFormatToExternal)
{
    formatToExternal_ = newFormatToExternal;
}


QDateTime CoreTransaction::clientDateTime() const
{
    return clientDateTime_;
}

void CoreTransaction::setClientDateTime(const QDateTime &newClientDateTime)
{
    clientDateTime_ = newClientDateTime;
}


QVariantMap CoreTransaction::toMap() const
{
    QVariantMap res;
    res.insert("operation", static_cast<quint32>(operation_));
    res.insert("uid", uid_);
    res.insert("externalId", externalId_);
    if(!clientId_.isEmpty()) res.insert("clientId", clientId_);
    if(!params_.isEmpty())res.insert("params", params_);
    res.insert("login", cashierLogin_);
    res.insert("password", cashierPassword_);
    res["toExt"] = formatToExternal_ ? 1 : 0;
    if(clientDateTime_.isValid())res["cdt"] = DT2STR_(clientDateTime_);
    return res;
}

void CoreTransaction::parseMap(const QVariantMap &map)
{
    operation_ = static_cast<Operation>(map["operation"].toUInt());
    if(operation_ > Operation::MAX_VALID_OPERATION) operation_ = Operation::INVALID;
    uid_ = map["uid"].toString().trimmed();
    externalId_ = map["externalId"].toString();
    clientId_ = map["clientId"].toString().trimmed();
    params_ = map["params"].toMap();
    cashierLogin_ = map["login"].toString();
    cashierPassword_ = map["password"].toString();
    formatToExternal_ = map["toExt"].toInt();
    if(map.contains("cdt"))
    {
        clientDateTime_ = STR2DT_(map["cdt"].toString());
    }
}

CoreTransaction &CoreTransaction::operator =(const CoreTransaction &other) noexcept
{
    operation_ = other.operation_;
    uid_ = other.uid_;
    externalId_ = other.externalId_;
    clientId_ = other.clientId_;
    params_ = other.params_;
    cashierLogin_ = other.cashierLogin_;
    cashierPassword_ = other.cashierPassword_;
    formatToExternal_ = other.formatToExternal_;
    clientDateTime_ = other.clientDateTime_;
    return *this;
}

CoreTransaction &CoreTransaction::operator =(CoreTransaction &&other)noexcept
{
    operation_ = other.operation_;
    uid_.swap(other.uid_);
    externalId_.swap(other.externalId_);
    clientId_.swap(other.clientId_);
    params_.swap(other.params_);
    cashierLogin_.swap(other.cashierLogin_);
    cashierPassword_.swap(other.cashierPassword_);
    formatToExternal_ = other.formatToExternal_;
    clientDateTime_.swap(other.clientDateTime_);
    return *this;
}

bool CoreTransaction::operator ==(const CoreTransaction &other) const noexcept
{
    return operation_ == other.operation_ &&
           uid_ == other.uid_ &&
           externalId_ == other.externalId_ &&
           clientId_ == other.clientId_ &&
           params_ == other.params_ &&
           cashierLogin_ == other.cashierLogin_ &&
           cashierPassword_ == other.cashierPassword_ &&
           clientDateTime_ == other.clientDateTime_;

}

bool CoreTransaction::operator !=(const CoreTransaction &other) const noexcept
{
    return !(*this == other);
}
