#include "devicestatus.h"

DeviceStatus::DeviceStatus() noexcept
    : dsn_()
    , hwHash_()
    , versionCode_(0)
    , versionText_()
    , ipAddress_()
    , interfaces_()
    , mem_()
    , storage_()
    , accumLevel_(10u, 500ll)
{

}

DeviceStatus::DeviceStatus(const DeviceStatus &other)noexcept
    : dsn_(other.dsn_)
    , hwHash_(other.hwHash_)
    , versionCode_(other.versionCode_)
    , versionText_(other.versionText_)
    , ipAddress_(other.ipAddress_)
    , interfaces_(other.interfaces_)
    , mem_(other.mem_)
    , storage_(other.storage_)
    , accumLevel_(other.accumLevel_)
{

}

DeviceStatus::DeviceStatus(DeviceStatus &&other)noexcept
    : dsn_()
    , hwHash_()
    , versionCode_(other.versionCode_)
    , versionText_()
    , ipAddress_()
    , interfaces_()
    , mem_()
    , storage_()
    , accumLevel_(other.accumLevel_)

{
    dsn_.swap(other.dsn_);
    hwHash_.swap(other.hwHash_);
    versionText_.swap(other.versionText_);
    ipAddress_.swap(other.ipAddress_);
    interfaces_.swap(other.interfaces_);
    mem_.swap(other.mem_);
    storage_.swap(other.storage_);

}

DeviceStatus::DeviceStatus(const QVariantMap &map)noexcept
    : dsn_()
    , hwHash_()
    , versionCode_(0)
    , versionText_()
    , ipAddress_()
    , interfaces_()
    , mem_()
    , storage_()
    , accumLevel_(10u, 500ll)

{
    parseMap(map);
}

DeviceStatus::~DeviceStatus()
{

}

bool DeviceStatus::isValid() const
{
    return !dsn_.isEmpty() &&
            !hwHash_.isEmpty() &&
            versionCode_ > 0 &&
            !versionText_.isEmpty();
}

QString DeviceStatus::dsn() const
{
    return dsn_;
}

void DeviceStatus::setDsn(const QString &dsn)
{
    dsn_ = dsn;
}

QString DeviceStatus::hwHash() const
{
    return hwHash_;
}

void DeviceStatus::setHwHash(const QString &hwHash)
{
    hwHash_ = hwHash;
}

quint32 DeviceStatus::versionCode() const
{
    return versionCode_;
}

void DeviceStatus::setVersionCode(const quint32 &versionCode)
{
    versionCode_ = versionCode;
}

QString DeviceStatus::versionText() const
{
    return versionText_;
}

void DeviceStatus::setVersionText(const QString &versionText)
{
    versionText_ = versionText;
}

QStringList DeviceStatus::ipAddresses() const
{
    return ipAddress_.split(",");
}


QString DeviceStatus::ipAddress() const
{
    return ipAddress_;
}

void DeviceStatus::setIpAddress(const QString &ipAddress)
{
    ipAddress_ = ipAddress;
}

QString DeviceStatus::interfaces() const
{
    return interfaces_;
}

void DeviceStatus::setInterfaces(const QString &newInterfaces)
{
    interfaces_ = newInterfaces;
}

QVariantMap DeviceStatus::storage() const
{
    return storage_;
}

void DeviceStatus::setStorage(const QVariantMap &newStorage)
{
    storage_ = newStorage;
}

QVariantMap DeviceStatus::mem() const
{
    return mem_;
}

void DeviceStatus::setMem(const QVariantMap &newMem)
{
    mem_ = newMem;
}

const FixNumber &DeviceStatus::accumLevel() const
{
    return accumLevel_;
}

void DeviceStatus::setAccumLevel(const FixNumber &newAccumLevel)
{
    accumLevel_ = newAccumLevel;
}





QVariantMap DeviceStatus::toMap() const
{
    QVariantMap res;
    res.insert("dsn", dsn_);
    res.insert("hw", hwHash_);
    res.insert("versionCode", versionCode_);
    res.insert("versionText", versionText_);
    res.insert("ipAddresses", ipAddress_);
    res.insert("accumLevel", accumLevel_.toMap());
    res.insert("interfaces", interfaces_);
    res.insert("mem", mem_);
    res.insert("storage", storage_);
    return res;
}

QVariantMap DeviceStatus::toExternalMap() const
{
    QVariantMap res;
    res.insert("dsn", dsn_);
    res.insert("versionCode", versionCode_);
    res.insert("versionText", versionText_);
    res.insert("ipAddresses", ipAddress_);
    res.insert("interfaces", interfaces_);
    res.insert("mem", mem_);
    res.insert("storage", storage_);
    res.insert("accumLevel", accumLevel_.toString() + "%");
    return res;
}

void DeviceStatus::parseMap(const QVariantMap &map)
{
    dsn_ = map["dsn"].toString();
    hwHash_ = map["hw"].toString();
    versionCode_ = map["versionCode"].toUInt();
    versionText_ = map["versionText"].toString();
    ipAddress_ = map["ipAddresses"].toString();
    interfaces_ = map["interfaces"].toString();
    mem_ = map["mem"].toMap();
    storage_ = map["storage"].toMap();
    accumLevel_.setMap(map["accumLevel"].toMap());
}

QStringList DeviceStatus::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<p align=\"center\"><b>Об устройстве</b></p>");
    res << QStringLiteral("<table align=\"center\" width=100%><tbody>")
        << QStringLiteral("<tr><td>DSN:</td><td align=\"right\">%1</td></tr>")
               .arg(dsn_)
        << QStringLiteral("<tr><td>APK:</td><td align=\"right\">%1/%2</td></tr>")
               .arg(versionCode_).arg(versionText_)
        << QStringLiteral("<tr><td>IP адреса:</td><td align=\"right\">%1</td></tr>")
               .arg(ipAddress_)
        << "</tbody></table><hr>";
    return res;
}

DeviceStatus &DeviceStatus::operator =(const DeviceStatus &other) noexcept
{
    dsn_ = other.dsn_;
    hwHash_ = other.hwHash_;
    versionCode_ = other.versionCode_;
    versionText_ = other.versionText_;
    ipAddress_ = other.ipAddress_;
    interfaces_ = other.interfaces_;
    mem_ = other.mem_;
    storage_ = other.storage_;
    accumLevel_ = other.accumLevel_;
    return *this;
}

DeviceStatus &DeviceStatus::operator =(DeviceStatus &&other)noexcept
{
    dsn_.swap(other.dsn_);
    hwHash_.swap(other.hwHash_);
    versionCode_ = other.versionCode_;
    versionText_.swap(other.versionText_);
    ipAddress_.swap(other.ipAddress_);
    interfaces_.swap(other.interfaces_);
    mem_.swap(other.mem_);
    storage_.swap(other.storage_);
    accumLevel_ = other.accumLevel_;
    return *this;
}


bool DeviceStatus::operator ==(const DeviceStatus &other) const noexcept
{
    return dsn_ == other.dsn_ &&
           hwHash_ == other.hwHash_ &&
           versionCode_ == other.versionCode_ &&
           versionText_ == other.versionText_ &&
           ipAddress_ == other.ipAddress_ &&
           interfaces_ == other.interfaces_ &&
           mem_ == other.mem_ &&
           storage_ == other.storage_ &&
           accumLevel_ == other.accumLevel_;
}

bool DeviceStatus::operator !=(const DeviceStatus &other) const noexcept
{
    return !(*this == other);
}
