#ifndef FDFCONSTANTS_H
#define FDFCONSTANTS_H

#include "qhashfunctions.h"
#include <QFlags>
#include <QString>
#include <QStringList>
#include <QSet>

namespace fdf {

    enum class RegFlag1290: qint32
    {
        ExternalPrinter             = 0x00000002,
        FosaCashbox                 = 0x00000004,
        InternetOnly                = 0x00000020,
        ExcisableProducts           = 0x00000040,
        LabledProducts              = 0x00000100,
        ServicesOnly                = 0x00000200,
        GamesCashbox                = 0x00000400,
        LotteryCashbox              = 0x00000800,
        Pawnshop                    = 0x00001000,
        Insurance                   = 0x00002000,
        AutoCashboxForLabled        = 0x00004000,
        Restaurant                  = 0x00008000,
        Wholesale                   = 0x00010000,
    };
    Q_DECLARE_FLAGS(RegFlags1290, RegFlag1290)
    Q_DECLARE_OPERATORS_FOR_FLAGS(RegFlags1290)

    Q_DECL_CONSTEXPR inline QFlags<RegFlag1290> operator~(RegFlag1290 f) Q_DECL_NOTHROW
    { return QFlags<RegFlag1290>(~static_cast<qint32>(f)) ;}

    inline QStringList regFlag1290ToStrings(const RegFlags1290 &flags) {
        if(static_cast<quint32>(flags) == 0u) return QStringList();
        QStringList sl;
        if(flags.testFlag(RegFlag1290::ExternalPrinter)) sl << QStringLiteral("ПРИНТЕР В АВТОМАТЕ");

        if(flags.testFlag(RegFlag1290::FosaCashbox      )) sl << QStringLiteral("АС БСО");
        if(flags.testFlag(RegFlag1290::InternetOnly     )) sl << QStringLiteral("ККТ ДЛЯ ИНТЕРНЕТ");
        if(flags.testFlag(RegFlag1290::ExcisableProducts)) sl << QStringLiteral("ПОДАКЦИЗНЫЕ ТОВАРЫ");
        if(flags.testFlag(RegFlag1290::LabledProducts   )) sl << QStringLiteral("ТМТ");
        if(flags.testFlag(RegFlag1290::ServicesOnly     )) sl << QStringLiteral("ККТ ДЛЯ УСЛУГ");
        if(flags.testFlag(RegFlag1290::GamesCashbox     )) sl << QStringLiteral("ПРОВЕДЕНИЕ АЗАРТНОЙ ИГРЫ");
        if(flags.testFlag(RegFlag1290::LotteryCashbox   )) sl << QStringLiteral("ПРОВЕДЕНИЕ ЛОТЕРЕИ");
        if(flags.testFlag(RegFlag1290::Pawnshop         )) sl << QStringLiteral("ЛОМБАРД");
        if(flags.testFlag(RegFlag1290::Insurance        )) sl << QStringLiteral("СТРАХОВАНИЕ");
        if(flags.testFlag(RegFlag1290::AutoCashboxForLabled   )) sl << QStringLiteral("ККТ С ТОРГ. АВТОМАТОМ");
        if(flags.testFlag(RegFlag1290::Restaurant         )) sl << QStringLiteral("ККТ В ОБЩ. ПИТАНИИ");
        if(flags.testFlag(RegFlag1290::Wholesale        )) sl << QStringLiteral("ККТ В ОПТ. ТОРГОВЛЕ С ОРГ. И ИП");

        return sl;
    }

    enum class RegFlag: qint32
    {
        AutoMode                    = 0x00000001,
        OfflineMode                 = 0x00000002,
        EncriptionMode              = 0x00000004
    };
    Q_DECLARE_FLAGS(RegFlags, RegFlag)
    Q_DECLARE_OPERATORS_FOR_FLAGS(RegFlags)

    Q_DECL_CONSTEXPR inline QFlags<RegFlag> operator~(RegFlag f) Q_DECL_NOTHROW
    { return QFlags<RegFlag>(~static_cast<qint32>(f)) ;}

    enum class Tax: quint8
    {
        Invalid                     = 0x00u,
        Basic                       = 0x01u,
        SimpleProfit                = 0x02u,
        SimpleProfitExpense         = 0x04u,
        SingleOnImputedProfit       = 0x08u,
        SingleAgriculturalTax       = 0x10u,
        PatentSystemOfTaxation      = 0x20u
    };
    Q_DECLARE_FLAGS(Taxes, Tax)
    Q_DECLARE_OPERATORS_FOR_FLAGS(Taxes)

    Q_DECL_CONSTEXPR inline QFlags<Tax> operator~(Tax f) Q_DECL_NOTHROW
    { return QFlags<Tax>(~static_cast<quint8>(f)) ;}

    inline QString taxesToString(const Taxes &tx) {
        QStringList sl;
        if(tx.testFlag(Tax::Basic)) sl << QStringLiteral("ОСН");
        if(tx.testFlag(Tax::SimpleProfit)) sl << QStringLiteral("УСН доход");
        if(tx.testFlag(Tax::SimpleProfitExpense)) sl << QStringLiteral("УСН доход-расход");
        if(tx.testFlag(Tax::SingleOnImputedProfit)) sl << QStringLiteral("ЕНВД");
        if(tx.testFlag(Tax::SingleAgriculturalTax)) sl << QStringLiteral("ЕСХН");
        if(tx.testFlag(Tax::PatentSystemOfTaxation)) sl << QStringLiteral("ПАТЕНТ");
        return sl.join(", ");
    }
    inline QString taxToString(const Tax&tx) {
        switch(tx)
        {
        case Tax::Basic : return QStringLiteral("ОСН");
        case Tax::SimpleProfit : return QStringLiteral("УСН доход");
        case Tax::SimpleProfitExpense : return QStringLiteral("УСН доход-расход");
        case Tax::SingleOnImputedProfit : return QStringLiteral("ЕНВД");
        case Tax::SingleAgriculturalTax : return QStringLiteral("ЕСХН");
        case Tax::PatentSystemOfTaxation : return QStringLiteral("ПАТЕНТ");
        default: return QString();
        }
    }
    inline bool taxIsValid(const Tax &t) {
        static const QSet<qint32> VALID {
            0x01u, 0x02u, 0x04u, /*0x08u,*/ 0x10u, 0x20u
        };
        return VALID.contains(static_cast<qint32>(t));
    }
    inline bool taxesIsValid(const Taxes &t) {
        static const quint8 INVALID= 0xC8U;
        return (INVALID & static_cast<quint8>(t)) == 0u;
    }
    inline bool validateTaxes(Taxes &t)
    {
        if(!taxesIsValid(t))
        {
            t = Taxes(static_cast<quint8>(t) & 0x37u);
        }
        return taxesIsValid(t);
    }
    enum class Reason:quint32
    {
        FsChange                             = 0x00000001u,// Замена фискального накопителя
        OfdChange                            = 0x00000002u,// Замена оператора фискальных данных
        UserNameChange                       = 0x00000004u,// Изменение наименования пользователя контрольно-кассовой техники
        AddressChange                        = 0x00000008u,// Изменение адреса и (или) места установки (применения) контрольно-кассовой техники
        OfflineToOnline                      = 0x00000010u,// Перевод ККТ из автономного режима в режим передачи данных
        OnlineToOffline                      = 0x00000020u,// Перевод ККТ из режима передачи данных в автономный режим
        CashboxVersionChange                 = 0x00000040u,// Изменение версии модели ККТ
        TaxesChange                          = 0x00000080u,// Изменение перечня систем налогообложения, применяемых при осуществлении расчетов
        BoxIdChange                          = 0x00000100u,// Изменение номера автоматического устройства для расчетов, в составе которого применяется ККТ
        AutoToSingle                         = 0x00000200u,// Перевод ККТ из автоматического режима в неавтоматический режим (осуществление расчетов кассиром)
        SingleToAuto                         = 0x00000400u,// Перевод ККТ из неавтоматического режима (осуществление расчетов кассиром) в автоматический режим
        BasicToFosa                          = 0x00000800u,// Перевод ККТ из режима, не позволяющего формировать БСО, в режим, позволяющий формировать БСО
        FosaToBasic                          = 0x00001000u,// Перевод ККТ из режима, позволяющего формировать БСО, в режим, не позволяющий формировать БСО
        InternetToBasic                      = 0x00002000u,//3 Перевод ККТ из режима расчетов в сети Интернет (позволяющего не печатать кассовый чек и БСО) в режим, позволяющий печатать кассовый чек и БСО
        BasicToInternet                      = 0x00004000u,// Перевод ККТ из режима, позволяющего печатать кассовый чек и БСО, в режим расчетов в сети Интернет (позволяющего не печатать кассовый чек и БСО)
        GamesToBasic                         = 0x00020000u,// Перевод ККТ из режима, позволяющего применять ККТ при приеме ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению азартных игр, в режим, не позволяющий применять ККТ при приеме ставок и выплате денежных среств в виде выигрыша при осуществлении деятельности по проведению азартных игр
        BasicToGames                         = 0x00040000u,// Перевод ККТ из режима, не позволяющего применять ККТ при приеме ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению азартных игр, в режим, позволяющий применять ККТ при приеме ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению азартных игр
        LotteryToBasic                       = 0x00080000u,// Перевод ККТ из режима, позволяющего применять ККТ при приеме денежных средств при реализации лотерейных билетов, электронных лотерейных билетов, приеме лотерейных ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению лотерей, в режим, не позволяющий применять ККТ при приеме денежных средств при реализации лотерейных билетов, электронных лотерейных билетов, приеме лотерейных ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению лотерей
        BasicToLottery                       = 0x00100000u,// Перевод ККТ из режима, не позволяющего применять ККТ при приеме денежных средств при реализации лотерейных билетов, электронных лотерейных билетов, приеме лотерейных ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению лотерей, в режим, позволяющий применять ККТ при приеме денежных средств при реализации лотерейных билетов, электронных лотерейных билетов, приеме лотерейных ставок и выплате денежных средств в виде выигрыша при осуществлении деятельности по проведению лотерей
        FFDChange                            = 0x00200000u,// Изменение версии ФФД
        AnotherReasons                       = 0x80000000u// Иные причины

    };

    Q_DECLARE_FLAGS(Reasons, Reason)
    Q_DECLARE_OPERATORS_FOR_FLAGS(Reasons)

    Q_DECL_CONSTEXPR inline QFlags<Reason> operator~(Reason f) Q_DECL_NOTHROW
    { return QFlags<Reason>(~static_cast<quint32>(f)) ;}


    inline QString reasonsToString(const Reasons &reasons){
        QStringList sl;
        if(reasons.testFlag(Reason::FsChange)) sl << "1";
        for(int i = 1; i < 32; ++i)
        {
            quint32 f = 1;
            f = (f << i);
            if(reasons.testFlag(static_cast<Reason>(f))) sl << QString::number(i + 1);
        }
        return sl.join(", ");
    }

    enum class Tag: quint16
    {
        Invalid                                 = 0,
        DocName                                 = 1000, //наименование документа
        AutoMode                                = 1001, //признак автоматического режима
        OfflineMode                             = 1002, //признак автономного режима
        TransferOperatorAddress                 = 1005, //адрес оператора перевода
        ClientPhone                             = 1008, //телефон или электронный адрес покупателя
        Address                                 = 1009, //адрес расчетов
        TimeZone                                = 1011, //Часовая зона
        DateTime                                = 1012, //дата, время
        CashboxSerial                           = 1013, //заводской номер ККТ
        TransferOperatorInn                     = 1016, //ИНН оператора перевода
        OfdInn                                  = 1017, //ИНН ОФД
        OwnerInn                                = 1018, //ИНН пользователя
        ReceiptTotal                            = 1020, //сумма расчета, указанного в чеке (БСО)
        Cashier                                 = 1021, //кассир
        OfdResultCode                           = 1022, //код ответа ОФД
        ItemQuantity                            = 1023, //количество предмета расчета
        TransferOperatorName                    = 1026, //наименование оператора перевода
        ItemName                                = 1030, //наименование предмета расчета
        ReceiptCash                             = 1031, //сумма по чеку (БСО) наличными
        BoxId                                   = 1036, //номер автомата
        CashboxRegNum                           = 1037, //регистрационный номер ККТ
        Cycle                                   = 1038, //номер смены
        Fd                                      = 1040, //номер ФД
        Fs                                      = 1041, //номер ФН
        Receipt                                 = 1042, //номер чека за смену
        ItemCost                                = 1043, //стоимость предмета расчета с учетом скидок и наценок
        BankPayingAgentOperation                = 1044, //операция банковского платежного агента
        OfdName                                 = 1046, //наименование ОФД
        OwnerName                               = 1048, //наименование пользователя
        FsResourceOff30                         = 1050, //признак исчерпания ресурса ФН
        FsChangeNecessity                       = 1051, //признак необходимости срочной замены ФН
        FsOverMemory                            = 1052, //признак заполнения памяти ФН
        OfdTimedOut                             = 1053, //признак превышения времени ожидания ответа ОФД
        ReceiptType                             = 1054, //признак расчета
        TaxSystem                               = 1055, //применяемая система налогообложения
        EncryptionMode                          = 1056, //признак шифрования
        AgentFlag                               = 1057, //признак агента
        ReceiptItem                             = 1059, //предмет расчета
        FnsSite                                 = 1060, //адрес сайта ФНС
        TaxSystems                              = 1062, //системы налогообложения
        OfdMessageForFs                         = 1068, //сообщение оператора для ФН
        PayingAgentPhone                        = 1073, //телефон платежного агента
        PayingOperatorPhone                     = 1074, //телефон оператора по приему платежей
        TransferOperatorPhone                   = 1075, //телефон оператора перевода
        FPD                                     = 1077, //ФПД
        FPO                                     = 1078, //ФПО
        ItemPrice                               = 1079, //цена за единицу предмета расчета с учетом скидок и наценок
        ReceiptCard                             = 1081, //сумма по чеку (БСО) безналичными
        ReceiptCardByWay                        = 1082,
        OwnerAdditionalParam                    = 1084, //дополнительный реквизит пользователя
        OwnerAdditionalParamName                = 1085, //наименование дополнительного реквизита пользователя
        OwnerAdditionalParamValue               = 1086, //значение дополнительного реквизита пользователя
        NotSendedDocsCount                      = 1097, //количество непереданных ФД
        FirstNotSendedDt                        = 1098, //дата первого из непереданных ФД
        ReregReasonCode                         = 1101, //код причины перерегистрации
        ReceiptVat20Amount                      = 1102, //сумма НДС чека по ставке 20%
        ReceiptVat10Amount                      = 1103, //сумма НДС чека по ставке 10%
        ReceiptVat0Amount                       = 1104, //сумма расчета по чеку с НДС по ставке 0%
        ReceiptVatNoneAmount                    = 1105, //сумма расчета по чеку без НДС
        ReceiptVat20_120Amount                  = 1106, //сумма НДС чека по расч. ставке 20/120
        ReceiptVat10_110Amount                  = 1107, //сумма НДС чека по расч. ставке 10/110
        InternetOnlyMode                        = 1108, //признак ККТ для расчетов только в Интернет
        ServiceMode                             = 1109, //признак расчетов за услуги
        FOSAMode                                = 1110, //признак АС БСО
        DocsForCycle                            = 1111, //общее количество ФД за смену
        VatAmounts                              = 1115,
        FirstNotSendedFd                        = 1116, //номер первого непереданного документа
        OwnerEmail                              = 1117, //адрес электронной почты отправителя чека
        ReceiptsForCycle                        = 1118, //количество кассовых чеков (БСО) за смену
        ReceiptVatAmount                        = 1119, //Сумма НДС чека
        VatAmount                               = 1120,// Сумма НДС, входит в 1119
        CashboxForInternetFlag                  = 1125, //признак расчета в сети Интернет
        LotteryMode                             = 1126, //признак проведения лотереи
        DebitCounter                            = 1129, //счетчики операций "приход"
        DebitRefundCounter                      = 1130, //счетчики операций "возврат прихода"
        CreditCounter                           = 1131, //счетчики операций "расход"
        CreditRefundCounter                     = 1132, //счетчики операций "возврат расхода"
        CorrectuinsCounters                     = 1133, //счетчики операций по чекам коррекции (БСО коррекции)
        ReceiptsAndCorrections                  = 1134, //количество чеков (БСО) и чеков коррекции (БСО коррекции) со всеми признаками расчетов
        ReceiptsCountByType                     = 1135, //количество чеков (БСО) по признаку расчетов
        ReceiptsCashTotal                       = 1136, //итоговая сумма в чеках (БСО) наличными денежными средствами
        ReceiptsCardTotal                       = 1138, //итоговая сумма в чеках (БСО) безналичными
        Vat20Amount                             = 1139, //сумма НДС по ставке 20%
        Vat10Amount                             = 1140, //сумма НДС по ставке 10%
        Vat20_120Amount                         = 1141, //сумма НДС по расч. ставке 20/120
        Vat10_110Amount                         = 1142, //сумма НДС по расч. ставке 10/110
        Vat0Amount                              = 1143, //сумма расчетов с НДС по ставке 0%
        NotSendedReceiptAndCorrectionsCount     = 1144, //количество чеков коррекции (БСО коррекции) или непереданных чеков (БСО) и чеков коррекции (БСО коррекции)
        DebitAmountCounter                      = 1145, //счетчики по признаку "приход"
        CreditAmountCounter                     = 1146, //счетчики по признаку "расход"
        FsTotalsCounters                        = 1157, //счетчики итогов ФН
        FsNotSendedTotalsCounters               = 1158, //счетчики итогов непереданных ФД
        ProductCode                             = 1163, //Код товара
        ProviderPhone                           = 1171, //телефон поставщика
        CorrectionType                          = 1173, //тип коррекции
        CorrectionReason                        = 1174, //основание для коррекции
        CorrecionDate                           = 1178, //дата совершения корректируемого расчета
        FnsResolutionNumber                     = 1179, //номер предписания налогового органа
        VatNoneAmount                           = 1183, //сумма расчетов без НДС
        Place                                   = 1187, //место расчетов
        CashboxVersion                          = 1188, //версия по ккт
        CashboxFFD                              = 1189, //версия ФФД ККТ
        FsFFD                                   = 1190, //версия ФФД ФН
        ItemAdditionalParam                     = 1191, //дополнительный реквизит предмета расчета
        ReceiptAdditionalParam                  = 1192, //дополнительный реквизит чека (БСО)
        GamesMode                               = 1193, //признак проведения азартных игр
        CycleTotalCounters                      = 1194, //счетчики итогов смены
        ReceiptQr                               = 1196, //QR-код
//        ItemUnit                                = 1197, //единица измерения предмета расчета
        VatForItemUnit                          = 1198, //размер НДС за единицу предмета расчета
        VatRate                                 = 1199, //ставка НДС
        VatForItem                              = 1200, //сумма НДС за предмет расчета
        TotalReceiptsAMount                     = 1201, //общая итоговая сумма в чеках (БСО)
        CashierInn                              = 1203, //ИНН кассира
        CashboxDataChangeReason                 = 1205, //коды причин изменения сведений о ККТ
        OfdMessage                              = 1206, //сообщение оператора
        ExcisableGoodsMode                      = 1207, //признак торговли подакцизными товарами
        ReceiptLoadSite                         = 1208, //сайт для получения чека
        FFD                                     = 1209, //номер версии ФФД
        ItemType                                = 1212, //признак предмета расчета
        KeysResource                            = 1213, //ресурс ключей ФП
        ItemPaymentType                         = 1214, //признак способа расчета
        ReceiptPrepaymentAmount                 = 1215, //сумма по чеку (БСО) предоплатой (зачетом аванса и (или) предыдущих платежей)
        ReceiptPostpaymentAmount                = 1216, //сумма по чеку (БСО) постоплатой (в кредит)
        ReceiptBarterAmount                     = 1217, //сумма по чеку (БСО) встречным предоставлением
        ReceiptsTotalPrepaymentAmount           = 1218, //итоговая сумма в чеках (БСО) предоплатами (авансами)
        ReceiptsTotalPostpaymentAmount          = 1219, //итоговая сумма в чеках (БСО) постоплатами (кредитами)
        ReceiptsTOtalBarterAmount               = 1220, //итоговая сумма в чеках (БСО) встречными предоставлениями
        ExternalPrinterMode                     = 1221, //признак установки принтера в автомате
        ItemAgentFlag                           = 1222, //признак агента по предмету расчета
        ItemAgentData                           = 1223, //данные агента
        ProviderData                            = 1224, //данные поставщика
        ProviderName                            = 1225, //наименование поставщика
        ProviderInn                             = 1226, //ИНН поставщика
        ClientName                              = 1227,
        ClientInn                               = 1228, //ИНН покупателя (клиента)
        ExciseTax                               = 1229, //акциз
        ProductCountryCode                      = 1230, //код страны происхождения товара
        ProductDeclarationNumber                = 1231, //номер декларации на товар
        DebitRefundAmountCounters               = 1232, //счетчики по признаку "возврат прихода"
        CreditRefundAmountCounters              = 1233, //счетчики по признаку "возврат расхода"
        AllCardPaysInfo                         = 1234,//сведения обо всех оплатах по чеку безналичными
        CardPayInfo                             = 1235,//сведения об оплате безналичными
        CardPayWay                              = 1236,//признак способа оплаты безналичными
        CardPayIds                              = 1237,//идентификаторы безналичной оплаты
        AdditionalCardPayInfo                   = 1238,//дополнительные сведения обезналичной оплате

        ClientBirthDay                          = 1243, //дата рождения покупателя (клиента)
        Citizenship                             = 1244, //гражданство
        PersonDocTypeCode                       = 1245, //код вида документа, удостоверяющего личность
        PersonDocData                           = 1246, //данные документа, удостоверяющего личность
        ClientAddress                           = 1254, //адрес покупателя (клиента)
        ClientInfo                              = 1256, //сведения о покупателе (клиенте)
        IndustryItemParam                       = 1260, //отраслевой реквизит предмета расчета
        IndustryReceiptParam                    = 1261, //отраслевой реквизит чека
        FEDID                                   = 1262, //идентификатор ФОИВ
        ReasonDocDt                             = 1263, //дата документа основания
        ReasonDocNumber                         = 1264, //номер документа основания
        IndustryParamValue                      = 1265, //значение отраслевого реквизита
        ReceiptOperationalParam                 = 1270, //операционный реквизит чека
        OperationId                             = 1271, //идентификатор операции
        OperationData                           = 1272, //данные операции
        OperationDt                             = 1273, //дата, время операции
        RegAdditionalParam                      = 1274, //дополнительный реквизит ОР
        RegAdditionalData                       = 1275, //дополнительные данные ОР
        CycleOpenAdditionalParam                = 1276, //дополнительный реквизит ООС
        CycleOpenAdditionalData                 = 1277, //дополнительные данные ООС
        CycleCloseAdditionalParam               = 1278, //дополнительный реквизит ОЗС
        CycleCloseAdditionalData                = 1279, //дополнительные данные ОЗС
        CalReportAdditionalParam                = 1280, //дополнительный реквизит ОТР
        CalcReportAdditionalData                = 1281, //дополнительные данные ОТР
        FsCloseAdditionalParam                  = 1282, //дополнительный реквизит ОЗФН
        FsCloseAdditionalData                   = 1283, //дополнительные данные ОЗФН
        CashboxUsageFlags                       = 1290, //признаки условий применения ККТ
        LabledProductFractionQuantity           = 1291, //дробное количество маркированного товара
        FractionPart                            = 1292, //дробная часть
        Nominator                               = 1293, //числитель
        Denominator                             = 1294, //знаменатель
        LabelUnknown                            = 1300, //КТ Н
        LabelEan8                               = 1301, //КТ EAN-8
        LabelEan13                              = 1302, //КТ EAN-13
        LabelItf14                              = 1303, //КТ ITF-14
        LabelGs10                               = 1304, //КТ GS1.0
        LabelGs1M                               = 1305, //КТ GS1.M
        LabelShortCode                          = 1306, //КТ КМК
        LabelFur                                = 1307, //КТ МИ
        LabelEgais2                             = 1308, //КТ ЕГАИС-2.0
        LabelEGais3                             = 1309, //КТ ЕГАИС-3.0
        LabelF1                                 = 1320, //КТ Ф.1
        LabelF2                                 = 1321, //КТ Ф.2
        LabelF3                                 = 1322, //КТ Ф.3
        LabelF4                                 = 1323, //КТ Ф.4
        LabelF5                                 = 1324, //КТ Ф.5
        LabelF6                                 = 1325, //КТ Ф.6
        Label                                   = 2000, //код маркировки
        RequestNumber                           = 2001, //номер запроса
        NotificationNumber                      = 2002, //номер уведомления
        PromiseProductStatus                    = 2003, //планируемый статус товара
        LabelCheckResult                        = 2004, //результат проверки КМ
        RequestProcessResult                    = 2005, //результаты обработки запроса
        NotificationProcessResult               = 2006, //результаты обработки уведомления
        LabledProductInfo                       = 2007, //данные о маркированном товаре
        FiscalDocNumberForNotification          = 2040, //номер ФД кассового чека
        LabelCodeType                           = 2100, //тип кода маркировки
        ProductId                               = 2101, //идентификатор товара
        LabelProcessMode                        = 2102, //режим обработки кода маркировки
        NotSendedNotificationsCount             = 2104, //количество непереданных уведомлений
        RequestProcessCodes                     = 2105, //коды обработки запроса
        ProductInfoCheckResult                  = 2106, //результат проверки сведений о товаре
        LabelProductsCheckresults               = 2107, //результаты проверки маркированных товаров
        ItemQuantityUnitType                    = 2108, //мера количества предмета расчета
        LabelSystemResponse                     = 2109, //ответ ОИСМ о статусе товара
        ProductStatus                           = 2110, //присвоенный статус товара
        NotificationsProcessCodes               = 2111, //коды обработки уведомления
        IncorrectLabelsFlag                     = 2112, //признак некорректных кодов маркировки
        IncorrectNotificationsFlag              = 2113, //признак некорректных запросов и уведомлений
        RequestDt                               = 2114, //дата и время запроса
        LabelCrc                                = 2115, //контрольный код КМ
        OperationType                           = 2116 //вид операции
    };

    enum class DocType
    {
        Registration                            = 1,
        ReRegistration                          = 11,
        CycleOpen                               = 2,
        CalcReport                              = 21,
        Receipt                                 = 3,
        Correction                              = 31,
        Fosa                                    = 4,
        FosaCorrection                          = 41,
        CycleClose                              = 5,
        FsClose                                 = 6,
        OfdResponse                             = 7,
        LabelRequest                            = 81,
        LabelNotification                       = 82,
        LabelResponse                           = 83,
        NotificationTicket                      = 84,


        CycleReport                             = 201,
        FsReport                                = 202,
        // Introduction                            = 203,
        // Payout                                  = 204,
        Invalid                                 = 255
    };


    inline bool docTypeIsValid(DocType t) {
        static const QSet<quint8> VALID_TYPES = {
            1, 11, 2, 21, 3,  31, 4,  41, 5,  6,  7, 81, 82, 83, 84
        };
        return VALID_TYPES.contains(static_cast<quint8>(t));
    }

    enum class PaymentType: quint8
    {
        Debit           = 0x01u,
        DebitReturn     = 0x02u,
        Credit          = 0x03u,
        CreditReturn    = 0x04u,

        Invalid         = 0x80u
    };

    inline bool paymentTypeIsValid(PaymentType t) {
        return t >= PaymentType::Debit && t <= PaymentType::CreditReturn;
    }
    inline QString paymentTypeToString(PaymentType t) {
        switch (t) {
        case PaymentType::Debit       : return QStringLiteral("ПРИХОД");
        case PaymentType::DebitReturn : return QStringLiteral("ВОЗВРАТ ПРИХОДА");
        case PaymentType::Credit      : return QStringLiteral("РАСХОД");
        case PaymentType::CreditReturn: return QStringLiteral("ВОЗВРАТ РАСХОДА");
        default: return QString();
        }
    }

    enum class OfdWarning: quint8
    {
        LogicalError                    = 0x02u,
        CheckOfdAccount                 = 0x04u,
        UpdateFFD                       = 0x08u,
        CashboxCheckPlan                = 0x10u,
        Call2Ofd                        = 0x40u,
        OfdIsClosed                     = 0x80u
    };
    Q_DECLARE_FLAGS(OfdWarnings, OfdWarning)
    Q_DECLARE_OPERATORS_FOR_FLAGS(OfdWarnings)
    Q_DECL_CONSTEXPR inline QFlags<OfdWarning> operator~(OfdWarning f) Q_DECL_NOTHROW
    { return QFlags<OfdWarning>(~static_cast<qint32>(f)) ;}
    inline QString ofdWarningsToString(const OfdWarnings &tx) {
        QStringList sl;
        if(tx.testFlag(OfdWarning::LogicalError)) sl << QStringLiteral("ОШИБКА ФЛК");
        if(tx.testFlag(OfdWarning::CheckOfdAccount)) sl << QStringLiteral("ПРОВЕРЬ КАБИНЕТ ККТ");
        if(tx.testFlag(OfdWarning::UpdateFFD)) sl << QStringLiteral("ТРЕБ. ОБНОВ. ФФД ККТ");
        if(tx.testFlag(OfdWarning::CashboxCheckPlan)) sl << QStringLiteral("ККТ ВКЛ. В ПЛАН НО");
        if(tx.testFlag(OfdWarning::Call2Ofd)) sl << QStringLiteral("ТРЕБ. НАСТР. ККТ");
        if(tx.testFlag(OfdWarning::OfdIsClosed)) sl << QStringLiteral("ОФД АННУЛИРОВАН");
        return sl.join(", ");
    }

    enum class IncorrectLabelsFlag: quint8
    {
        IncorrectLabelResponse              = 0x02,
        IncorrectLabelTicket                = 0x04
    };
    Q_DECLARE_FLAGS(IncorrectLabelsFlags, IncorrectLabelsFlag)
    Q_DECLARE_OPERATORS_FOR_FLAGS(IncorrectLabelsFlags)
    Q_DECL_CONSTEXPR inline QFlags<IncorrectLabelsFlag> operator~(IncorrectLabelsFlag f) Q_DECL_NOTHROW
    { return QFlags<IncorrectLabelsFlag>(~static_cast<qint8>(f)) ;}
    inline QStringList incorrectLabelsFlagToString(const IncorrectLabelsFlags &tx) {
        QStringList sl;
        if(tx.testFlag(IncorrectLabelsFlag::IncorrectLabelResponse)) sl << QStringLiteral("ПОЛУЧЕН ОТВЕТ О НЕКОРРЕКТНОМ КМ");
        if(tx.testFlag(IncorrectLabelsFlag::IncorrectLabelTicket)) sl << QStringLiteral("ПОЛУЧЕНА КВИТАНЦИЯ О НЕКОРРЕКТНОМ КМ");
        return sl;
    }
    enum class IncorrectNotificationsFlag:quint8
    {
        IcorrectRequest                     = 0x01,
        IncorrectNotification               = 0x02
    };
    Q_DECLARE_FLAGS(IncorrectNotificationsFlags,IncorrectNotificationsFlag)
    Q_DECLARE_OPERATORS_FOR_FLAGS(IncorrectNotificationsFlags)
    Q_DECL_CONSTEXPR inline QFlags<IncorrectNotificationsFlag> operator~(IncorrectNotificationsFlag f) Q_DECL_NOTHROW
    { return QFlags<IncorrectNotificationsFlag>(~static_cast<qint8>(f)) ;}
    inline QStringList incorrectNotificationsFlagToString(const IncorrectNotificationsFlags &tx) {
        QStringList sl;
        if(tx.testFlag(IncorrectNotificationsFlag::IcorrectRequest)) sl << QStringLiteral("НЕКОРРЕКТНЫЙ ЗАПРОС");
        if(tx.testFlag(IncorrectNotificationsFlag::IncorrectNotification)) sl << QStringLiteral("НЕКОРРЕКТНОЕ УВЕДОМЛЕНИЕ");
        return sl;
    }

    enum class OfdAnswerCode: quint8
    {
        Ok                      = 0,
        InvalidDocument         = 11,
        LogicalError            = 14
    };

    enum class FsTicketAnswerCode: quint8
    {
        Ok                                  = 0,
        InvalidFiscalCode                   = 1,
        IncalidTicketFormat                 = 2,
        InvalidFd                           = 3,
        InvalidFsNumber                     = 4,
        InvalidCrc                          = 5,
        InvalidOfdInn                       = 6
    };

    enum class ItemPayAgentFlag: quint8
    {
        NotAgent                            = 0x00,
        BankAgent                           = 0x01,
        BankSubAgent                        = 0x02,
        PayAgent                            = 0x04,
        PaySubAgent                         = 0x08,
        Attorney                            = 0x10,
        Broker                              = 0x20,
        Agent                               = 0x40
    };
    inline bool itemPayAgentFlagIsValid(ItemPayAgentFlag flag) {
        QSet<quint8>SUPPORTED = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40};
        return SUPPORTED.contains(static_cast<quint8>(flag));
    }
    Q_DECLARE_FLAGS(ItemPayAgentFlags, ItemPayAgentFlag)
    Q_DECLARE_OPERATORS_FOR_FLAGS(ItemPayAgentFlags)
    Q_DECL_CONSTEXPR inline QFlags<ItemPayAgentFlag> operator~(ItemPayAgentFlag f) Q_DECL_NOTHROW
    { return QFlags<ItemPayAgentFlag>(~static_cast<qint8>(f)) ;}

    inline QString itemPayAgentFlagsToString(const ItemPayAgentFlags &f) {
        QStringList sl;
        if(f.testFlag(ItemPayAgentFlag::BankAgent)) sl << QStringLiteral("БАНК. ПЛ. АГЕНТ");
        if(f.testFlag(ItemPayAgentFlag::BankSubAgent)) sl << QStringLiteral("БАНК. ПЛ.СУБАГЕНТ");
        if(f.testFlag(ItemPayAgentFlag::PayAgent)) sl << QStringLiteral("ПЛ. АГЕНТ");
        if(f.testFlag(ItemPayAgentFlag::PaySubAgent)) sl << QStringLiteral("ПЛ. СУБАГЕНТ");
        if(f.testFlag(ItemPayAgentFlag::Attorney)) sl << QStringLiteral("ПОВЕРЕННЫЙ");
        if(f.testFlag(ItemPayAgentFlag::Broker)) sl << QStringLiteral("КОМИССИОНЕР");
        if(f.testFlag(ItemPayAgentFlag::Agent)) sl << QStringLiteral("АГЕНТ");
        return sl.join(", ");
    }
    inline QString itemPayAgentFlagToString(const ItemPayAgentFlag &f) {
        switch (f)
        {
        case ItemPayAgentFlag::BankAgent   : return QStringLiteral("БАНК. ПЛ. АГЕНТ");
        case ItemPayAgentFlag::BankSubAgent: return QStringLiteral("БАНК. ПЛ.СУБАГЕНТ");
        case ItemPayAgentFlag::PayAgent    : return QStringLiteral("ПЛ. АГЕНТ");
        case ItemPayAgentFlag::PaySubAgent : return QStringLiteral("ПЛ. СУБАГЕНТ");
        case ItemPayAgentFlag::Attorney    : return QStringLiteral("ПОВЕРЕННЫЙ");
        case ItemPayAgentFlag::Broker      : return QStringLiteral("КОМИССИОНЕР");
        case ItemPayAgentFlag::Agent       : return QStringLiteral("АГЕНТ");
        default: return QString();
        }
    }

    enum class VatRate: quint8
    {
        Invalid                 = 0x00,
        Vat20                   = 0x01,
        Vat10                   = 0x02,
        Vat20_120               = 0x03,
        Vat10_110               = 0x04,
        Vat0                    = 0x05,
        VatNone                 = 0x06,
        Vat5                    = 0x07,
        Vat7                    = 0x08,
        Vat5_105                = 0x09,
        Vat7_107                = 0x0A
    };

    inline bool vatIsValid(VatRate rate) {
        return rate > VatRate::Invalid && rate <= VatRate::Vat7_107;
    }

    inline QString vatPrintable(VatRate v) {
        switch (v) {
        case VatRate::Vat20    : return "НДС 20%"   ;
        case VatRate::Vat10    : return "НДС 10%"   ;
        case VatRate::Vat20_120: return "НДС 20/120";
        case VatRate::Vat10_110: return "НДС 10/110";
        case VatRate::Vat0     : return "НДС 0%"    ;
        case VatRate::Vat5     : return "НДС 5%"    ;
        case VatRate::Vat7     : return "НДС 7%"    ;
        case VatRate::Vat5_105 : return "НДС 5/105" ;
        case VatRate::Vat7_107 : return "НДС 7/107";

        default: return QString();
        }
    }
    inline QString vatLabel(VatRate v) {
        switch (v) {
        case VatRate::Vat20    : return "НДС 20%"   ;
        case VatRate::Vat10    : return "НДС 10%"   ;
        case VatRate::Vat20_120: return "НДС 20/120";
        case VatRate::Vat10_110: return "НДС 10/110";
        case VatRate::Vat0     : return "НДС 0%"    ;
        case VatRate::VatNone  : return "БЕЗ НДС"   ;
        case VatRate::Vat5     : return "НДС 5%"    ;
        case VatRate::Vat7     : return "НДС 7%"    ;
        case VatRate::Vat5_105 : return "НДС 5/105" ;
        case VatRate::Vat7_107 : return "НДС 7/107";
        default: return QString();
        }
    }


    enum class ReceiptOperationType: quint8
    {
        Invalid                             = 0,
        Product                             = 1,
        ExcisableProduct                    = 2,
        Job                                 = 3,
        Service                             = 4,
        GameRate                            = 5,
        GamePrize                           = 6,
        Lottery                             = 7,
        LotteryPrize                        = 8,
        RiaRights                           = 9,
        Prepayment                          = 10,
        AgentCommission                     = 11,
        Fine                                = 12,
        Another                             = 13,
        PropertyRight                       = 14,
        NonReleaseIncome                    = 15,
        AnotherPayments                     = 16,
        TradeFee                            = 17,
        SpaFee                              = 18,
        Deposit                             = 19,
        Cconsumption                        = 20,
        PensionContribution                 = 21,
        PensionContribution2                = 22,
        MedicalContribution                 = 23,
        MedicalContribution2                = 24,
        SocialContribution                  = 25,
        CasinoPayment                       = 26,
        PaymentOfMoney                      = 27,
        ExcisableLabledWitoutCodeProduct    = 30,
        ExcisableLabledProduct              = 31,
        LabledWitoutCodeProduct             = 32,
        LabledProduct                       = 33
    };
    inline bool opTypeIsValid(ReceiptOperationType t) {
        quint8 buf = static_cast<quint8>(t);
        return (buf > 0 && buf <= 27) ||(buf >= 30 && buf <= 33);
    }

    inline bool opTypeForLable(ReceiptOperationType t) {
        return t == ReceiptOperationType::ExcisableLabledWitoutCodeProduct ||
                t == ReceiptOperationType::ExcisableLabledProduct ||
                t == ReceiptOperationType::LabledWitoutCodeProduct ||
                t == ReceiptOperationType::LabledProduct;
    }
    inline bool opTypeForExcisable(ReceiptOperationType t) {
        return t == ReceiptOperationType::ExcisableLabledWitoutCodeProduct ||
                t == ReceiptOperationType::ExcisableLabledProduct ||
                t == ReceiptOperationType::ExcisableProduct;
    }

    inline QString opTypeCaption (ReceiptOperationType t){
        switch (t) {
        case ReceiptOperationType::Product                          :
            return "ТОВАР";
        case ReceiptOperationType::ExcisableProduct                 :
            return "ПОДАКЦИЗНЫЙ ТОВАР";
        case ReceiptOperationType::Job                              :
            return "РАБОТА";
        case ReceiptOperationType::Service                          :
            return "УСЛУГА";
        case ReceiptOperationType::GameRate                         :
            return "СТАВКА ИГРЫ";
        case ReceiptOperationType::GamePrize                        :
            return "ВЫИГРЫШ АИ";
        case ReceiptOperationType::Lottery                          :
            return "СТАВКА ЛОТЕРЕИ";
        case ReceiptOperationType::LotteryPrize                     :
            return "ВЫИГРЫШ ЛОТЕРЕИ";
        case ReceiptOperationType::RiaRights                        :
            return "ПРЕДОСТАВЛЕНИЕ РИД";
        case ReceiptOperationType::Prepayment                       :
            return "ПЛАТЕЖ";
        case ReceiptOperationType::AgentCommission                  :
            return "АГЕНСТКОЕ ВОЗНАГРАЖДЕНИЕ";
        case ReceiptOperationType::Fine                             :
            return "ВЫПЛАТА";
        case ReceiptOperationType::Another                          :
            return "ИНОЙ ПРЕДМЕТ РАСЧЕТА";
        case ReceiptOperationType::PropertyRight                    :
            return "ИМУЩЕСТВЕННОЕ ПРАВО";
        case ReceiptOperationType::NonReleaseIncome                 :
            return "ВНЕРЕЛИЗАЦИОННЫЙ ДОХОД";
        case ReceiptOperationType::AnotherPayments                  :
            return "ИНЫЕ ПЛАТЕЖИ И ВЗНОСЫ";
        case ReceiptOperationType::TradeFee                         :
            return "ТОРГОВЫЙ СБОР";
        case ReceiptOperationType::SpaFee                           :
            return "ТУРИСТИЧЕСКИЙ НАЛОГ";
        case ReceiptOperationType::Deposit                          :
            return "ЗАЛОГ";
        case ReceiptOperationType::Cconsumption                     :
            return "РАСХОД";
        case ReceiptOperationType::PensionContribution              :
            return "ВЗНОСЫ НА ОПС ИП";
        case ReceiptOperationType::PensionContribution2             :
            return "ВЗНОСЫ НА ОПС";
        case ReceiptOperationType::MedicalContribution              :
            return "ВЗНОСЫ НА ОМС ИП";
        case ReceiptOperationType::MedicalContribution2             :
            return "ВЗНОСЫ НА ОМС";
        case ReceiptOperationType::SocialContribution               :
            return "ВЗНОСЫ НА ОСС";
        case ReceiptOperationType::CasinoPayment                    :
            return "ПЛАТЕЖ КАЗИНО";
        case ReceiptOperationType::PaymentOfMoney                   :
            return "ВЫДАЧА ДС";
        case ReceiptOperationType::ExcisableLabledWitoutCodeProduct :
            return "АТНМ";
        case ReceiptOperationType::ExcisableLabledProduct           :
            return "АТМ";
        case ReceiptOperationType::LabledWitoutCodeProduct          :
            return "ТНМ";
        case ReceiptOperationType::LabledProduct                    :
            return "ТМ";
        default: return QString();
        }
    }
    inline QString opTypePrint (ReceiptOperationType t){
        switch (t) {
        case ReceiptOperationType::Product                          :
            return "Т";
        case ReceiptOperationType::ExcisableProduct                 :
            return "АТ";
        case ReceiptOperationType::Job                              :
            return "Р";
        case ReceiptOperationType::Service                          :
            return "У";
        case ReceiptOperationType::GameRate                         :
            return "СА";
        case ReceiptOperationType::GamePrize                        :
            return "ВА";
        case ReceiptOperationType::Lottery                          :
            return "СЛ";
        case ReceiptOperationType::LotteryPrize                     :
            return "ВЛ";
        case ReceiptOperationType::RiaRights                        :
            return "РИД";
        case ReceiptOperationType::Prepayment                       :
            return "П";
        case ReceiptOperationType::AgentCommission                  :
            return "АВ";
        case ReceiptOperationType::Fine                             :
            return "В";
        case ReceiptOperationType::Another                          :
            return "ИПР";
        case ReceiptOperationType::PropertyRight                    :
            return "ИМУЩЕСТВЕННОЕ ПРАВО";
        case ReceiptOperationType::NonReleaseIncome                 :
            return "ВНЕРЕЛИЗАЦИОННЫЙ ДОХОД";
        case ReceiptOperationType::AnotherPayments                  :
            return "ИНЫЕ ПЛАТЕЖИ И ВЗНОСЫ";
        case ReceiptOperationType::TradeFee                         :
            return "ТОРГОВЫЙ СБОР";
        case ReceiptOperationType::SpaFee                           :
            return "ТУРИСТИЧЕСКИЙ НАЛОГ";
        case ReceiptOperationType::Deposit                          :
            return "ЗАЛОГ";
        case ReceiptOperationType::Cconsumption                     :
            return "РАСХОД";
        case ReceiptOperationType::PensionContribution              :
            return "ВЗНОСЫ НА ОПС ИП";
        case ReceiptOperationType::PensionContribution2             :
            return "ВЗНОСЫ НА ОПС";
        case ReceiptOperationType::MedicalContribution              :
            return "ВЗНОСЫ НА ОМС ИП";
        case ReceiptOperationType::MedicalContribution2             :
            return "ВЗНОСЫ НА ОМС";
        case ReceiptOperationType::SocialContribution               :
            return "ВЗНОСЫ НА ОСС";
        case ReceiptOperationType::CasinoPayment                    :
            return "ПК";
        case ReceiptOperationType::PaymentOfMoney                   :
            return "ВЫДАЧА ДС";
        case ReceiptOperationType::ExcisableLabledWitoutCodeProduct :
            return "АТНМ";
        case ReceiptOperationType::ExcisableLabledProduct           :
            return "АТМ";
        case ReceiptOperationType::LabledWitoutCodeProduct          :
            return "ТНМ";
        case ReceiptOperationType::LabledProduct                    :
            return "ТМ";
        default: return QString();
        }
    }

    enum class ItemPaymentType :quint8
    {
        FullPrepayment                  = 1,
        Prepayment                      = 2,
        Advance                         = 3,
        FullPayment                     = 4,
        PartialPayment                  = 5,
        Credit                          = 6,
        CrediPayment                    = 7
    };

    inline bool itemPaymenTypeIsValid(ItemPaymentType t) {
        return t >= ItemPaymentType::FullPrepayment &&
                t <= ItemPaymentType::CrediPayment;
    }
    inline QString itemPaymentTypeToString(ItemPaymentType t) {
        switch(t)
        {
        case ItemPaymentType::FullPrepayment: return "ПРЕДОПЛАТА 100%";
        case ItemPaymentType::Prepayment    : return "ПРЕДОПЛАТА";
        case ItemPaymentType::Advance       : return "АВАНС";
        case ItemPaymentType::FullPayment   : return "ПОЛНЫЙ РАСЧЕТ";
        case ItemPaymentType::PartialPayment: return "ЧАСТИЧНЫЙ РАСЧЕТ И КРЕДИТ";
        case ItemPaymentType::Credit        : return "ПЕРЕДАЧА В КРЕДИТ";
        case ItemPaymentType::CrediPayment  : return "ОПЛАТА КРЕДИТА";
        default: return  QString();
        }
    }


    enum class CheckLabelFlag: quint8
    {
        LabelChecked                    = 0x01,
        LabelValid                      = 0x02,
        StatusChecked                   = 0x04,
        StatusValid                     = 0x08,
        OfflineMode                     = 0x10
    };
    Q_DECLARE_FLAGS(CheckLabelFlags, CheckLabelFlag)
    Q_DECLARE_OPERATORS_FOR_FLAGS(CheckLabelFlags)
    Q_DECL_CONSTEXPR inline QFlags<CheckLabelFlag> operator~(CheckLabelFlag f) Q_DECL_NOTHROW
    { return QFlags<CheckLabelFlag>(~static_cast<qint8>(f)) ;}

    inline QString checkLabelFlagToString(CheckLabelFlags f) {
        switch (static_cast<quint32>(f)) {
        case 0b00000000:
        case 0b00000011:
        case 0b00010000:
        case 0b00010011:
            return "[М]";
        case 0b00000001:
        case 0b00010001:
        case 0b00000101:
        case 0b00000111:
            return "[М-]";
        case 0b00001111:
            return "[М+]";
        }
        return QString();
    }

    enum class ItemUnit: quint8
    {
        Ut                      = 0,
        g                       = 10,
        Kg                      = 11,
        t                       = 12,
        sm                      = 20,
        dm                      = 21,
        m                       = 22,
        sm2                     = 30,
        dm2                     = 31,
        m2                      = 32,
        ml                      = 40,
        l                       = 41,
        m3                      = 42,
        KWth                    = 50,
        GCal                    = 51,
        day                     = 70,
        hour                    = 71,
        minute                  = 72,
        second                  = 73,
        Kb                      = 80,
        Mb                      = 81,
        Gb                      = 82,
        Tb                      = 83,
        Another                 = 255
    };
    inline bool unitIsValid(ItemUnit u) {
        static const QSet<quint8> SUPPORTED = {
            0, 10, 11, 12, 20, 21, 22, 30, 31, 32, 40, 41, 42, 50,
            51, 70, 71, 72, 73, 80, 81, 82, 83, 255
        };
        return SUPPORTED.contains(static_cast<quint8>(u));
    }
    inline QString unitText(ItemUnit u) {
        switch (u) {
        case ItemUnit::Ut    : return "шт.";
        case ItemUnit::g     : return "г";
        case ItemUnit::Kg    : return "кг";
        case ItemUnit::t     : return "т";
        case ItemUnit::sm    : return "см";
        case ItemUnit::dm    : return "дм";
        case ItemUnit::m     : return "м";
        case ItemUnit::sm2   : return "кв.см";
        case ItemUnit::dm2   : return "кв.дм";
        case ItemUnit::m2    : return "кв.м";
        case ItemUnit::ml    : return "мл";
        case ItemUnit::l     : return "л";
        case ItemUnit::m3    : return "куб.м";
        case ItemUnit::KWth  : return "кВт*ч";
        case ItemUnit::GCal  : return "ГКал";
        case ItemUnit::day   : return "сутки";
        case ItemUnit::hour  : return "час";
        case ItemUnit::minute: return "мин";
        case ItemUnit::second: return "с";
        case ItemUnit::Kb    : return "Кбайт";
        case ItemUnit::Mb    : return "Мбайт";
        case ItemUnit::Gb    : return "Гбайт";
        case ItemUnit::Tb    : return "Тбайт";
        default: return QString();
        }
    }

    enum class LabelCodeType: quint8
    {
        Unknown                     = 0,
        Short                       = 1,
        Code88Checked               = 2,
        Code44NotChecked            = 3,
        Code44Checked               = 4,
        Code4NotChecked             = 5
    };

    inline bool labelCodeTypeIsValid(LabelCodeType t) {
        return t >= LabelCodeType::Unknown && t <= LabelCodeType::Code4NotChecked;
    }

    enum class LabelCodeCheckFsResult :quint8
    {
        NotChecked                          = 0,
        CheckedWrong                        = 1,
        CheckedSuccess                      = 3
    };

    enum class LabelCodeCheckFsStatus: quint8
    {
        Checked                             = 0,
        NotNeedCheck                        = 1,
        FsHasNoValidKey                     = 2,
        No91And92                           = 3,
        AnotherReason                       = 4
    };

    enum class LabelProductStatus:  quint8
    {
        Invalid                     = 0,
        Piece                       = 1,
        Measured                    = 2,
        PieceRefund                 = 3,
        MeasuredRefund              = 4,
        PieceReleasing              = 5,
        MeasuredReleased            = 6,
        NotChanged                  = 255
    };
    inline bool labelProductStatusIsValid(LabelProductStatus st) {
        return st == LabelProductStatus::NotChanged || (
                st >= LabelProductStatus::Piece && st <= LabelProductStatus::MeasuredReleased);
    }
}
#endif // FDFCONSTANTS_H
