#include "inn.h"

Inn::Inn() noexcept
    : value_()
{

}

Inn::Inn(const QString &val) noexcept
    : value_(val.trimmed())
{
    if(value_.startsWith("0000000000")) value_.clear();
}

Inn::Inn(const Inn &other)noexcept
    : value_(other.value_)
{

}

Inn::Inn(Inn &&other)noexcept
    : value_()
{
    value_.swap(other.value_);
}

Inn::~Inn()
{

}

bool Inn::isNull() const
{
    return value_.trimmed().isEmpty();
}

Inn::Type Inn::type() const
{
    if(isNull()) return Type::Null;
    if(!isValid()) return Type::Invalid;
    if(value_.size() == 10) return Type::Organization;
    if(value_.size() == 12) return Type::Person;
    return Type::Invalid;
}

bool Inn::isValid(bool canBeNull) const
{
    if(canBeNull && (isNull() || value_ == "000000000000")) return true;
    if(value_.size() != 12 && value_.size() != 10) return false;
    return check();
}

const QString &Inn::value() const
{
    return value_;
}

void Inn::setValue(const QString &newValue)
{
    value_ = newValue.trimmed();
    if(value_.startsWith("0000000000")) value_.clear();
}

QByteArray Inn::data() const
{
    if(isNull()) return "000000000000";
    if(!isValid()) return QByteArray();
    if(value_.size() == 12) return value_.toLatin1();
    else return value_.toLatin1() + "  ";
}

std::optional<Inn> Inn::create(const QString &val)
{
    Inn inn(val);
    if(inn.isNull() || !inn.isValid()) return {};
    return std::optional<Inn>(inn);
}

Inn &Inn::operator =(const Inn &other) noexcept
{
    value_ = other.value_;
    return *this;
}
Inn &Inn::operator =(Inn &&other) noexcept
{
    value_.swap(other.value_);
    return *this;
}

bool Inn::operator ==(const Inn &other) const noexcept
{
    return value_ == other.value_;
}

bool Inn::operator !=(const Inn &other) const noexcept
{
    return value_!= other.value_;
}

bool Inn::check() const
{
    if(isNull()) return false;
    bool ok = false;
    qint64 testval = value_.toLongLong(&ok);
    if(!ok || testval <= 0) return false;

    QByteArray inn = value_.toUtf8();
    if(inn.size() == 10)
    {
        qint16 cs = 2 * (inn[0] - '0') +
                4 * (inn[1] - '0') +
                10 * (inn[2] - '0') +
                3 * (inn[3] - '0') +
                5 * (inn[4] - '0') +
                9 * (inn[5] - '0') +
                4 * (inn[6] - '0') +
                6 * (inn[7] - '0') +
                8 * (inn[8] - '0');
        cs = cs % 11;
        cs = cs % 10;
        return cs == (inn[9] - '0');
    }
    else if(inn.size() == 12)
    {
        qint16 cs1 = ((7 * (inn[0] - '0') +
                2 * (inn[1] - '0') +
                4 * (inn[2] - '0') +
                10 * (inn[3] - '0') +
                3 * (inn[4] - '0') +
                5 * (inn[5] - '0') +
                9 * (inn[6] - '0') +
                4 * (inn[7] - '0') +
                6 * (inn[8] - '0') +
                8 * (inn[9] - '0') +
                0 * (inn[10] - '0')
                ) % 11) % 10;
        qint16 cs2 = ((3 * (inn[0] - '0') +
                7 * (inn[1] - '0') +
                2 * (inn[2] - '0') +
                4 * (inn[3] - '0') +
                10 * (inn[4] - '0') +
                3 * (inn[5] - '0') +
                5 * (inn[6] - '0') +
                9 * (inn[7] - '0') +
                4 * (inn[8] - '0') +
                6 * (inn[9] - '0') +
                8 * (inn[10] - '0') +
                0 * (inn[11] - '0')
                ) % 11) % 10;
        return (cs1 == (inn[10] - '0')) && (cs2 == (inn[11] - '0'));
    }
    return false;
}
