#include "labelcheckdata.h"

#include "productcodecreator.h"
#include "formatutils.h"
#include "cashboxbuildconfig.h"
#include <cmath>
#include <QSet>


LabelCheckData::LabelCheckData() noexcept
    : labelCode_()
    , excpectedStatus_(fdf::LabelProductStatus::Invalid)
    , mode_(0)
    , quantity_()
    , unit_()
    , fraction_()
    , checkFlags_()
{

}

LabelCheckData::LabelCheckData(const LabelCheckData &other) noexcept
    : labelCode_(other.labelCode_)
    , excpectedStatus_(other.excpectedStatus_)
    , mode_(other.mode_)
    , quantity_(other.quantity_)
    , unit_(other.unit_)
    , fraction_(other.fraction_)
    , checkFlags_(other.checkFlags_)

{

}

LabelCheckData::LabelCheckData(LabelCheckData &&other) noexcept
    : labelCode_()
    , excpectedStatus_(other.excpectedStatus_)
    , mode_(other.mode_)
    , quantity_()
    , unit_()
    , fraction_()
    , checkFlags_(other.checkFlags_)

{
    labelCode_.swap(other.labelCode_);
    quantity_.swap(other.quantity_);
    unit_.swap(other.unit_);
    fraction_.swap(other.fraction_);
}

LabelCheckData::LabelCheckData(const QVariantMap &map) noexcept
    : labelCode_()
    , excpectedStatus_(fdf::LabelProductStatus::Invalid)
    , mode_(0)
    , quantity_()
    , unit_()
    , fraction_()
    , checkFlags_()

{
    parseMap(map);
}

LabelCheckData::~LabelCheckData()
{

}

bool LabelCheckData::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(!labelCode_.has_value() || !labelCode_.value().readyToB())
    {
        err = CoreApiConst::ErrorCode::NoAnyLabel;
        msg = QStringLiteral("В запросе нет марки");
        return false;
    }
    if(!fdf::labelProductStatusIsValid(excpectedStatus_))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 2003 (планируемый статус товара) отсутствует или имеет некорректное значение");
        return false;
    }

    if(!unit_.has_value() || !fdf::unitIsValid(unit_.value()))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 2108 ( мера количества предмета расчета) отсутствует или имеет некорректное значение");
        return false;
    }
    if(fraction_.has_value() && (!fraction_.value().isValid() || (quantity_.value().value() != 0ll && quantity_.value().value() != DEFAULT_QUANTITY_MULT)))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 1291 ( дробное количество маркированного товара)  имеет некорректное значение");
        return false;
    }
    if(!quantity_.has_value() || quantity_.value().value() <= 0)
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 1023 (количество предмета расчета) отсутствует или имеет некорректное значение");
        return false;
    }

    return true;
}

bool LabelCheckData::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(err, msg);
}

QString LabelCheckData::rawLabel() const
{
    return labelCode_.value_or(LabelCode()).tradeMark();
}

void LabelCheckData::setRawLabel(const QString &newRawLabel)
{
    QString newLabel = newRawLabel;
    if(newLabel.startsWith("\u001d"))newLabel = newLabel.mid(1);
    if(newLabel.trimmed().isEmpty()) labelCode_.reset();
    else
    {
        ProductCodeCreator pcc;
        labelCode_ = pcc.create(newLabel);
    }
}

LabelCode LabelCheckData::labelCode() const
{
    return labelCode_.value_or(LabelCode());
}

void LabelCheckData::setLabelCode(const std::optional<LabelCode> &newLabelCode)
{
    labelCode_ = newLabelCode;
}

fdf::LabelProductStatus LabelCheckData::excpectedStatus() const
{
    return excpectedStatus_;
}

void LabelCheckData::setExcpectedStatus(fdf::LabelProductStatus newExcpectedStatus)
{
    excpectedStatus_ = newExcpectedStatus;
}

quint8 LabelCheckData::mode() const
{
    return mode_;
}

void LabelCheckData::setMode(quint8 newMode)
{
    mode_ = newMode;
}

std::optional<FixNumber> LabelCheckData::quantity() const
{
    return quantity_;
}

void LabelCheckData::setQuantity(std::optional<FixNumber> newQuantity)
{
    quantity_ = newQuantity;
}

std::optional<fdf::ItemUnit> LabelCheckData::unit() const
{
    return unit_;
}

void LabelCheckData::setUnit(std::optional<fdf::ItemUnit> newUnit)
{
    unit_ = newUnit;
}

std::optional<QuantityFraction> LabelCheckData::fraction() const
{
    return fraction_;
}

void LabelCheckData::setFraction(std::optional<QuantityFraction> newFraction)
{
    fraction_ = newFraction;
}

const LabelCheckData::LabelCheckFlags &LabelCheckData::checkFlags() const
{
    return checkFlags_;
}

void LabelCheckData::setCheckFlags(const LabelCheckFlags &newCheckFlags)
{
    checkFlags_ = newCheckFlags;
}

QVariantMap LabelCheckData::toMap() const
{
    QVariantMap res;
    if(labelCode_.has_value() && labelCode_.value().isValid())
    {
        res["rawLabel"] = FormatUtils::rawLabelToTransport(rawLabel());
    }
    res["excpectedStatus"] = static_cast<qint32>(excpectedStatus_);
    if(mode_) res["mode"] = static_cast<qint32>(mode_);
    if(quantity_.has_value()) res["quantity"] = quantity_.value().toMap();
    if(unit_.has_value()) res["unit"] = static_cast<qint32>(unit_.value());
    if(fraction_.has_value()) res["fraction"] = fraction_.value().toMap();
    res["checkFlags"] = static_cast<qint32>(checkFlags_);
    return res;
}

void LabelCheckData::parseMap(const QVariantMap &map)
{
    clean();
    if(map.contains("rawLabel")) setRawLabel(FormatUtils::rawLabelFromTransport(map["rawLabel"].toString()));
    else if(map.contains("label"))setRawLabel(FormatUtils::rawLabelFromTransport(map["label"].toString()));
    if(map.contains("excpectedStatus"))
    {
        excpectedStatus_ = static_cast<fdf::LabelProductStatus>(map["excpectedStatus"].toInt());
    }
    if(map.contains("mode")) mode_ = map["mode"].toInt();
    if(map.contains("quantity")) quantity_ = FormatUtils::parseSumm("quantity", map, DEFAULT_QUANTITY_DEC);
    if(map.contains("unit")) unit_ = static_cast<fdf::ItemUnit>(map["unit"].toInt());
    if(map.contains("fraction")) fraction_ = QuantityFraction(map["fraction"].toMap());
    checkFlags_ = LabelCheckFlags(map["checkFlags"].toInt());
}

void LabelCheckData::clean()
{
    labelCode_.reset();
    excpectedStatus_ = fdf::LabelProductStatus::Invalid;
    mode_ = 0;
    quantity_.reset();
    unit_.reset();
    fraction_.reset();
    checkFlags_ = LabelCheckFlags();
}

Tlv::Stlv LabelCheckData::forB5() const
{
    Tlv::Stlv res;
    if(!isValid()) return res;
    Tlv buf;
    buf.setByte(fdf::Tag::PromiseProductStatus, static_cast<quint8>(excpectedStatus_));
    res << buf;
    buf.setByte(fdf::Tag::LabelProcessMode, mode_);
    res << buf;
    if(/*(excpectedStatus_ == fdf::LabelProductStatus::Measured ||
            excpectedStatus_ == fdf::LabelProductStatus::MeasuredRefund) &&*/
            unit_.has_value())
    {
        FixNumber quantity = quantity_.has_value() && quantity_.value().value() > 0ll ?
                    quantity_.value() : FixNumber(DEFAULT_QUANTITY_MULT, static_cast<qint64>(DEFAULT_AMOUNT_MULT));
        buf.setByte(fdf::Tag::ItemQuantityUnitType, static_cast<quint8>(unit_.value()));
        res << buf;
        if(/*unit_.value() != fdf::ItemUnit::Ut ||*/ !fraction_.has_value() || !fraction_->isValid())
        {
            buf.setFvln(fdf::Tag::ItemQuantity, quantity);
            res << buf;
        }
        else if(/*unit_.value() == fdf::ItemUnit::Ut || */fraction_.has_value() && fraction_->isValid())
        {
            res << fraction_.value().toTlv();
        }
    }
    return res;
}

QList<LabelCheckData> LabelCheckData::withAlternativeStatuses() const
{
    QList<LabelCheckData> res;
    if(!isValid()) return res;
    switch (excpectedStatus_) {
    case fdf::LabelProductStatus::Piece          :
        res << withAlternativeStatus(fdf::LabelProductStatus::Measured)
            << withAlternativeStatus(fdf::LabelProductStatus::NotChanged);
        break;
    case fdf::LabelProductStatus::Measured       :
        res << withAlternativeStatus(fdf::LabelProductStatus::Piece)
            << withAlternativeStatus(fdf::LabelProductStatus::NotChanged);
        break;
    case fdf::LabelProductStatus::PieceRefund    :
        res << withAlternativeStatus(fdf::LabelProductStatus::MeasuredRefund)
            << withAlternativeStatus(fdf::LabelProductStatus::NotChanged);
        break;
    case fdf::LabelProductStatus::MeasuredRefund :
        res << withAlternativeStatus(fdf::LabelProductStatus::PieceRefund)
            << withAlternativeStatus(fdf::LabelProductStatus::NotChanged);
        break;
    case fdf::LabelProductStatus::NotChanged     :
        res << withAlternativeStatus(fdf::LabelProductStatus::Piece)
            << withAlternativeStatus(fdf::LabelProductStatus::Measured)
            << withAlternativeStatus(fdf::LabelProductStatus::PieceRefund)
            << withAlternativeStatus(fdf::LabelProductStatus::MeasuredRefund);
        break;
    default:
        break;
    }
    return res;
}

LabelCheckData &LabelCheckData::operator =(const LabelCheckData &other) noexcept
{
    labelCode_ = other.labelCode_;
    excpectedStatus_ = other.excpectedStatus_;
    mode_ = other.mode_;
    quantity_ = other.quantity_;
    unit_ = other.unit_;
    fraction_ = other.fraction_;
    checkFlags_ = other.checkFlags_;
    return *this;
}


LabelCheckData &LabelCheckData::operator =(LabelCheckData &&other) noexcept
{
    labelCode_.swap(other.labelCode_);
    excpectedStatus_ = other.excpectedStatus_;
    mode_ = other.mode_;
    quantity_.swap(other.quantity_);
    unit_.swap(other.unit_);
    fraction_.swap(other.fraction_);
    checkFlags_ = other.checkFlags_;
    return *this;
}

bool LabelCheckData::operator ==(const LabelCheckData &other) const noexcept
{
    return labelCode_ == other.labelCode_ &&
            excpectedStatus_ == other.excpectedStatus_ &&
            mode_ == other.mode_ &&
            quantity_ == other.quantity_ &&
            unit_ == other.unit_ &&
            fraction_ == other.fraction_ &&
            checkFlags_ == other.checkFlags_;
}

bool LabelCheckData::operator !=(const LabelCheckData &other) const noexcept
{
    return !(*this == other);
}

LabelCheckData LabelCheckData::withAlternativeStatus(fdf::LabelProductStatus s) const
{
    LabelCheckData res = *this;
    res.excpectedStatus_ = s;
    return res;
}
