#include "labelcheckresult.h"

#include "productcodecreator.h"

#include "formatutils.h"
#include "cashboxbuildconfig.h"
#include "crc.h"
#include <cmath>
#include <QSet>



LabelCheckResult::LabelCheckResult() noexcept
    : labelCode_()
    , actualStatus_(fdf::LabelProductStatus::Invalid)
    , mode_(0)
    , quantity_()
    , unit_()
    , fraction_()
    , lct2100_()
    , productId_()
    , clFlags_()
    , reqDt_()
    , st2109_(Status2109::Correct)
    , reqCode2105_(ReqCode2105::Correct)
    , reqResult_(5)

{

}

LabelCheckResult::LabelCheckResult(const LabelCheckResult &other) noexcept
    : labelCode_(other.labelCode_)
    , actualStatus_(other.actualStatus_)
    , mode_(other.mode_)
    , quantity_(other.quantity_)
    , unit_(other.unit_)
    , fraction_(other.fraction_)
    , lct2100_(other.lct2100_)
    , productId_(other.productId_)
    , clFlags_(other.clFlags_)
    , reqDt_(other.reqDt_)
    , st2109_(other.st2109_)
    , reqCode2105_(other.reqCode2105_)
    , reqResult_(other.reqResult_)
{

}

LabelCheckResult::LabelCheckResult(LabelCheckResult &&other) noexcept
    : labelCode_()
    , actualStatus_(other.actualStatus_)
    , mode_(other.mode_)
    , quantity_()
    , unit_()
    , fraction_()
    , lct2100_()
    , productId_()
    , clFlags_(other.clFlags_)
    , reqDt_()
    , st2109_(other.st2109_)
    , reqCode2105_(other.reqCode2105_)
    , reqResult_(other.reqResult_)

{
    labelCode_.swap(other.labelCode_);
    quantity_.swap(other.quantity_);
    unit_.swap(other.unit_);
    fraction_.swap(other.fraction_);
    lct2100_.swap(other.lct2100_);
    productId_.swap(other.productId_);
    reqDt_.swap(other.reqDt_);

}

LabelCheckResult::LabelCheckResult(const LabelCheckData &in) noexcept
    : labelCode_(in.labelCode())
    , actualStatus_(in.excpectedStatus())
    , mode_(in.mode())
    , quantity_(in.quantity())
    , unit_(in.unit())
    , fraction_(in.fraction())
    , lct2100_()
    , productId_()
    , clFlags_()
    , reqDt_()
    , st2109_(Status2109::Correct)
    , reqCode2105_(ReqCode2105::Correct)
    , reqResult_(5)

{

}

LabelCheckResult::LabelCheckResult(const QVariantMap &map) noexcept
    : labelCode_()
    , actualStatus_(fdf::LabelProductStatus::Invalid)
    , mode_(0)
    , quantity_()
    , unit_()
    , fraction_()
    , lct2100_()
    , productId_()
    , clFlags_()
    , reqDt_()
    , st2109_(Status2109::Correct)
    , reqCode2105_(ReqCode2105::Correct)
    , reqResult_(5)

{
    parseMap(map);
}

LabelCheckResult::~LabelCheckResult()
{

}

QString LabelCheckResult::rawLabel() const
{
    return labelCode_.value_or(LabelCode()).tradeMark();
}

void LabelCheckResult::setRawLabel(const QString &newRawLabel)
{
    if(newRawLabel.trimmed().isEmpty()) labelCode_.reset();
    else
    {
        ProductCodeCreator pcc;
        labelCode_ = pcc.create(newRawLabel);
    }
}

LabelCode LabelCheckResult::labelCode() const
{
    return labelCode_.value_or(LabelCode());
}

void LabelCheckResult::setLabelCode(const std::optional<LabelCode> &newLabelCode)
{
    labelCode_ = newLabelCode;
}

QString LabelCheckResult::calc2115() const
{
    return labelCode_.value_or(LabelCode()).calc2115();
}

fdf::LabelProductStatus LabelCheckResult::actualStatus() const
{
    return actualStatus_;
}

void LabelCheckResult::setActualStatus(fdf::LabelProductStatus newExcpectedStatus)
{
    actualStatus_ = newExcpectedStatus;
}

quint8 LabelCheckResult::mode() const
{
    return mode_;
}

void LabelCheckResult::setMode(quint8 newMode)
{
    mode_ = newMode;
}

std::optional<FixNumber> LabelCheckResult::quantity() const
{
    return quantity_;
}

void LabelCheckResult::setQuantity(std::optional<FixNumber> newQuantity)
{
    quantity_ = newQuantity;
}

std::optional<fdf::ItemUnit> LabelCheckResult::unit() const
{
    return unit_;
}

void LabelCheckResult::setUnit(std::optional<fdf::ItemUnit> newUnit)
{
    unit_ = newUnit;
}

std::optional<QuantityFraction> LabelCheckResult::fraction() const
{
    return fraction_;
}

void LabelCheckResult::setFraction(std::optional<QuantityFraction> newFraction)
{
    fraction_ = newFraction;
}

fdf::LabelCodeType LabelCheckResult::lct2100() const
{
    if(lct2100_.has_value()) return lct2100_.value();
    if(labelCode_.has_value()) return labelCode_.value().lct2100();
    return fdf::LabelCodeType::Unknown;
}

void LabelCheckResult::setLct2100(std::optional<fdf::LabelCodeType> newLct2100)
{
    lct2100_ = newLct2100;
}

void LabelCheckResult::setLct2100(fdf::LabelCodeType lct)
{
    lct2100_ = lct;
}

QString LabelCheckResult::productId() const
{
    if(productId_.has_value()) return productId_.value();
    if(!labelCode_.has_value()) return QString();
    if(labelCode_.value().isEncodedTradeMark()) return labelCode_.value().productCode().data();
    return QString();
}

void LabelCheckResult::setProductId(std::optional<QString> newProductId)
{
    productId_ = newProductId;
}

void LabelCheckResult::setProductId(const QString &newProductId)
{
    productId_ = newProductId;
}

const fdf::CheckLabelFlags &LabelCheckResult::clFlags() const
{
    return clFlags_;
}

void LabelCheckResult::setClFlags(const fdf::CheckLabelFlags &newClFlags)
{
    clFlags_ = newClFlags;
}

const QDateTime &LabelCheckResult::reqDt() const
{
    return reqDt_;
}

void LabelCheckResult::setReqDt(const QDateTime &newReqDt)
{
    reqDt_ = newReqDt;
}

LabelCheckResult::Status2109 LabelCheckResult::st2109() const
{
    return st2109_;
}

void LabelCheckResult::setSt2109(Status2109 newSt2109)
{
    st2109_ = newSt2109;
}

LabelCheckResult::ReqCode2105 LabelCheckResult::reqCode2105() const
{
    return reqCode2105_;
}

void LabelCheckResult::setReqCode2105(ReqCode2105 newReqCode2105)
{
    reqCode2105_ = newReqCode2105;
}

const LabelCheckResult::ReqResult &LabelCheckResult::reqResult() const
{
    return reqResult_;
}

void LabelCheckResult::setReqResult(const ReqResult &newReqResult)
{
    reqResult_ = newReqResult;
}

QVariantMap LabelCheckResult::toMap() const
{
    QVariantMap res;
    if(labelCode_.has_value() && labelCode_.value().isValid())
    {
        res["rawLabel"] = FormatUtils::rawLabelToTransport(rawLabel());
    }
    res["actualStatus"] = static_cast<qint32>(actualStatus_);
    if(mode_) res["mode"] = static_cast<qint32>(mode_);
    if(quantity_.has_value()) res["quantity"] = quantity_.value().toMap();
    if(unit_.has_value()) res["unit"] = static_cast<qint32>(unit_.value());
    if(fraction_.has_value()) res["fraction"] = fraction_.value().toMap();
    if(lct2100_.has_value()) res["lct2100"] = static_cast<qint32>(lct2100_.value());
    if(productId_.has_value()) res["productId"] = productId();
    res["clFlags"] = static_cast<qint32>(clFlags_);
    if(reqDt_.isValid()) res["reqDt"] = DT2STR_(reqDt_);
    res["st2109"] = static_cast<qint32>(st2109_);
    res["reqCode2105"] = static_cast<qint32>(reqCode2105_);
    res["reqResult"] = static_cast<qint32>(reqResult_);

    return res;
}

QVariantMap LabelCheckResult::toExternalMap() const
{
    QVariantMap res;
    if(labelCode_.has_value() && labelCode_.value().isValid())
    {
        res["rawLabel"] = FormatUtils::rawLabelToTransport(rawLabel());
    }
    res["actualStatus"] = static_cast<qint32>(actualStatus_);
    if(mode_) res["mode"] = static_cast<qint32>(mode_);
    if(quantity_.has_value()) res["quantity"] = quantity_.value().toString(QLatin1Char('.'));
    if(unit_.has_value()) res["unit"] = static_cast<qint32>(unit_.value());
    if(fraction_.has_value()) res["fraction"] = fraction_.value().toMap();
    if(lct2100_.has_value()) res["lct2100"] = static_cast<qint32>(lct2100_.value());
    if(productId_.has_value()) res["productId"] = productId();
    res["clFlags"] = static_cast<qint32>(clFlags_);
    if(reqDt_.isValid()) res["reqDt"] = DT2STR_(reqDt_);
    res["st2109"] = static_cast<qint32>(st2109_);
    res["reqCode2105"] = static_cast<qint32>(reqCode2105_);
    res["reqResult"] = static_cast<qint32>(reqResult_);

    return res;
}

void LabelCheckResult::parseMap(const QVariantMap &map)
{
    clean();
    if(map.contains("rawLabel")) setRawLabel(FormatUtils::rawLabelFromTransport(map["rawLabel"].toString()));
    else if(map.contains("label"))setRawLabel(FormatUtils::rawLabelFromTransport(map["label"].toString()));
    if(map.contains("actualStatus"))
    {
        actualStatus_ = static_cast<fdf::LabelProductStatus>(map["actualStatus"].toInt());
    }
    if(map.contains("mode")) mode_ = map["mode"].toInt();
    if(map.contains("quantity")) quantity_ = FormatUtils::parseSumm("quantity", map, DEFAULT_QUANTITY_DEC);
    if(map.contains("unit")) unit_ = static_cast<fdf::ItemUnit>(map["unit"].toInt());
    if(map.contains("fraction")) fraction_ = QuantityFraction(map["fraction"].toMap());
    if(map.contains("lct2100"))
    {
        lct2100_ = static_cast<fdf::LabelCodeType>(map["lct2100"].toInt());
    }
    if(map.contains("productId")) productId_ = map["productId"].toString();
    clFlags_ = fdf::CheckLabelFlags(map["clFlags"].toInt());
    reqDt_ = STR2DT_(map["reqDt"].toString());
    st2109_ = static_cast<Status2109>(map["st2109"].toInt());
    reqCode2105_ = static_cast<ReqCode2105>(map["reqCode2105"].toInt());
    reqResult_ = ReqResult(map["reqResult"].toInt());
}

void LabelCheckResult::clean()
{
    labelCode_.reset();
    actualStatus_ = fdf::LabelProductStatus::Invalid;
    mode_ = 0;
    quantity_.reset();
    unit_.reset();
    fraction_.reset();
    lct2100_.reset();
    productId_.reset();
    clFlags_ = fdf::CheckLabelFlags();
    reqDt_ = QDateTime();
    st2109_ = Status2109::Correct;
    reqCode2105_ = ReqCode2105::Correct;
    reqResult_ = ReqResult(5);

}

void LabelCheckResult::parseB6(const fdf::CheckLabelFlags &flags, const Tlv::Stlv &list)
{
    QStringList sl;
    for(const Tlv &t : list)
    {
         sl << QString::number(static_cast<qint32>(t.tag())) + ": " + t.value().toHex();
    }

    for(const Tlv &t : list)
    {
        switch (t.tag()) {
        case fdf::Tag::RequestDt: reqDt_ = t.toDt(); break;
        case fdf::Tag::LabelProcessMode: mode_ = t.toByte(); break;
        case fdf::Tag::LabelCodeType: lct2100_ = static_cast<fdf::LabelCodeType>(t.toByte()); break;
        case fdf::Tag::LabelSystemResponse: st2109_ = static_cast<Status2109>(t.toByte()); break;
        case fdf::Tag::ProductId: productId_ = t.toString(); break;
        case fdf::Tag::RequestProcessCodes: reqCode2105_ = static_cast<ReqCode2105>(t.toByte()); break;
        case fdf::Tag::RequestProcessResult: reqResult_ = ReqResult(t.toByte()); break;
        default:
            break;
        }
    }
    clFlags_ = flags;

}

LabelCheckResult &LabelCheckResult::operator =(const LabelCheckResult &other) noexcept
{
    labelCode_ = other.labelCode_;
    actualStatus_ = other.actualStatus_;
    mode_ = other.mode_;
    quantity_ = other.quantity_;
    unit_ = other.unit_;
    fraction_ = other.fraction_;
    lct2100_ = other.lct2100_;
    productId_ = other.productId_;
    clFlags_ = other.clFlags_;
    reqDt_ = other.reqDt_;
    st2109_ = other.st2109_;
    reqCode2105_ =other.reqCode2105_;
    reqResult_ = other.reqResult_;

    return *this;
}


LabelCheckResult &LabelCheckResult::operator =(LabelCheckResult &&other) noexcept
{
    labelCode_.swap(other.labelCode_);
    actualStatus_ = other.actualStatus_;
    mode_ = other.mode_;
    quantity_.swap(other.quantity_);
    unit_.swap(other.unit_);
    fraction_.swap(other.fraction_);
    lct2100_.swap(other.lct2100_);
    productId_.swap(other.productId_);
    clFlags_ = other.clFlags_;
    reqDt_.swap(other.reqDt_);
    st2109_ = other.st2109_;
    reqCode2105_ =other.reqCode2105_;
    reqResult_ = other.reqResult_;
    return *this;
}

bool LabelCheckResult::operator ==(const LabelCheckResult &other) const noexcept
{
    return labelCode_ == other.labelCode_ &&
            actualStatus_ == other.actualStatus_ &&
            mode_ == other.mode_ &&
            quantity_ == other.quantity_ &&
            unit_ == other.unit_ &&
            fraction_ == other.fraction_ &&
            lct2100_ == other.lct2100_ &&
            productId_ == other.productId_ &&
            clFlags_ == other.clFlags_ &&
            reqDt_ == other.reqDt_ &&
            st2109_ == other.st2109_ &&
            reqCode2105_ == other.reqCode2105_ &&
            reqResult_ ==  other.reqResult_;

}

bool LabelCheckResult::operator !=(const LabelCheckResult &other) const noexcept
{
    return !(*this == other);
}
