#include "modeldata.h"
#include "cashboxbuildconfig.h"


#include <QCryptographicHash>

const qint32 ModelData::SERIAL_LENGTH = DEFAULT_SERIAL_LENGTH;

ModelData::ModelData() noexcept
    : model_(CASHBOX_MODEL)
    , srvModel_(CASHBOX_SRV_MODEL)
    , modelName_(CASHBOX_MODEL_NAME)
    , serial_()
    , version_(CASHBOX_VERSION)
    , sign_()
{

}

ModelData::ModelData(const ModelData &other)noexcept
    : model_(other.model_)
    , srvModel_(other.srvModel_)
    , modelName_(other.modelName_)
    , serial_(other.serial_)
    , version_(other.version_)
    , sign_(other.sign_)
{

}

ModelData::ModelData(ModelData &&other)noexcept
    : model_(other.model_)
    , srvModel_(other.srvModel_)
    , modelName_()
    , serial_()
    , version_()
    , sign_()
{
    modelName_.swap(other.modelName_);
    serial_.swap(other.serial_);
    version_.swap(other.version_);
    sign_.swap(other.sign_);
}

ModelData::ModelData(const QVariantMap &map)noexcept
    : model_(CASHBOX_MODEL)
    , srvModel_(CASHBOX_SRV_MODEL)
    , modelName_(CASHBOX_MODEL_NAME)
    , serial_()
    , version_()
    , sign_()
{
    parseMap(map);
}

ModelData::~ModelData()
{

}

bool ModelData::isValid() const
{
    return model_ > 0 &&
            srvModel_ > 0 &&
            !modelName_.isEmpty() &&
            !serial_.trimmed().isEmpty() &&
            serial_.trimmed().size() < 21 &&
            !sign_.isEmpty();
}

bool ModelData::checkSign(const QString &hwHash) const
{
    if(serial_.trimmed().isEmpty() || serial_.trimmed().size() >20 || modelName_.isEmpty() || model_<= 0 || srvModel_ <=0) return false;
    QByteArray buf = (
                QString::number(model_) +
                QString::number(srvModel_) +
                serial_ + hwHash).toUtf8();
    return sign_ == QString::fromLatin1(QCryptographicHash::hash(buf, QCryptographicHash::Sha1).toBase64());
}

bool ModelData::signData(const QString &hwHash)
{
    if(serial_.trimmed().isEmpty() || serial_.trimmed().size() >20 || modelName_.isEmpty() || model_<= 0 || srvModel_ <=0) return false;
    QByteArray buf = (
                QString::number(model_) +
                QString::number(srvModel_) +
                serial_ + hwHash).toUtf8();
    sign_ = QString::fromLatin1(QCryptographicHash::hash(buf, QCryptographicHash::Sha1).toBase64());
    return isValid();
}

quint16 ModelData::model() const
{
    return model_;
}

void ModelData::setModel(const quint16 &model)
{
    model_ = model;
}

qint32 ModelData::srvModel() const
{
    return srvModel_;
}

void ModelData::setSrvModel(const qint32 &srvModel)
{
    srvModel_ = srvModel;
}

QString ModelData::modelName() const
{
    return modelName_;
}

void ModelData::setModelName(const QString &modelName)
{
    modelName_ = modelName;
}

QString ModelData::serial() const
{
    return serial_;
}

void ModelData::setSerial(const QString &serial)
{
    serial_ = serial;
}

QString ModelData::sign() const
{
    return sign_;
}

void ModelData::setSign(const QString &sign)
{
    sign_ = sign;
}

QString ModelData::version() const
{
    return QString(version_.data(), 3);
}

void ModelData::setVersion(const QString &ver)
{
    if(ver.size() !=3)
    {
        version_ = std::array<QChar, 3>(CASHBOX_VERSION);
    }
    else
    {
        version_[0] = ver[0];
        version_[1] = ver[1];
        version_[2] = ver[2];
    }
}

QVariantMap ModelData::toMap() const
{
    QVariantMap res;
    res.insert("model", model_);
    res.insert("srvModel", srvModel_);
    res.insert("modelName", modelName_);
    res.insert("serial", serial_);
    res.insert("version", version());
    res.insert("sign", sign_);
    return res;
}

QVariantMap ModelData::toExternalMap() const
{
    QVariantMap res;
    res.insert("model", model_);
    res.insert("srvModel", srvModel_);
    res.insert("modelName", modelName_);
    res.insert("serial", serial_);
    res.insert("version", version());
    return res;
}

void ModelData::parseMap(const QVariantMap &map)
{
    model_ = static_cast<quint16> (map["model"].toUInt());
    srvModel_ = map["srvModel"].toInt();
    modelName_ = map["modelName"].toString().trimmed();
    serial_ = map["serial"].toString().trimmed();
    setVersion(map["version"].toString().trimmed());
    sign_ = map["sign"].toString();
}

QString ModelData::toHtml(bool withHead) const
{
    QStringList res;
    if(withHead)
    {
        res << "<html><head>"
            << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">"
            << "</head><body>" ;
    }
    res << QStringLiteral("<h1 align=\"center\">%1</h1>").arg(modelName_)
        << QStringLiteral("<p align=\"center\"><b>%1</b></p>").arg(serial_)
           << QStringLiteral("<p align=\"center\">ВЕРСИЯ<b>: %1</b></p>").arg(version())
        << QStringLiteral("<hr>");
    if(withHead)
    {
        res	<< "</body></html>";
    }

    return res.join("\n");
}

ModelData &ModelData::operator =(const ModelData &other) noexcept
{
    model_ = other.model_;
    srvModel_ = other.srvModel_;
    modelName_ = other.modelName_;
    serial_ = other.serial_;
    version_ = other.version_;
    sign_ = other.sign_;
    return *this;
}

ModelData &ModelData::operator =(ModelData &&other)noexcept
{
    model_ = other.model_;
    srvModel_ = other.srvModel_;
    modelName_.swap(other.modelName_);
    serial_.swap(other.serial_);
    version_.swap(other.version_);
    sign_.swap(other.sign_);
    return *this;

}

bool ModelData::operator ==(const ModelData &other) const noexcept
{
    return model_ == other.model_ &&
            srvModel_ == other.srvModel_ &&
            modelName_ == other.modelName_ &&
            serial_ == other.serial_ &&
            version_ == other.version_ &&
            sign_ == other.sign_;

}

bool ModelData::operator !=(const ModelData &other) const noexcept
{
    return !(*this == other);
}
