#include "ofdsettings.h"

const QString OfdSettings::SECTION = "OFD";
const QString OfdSettings::INN = "inn";
const QString OfdSettings::ADDRESS = "addr";
const QString OfdSettings::PORT = "port";
const QString OfdSettings::QUERYI_NTERVAL = "interval";
const QString OfdSettings::TIMER_S = "timerS";
const QString OfdSettings::URL = "url";
const QString OfdSettings::NAME = "name";
const QString OfdSettings::SHORT_NAME = "sname";
const QString OfdSettings::LABELS_ADDRESS = "labelsAddress";
const QString OfdSettings::LABELS_PORT = "labelsPort";
const QString OfdSettings::KEYS_ADDRESS = "keysAddress";
const QString OfdSettings::KEYS_PORT = "keysPort";
const QString OfdSettings::TEST_TIMEOUT = "testTout";
const QString OfdSettings::KEYS_TIMEOUT = "keysTout";


const QMap<QString, OfdSettings> OfdSettings::SUPPORTED_OFD = {
    std::make_pair(QStringLiteral("7704211201"), OfdSettings(
    QStringLiteral("7704211201"),
    "f1.taxcom.ru", 7777, "http://receipt.taxcom.ru/",
    QStringLiteral("Общество с ограниченной ответственностью Такском"),
    QStringLiteral("Такском"))),

    std::make_pair(QStringLiteral("9715260691"), OfdSettings(
    QStringLiteral("9715260691"),
    "ofdp.platformaofd.ru", 21101,
    "https://lk.platformaofd.ru/web/noauth/cheque/search/",
    QStringLiteral("Общество с ограниченной ответственностью Эвотор ОФД"),
    QStringLiteral("Эвотор ОФД"))),

    std::make_pair(QStringLiteral("7709364346"), OfdSettings(
    QStringLiteral("7709364346"),
    "k-server.1-ofd.ru", 7777, "http://consumer.1-ofd.ru/",
    QStringLiteral("Акционерное общество Энергетические системы и коммуникации"),
    QStringLiteral("Энергетические системы и коммуникации"))),

    std::make_pair(QStringLiteral("7728699517"), OfdSettings(
    QStringLiteral("7728699517"),
    "connect.ofd-ya.ru", 7779, "https://ofd-ya.ru/open_service/",
    QStringLiteral("Общество с ограниченной ответственностью Ярус"),
    QStringLiteral("Ярус"))),

    std::make_pair(QStringLiteral("7841465198"), OfdSettings(
    QStringLiteral("7841465198"),
    "gate.ofd.ru", 4000, "http://ofd.ru/checkinfo/",
    QStringLiteral("Общество с ограниченной ответственностью ПЕТЕР-СЕРВИС Спецтехнологии"),
    QStringLiteral("ПЕТЕР-СЕРВИС Спецтехнологии"))),

    std::make_pair(QStringLiteral("7605016030"), OfdSettings(
    QStringLiteral("7605016030"),
    "kkt.sbis.ru", 7777, "https://ofd.sbis.ru/",
    QStringLiteral("Общества с ограниченной ответственностью Компания Тензор"),
    QStringLiteral("Компания Тензор"))),

    std::make_pair(QStringLiteral("4029017981"), OfdSettings(
    QStringLiteral("4029017981"),
    "ofd.astralnalog.ru", 7777, "https://ofd.astralnalog.ru/#find-ticket/",
    QStringLiteral("Закрытое акционерное общество КАЛУГА АСТРАЛ"),
    QStringLiteral("КАЛУГА АСТРАЛ"))),

    std::make_pair(QStringLiteral("6663003127"), OfdSettings(
    QStringLiteral("6663003127"),
    "ofd.kontur.ru", 7777, "https://cash.kontur.ru/CashReceipt/Search/",
    QStringLiteral("Закрытое акционерное общество Производственная фирма СКБ Контур"),
    QStringLiteral("СКБ Контур"))),


    std::make_pair(QStringLiteral("7704358518"), OfdSettings(
    QStringLiteral("7704358518"),
    "kkt.ofd.yandex.net", 12345, "https://ofd.yandex.ru/",
    QStringLiteral("Общество с ограниченной ответственностью Яндекс.ОФД"),
    QStringLiteral("Яндекс.ОФД"))),

    std::make_pair(QStringLiteral("7729633131"), OfdSettings(
    QStringLiteral("7729633131"),
    "ofd.garantexpress.ru", 30801, "http://www.garantexpress.ru/",
    QStringLiteral("Общество с ограниченной ответственностью Электронный экспресс"),
    QStringLiteral("Электронный экспресс"))),

    std::make_pair(QStringLiteral("7801392271"), OfdSettings(
    QStringLiteral("7801392271"),
    "92.38.2.202", 7001, "https://www.esphere.ru/",
    QStringLiteral("Общества с ограниченной ответственностью КОРУС Консалтинг СНГ"),
    QStringLiteral("КОРУС Консалтинг СНГ"))),

    std::make_pair(QStringLiteral("2310031475"), OfdSettings(
    QStringLiteral("2310031475"),
    "kkt.ofd-magnit.ru", 7001, "https://check.ofd-magnit.ru/CheckWebApp/fiscaldocsearch.zul/",
    QStringLiteral("Акционерное общество Тандер"),QStringLiteral("Тандер"))),//АО ТАНДЕР - не правильные данные

    std::make_pair(QStringLiteral("5902034504"), OfdSettings(
    QStringLiteral("5902034504"),
    "kkt.ofd-initpro.ru", 9999, "https://ofd-initpro.ru/check-bill/",
    QStringLiteral("Общество с ограниченной ответственностью Удостоверяющий центр \"ИнитПро\""),
    QStringLiteral("Удостоверяющий центр \"ИнитПро\""))),//Общество с ограниченной ответственностью Удостоверяющий центр «ИнитПро»

    std::make_pair(QStringLiteral("7729642175"), OfdSettings(
    QStringLiteral("7729642175"),
    "kkt.e-ofd.ru", 7777, "https://e-ofd.ru/check/",
    QStringLiteral("Общество с ограниченной ответственностью \"ГРУППА ЭЛЕМЕНТ\""),
    QStringLiteral("ГРУППА ЭЛЕМЕНТ"))),//Общество с ограниченной ответственностью «ГРУППА ЭЛЕМЕНТ»


    std::make_pair(QStringLiteral("7703282175"), OfdSettings(
    QStringLiteral("7703282175"),
    "127.0.0.1", 7777, "http://ofd.nvg.ru/receipt/",
    QStringLiteral("Акционерное общество \"Энвижн Груп\""), QStringLiteral("Энвижн Груп"))),//Акционерное общество «Энвижн Груп»

    std::make_pair(QStringLiteral("7713076301"), OfdSettings(
    QStringLiteral("7713076301"),
    "217.118.87.210", 7765, "http://ofd.beeline.ru:81/check-order/",
    QStringLiteral("Публичное акционерное общество \"Вымпел-Коммуникации\""),
    QStringLiteral("Вымпел-Коммуникации"))),//Публичное акционерное общество «Вымпел-Коммуникации»

    std::make_pair(QStringLiteral("7710007966"), OfdSettings(
    QStringLiteral("7710007966"),
    "ofd.multicarta.ru", 21101, "https://ofdcheck.multicarta.ru/",
    QStringLiteral("Общество с ограниченной ответственностью \"МультиКарта\""),
    QStringLiteral("МультиКарта"))),//Общество с ограниченной ответственностью «МультиКарта»

    std::make_pair(QStringLiteral("7802870820"), OfdSettings(
    QStringLiteral("7802870820"),
    "185.241.176.4", 21101, "https://ofdcheck.multicarta.ru/",
    QStringLiteral("Общество с ограниченной ответственностью \"Дримкас\""),
    QStringLiteral("Дримкас"))),//Общество с ограниченной ответственностью «Дримкас»

    std::make_pair(QStringLiteral("6658497833"), OfdSettings(
    QStringLiteral("6658497833"),
    "ofd.kontur.ru", 7777, "cash-ntt.kontur.ru",
    QStringLiteral("Общество с ограниченной ответственностью \"Контур НТТ\""),
    QStringLiteral("Контур НТТ"))),//Общество с ограниченной ответственностью «Контур НТТ»

    std::make_pair(QStringLiteral("7714731464"), OfdSettings(
    QStringLiteral("7714731464"),
    "109.73.43.4", 19086, "109.73.43.4:63188",
    QStringLiteral("Тестовый ОФД Атлас кард"),
    QStringLiteral("Атлас (ТЕСТ)"),
    QStringLiteral("82.202.183.18"), 21701, QStringLiteral("31.44.83.188"), 31101)),

};

const QMap<QString, OfdSettings> OfdSettings::SUPPORTED_DEBUG_OFD = {
    std::make_pair(QStringLiteral("7714731464"), OfdSettings(
     QStringLiteral("7714731464"),
     "109.73.43.4", 19086, "109.73.43.4:63188",
     QStringLiteral("Тестовый ОФД Атлас кард"),
     QStringLiteral("Атлас (ТЕСТ)"),
     QStringLiteral("82.202.183.18"), 21701, QStringLiteral("31.44.83.188"), 31101)),
    std::make_pair(QStringLiteral("9715260691"), OfdSettings(
     QStringLiteral("9715260691"),
     "185.170.204.85", 19081, "ofdt.platformaofd.ru:19081",
     QStringLiteral("Общество с ограниченной ответственностью Эвотор ОФД"),
     QStringLiteral("ЭВОТОР (ТЕСТ)"),
     QStringLiteral("185.170.204.85"), 19082, QStringLiteral("31.44.83.188"), 31101)),
};


OfdSettings::OfdSettings(const QString &inn, const QString &address, quint16 port,
                         const QString &url,
                         const QString &name, const QString &shortName) noexcept
    : inn_(inn)
    , address_(address.trimmed())
    , port_(port)
    , queryInterval_(0)
    , timerS_(12000)
    , url_(url)
    , name_(name)
    , shortName_(shortName)
    , labelsAddress_()
    , labelsPort_(0)
    , keysAddress_()
    , keysPort_(0)
    , testTimeout_(3000)
    , keysTimeout_(30000)

{

}

OfdSettings::OfdSettings(const QString &inn, const QString &address, quint16 port,
                         const QString &url, const QString &name,
                         const QString &shortName, const QString &labelsAddress, quint16 labelsPort,
                         const QString &keysAddress, quint16 keysPort) noexcept
    : inn_(inn)
    , address_(address.trimmed())
    , port_(port)
    , queryInterval_(0)
    , timerS_(12000)
    , url_(url)
    , name_(name)
    , shortName_(shortName)
    , labelsAddress_(labelsAddress)
    , labelsPort_(labelsPort)
    , keysAddress_(keysAddress)
    , keysPort_(keysPort)
    , testTimeout_(3000)
    , keysTimeout_(30000)

{

}

OfdSettings::OfdSettings() noexcept
    : inn_()
    , address_(QString(64, QLatin1Char(' ')))
    , port_(0)
    , queryInterval_(0)
    , timerS_(12000)
    , url_(QStringLiteral("http://URL/par"))
    , name_()
    , shortName_()
    , labelsAddress_()
    , labelsPort_(0)
    , keysAddress_()
    , keysPort_(0)
    , testTimeout_(3000)
    , keysTimeout_(30000)



{

}

OfdSettings::OfdSettings(const OfdSettings &other) noexcept
    : inn_(other.inn_)
    , address_(other.address_)
    , port_(other.port_)
    , queryInterval_(other.queryInterval_)
    , timerS_(other.timerS_)
    , url_(other.url_)
    , name_(other.name_)
    , shortName_(other.shortName_)
    , labelsAddress_(other.labelsAddress_)
    , labelsPort_(other.labelsPort_)
    , keysAddress_(other.keysAddress_)
    , keysPort_(other.keysPort_)
    , testTimeout_(other.testTimeout_)
    , keysTimeout_(other.keysTimeout_)


{

}

OfdSettings::OfdSettings(OfdSettings &&other) noexcept
    : inn_()
    , address_()
    , port_(other.port_)
    , queryInterval_(other.queryInterval_)
    , timerS_(other.timerS_)
    , url_()
    , name_()
    , shortName_()
    , labelsAddress_()
    , labelsPort_(other.labelsPort_)
    , keysAddress_()
    , keysPort_(other.keysPort_)
    , testTimeout_(other.testTimeout_)
    , keysTimeout_(other.keysTimeout_)


{
    inn_.swap(other.inn_);
    address_.swap(other.address_);
    url_.swap(other.url_);
    name_.swap(other.name_);
    shortName_.swap(other.shortName_);
    labelsAddress_.swap(other.labelsAddress_);
    keysAddress_.swap(other.keysAddress_);
}

OfdSettings::OfdSettings(const QVariantMap &other) noexcept
    : inn_()
    , address_()
    , port_()
    , queryInterval_()
    , timerS_(12000)
    , url_()
    , name_()
    , shortName_()
    , labelsAddress_()
    , labelsPort_(0)
    , keysAddress_()
    , keysPort_(0)

{
    OfdSettings::parseMap(other);
}

OfdSettings::~OfdSettings()
{

}

bool OfdSettings::isValid() const
{
    return !address_.isEmpty() && port_ > 0;
}

QString OfdSettings::address() const
{
    return address_.trimmed();
}

void OfdSettings::setAddress(const QString &address)
{
    address_ = address.trimmed();
}

quint16 OfdSettings::port() const
{
    return port_;
}

void OfdSettings::setPort(const quint16 &port)
{
    port_ = port;
}


quint32 OfdSettings::queryInterval() const
{
    return queryInterval_;
}

void OfdSettings::setQueryInterval(const quint32 &queryInterval)
{
    queryInterval_ = queryInterval;
    if(queryInterval_ > 60000)
    {
        queryInterval_ = 10000;
    }
}

quint32 OfdSettings::timerS() const
{
    return timerS_;
}

void OfdSettings::setTimerS(quint32 newTimerS)
{
    timerS_ = newTimerS;
    if(timerS_ < 12000) timerS_ = 12000;
    else if(timerS_ > 3600000) timerS_ = 3600000;
}

QString OfdSettings::url() const
{
    return url_.trimmed();
}

void OfdSettings::setUrl(const QString &url)
{
    url_ = url.trimmed();
}
QString OfdSettings::name() const
{
    return name_;
}

void OfdSettings::setName(const QString &name)
{
    name_ = name;
}

QString OfdSettings::inn() const
{
    return inn_;
}

void OfdSettings::setInn(const QString &inn)
{
    inn_ = inn.trimmed();
}


QString OfdSettings::shortName() const
{
    return shortName_.isEmpty() ? name_ : shortName_;
}

void OfdSettings::setShortName(const QString &shortName)
{
    shortName_ = shortName;
}

const QString &OfdSettings::labelsAddress() const
{
    return labelsAddress_;
}

void OfdSettings::setLabelsAddress(const QString &newLabelsAddress)
{
    labelsAddress_ = newLabelsAddress;
}

quint16 OfdSettings::labelsPort() const
{
    return labelsPort_;
}

void OfdSettings::setLabelsPort(quint16 newLabelsPort)
{
    labelsPort_ = newLabelsPort;
}

const QString &OfdSettings::keysAddress() const
{
    return keysAddress_;
}

void OfdSettings::setKeysAddress(const QString &newKeysAddress)
{
    keysAddress_ = newKeysAddress;
}

quint16 OfdSettings::keysPort() const
{
    return keysPort_;
}

void OfdSettings::setKeysPort(quint16 newKeysPort)
{
    keysPort_ = newKeysPort;
}


qint32 OfdSettings::testTimeout() const
{
    return testTimeout_> 1000 ?
                testTimeout_ : 3000;
}

void OfdSettings::setTestTimeout(qint32 newTestTimeout)
{
    testTimeout_ = newTestTimeout;
}

qint32 OfdSettings::keysTimeout() const
{
    return keysTimeout_ > 1000 ? keysTimeout_ : 30000;
}

void OfdSettings::setKeysTimeout(qint32 newKeysTimeout)
{
    keysTimeout_ = newKeysTimeout;
}



QVariantMap OfdSettings::toMap() const
{
    return {
        {INN, inn_.trimmed()},
        {ADDRESS, address_},
        {PORT, port_},
        {QUERYI_NTERVAL, queryInterval_},
        {TIMER_S, timerS_},
        {URL, url_},
        {NAME, name_},
        {SHORT_NAME, shortName_},
        {LABELS_ADDRESS, labelsAddress_},
        {LABELS_PORT, labelsPort_},
        {KEYS_ADDRESS, keysAddress_},
        {KEYS_PORT, keysPort_},
        {TEST_TIMEOUT, testTimeout_},
        {KEYS_TIMEOUT, keysTimeout_}
    };
}

void OfdSettings::parseMap(const QVariantMap &map)
{
    inn_ = map[INN].toString().trimmed();
    address_ = map[ADDRESS].toString();
    port_ = map[PORT].toUInt();
    setQueryInterval(map[QUERYI_NTERVAL].toUInt());
    setTimerS(map[TIMER_S].toUInt());
    url_ = map[URL].toString();
    name_ = map[NAME].toString();
    shortName_ = map[SHORT_NAME].toString();
    labelsAddress_ = map[LABELS_ADDRESS].toString().trimmed();
    labelsPort_ = map[LABELS_PORT].toUInt();
    keysAddress_ = map[KEYS_ADDRESS].toString();
    keysPort_ = map[KEYS_PORT].toUInt();
    testTimeout_ = map[TEST_TIMEOUT].toInt();
    keysTimeout_ = map[KEYS_TIMEOUT].toInt();
}

QString OfdSettings::section() const
{
    return SECTION;
}

QStringList OfdSettings::fields() const
{
    return {
        INN,
        ADDRESS,
        PORT,
        QUERYI_NTERVAL,
        TIMER_S,
        URL,
        NAME,
        SHORT_NAME,
        LABELS_ADDRESS,
        LABELS_PORT,
        KEYS_ADDRESS,
        KEYS_PORT,
        TEST_TIMEOUT,
        KEYS_TIMEOUT
    };
}

QMap<QString, QString> OfdSettings::toStringMap() const
{
    return {
        {INN, inn_.trimmed()},
        {ADDRESS, address_},
        {PORT, QString::number(port_)},
        {QUERYI_NTERVAL, QString::number(queryInterval_)},
        {TIMER_S, QString::number(timerS_)},
        {URL, url_},
        {NAME, name_},
        {SHORT_NAME, shortName_},
        {LABELS_ADDRESS, labelsAddress_},
        {LABELS_PORT, QString::number(labelsPort_)},
        {KEYS_ADDRESS, keysAddress_},
        {KEYS_PORT, QString::number(keysPort_)},
        {TEST_TIMEOUT, QString::number(testTimeout_)},
        {KEYS_TIMEOUT, QString::number(keysTimeout_)}
    };
}

void OfdSettings::parseStringMap(const QMap<QString, QString> &map)
{
    inn_ = map[INN].trimmed();
    address_ = map[ADDRESS].trimmed();
    port_ = map[PORT].toUInt();
    setQueryInterval(map[QUERYI_NTERVAL].toUInt());
    setTimerS(map[TIMER_S].toUInt());
    url_ = map[URL].trimmed();
    name_ = map[NAME].trimmed();
    shortName_ = map[SHORT_NAME].trimmed();
    labelsAddress_ = map[LABELS_ADDRESS].trimmed();
    labelsPort_ = map[LABELS_PORT].toUInt();
    keysAddress_ = map[KEYS_ADDRESS].trimmed();
    keysPort_ = map[KEYS_PORT].toUInt();
    testTimeout_ = map[TEST_TIMEOUT].toInt();
    keysTimeout_ = map[KEYS_TIMEOUT].toInt();
}

CoreApiResult OfdSettings::toSuccessResult() const
{
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), toMap());
}

void OfdSettings::parseTransactionParams(const QVariantMap &params)
{
    parseMap(params);
}

OfdSettings &OfdSettings::operator=(const OfdSettings &other) noexcept
{
    inn_ = other.inn_;
    address_ = other.address_;
    port_ = other.port_;
    queryInterval_ = other.queryInterval_;
    timerS_ = other.timerS_;
    url_ = other.url_;
    name_ = other.name_;
    shortName_ = other.shortName_;
    labelsAddress_ = other.labelsAddress_;
    labelsPort_ = other.labelsPort_;
    keysAddress_ = other.keysAddress_;
    keysPort_ = other.keysPort_;
    testTimeout_ = other.testTimeout_;
    keysTimeout_ = other.keysTimeout_;
    return *this;
}

OfdSettings &OfdSettings::operator=(OfdSettings &&other)noexcept
{
    inn_.swap(other.inn_);
    address_.swap(other.address_);
    port_ = other.port_;
    queryInterval_ = other.queryInterval_;
    timerS_ = other.timerS_;
    url_.swap(other.url_);
    name_.swap(other.name_);
    shortName_.swap(other.shortName_);
    labelsAddress_.swap(other.labelsAddress_);
    labelsPort_ = other.labelsPort_;
    keysAddress_.swap(other.keysAddress_);
    keysPort_ = other.keysPort_;
    testTimeout_ = other.testTimeout_;
    keysTimeout_ = other.keysTimeout_;

    return *this;

}

bool OfdSettings::operator==(const OfdSettings &other) const
{
    return	inn_ == other.inn_ &&
            address_ == other.address_ &&
            port_ == other.port_ &&
            queryInterval_ == other.queryInterval_ &&
            timerS_ == other.timerS_ &&
            url_ == other.url_ &&
            name_ == other.name_ &&
            shortName_ == other.shortName_ &&
            labelsAddress_ == other.labelsAddress_ &&
            labelsPort_ == other.labelsPort_ &&
            keysAddress_ == other.keysAddress_ &&
            keysPort_ == other.keysPort_ &&
            testTimeout_ == other.testTimeout_ &&
            keysTimeout_ == other.keysTimeout_;




}

bool OfdSettings::operator!=(const OfdSettings &other) const
{
    return !(*this == other);
}



//-----------------------------------------------------------------------------------------------------
