#include "receipt.h"

#include "formatutils.h"
#include "cashboxbuildconfig.h"


#include <algorithm>

Receipt::Receipt() noexcept
    : BaseWorkIncoming()
    , options_()
    , paymentAttr_(fdf::PaymentType::Invalid)
    , tax_(fdf::Tax::Invalid)
    , amount_()
    , cash_()
    , card_()
    , prepay_()
    , postpay_()
    , barter_()
    , vat20_()
    , vat10_()
    , vat0_()
    , vatNone_()
    , vat20_120_()
    , vat10_110_()
    , vat5_()
    , vat7_()
    , vat5_105_()
    , vat7_107_()
    , calculatedvat20_()
    , calculatedvat10_()
    , calculatedvat0_()
    , calculatedvatNone_()
    , calculatedvat20_120_()
    , calculatedvat10_110_()
    , calculatedvat5_()
    , calculatedvat7_()
    , calculatedvat5_105_()
    , calculatedvat7_107_()
    , userParam_()
    , buyerInfo_()
    , industryProperties_()
    , operationalParam_()
    , correctionReason_()
    , buyerPhone_()
    , boxId_()
    , operations_()
    , containsLabels_()
    , t2107_(255)
    , internetPayment_(false)
    , cardPayments_{}
{

}

Receipt::Receipt(const Receipt &other) noexcept
    : BaseWorkIncoming(other)
    , options_(other.options_)
    , paymentAttr_(other.paymentAttr_)
    , tax_(other.tax_)
    , amount_(other.amount_)
    , cash_(other.cash_)
    , card_(other.card_)
    , prepay_(other.prepay_)
    , postpay_(other.postpay_)
    , barter_(other.barter_)
    , vat20_(other.vat20_)
    , vat10_(other.vat10_)
    , vat0_(other.vat0_)
    , vatNone_(other.vatNone_)
    , vat20_120_(other.vat20_120_)
    , vat10_110_(other.vat10_110_)
    , vat5_(other.vat5_)
    , vat7_(other.vat7_)
    , vat5_105_(other.vat5_105_)
    , vat7_107_(other.vat7_107_)
    , userParam_(other.userParam_)
    , buyerInfo_(other.buyerInfo_)
    , industryProperties_(other.industryProperties_)
    , operationalParam_(other.operationalParam_)
    , correctionReason_(other.correctionReason_)
    , buyerPhone_(other.buyerPhone_)
    , boxId_(other.boxId_)
    , operations_(other.operations_)
    , containsLabels_(other.containsLabels_)
    , t2107_(other.t2107_)
    , internetPayment_(other.internetPayment_)
    , cardPayments_{other.cardPayments_}

{

}

Receipt::Receipt(Receipt &&other) noexcept
    : BaseWorkIncoming(other)
    , options_(other.options_)
    , paymentAttr_(other.paymentAttr_)
    , tax_(other.tax_)
    , amount_()
    , cash_()
    , card_()
    , prepay_()
    , postpay_()
    , barter_()
    , vat20_()
    , vat10_()
    , vat0_()
    , vatNone_()
    , vat20_120_()
    , vat10_110_()
    , vat5_()
    , vat7_()
    , vat5_105_()
    , vat7_107_()
    , userParam_()
    , buyerInfo_()
    , industryProperties_()
    , operationalParam_()
    , correctionReason_()
    , buyerPhone_()
    , boxId_()
    , operations_()
    , containsLabels_(other.containsLabels_)
    , t2107_(other.t2107_)
    , cardPayments_{}

{
    amount_.swap(other.amount_);
    cash_.swap(other.cash_);
    card_.swap(other.card_);
    prepay_.swap(other.prepay_);
    postpay_.swap(other.postpay_);
    barter_.swap(other.barter_);
    vat20_.swap(other.vat20_);
    vat10_.swap(other.vat10_);
    vat0_.swap(other.vat0_);
    vatNone_.swap(other.vatNone_);
    vat20_120_.swap(other.vat20_120_);
    vat10_110_.swap(other.vat10_110_);
    vat5_.swap(other.vat5_);
    vat7_.swap(other.vat7_);
    vat5_105_.swap(other.vat5_105_);
    vat7_107_.swap(other.vat7_107_);

    userParam_.swap(other.userParam_);
    buyerInfo_.swap(other.buyerInfo_);
    industryProperties_.swap(other.industryProperties_);
    operationalParam_.swap(other.operationalParam_);
    correctionReason_.swap(other.correctionReason_);
    buyerPhone_.swap(other.buyerPhone_);
    boxId_.swap(other.boxId_);
    operations_.swap(other.operations_);
    cardPayments_.swap(other.cardPayments_);
}

Receipt::Receipt(const QVariantMap &map) noexcept
    : BaseWorkIncoming()
    , options_()
    , paymentAttr_(fdf::PaymentType::Invalid)
    , tax_(fdf::Tax::Invalid)
    , amount_()
    , cash_()
    , card_()
    , prepay_()
    , postpay_()
    , barter_()
    , vat20_()
    , vat10_()
    , vat0_()
    , vatNone_()
    , vat20_120_()
    , vat10_110_()
    , vat5_()
    , vat7_()
    , vat5_105_()
    , vat7_107_()
    , userParam_()
    , buyerInfo_()
    , industryProperties_()
    , operationalParam_()
    , correctionReason_()
    , buyerPhone_()
    , boxId_()
    , operations_()
    , containsLabels_()
    , t2107_(255)
{
    Receipt::parseMap(map);
}

Receipt::~Receipt()
{

}

bool Receipt::isValid(const RegData &rd) const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(rd, err, msg);
}

bool Receipt::isValid(const RegData &rd, CoreApiConst::ErrorCode &err,
                      QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();

    if(!fdf::paymentTypeIsValid(paymentAttr_))
    {
        err = CoreApiConst::ErrorCode::InvalidParamValue;
        msg = QStringLiteral("Некорректное значение реквизита 'признак расчета' (1054)");
        return false;
    }

    if(operations_.isEmpty())
    {
        err = CoreApiConst::ErrorCode::NoAnyReceiptOperations;
        msg = QStringLiteral("Чек не содержит ни одного предмета расчета");
        return false;
    }
    if(!mayBeLabled() && std::any_of(operations_.constBegin(), operations_.constEnd(),
                                     [](const PSimpleReceiptOperation &o) -> bool{
                     return o.dynamicCast<ReceiptOperation>();
                 }))
    {
        err = CoreApiConst::ErrorCode::LabledProductsNotSupported;
        if(options_.testFlag(ReceiptOption::SupportLabledTrandes))
        {
            msg = QStringLiteral("Маркированные товары не поддерживаются в чеках расхода и возврата расхода");
        }
        else
        {
            msg = QStringLiteral("Маркированные товары не поддерживаются");
        }
        return false;
    }
    if(!std::all_of(operations_.constBegin(), operations_.constEnd(),
                    [&err, &msg](const PSimpleReceiptOperation &o) ->bool {
                    return o && o->isValid(err, msg);
    }))
    {
        if(err == CoreApiConst::ErrorCode::Ok) err = CoreApiConst::ErrorCode::InvalidItemData;
        return false;
    }
    if(!fdf::taxIsValid(tax_) || !rd.taxes().testFlag(tax_))
    {
        err = CoreApiConst::ErrorCode::InvalidTaxSystem;
        msg = QStringLiteral("Указана неправильная система налогообложения (тег 1055)");
        return false;
    }


    if(!rd.getAutoMode())
    {
//        lmWarning() << cashier_ << cashier_.isEmpty() << cashier_.size();
        if(!FormatUtils::checkString(cashier_, 256, false, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParamValue;
            msg = QStringLiteral("Реквизит 'Кассир' (1021) ") + msg;
            return false;
        }
        Inn cinn(cashierInn_);
        if(!cinn.isValid(true))
        {
            err = CoreApiConst::ErrorCode::InvalidParamValue;
            msg = QStringLiteral("Реквизит 'ИНН кассира' (1203) содержит некорректное значение: ")  + cashierInn_;
            return false;
        }
    }
    else
    {
        if(rd.boxId().isEmpty() &&
                !FormatUtils::checkString(boxId_, 20 , false, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParamValue;
            msg = QStringLiteral("Реквизит 'Номер автомата' (1036) ") + msg;
            return false;
        }
    }
    if(!rd.getOfflineMode())
    {
        if(!FormatUtils::checkPhone(buyerPhone_, 64, !internetPayment_, msg) && !
                FormatUtils::checkEmail(buyerPhone_, 64, !internetPayment_, msg) &&
            !(buyerPhone_.trimmed().toLower() == "none" && !internetPayment_))
        {
            err = CoreApiConst::ErrorCode::InvalidParamValue;
            msg = QStringLiteral("Реквизит 'телефон или электронный адрес покупателя' (1008) ") + msg;
            return false;
        }
    }

    if(!FormatUtils::checkString(address_, 256, false, !rd.address().isEmpty(), msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParamValue;
        msg = QStringLiteral("Реквизит 'Адресс расчетов' (1009) ") + msg;
        return false;
    }
    if(!FormatUtils::checkString(place_, 256, false, !rd.place().isEmpty() && !internetPayment_, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParamValue;
        msg = QStringLiteral("Реквизит 'Место расчетов' (1187) ") + msg;
        return false;
    }
    if(!FormatUtils::checkString(additionalParam_, 16, false, true, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParamValue;
        msg = QStringLiteral("Реквизит 'Дополнительный реквизит чека' (1192) ") + msg;
        return false;
    }
    if(isCorrection())
    {
        if(!correctionReason_.has_value())
        {
            err = CoreApiConst::ErrorCode::NotContainsCorrrectionReason;
            msg = QStringLiteral("Чек не содержит основания для коррекции (1174) ");
            return false;
        }
        if(!correctionReason_.value().isValid(err, msg))
        {
            return false;
        }
    }
    if(!checkAmount(err, msg) ||
       (userParam_.has_value() && !userParam_.value().isValid(false, err, msg)) ||
            (buyerInfo_.has_value() && !buyerInfo_.value().isValid(false, err, msg)) ||
            !checkIndustryCode(err, msg) ||
            (operationalParam_.has_value() && !operationalParam_.value().isValid(false, err, msg)))
    {
        return false;
    }

    if(!mayBeLabled() && containsLabels())
    {
        err= CoreApiConst::ErrorCode::LabelsNotSupported;
        return false;
    }
    if(cardPayments_.has_value() && !cardPayments_->isValid(err, msg))
    {
        return false;
    }
    return true;

}

bool Receipt::isRefund() const
{
    return paymentAttr_ == fdf::PaymentType::DebitReturn ||
           paymentAttr_ == fdf::PaymentType::CreditReturn;
}

bool Receipt::isCredit() const
{
    return paymentAttr_ == fdf::PaymentType::DebitReturn ||
           paymentAttr_ == fdf::PaymentType::Credit;
}

bool Receipt::mayBeLabled() const
{
    return !isFosa() && (paymentAttr_ == fdf::PaymentType::Debit ||
                         paymentAttr_ == fdf::PaymentType::DebitReturn) &&
            supportLabledTrandes();
}

bool Receipt::containsLabels() const
{
    if(containsLabels_.has_value()) return containsLabels_.value();
    containsLabels_ = std::make_optional(mayBeLabled() &&
            std::any_of(operations_.constBegin(), operations_.constEnd(),
                        [](const PSimpleReceiptOperation &o) -> bool{
        return o.dynamicCast<ReceiptOperation>();
    }));
    return containsLabels_.value();
}

const Receipt::Options &Receipt::options() const
{
    return options_;
}

void Receipt::setOptions(const Options &newOptions)
{
    options_ = newOptions;
}

bool Receipt::hasOption(Option o) const
{
    return options_.testFlag(o);
}

void Receipt::setOption(Option o, bool set)
{
    options_.setFlag(o, set);
}

bool Receipt::isCorrection() const
{
    return options_.testFlag(Option::IsCorrection);
}

void Receipt::setCorrection(bool set)
{
    options_.setFlag(Option::IsCorrection, set);
}

bool Receipt::isFosa() const
{
    return options_.testFlag(Option::IsFosa);
}

void Receipt::setFosa(bool set)
{
    options_.setFlag(Option::IsFosa, set);
}

bool Receipt::internatCashbox() const
{
    return options_.testFlag(Option::InternetCashbox);
}

void Receipt::setInternetCashbox(bool set)
{
    options_.setFlag(Option::InternetCashbox, set);
}

bool Receipt::supportLabledTrandes() const
{
    return options_.testFlag(Option::SupportLabledTrandes);
}

void Receipt::setSupportLabledTrandes(bool set)
{
    options_.setFlag(Option::SupportLabledTrandes, set);
}

bool Receipt::supportExciseTrandes() const
{
    return options_.testFlag(Option::SupportExciseTrades);
}

void Receipt::setSupportExciseTrandes(bool set)
{
    options_.setFlag(Option::SupportExciseTrades, set);
}

fdf::DocType Receipt::needDocType() const
{
    if(hasOption(Option::IsFosa))
    {
        return hasOption(Option::IsCorrection) ? fdf::DocType::FosaCorrection : fdf::DocType::Fosa;
    }
    else
    {
        return hasOption(Option::IsCorrection) ? fdf::DocType::Correction : fdf::DocType::Receipt;
    }
}

fdf::PaymentType Receipt::paymentAttr() const
{
    return paymentAttr_;
}

void Receipt::setPaymentAttr(fdf::PaymentType newPaymentAttr)
{
    paymentAttr_ = newPaymentAttr;
}

fdf::Tax Receipt::tax() const
{
    return tax_;
}

void Receipt::setTax(fdf::Tax newTax)
{
    tax_ = newTax;
}

FixNumber Receipt::amount() const
{
    if(amount_.has_value()) return amount_.value();
    return calculatedAmount();
}

FixNumber Receipt::calculatedAmount() const
{
    FixNumber res = FixNumber(DEFAULT_AMOUNT_MULT, 0ll);
    return std::accumulate(operations_.constBegin(), operations_.constEnd(), res,
                    [](const FixNumber &sum, const PSimpleReceiptOperation &o) ->FixNumber {
        return sum + o->cost();
    });
}

void Receipt::setAmount(std::optional<FixNumber> newAmount)
{
    amount_ = newAmount;
}

FixNumber Receipt::change() const
{
    FixNumber res (DEFAULT_AMOUNT_MULT, 0ll);
    if(paymentAttr_ == fdf::PaymentType::Debit && cash_.has_value())
    {
        FixNumber notCash = card_.value_or(res) + prepay_.value_or(res) +
                postpay_.value_or(res) + barter_.value_or(res);
        FixNumber a = amount();
        if(a <= notCash) res = cash_.value();
        else res = cash_.value() + notCash - a;
    }
    return res;
}

std::optional<FixNumber> Receipt::cash() const
{
    return cash_;
}

void Receipt::setCash(std::optional<FixNumber> newCash)
{
    cash_ = newCash;
}

void Receipt::setCash(const FixNumber cash)
{
    if(cash.realMultiplier() != DEFAULT_AMOUNT_MULT || cash.value() <= 0ll)
    {
        cash_.reset();
    }
    else
    {
        cash_ = cash;
    }
}

std::optional<FixNumber> Receipt::card() const
{
    return card_;
}

void Receipt::setCard(std::optional<FixNumber> newCard)
{
    card_ = newCard;
}

void Receipt::setCard(const FixNumber &card)
{
    if(card.realMultiplier() != DEFAULT_AMOUNT_MULT || card.value() <= 0ll)
    {
        card_.reset();
    }
    else
    {
        card_ = card;
    }

}

std::optional<FixNumber> Receipt::prepay() const
{
    return prepay_;
}

void Receipt::setPrepay(std::optional<FixNumber> newPrepay)
{
    prepay_ = newPrepay;
}

void Receipt::setPrepay(const FixNumber &prepay)
{
    if(prepay.realMultiplier() != DEFAULT_AMOUNT_MULT || prepay.value() <= 0ll)
    {
        prepay_.reset();
    }
    else
    {
        prepay_ = prepay;
    }

}

std::optional<FixNumber> Receipt::postpay() const
{
    return postpay_;
}

void Receipt::setPostpay(std::optional<FixNumber> newPostpay)
{
    postpay_ = newPostpay;
}

void Receipt::setPostpay(const FixNumber &postpay)
{
    if(postpay.realMultiplier() != DEFAULT_AMOUNT_MULT || postpay.value() <= 0ll)
    {
        postpay_.reset();
    }
    else
    {
        postpay_ = postpay;
    }

}

std::optional<FixNumber> Receipt::barter() const
{
    return barter_;
}

void Receipt::setBarter(std::optional<FixNumber> newBarter)
{
    barter_ = newBarter;
}

void Receipt::setBarter(const FixNumber &barter)
{
    if(barter.realMultiplier() != DEFAULT_AMOUNT_MULT || barter.value() <= 0ll)
    {
        barter_.reset();
    }
    else
    {
        barter_ = barter;
    }

}

std::optional<FixNumber> Receipt::vat20() const
{
    return vat20_.has_value() ? vat20_ : calculatedvat20_;
}

void Receipt::setVat20(std::optional<FixNumber> newVat20)
{
    vat20_ = newVat20;
}

std::optional<FixNumber> Receipt::vat10() const
{
    return vat10_.has_value() ? vat10_ : calculatedvat10_;;
}

void Receipt::setVat10(std::optional<FixNumber> newVat10)
{
    vat10_ = newVat10;
}

std::optional<FixNumber> Receipt::vat0() const
{
    return vat0_.has_value() ? vat0_ : calculatedvat0_;;
}

void Receipt::setVat0(std::optional<FixNumber> newVat0)
{
    vat0_ = newVat0;
}

std::optional<FixNumber> Receipt::vatNone() const
{
    return vatNone_.has_value() ? vatNone_ : calculatedvatNone_;
}

void Receipt::setVatNone(std::optional<FixNumber> newVatNone)
{
    vatNone_ = newVatNone;
}

std::optional<FixNumber> Receipt::vat20_120() const
{
    return vat20_120_.has_value() ? vat20_120_ : calculatedvat20_120_;
}

void Receipt::setVat20_120(std::optional<FixNumber> newVat20_120)
{
    vat20_120_ = newVat20_120;
}

std::optional<FixNumber> Receipt::vat10_110() const
{
    return vat10_110_.has_value() ? vat10_110_ : calculatedvat10_110_;
}

void Receipt::setVat10_110(std::optional<FixNumber> newVat10_110)
{
    vat10_110_ = newVat10_110;
}

std::optional<FixNumber> Receipt::vat5() const
{
    return vat5_.has_value() ? vat5_ : calculatedvat5_;
}

void Receipt::setVat5(std::optional<FixNumber> newVat5)
{
    vat5_ = newVat5;
}

std::optional<FixNumber> Receipt::vat7() const
{
    return vat7_.has_value() ? vat7_ : calculatedvat7_;
}

void Receipt::setVat7(std::optional<FixNumber> newVat7)
{
    vat7_ = newVat7;
}

std::optional<FixNumber> Receipt::vat5_105() const
{
    return vat5_105_.has_value() ? vat5_105_ : calculatedvat5_105_;
}

void Receipt::setVat5_105(std::optional<FixNumber> newVat5_105)
{
    vat5_105_ = newVat5_105;
}

std::optional<FixNumber> Receipt::vat7_107() const
{
    return vat7_107_.has_value() ? vat7_107_ : calculatedvat7_107_;
}

void Receipt::setVat7_107(std::optional<FixNumber> newVat7_107)
{
    vat7_107_ = newVat7_107;
}

std::optional<ReceiptAdditionalUserParam> Receipt::userParam() const
{
    return userParam_;
}

void Receipt::setUserParam(std::optional<ReceiptAdditionalUserParam> newUserParam)
{
    userParam_ = newUserParam;
}

std::optional<BuyerInfo> Receipt::buyerInfo() const
{
    return buyerInfo_;
}

void Receipt::setBuyerInfo(std::optional<BuyerInfo> newBuyerInfo)
{
    buyerInfo_ = newBuyerInfo;
}

const QList<IndustryProperty> &Receipt::industryProperties() const
{
    return industryProperties_;
}

void Receipt::setIndustryProperties(const QList<IndustryProperty> &newIndustryProperties)
{
    industryProperties_ = newIndustryProperties;
}

std::optional<ReceiptOperationalParam> Receipt::operationalParam() const
{
    return operationalParam_;
}

void Receipt::setOperationalParam(std::optional<ReceiptOperationalParam> newOperationalParam)
{
    operationalParam_ = newOperationalParam;
}

std::optional<CorrectionReason> Receipt::correctionReason() const
{
    return correctionReason_;
}

void Receipt::setCorrectionReason(std::optional<CorrectionReason> newCorrectionReason)
{
    correctionReason_ = newCorrectionReason;
}

const QString &Receipt::buyerPhone() const
{
    return buyerPhone_;
}

void Receipt::setBuyerPhone(const QString &newBuyerPhone)
{
    buyerPhone_ = newBuyerPhone;
}

const QString &Receipt::boxId() const
{
    return boxId_;
}

void Receipt::setBoxId(const QString &newBoxId)
{
    boxId_ = STR4FS_(newBoxId);
}

const PSimpleReceiptOperations &Receipt::operations() const
{
    return operations_;
}

void Receipt::setOperations(const PSimpleReceiptOperations &newOperations)
{
    containsLabels_.reset();
    operations_ = newOperations;
}

SimpleReceiptOperations Receipt::simpleOperations() const
{
    SimpleReceiptOperations res;
    return std::accumulate(operations_.constBegin(), operations_.constEnd(), res,
                           [](const SimpleReceiptOperations &list, const PSimpleReceiptOperation &o)
                           -> SimpleReceiptOperations {
            SimpleReceiptOperations rlist = list;
            if(!o.dynamicCast<ReceiptOperation>())  rlist << *o;
            return rlist;
    });
}

ReceiptOperations Receipt::labledOperattions(bool includeOisn, bool includeSimple) const
{
    ReceiptOperations res;
    return std::accumulate(operations_.constBegin(), operations_.constEnd(), res,
                           [includeOisn, includeSimple](const ReceiptOperations &list,
                                                        const PSimpleReceiptOperation &o)
                           -> ReceiptOperations {
            ReceiptOperations rlist = list;
            if(o.dynamicCast<ReceiptOperation>() &&
                    ((includeOisn && o.dynamicCast<ReceiptOperation>()->isLabled()) ||
                     (includeSimple && !o.dynamicCast<ReceiptOperation>()->isLabled())))
            {
                rlist << *(o.dynamicCast<ReceiptOperation>());
            }
            return rlist;
    });
}

QVariantMap Receipt::toMap() const
{
    QVariantMap res = BaseWorkIncoming::toMap();
    res["options"] = static_cast<qint32>(options_);
    res["paymentAttr"] = static_cast<qint32>(paymentAttr_);
    res["tax"] = static_cast<qint32>(tax_);
    if(amount_   .has_value())res["amount"]    = amount_   .value().toMap();
    if(cash_     .has_value())res["cash"]      = cash_     .value().toMap();
    if(card_     .has_value())res["card"]      = card_     .value().toMap();
    if(prepay_   .has_value())res["prepay"]    = prepay_   .value().toMap();
    if(postpay_  .has_value())res["postpay"]   = postpay_  .value().toMap();
    if(barter_   .has_value())res["barter"]    = barter_   .value().toMap();
    if(vat20_    .has_value())res["vat20"]     = vat20_    .value().toMap();
    if(vat10_    .has_value())res["vat10"]     = vat10_    .value().toMap();
    if(vat0_     .has_value())res["vat0"]      = vat0_     .value().toMap();
    if(vatNone_  .has_value())res["vatNone"]   = vatNone_  .value().toMap();
    if(vat20_120_.has_value())res["vat20_120"] = vat20_120_.value().toMap();
    if(vat10_110_.has_value())res["vat10_110"] = vat10_110_.value().toMap();
    if(vat5_     .has_value())res["vat5"]      = vat5_     .value().toMap();
    if(vat7_     .has_value())res["vat7"]      = vat7_     .value().toMap();
    if(vat5_105_ .has_value())res["vat5_105"]  = vat5_105_ .value().toMap();
    if(vat7_107_ .has_value())res["vat7_107"]  = vat7_107_ .value().toMap();

    if(userParam_       .has_value())res["userParam"] = userParam_.value().toMap();
    if(buyerInfo_       .has_value())res["buyerInfo"] = buyerInfo_.value().toMap();
    if(operationalParam_.has_value())res["operationalParam"] = operationalParam_.value().toMap();
    if(correctionReason_.has_value())res["correctionReason"] = correctionReason_.value().toMap();
    if(!industryProperties_.isEmpty())
    {
        QVariantList list;
        for(const IndustryProperty &p :industryProperties_)
        {
            list << p.toMap();
        }
        res["industryProperties"] = list;
    }
    if(!buyerPhone_.isEmpty())res["buyerPhone"] = buyerPhone_;
    if(!boxId_     .isEmpty())res["boxId"]      = boxId_     ;
    if(!operations_.isEmpty())
    {
        QVariantList list;
        QVariantList labledList;
        for(const PSimpleReceiptOperation &o :operations_)
        {
            if(o.dynamicCast<ReceiptOperation>()) labledList << o->toMap();
            else list << o->toMap();

        }
        if(!list.isEmpty())res["operations"] = list;
        if(!labledList.isEmpty())res["labledOperations"] = labledList;
    }
    if(internetPayment_) res["internetPayment"] = 1;
    if(cardPayments_.has_value()) res["cardPayments"] = cardPayments_->toList();
    return res;
}

QVariantMap Receipt::toExternalMap() const
{
    QVariantMap res = BaseWorkIncoming::toMap();
//    res["options"] = static_cast<qint32>(options_);
    res["paymentAttr"] = static_cast<qint32>(paymentAttr_);
    res["tax"] = static_cast<qint32>(tax_);
    if(amount_   .has_value())res["amount"]    = amount_   .value().toString();
    if(cash_     .has_value())res["cash"]      = cash_     .value().toString();
    if(card_     .has_value())res["card"]      = card_     .value().toString();
    if(prepay_   .has_value())res["prepay"]    = prepay_   .value().toString();
    if(postpay_  .has_value())res["postpay"]   = postpay_  .value().toString();
    if(barter_   .has_value())res["barter"]    = barter_   .value().toString();
    if(vat20_    .has_value())res["vat20"]     = vat20_    .value().toString();
    if(vat10_    .has_value())res["vat10"]     = vat10_    .value().toString();
    if(vat0_     .has_value())res["vat0"]      = vat0_     .value().toString();
    if(vatNone_  .has_value())res["vatNone"]   = vatNone_  .value().toString();
    if(vat20_120_.has_value())res["vat20_120"] = vat20_120_.value().toString();
    if(vat10_110_.has_value())res["vat10_110"] = vat10_110_.value().toString();
    if(vat5_     .has_value())res["vat5"]      = vat5_     .value().toString();
    if(vat7_     .has_value())res["vat7"]      = vat7_     .value().toString();
    if(vat5_105_ .has_value())res["vat5_105"]  = vat5_105_ .value().toString();
    if(vat7_107_ .has_value())res["vat7_107"]  = vat7_107_ .value().toString();

    if(userParam_       .has_value())res["userParam"] = userParam_.value().toMap();
    if(buyerInfo_       .has_value())res["buyerInfo"] = buyerInfo_.value().toExternalMap();
    if(operationalParam_.has_value())res["operationalParam"] = operationalParam_.value().toMap();
    if(correctionReason_.has_value())res["correctionReason"] = correctionReason_.value().toMap();
    if(!industryProperties_.isEmpty())
    {
        QVariantList list;
        for(const IndustryProperty &p :industryProperties_)
        {
            list << p.toMap();
        }
        res["industryProperties"] = list;
    }
    if(!buyerPhone_.isEmpty())res["buyerPhone"] = buyerPhone_;
    if(!boxId_     .isEmpty())res["boxId"]      = boxId_     ;
    if(!operations_.isEmpty())
    {
        QVariantList list;
        QVariantList labledList;
        for(const PSimpleReceiptOperation &o :operations_)
        {
            if(o.dynamicCast<ReceiptOperation>()) labledList << o->toExternalMap();
            else list << o->toExternalMap();

        }
        if(!list.isEmpty())res["operations"] = list;
        if(!labledList.isEmpty())res["labledOperations"] = labledList;
    }
    if(internetPayment_) res["internetPayment"] = true;
    if(cardPayments_.has_value()) res["cardPayments"] = cardPayments_->toExtList();
    return res;
}
void Receipt::parseMap(const QVariantMap &map)
{

    Receipt::clean();
    BaseWorkIncoming::parseMap(map);

    options_ = Options(map["options"].toInt());
    paymentAttr_ = static_cast<fdf::PaymentType>(map["paymentAttr"].toInt());
    tax_ = static_cast<fdf::Tax>(map["tax"].toInt());
    if(map.contains("amount")   )amount_   = FormatUtils::parseSumm("amount"   , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("cash")     )cash_     = FormatUtils::parseSumm("cash"     , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("card")     )card_     = FormatUtils::parseSumm("card"     , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("prepay")   )prepay_   = FormatUtils::parseSumm("prepay"   , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("postpay")  )postpay_  = FormatUtils::parseSumm("postpay"  , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("barter")   )barter_   = FormatUtils::parseSumm("barter"   , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat20")    )vat20_    = FormatUtils::parseSumm("vat20"    , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat10")    )vat10_    = FormatUtils::parseSumm("vat10"    , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat0")     )vat0_     = FormatUtils::parseSumm("vat0"     , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vatNone")  )vatNone_  = FormatUtils::parseSumm("vatNone"  , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat20_120"))vat20_120_= FormatUtils::parseSumm("vat20_120", map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat10_110"))vat10_110_= FormatUtils::parseSumm("vat10_110", map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat5_")    )vat5_     = FormatUtils::parseSumm("vat5"     , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat7_")    )vat7_     = FormatUtils::parseSumm("vat7"     , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat5_105") )vat5_105_ = FormatUtils::parseSumm("vat5_105" , map, DEFAULT_AMOUNT_DEC);
    if(map.contains("vat7_107_"))vat7_107_ = FormatUtils::parseSumm("vat7_107" , map, DEFAULT_AMOUNT_DEC);



    if(map.contains("userParam"       ) && !map["userParam"       ].toMap().isEmpty())userParam_        = ReceiptAdditionalUserParam(map["userParam"       ].toMap());
    if(map.contains("buyerInfo"       ) && !map["buyerInfo"       ].toMap().isEmpty())buyerInfo_        = BuyerInfo(map["buyerInfo"       ].toMap());
    if(map.contains("operationalParam") && !map["operationalParam"].toMap().isEmpty())operationalParam_ = ReceiptOperationalParam(map["operationalParam"].toMap());
    if(map.contains("correctionReason") && !map["correctionReason"].toMap().isEmpty())correctionReason_ = CorrectionReason(map["correctionReason"].toMap());
    if(map.contains("buyerPhone"))buyerPhone_ = map["buyerPhone"].toString().trimmed();
    if(map.contains("boxId"     ))boxId_      = STR4FS_(map["boxId"     ].toString().trimmed());
    if(map.contains("industryProperties"))
    {
        QVariantList list = map["industryProperties"].toList();
        for(const QVariant &v :list) if(!v.toMap().isEmpty())industryProperties_ << IndustryProperty(v.toMap());
    }
    if(map.contains("labledOperations"))
    {
        QVariantList list = map["labledOperations"].toList();
        for(const QVariant &v :list) if(!v.toMap().isEmpty())operations_ << PSimpleReceiptOperation(new ReceiptOperation(v.toMap()));
    }
    if(map.contains("operations"))
    {
        QVariantList list = map["operations"].toList();
        for(const QVariant &v :list)if(!v.toMap().isEmpty()) operations_ << PSimpleReceiptOperation(new SimpleReceiptOperation(v.toMap()));
    }
    internetPayment_ = map.contains("internetPayment") && map["internetPayment"].toBool();
    cardPayments_.reset();
    if(map.contains("cardPayments"))
    {
        QVariantList list = map["cardPayments"].toList();
        if(!list.isEmpty())
        {
            CardPaymentsInfo i;
            i.parseList(list);
            cardPayments_ = i;
        }
    }

    calcAmounts();
}

Tlv::Stlv Receipt::toTlv(const RegData &rd) const
{
    Tlv::Stlv res;
    Tlv buf;

    for(const PSimpleReceiptOperation &o :operations_)
    {
        if(o.dynamicCast<ReceiptOperation>()) continue;
        res << o->to1059();
    }

    buf.setByte(fdf::Tag::TaxSystem, static_cast<quint8>(tax_));
    res << buf;

    FixNumber cash = cash_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) ;
    FixNumber coins = change();
    if(cash.value() - coins.value() >= 0)
    {
        cash = cash - coins;
    }

    buf.setVln(fdf::Tag::ReceiptCash, cash.value());
    res << buf;
    buf.setVln(fdf::Tag::ReceiptCard, card_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)).value());
    res << buf;
    buf.setVln(fdf::Tag::ReceiptPrepaymentAmount, prepay_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)).value());
    res << buf;
    buf.setVln(fdf::Tag::ReceiptPostpaymentAmount, postpay_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)).value());
    res << buf;
    buf.setVln(fdf::Tag::ReceiptBarterAmount, barter_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)).value());
    res << buf;

    bool hasOldVat = false;
    if(vat20().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVat20Amount, vat20().value().value());
        res << buf;
        hasOldVat = true;
    }
    if(vat10().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVat10Amount, vat10().value().value());
        res << buf;
        hasOldVat = true;
    }
    if(vat0().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVat0Amount, vat0().value().value());
        res << buf;
        hasOldVat = true;
    }
    if(vatNone().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVatNoneAmount, vatNone().value().value());
        res << buf;
        hasOldVat = true;
    }
    if(vat20_120().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVat20_120Amount, vat20_120().value().value());
        res << buf;
        hasOldVat = true;
    }
    if(vat10_110().has_value())
    {
        buf.setVln(fdf::Tag::ReceiptVat10_110Amount, vat10_110().value().value());
        res << buf;
        hasOldVat = true;
    }

    ReceiptsVatAmounts newVats;
    if(vat5().has_value())
    {
        newVats.appendVat(fdf::VatRate::Vat5, vat5().value());
    }
    if(vat7().has_value())
    {
        newVats.appendVat(fdf::VatRate::Vat7, vat7().value());
    }
    if(vat5_105().has_value())
    {
        newVats.appendVat(fdf::VatRate::Vat5_105, vat5_105().value());
    }
    if(vat7_107().has_value())
    {
        newVats.appendVat(fdf::VatRate::Vat7_107, vat7_107().value());
    }
    if(newVats.isValid())
    {
        if(!hasOldVat)
        {
            buf.setVln(fdf::Tag::ReceiptVat0Amount, FixNumber(DEFAULT_AMOUNT_MULT, 0ll).value());
            res << buf;
        }
        Tlv t = newVats.toTlv();
        if(t.tag() == fdf::Tag::VatAmounts) res << t;
        else {
            //INCORRECT VAT TAGGING
        }
    }

    QString address = address_.isEmpty() ? rd.address() : address_;
    QString place = place_.isEmpty() ? rd.place() : place_;
    if(!address.isEmpty())
    {
        buf.setString(fdf::Tag::Address, address);
        res << buf;
    }
    if(!place.isEmpty() )
    {
        buf.setString(fdf::Tag::Place, place);
        res << buf;
    }

    if(rd.ownerName().isEmpty())
    {
        buf.setString(fdf::Tag::OwnerName, ownerName_);
        res << buf;
    }
    if(!cashier_.isEmpty() && !rd.getAutoMode())
    {
        buf.setString(fdf::Tag::Cashier, cashier_);
        res << buf;
        Inn inn(cashierInn_);
        if(inn.isValid())
        {
            buf.setInn(fdf::Tag::CashierInn, inn);
            res << buf;
        }
    }
    if(!rd.getOfflineMode())
    {
        if(!buyerPhone_.isEmpty())
        {
            buf.setString(fdf::Tag::ClientPhone, buyerPhone_);
            res << buf;
//OwnerEmail, видимо, берется из регистрации
            if(buyerPhone_.trimmed().toLower() == "none")
            {
                buf.setString(fdf::Tag::OwnerEmail, "none");
            }
            else
            {
                buf.setString(fdf::Tag::OwnerEmail, rd.email());
            }
            res << buf;
        }
    }
    if(rd.getAutoMode() && rd.boxId().isEmpty())
    {
        buf.setString(fdf::Tag::BoxId, boxId_);
        res << buf;
    }
    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::ReceiptAdditionalParam, additionalParam_);
        res << buf;
    }
    if(userParam_.has_value())
    {
        res << userParam_.value().toTlv();
    }
    if(operationalParam_.has_value())
    {
        res << operationalParam_.value().toTlv();
    }
    if(buyerInfo_.has_value())
    {
        res << buyerInfo_.value().toTlv();
    }
    for(const IndustryProperty &p : industryProperties_)
    {
        res << p.toTlv(fdf::Tag::IndustryReceiptParam);
    }
    if(containsLabels())
    {

    }
    if(isCorrection() && correctionReason_.has_value())
    {
        res << correctionReason_.value().toTlv();

        buf.setBool(fdf::Tag::CorrectionType, !correctionReason_->isIndependent());
        res << buf;
    }
    if(t2107_ < 2)
    {
        buf.setByte(fdf::Tag::LabelProductsCheckresults, t2107_);
        res << buf;
    }
    if(internetPayment_)
    {
        buf.setBool(fdf::Tag::CashboxForInternetFlag, internetPayment_);
        res << buf;
    }
    if(cardPayments_.has_value() && cardPayments_->isValid())
    {
        res << cardPayments_->toTlv();
    }
    return res;
}

Tlv::Stlv Receipt::toTlvB7_3(const RegData &rd, quint32 fd) const
{
    Tlv::Stlv res;
    Tlv buf;
    for(const IndustryProperty &p :industryProperties_)
    {
        res << p.toTlv(fdf::Tag::IndustryReceiptParam);
    }
    if(buyerInfo_.has_value())
    {
        Inn i(buyerInfo_.value().inn());
        if(i.isValid())
        {
            buf.setInn(fdf::Tag::ClientInn, i);
            res << buf;
        }
    }
    buf.setByte(fdf::Tag::TaxSystem, static_cast<quint8>(tax_));
    res << buf;
    buf.setString(fdf::Tag::Address, address_.isEmpty() ? rd.address() : address_);
    res << buf;
    buf.setUInt32(fdf::Tag::FiscalDocNumberForNotification, fd);
    res << buf;
    quint8 tz =QDateTime::currentDateTime().offsetFromUtc() / 3600 - 1;
    if(tz > 11 || tz < 1) tz = 2;
    buf.setByte(fdf::Tag::TimeZone, tz);
    res << buf;
    return res;
}

void Receipt::clean()
{
    BaseWorkIncoming::clean();
    options_ = Options();
    paymentAttr_ = fdf::PaymentType::Invalid;
    tax_ = fdf::Tax::Invalid;
    amount_.reset();
    cash_.reset();
    card_.reset();
    prepay_.reset();
    postpay_.reset();
    barter_.reset();
    vat20_.reset();
    vat10_.reset();
    vat0_.reset();
    vatNone_.reset();
    vat20_120_.reset();
    vat10_110_.reset();
    vat5_.reset();
    vat7_.reset();
    vat5_105_.reset();
    vat7_107_.reset();
    calculatedvat20_.reset();
    calculatedvat10_.reset();
    calculatedvat0_.reset();
    calculatedvatNone_.reset();
    calculatedvat20_120_.reset();
    calculatedvat10_110_.reset();
    calculatedvat5_.reset();
    calculatedvat7_.reset();
    calculatedvat5_105_.reset();
    calculatedvat7_107_.reset();
    userParam_.reset();
    buyerInfo_.reset();
    industryProperties_.clear();
    operationalParam_.reset();
    correctionReason_.reset();
    buyerPhone_.clear();
    boxId_.clear();
    operations_.clear();
    containsLabels_.reset();
    t2107_ = 255;
    internetPayment_ = false;
    cardPayments_.reset();
}

Receipt &Receipt::operator =(const Receipt &other) noexcept
{
    BaseWorkIncoming::operator =(other);
    options_ = other.options_;
    paymentAttr_ = other.paymentAttr_;
    tax_ = other.tax_;
    amount_ = other.amount_;
    cash_ = other.cash_;
    card_ = other.card_;
    prepay_ = other.prepay_;
    postpay_ = other.postpay_;
    barter_ = other.barter_;
    vat20_ = other.vat20_;
    vat10_ = other.vat10_;
    vat0_ = other.vat0_;
    vatNone_ = other.vatNone_;
    vat20_120_ = other.vat20_120_;
    vat10_110_ = other.vat10_110_;
    vat5_ = other.vat5_;
    vat7_ = other.vat7_;
    vat5_105_ = other.vat5_105_;
    vat7_107_ = other.vat7_107_;
    calculatedvat20_ = other.calculatedvat20_;
    calculatedvat10_ = other.calculatedvat10_;
    calculatedvat0_ = other.calculatedvat0_;
    calculatedvatNone_ = other.calculatedvatNone_;
    calculatedvat20_120_ = other.calculatedvat20_120_;
    calculatedvat10_110_ = other.calculatedvat10_110_;
    calculatedvat5_ = other.calculatedvat5_;
    calculatedvat7_ = other.calculatedvat7_;
    calculatedvat5_105_ = other.calculatedvat5_105_;
    calculatedvat7_107_ = other.calculatedvat7_107_;
    userParam_ = other.userParam_;
    buyerInfo_ = other.buyerInfo_;
    industryProperties_ = other.industryProperties_;
    operationalParam_ = other.operationalParam_;
    correctionReason_ = other.correctionReason_;
    buyerPhone_ = other.buyerPhone_;
    boxId_ = other.boxId_;
    operations_ = other.operations_;
    containsLabels_ = other.containsLabels_;
    t2107_ = other.t2107_;
    internetPayment_ = other.internetPayment_;
    cardPayments_ = other.cardPayments_;
    return *this;
}

Receipt &Receipt::operator =(Receipt &&other) noexcept
{
    BaseWorkIncoming::operator =(other);
    options_ = other.options_;
    paymentAttr_ = other.paymentAttr_;
    tax_ = other.tax_;
    amount_.swap(other.amount_);
    cash_.swap(other.cash_);
    card_.swap(other.card_);
    prepay_.swap(other.prepay_);
    postpay_.swap(other.postpay_);
    barter_.swap(other.barter_);
    vat20_.swap(other.vat20_);
    vat10_.swap(other.vat10_);
    vat0_.swap(other.vat0_);
    vatNone_.swap(other.vatNone_);
    vat20_120_.swap(other.vat20_120_);
    vat10_110_.swap(other.vat10_110_);
    vat5_.swap(other.vat5_);
    vat7_.swap(other.vat7_);
    vat5_105_.swap(other.vat5_105_);
    vat7_107_.swap(other.vat7_107_);
    calculatedvat20_.swap(other.calculatedvat20_);
    calculatedvat10_.swap(other.calculatedvat10_);
    calculatedvat0_.swap(other.calculatedvat0_);
    calculatedvatNone_.swap(other.calculatedvatNone_);
    calculatedvat20_120_.swap(other.calculatedvat20_120_);
    calculatedvat10_110_.swap(other.calculatedvat10_110_);
    calculatedvat5_.swap(other.calculatedvat5_);
    calculatedvat7_.swap(other.calculatedvat7_);
    calculatedvat5_105_.swap(other.calculatedvat5_105_);
    calculatedvat7_107_.swap(other.calculatedvat7_107_);
    userParam_.swap(other.userParam_);
    buyerInfo_.swap(other.buyerInfo_);
    industryProperties_.swap(other.industryProperties_);
    operationalParam_.swap(other.operationalParam_);
    correctionReason_.swap(other.correctionReason_);
    buyerPhone_.swap(other.buyerPhone_);
    boxId_.swap(other.boxId_);
    operations_.swap(other.operations_);
    containsLabels_.swap(other.containsLabels_);
    t2107_ = other.t2107_;
    internetPayment_ = other.internetPayment_;
    cardPayments_.swap(other.cardPayments_);
    return *this;
}

bool Receipt::operator ==(const Receipt &other) const noexcept
{
    return options_ == other.options_ &&
            paymentAttr_ == other.paymentAttr_ &&
            tax_ == other.tax_ &&
            amount_ == other.amount_ &&
            cash_ == other.cash_ &&
            card_ == other.card_ &&
            prepay_ == other.prepay_ &&
            postpay_ == other.postpay_ &&
            barter_ == other.barter_ &&
            vat20_ == other.vat20_ &&
            vat10_ == other.vat10_ &&
            vat0_ == other.vat0_ &&
            vatNone_ == other.vatNone_ &&
            vat20_120_ == other.vat20_120_ &&
            vat10_110_ == other.vat10_110_ &&
           vat5_ == other.vat5_ &&
           vat7_ == other.vat7_ &&
           vat5_105_ == other.vat5_105_ &&
           vat7_107_ == other.vat7_107_ &&
           userParam_ == other.userParam_ &&
            buyerInfo_ == other.buyerInfo_ &&
            industryProperties_ == other.industryProperties_ &&
            operationalParam_ == other.operationalParam_ &&
            correctionReason_ == other.correctionReason_ &&
            buyerPhone_ == other.buyerPhone_ &&
            boxId_ == other.boxId_ &&
            operations_ == other.operations_ &&
            t2107_ == other.t2107_ &&
           internetPayment_ == other.internetPayment_ &&
           cardPayments_ == other.cardPayments_ &&
            BaseWorkIncoming::operator ==(other);
}

bool Receipt::operator !=(const Receipt &other) const noexcept
{
    return !(*this == other);
}

const PSimpleReceiptOperation &Receipt::operator[](int i) const
{
    static const PSimpleReceiptOperation EMPTY_DEFAULT;
    if(i < 0 || i >= operations_.size()) return EMPTY_DEFAULT;
    return operations_[i];
}

qint32 Receipt::count() const
{
    return operations_.count();
}

qint32 Receipt::addOperation(const ReceiptOperation &op)
{
    PSimpleReceiptOperation o = PSimpleReceiptOperation(new ReceiptOperation(op));
    operations_.push_back(o);
    return operations_.count() -1;
}

qint32 Receipt::setOperation(const ReceiptOperation &op, qint32 idx)
{

    if(idx < 0 || idx > operations_.size()) return -1;
    if(idx == operations_.size()) return addOperation(op);
    PSimpleReceiptOperation o = PSimpleReceiptOperation(new ReceiptOperation(op));
    operations_[idx] = o;
    return idx;
}

qint32 Receipt::addOperation(const SimpleReceiptOperation &op)
{
    if(containsLabels_.has_value() && !containsLabels_.value())containsLabels_.reset();
    PSimpleReceiptOperation o = PSimpleReceiptOperation(new SimpleReceiptOperation(op));
    operations_.push_back(o);
    return operations_.count() -1;;
}

qint32 Receipt::setOperation(const SimpleReceiptOperation &op, qint32 idx)
{
    if(containsLabels_.has_value() && !containsLabels_.value())containsLabels_.reset();
    if(idx < 0 || idx > operations_.size()) return -1;
    if(idx == operations_.size()) return addOperation(op);
    PSimpleReceiptOperation o = PSimpleReceiptOperation(new SimpleReceiptOperation(op));
    operations_[idx] = o;
    return idx;
}

qint32 Receipt::removeOpearation(qint32 idx)
{
    containsLabels_.reset();
    if(idx >= 0 || idx < operations_.size())
    {
        operations_.removeAt(idx);
    }
    return operations_.count();
}

void Receipt::calcAmounts()
{
    calculatedvat20_        .reset();
    calculatedvat10_        .reset();
    calculatedvat0_         .reset();
    calculatedvatNone_      .reset();
    calculatedvat20_120_    .reset();
    calculatedvat10_110_    .reset();
    calculatedvat5_         .reset();
    calculatedvat7_         .reset();
    calculatedvat5_105_     .reset();
    calculatedvat7_107_     .reset();
    std::for_each(operations_.constBegin(), operations_.constEnd(),
                  [this](const PSimpleReceiptOperation &o) {
        switch (o->vatRate()) {
        case fdf::VatRate::Vat20    :
            calculatedvat20_ = calculatedvat20_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat10    :
            calculatedvat10_ = calculatedvat10_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat20_120:
            calculatedvat20_120_= calculatedvat20_120_ .value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
           break;
        case fdf::VatRate::Vat10_110:
            calculatedvat10_110_ = calculatedvat10_110_ .value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat0     :
            calculatedvat0_ = calculatedvat0_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::VatNone :
            calculatedvatNone_ = calculatedvatNone_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat5    :
            calculatedvat5_ = calculatedvat5_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat7    :
            calculatedvat7_ = calculatedvat7_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat5_105    :
            calculatedvat5_105_ = calculatedvat5_105_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        case fdf::VatRate::Vat7_107    :
            calculatedvat7_107_ = calculatedvat7_107_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll)) + o->costVat();
            break;
        default:break;
        }
    });
}

void Receipt::addQuantity(qint32 index)
{
    if(index >= 0 && index < operations_.size())
    {
        operations_[index]->addQuantity();
    }
}

void Receipt::subQuantity(qint32 index)
{
    if(index >= 0 && index < operations_.size())
    {
        operations_[index]->subQuantity();
    }

}

quint8 Receipt::t2107() const
{
    return t2107_;
}

void Receipt::setT2107(quint8 newT2107)
{
    t2107_ = newT2107;
}

bool Receipt::internetPayment() const
{
    return internetPayment_;
}

void Receipt::setInternetPayment(bool newInternetPayment)
{
    internetPayment_ = newInternetPayment;
}

std::optional<CardPaymentsInfo> Receipt::cardPayments() const
{
    return cardPayments_;
}

void Receipt::setCardPayments(std::optional<CardPaymentsInfo> newCardPayments)
{
    cardPayments_ = newCardPayments;
}

bool Receipt::checkAmount(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    FixNumber nul (DEFAULT_AMOUNT_MULT, 0ll);
    FixNumber pays = card_.value_or(nul) + prepay_.value_or(nul) +
            postpay_.value_or(nul) + barter_.value_or(nul);
    FixNumber a = amount();
    if(paymentAttr_ == fdf::PaymentType::Debit)
    {
        if(pays > a)
        {
            err = CoreApiConst::ErrorCode::ReceiptOverPayed;
            msg = QStringLiteral("Сумма оплат по чеку больше суммы чека");
            return false;
        }
        pays = pays + cash_.value_or(nul);
        if(pays < a)
        {
            err = CoreApiConst::ErrorCode::ReceiptNotPayed;
            msg = QStringLiteral("Сумма оплат по чеку меньше суммы чека");
            return false;
        }
    }
    else
    {
        pays = pays + cash_.value_or(nul);

        if(pays > a)
        {
            err = CoreApiConst::ErrorCode::ReceiptOverPayed;
            msg = QStringLiteral("Сумма оплат по чеку больше суммы чека");
            return false;
        }
        if(pays < a)
        {
            err = CoreApiConst::ErrorCode::ReceiptNotPayed;
            msg = QStringLiteral("Сумма оплат по чеку меньше суммы чека");
            return false;
        }
    }
    if(amount_.has_value() &&
            (amount_.value().value() / DEFAULT_AMOUNT_MULT) !=
            calculatedAmount().value() / DEFAULT_AMOUNT_MULT)
    {
        err = CoreApiConst::ErrorCode::InvalidReceiptAmount;
        msg = QStringLiteral("Сумма оплат по чеку не равна суммам стоимостей предметов расчета");
        return false;

    }
    return true;
}

bool Receipt::checkIndustryCode(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(!containsLabels() || industryProperties_.isEmpty()) return true;
    return std::all_of(industryProperties_.constBegin(), industryProperties_.constEnd(),
                       [&err, &msg](const IndustryProperty &p) -> bool {
        return p.isValid(false, err, msg);
    });
}
