#include "receiptdataclasses.h"
#include "cashboxbuildconfig.h"
#include "formatutils.h"
#include "inn.h"



#include <QObject>

ReceiptAdditionalUserParam::ReceiptAdditionalUserParam() noexcept
    : name_()
    , value_()
{

}

ReceiptAdditionalUserParam::ReceiptAdditionalUserParam(const ReceiptAdditionalUserParam &other) noexcept
    : name_(other.name_)
    , value_(other.value_)

{

}

ReceiptAdditionalUserParam::ReceiptAdditionalUserParam(ReceiptAdditionalUserParam &&other) noexcept
    : name_()
    , value_()

{
    name_.swap(other.name_);
    value_.swap(other.value_);
}

ReceiptAdditionalUserParam::ReceiptAdditionalUserParam(const QVariantMap &map) noexcept
    : name_()
    , value_()

{
    parseMap(map);
}

ReceiptAdditionalUserParam::~ReceiptAdditionalUserParam()
{

}

bool ReceiptAdditionalUserParam::isEmpty() const
{
    return name_.isEmpty() && value_.isEmpty();
}

bool ReceiptAdditionalUserParam::isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err,
                                         QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(canBeEmpty && isEmpty()) return true;
    QString suffix;
    if(!FormatUtils::checkString(name_, 64, false, false, suffix))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Наименование дополнительного реквизита пользователя (1085)") + suffix;
        return false;
    }
    if(!FormatUtils::checkString(value_, 256, false, false, suffix))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Значение дополнительного реквизита пользователя (1086)") + suffix;
        return false;
    }
    return true;
}

const QString &ReceiptAdditionalUserParam::name() const
{
    return name_;
}

void ReceiptAdditionalUserParam::setName(const QString &newName)
{
    name_ = STR4FS_(newName);
}

const QString &ReceiptAdditionalUserParam::value() const
{
    return value_;
}

void ReceiptAdditionalUserParam::setValue(const QString &newValue)
{
    value_ = STR4FS_(newValue);
}

QVariantMap ReceiptAdditionalUserParam::toMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"name", name_},
        {"value", value_}
    };
}

void ReceiptAdditionalUserParam::parseMap(const QVariantMap &map)
{
    name_ = STR4FS_(map["name"].toString());
    value_ = STR4FS_(map["value"].toString());
}

Tlv ReceiptAdditionalUserParam::toTlv() const
{
    if(isEmpty()) return Tlv();
    Tlv::Stlv list;
    Tlv buf;
    buf.setString(fdf::Tag::OwnerAdditionalParamName, name_);
    list << buf;
    buf.setString(fdf::Tag::OwnerAdditionalParamValue, value_);
    list << buf;
    buf.setStlv(fdf::Tag::OwnerAdditionalParam, list);
    return buf;
}

void ReceiptAdditionalUserParam::parseTlv(const Tlv &tlv)
{
    name_.clear();
    value_.clear();
    if(tlv.tag() == fdf::Tag::OwnerAdditionalParam)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::OwnerAdditionalParamName))
        {
            name_ = map[fdf::Tag::OwnerAdditionalParamName].toString();
        }
        if(map.contains(fdf::Tag::OwnerAdditionalParamValue))
        {
            value_ = map[fdf::Tag::OwnerAdditionalParamValue].toString();
        }
    }
}

ReceiptAdditionalUserParam &ReceiptAdditionalUserParam::operator =(const ReceiptAdditionalUserParam &other) noexcept
{
    name_ = other.name_;
    value_ = other.value_;
    return *this;
}
ReceiptAdditionalUserParam &ReceiptAdditionalUserParam::operator =(ReceiptAdditionalUserParam &&other) noexcept
{
    name_.swap(other.name_);
    value_.swap(other.value_);
    return *this;

}
bool ReceiptAdditionalUserParam::operator ==(const ReceiptAdditionalUserParam &other) const noexcept
{
    return name_ == other.name_ &&
            value_ == other.value_;
}

bool ReceiptAdditionalUserParam::operator !=(const ReceiptAdditionalUserParam &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

QuantityFraction::QuantityFraction() noexcept
    : nominator_(0ll)
    , denominator_(0ll)
{

}

QuantityFraction::QuantityFraction(const QVariantMap &map)
    : nominator_(0ll)
    , denominator_(0ll)
{
    parseMap(map);
}

QuantityFraction::~QuantityFraction()
{

}

bool QuantityFraction::is1() const
{
    return nominator_ == 1 && denominator_ == 1;
}


bool QuantityFraction::isValid() const
{
    return denominator_ > 0ll && nominator_ > 0ll && nominator_ < denominator_;
}

bool QuantityFraction::isValid(CoreApiConst::ErrorCode &err, QString &msg)
{
    msg.clear();
    err = CoreApiConst::ErrorCode::Ok;
    if(!isValid())
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректное значение дробного количества маркированного товара (1291)";
        return false;
    }
    return true;
}

qint64 QuantityFraction::nominator() const
{
    return nominator_;
}

void QuantityFraction::setNominator(qint64 newNominator)
{
    nominator_ = newNominator;
}

qint64 QuantityFraction::denominator() const
{
    return denominator_;
}

void QuantityFraction::setDenominator(qint64 newDenominator)
{
    denominator_ = newDenominator;
}

QString QuantityFraction::text() const
{
    return isValid() ? QStringLiteral("%1/%2").arg(nominator_).arg(denominator_) : QString();
}

FixNumber QuantityFraction::fixNumber() const
{
    if(!isValid()) return FixNumber();
    return FixNumber(DEFAULT_QUANTITY_MULT,
                     static_cast<double>(nominator_) /
                     static_cast<double>(denominator_));
}

QVariantMap QuantityFraction::toMap() const
{
    if(!isValid()) return QVariantMap();
    if(nominator_ < 0x7FFFFFFFll && denominator_ < 0x7FFFFFFFll)
    {
        return {
            {"nominator", nominator_},
            {"denominator", denominator_},
        };
    }
    return {
        {"nominator", QString::number(nominator_)},
        {"denominator", QString::number(denominator_)},
    };
}

void QuantityFraction::parseMap(const QVariantMap &map)
{
    QVariant n = map["nominator"], d = map["denominator"];
    nominator_ = n.type() == QVariant::String ? n.toString().toLongLong() : n.toLongLong();
    denominator_ = d.type() == QVariant::String ? d.toString().toLongLong() : d.toLongLong();
}


Tlv QuantityFraction::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv buf;
    Tlv::Stlv list;
    buf.setVln(fdf::Tag::Nominator, nominator_);
    list << buf;
    buf.setVln(fdf::Tag::Denominator, denominator_);
    list << buf;
    buf.setStlv(fdf::Tag::LabledProductFractionQuantity, list);
    return buf;
}

void QuantityFraction::parseTlv(const Tlv &tlv)
{
    nominator_ = 0ll;
    denominator_ = 0ll;
    if(tlv.tag() == fdf::Tag::LabledProductFractionQuantity)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::Nominator)) nominator_ = map[fdf::Tag::Nominator].toVln();
        if(map.contains(fdf::Tag::Denominator)) denominator_ = map[fdf::Tag::Denominator].toVln();
    }
}

bool QuantityFraction::operator ==(const QuantityFraction &other) const noexcept
{
    return nominator_ == other.nominator_ &&
            denominator_ == other.denominator_;
}

bool QuantityFraction::operator !=(const QuantityFraction &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------

ReceiptOperationalParam::ReceiptOperationalParam() noexcept
    : code_(0)
    , dt_()
    , data_()
{

}

ReceiptOperationalParam::ReceiptOperationalParam(const ReceiptOperationalParam &other) noexcept
    : code_(other.code_)
    , dt_(other.dt_)
    , data_(other.data_)
{

}

ReceiptOperationalParam::ReceiptOperationalParam(ReceiptOperationalParam &&other) noexcept
    : code_(other.code_)
    , dt_()
    , data_()

{
    dt_.swap(other.dt_);
    data_.swap(other.data_);
}

ReceiptOperationalParam::ReceiptOperationalParam(const QVariantMap &map)
    : code_(0)
    , dt_()
    , data_()
{
    parseMap(map);
}

ReceiptOperationalParam::~ReceiptOperationalParam()
{

}

bool ReceiptOperationalParam::isValid() const
{
    return dt_.isValid() && !data_.isEmpty();
}

bool ReceiptOperationalParam::isEmpty() const
{
    return code_ == 0u && !dt_.isValid() && data_.isEmpty();
}

bool ReceiptOperationalParam::isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err,
                                      QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(canBeEmpty && isEmpty()) return true;
    if(!dt_.isValid())
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректное значение даты операции (1273) операционного реквизита чека (1270)";
        return false;
    }
    if(!FormatUtils::checkString(data_, 64, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 'данные операции' (1272) операционного реквизита чека (1270)") + msg;
        return false;
    }
    return true;
}

quint8 ReceiptOperationalParam::code() const
{
    return code_;
}

void ReceiptOperationalParam::setCode(quint8 newCode)
{
    code_ = newCode;
}

const QDateTime &ReceiptOperationalParam::dt() const
{
    return dt_;
}

void ReceiptOperationalParam::setDt(const QDateTime &newDt)
{
    dt_ = newDt;
}

const QString &ReceiptOperationalParam::data() const
{
    return data_;
}

void ReceiptOperationalParam::setData(const QString &newData)
{
    data_ = STR4FS_(newData);
}

QVariantMap ReceiptOperationalParam::toMap() const
{
    if(!isValid()) return QVariantMap();
    return {
        {"code", static_cast<qint32>(code_)},
        {"dt", DT2STR_(dt_)},
        {"data", data_}
    };
}

void ReceiptOperationalParam::parseMap(const QVariantMap &map)
{
    code_ = map["code"].toUInt();
    dt_ = STR2DT_(map["dt"].toString());
    data_ = STR4FS_(map["data"].toString().trimmed());
}

Tlv ReceiptOperationalParam::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv buf;
    Tlv::Stlv list;
    buf.setByte(fdf::Tag::OperationId, code_);
    list << buf;
    buf.setDt(fdf::Tag::OperationDt, dt_);
    list << buf;
    buf.setString(fdf::Tag::OperationData, data_);
    list << buf;
    buf.setStlv(fdf::Tag::ReceiptOperationalParam, list);
    return buf;
}

void ReceiptOperationalParam::parseTlv(const Tlv &tlv)
{
    dt_ = QDateTime();
    data_.clear();
    code_ = 0u;
    if(tlv.tag() == fdf::Tag::ReceiptOperationalParam)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::OperationId)) code_ = map[fdf::Tag::OperationId].toByte();
        if(map.contains(fdf::Tag::OperationDt)) dt_ = map[fdf::Tag::OperationDt].toDt();
        if(map.contains(fdf::Tag::OperationData)) data_ = map[fdf::Tag::OperationData].toString();
    }
}

ReceiptOperationalParam &ReceiptOperationalParam::operator =(const ReceiptOperationalParam &other)noexcept
{
    code_ = other.code_;
    dt_ = other.dt_;
    data_ = other.data_;
    return *this;
}
ReceiptOperationalParam &ReceiptOperationalParam::operator =(ReceiptOperationalParam &&other)noexcept
{
    code_ = other.code_;
    dt_.swap(other.dt_);
    data_.swap(other.data_);
    return *this;
}
bool ReceiptOperationalParam::operator ==(const ReceiptOperationalParam &other) const noexcept
{
    return code_ == other.code_ &&
            dt_ == other.dt_ &&
            data_ == other.data_;
}

bool ReceiptOperationalParam::operator !=(const ReceiptOperationalParam &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------

const QSet<QString> BuyerInfo::RUSSIAN_DOCS = {
    "21", "22", "26", "27", "28"
};
const QSet<QString> BuyerInfo::FOREIGH_DOCS = {
    "31", "32", "33", "34", "35", "36", "37", "38", "40"
};

BuyerInfo::BuyerInfo() noexcept
    : name_()
    , inn_()
    , birthDay_()
    , citizenship_()
    , docCode_()
    , docData_()
    , address_()
{

}

BuyerInfo::BuyerInfo(const BuyerInfo &other) noexcept
    : name_(other.name_)
    , inn_(other.inn_)
    , birthDay_(other.birthDay_)
    , citizenship_(other.citizenship_)
    , docCode_(other.docCode_)
    , docData_(other.docData_)
    , address_(other.address_)

{

}

BuyerInfo::BuyerInfo(BuyerInfo &&other) noexcept
    : name_()
    , inn_()
    , birthDay_(other.birthDay_)
    , citizenship_()
    , docCode_()
    , docData_()
    , address_()
{
    name_.swap(other.name_);
    inn_.swap(other.inn_);
    citizenship_.swap(other.citizenship_);
    docCode_.swap(other.docCode_);
    docData_.swap(other.docData_);
    address_.swap(other.address_);
}

BuyerInfo::BuyerInfo(const QVariantMap &map) noexcept
    : name_()
    , inn_()
    , birthDay_()
    , citizenship_()
    , docCode_()
    , docData_()
    , address_()
{
    parseMap(map);
}

BuyerInfo::~BuyerInfo()
{

}

bool BuyerInfo::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(false, err, msg);
}

bool BuyerInfo::isEmpty() const
{
    return name_.isEmpty() || (inn_.isEmpty() && (!birthDay_.isValid() || docCode_.isEmpty() || docData_.isEmpty()));
}
/**
 * @brief Замечания павлова от 28.07.2021
 * Добрый день. Подскажите, пожалуйста, не было ли разъяснений по поводу реквидита 1256 (сведения о покупателе). Дело в том, что в его составе есть рквизит 1245 (код документа). В описании тегов в ФФД написано, что это строка 32 символа с фиксированной длиной.  но в таблице значений (116) - это двухсимвольные коды. Где ошибка? каков правильный формат передачи? если ошибки нет, то чем заполнять оставшиеся 30 символов строки (длина, написано, фиксированная)
Владимир Павлов
В действующей версии ФФД это не единственная ошибка. Рекомендуем передавать значение 1245 как строку нефиксированной длины (два символа)
 */
bool BuyerInfo::isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(canBeEmpty && isEmpty()) return true;
    if(!FormatUtils::checkString(name_, 256, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Покупатель (1227) " + msg;
        return false;
    }
    if(inn_.isEmpty())
    {
        if(!birthDay_.isValid())
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Не указан реквизит 1243 (дата рождения покупателя)";
            return false;
        }
        if(!FormatUtils::checkString(docCode_, 32, false, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1245 (код вида документа)" + msg;
            return false;
        }
        if(!FOREIGH_DOCS.contains(docCode_) && !RUSSIAN_DOCS.contains(docCode_))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1245 (код вида документа) имеет некорректное значение: " + docCode_;
            return false;
        }
        if(!FormatUtils::checkString(docData_, 64, false, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1246 (данные документа) " + msg;
            return false;
        }
        if(FOREIGH_DOCS.contains(docCode_))
        {
            if(!FormatUtils::checkString(citizenship_, 3, true, true, msg))
            {
                err = CoreApiConst::ErrorCode::InvalidParameter;
                msg = "Реквизит 1244 (гражданство) " + msg;
                return false;
            }
        }
    }
    else
    {
        Inn i(inn_);
        if(!i.isValid())
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1228 (ИНН покупателя) имеет некорректное значение: " + inn_;
            return false;
        }
        if(!FormatUtils::checkString(address_, 256, false, true, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1254 (адрес покупателя) " + msg;
            return false;
        }
    }
    return true;
}

const QString &BuyerInfo::name() const
{
    return name_;
}

void BuyerInfo::setName(const QString &newName)
{
    name_ = STR4FS_(newName);
}

const QString &BuyerInfo::inn() const
{
    return inn_;
}

void BuyerInfo::setInn(const QString &newInn)
{
    inn_ = newInn;
}

const QDate &BuyerInfo::birthDay() const
{
    return birthDay_;
}

void BuyerInfo::setBirthDay(const QDate &d)
{
    birthDay_ = d;
}

const QString &BuyerInfo::citizenship() const
{
    return citizenship_;
}

void BuyerInfo::setCitizenship(const QString &newCitizenship)
{
    citizenship_ = STR4FS_(newCitizenship);
}

const QString &BuyerInfo::docCode() const
{
    return docCode_;
}

void BuyerInfo::setDocCode(const QString &newDocCode)
{
    docCode_ = STR4FS_(newDocCode);
}

const QString &BuyerInfo::docData() const
{
    return docData_;
}

void BuyerInfo::setDocData(const QString &newDocData)
{
    docData_ = STR4FS_(newDocData);
}

const QString &BuyerInfo::address() const
{
    return address_;
}

void BuyerInfo::setAddress(const QString &newAddress)
{
    address_ = STR4FS_(newAddress);
}

QVariantMap BuyerInfo::toMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"name", name_},
        {"inn", inn_},
        {"birthDay", birthDay_.toString(Qt::ISODate)},
        {"citizenship", citizenship_},
        {"docCode", docCode_},
        {"docData", docData_},
        {"address", address_}
    };
}

QVariantMap BuyerInfo::toExternalMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"name", name_.isEmpty() ? QVariant() : QVariant(name_)},
        {"inn", inn_.isEmpty() ? QVariant() : QVariant(inn_)},
        {"birthDay", birthDay_.isValid() ? QVariant(birthDay_.toString(Qt::ISODate)) : QVariant()},
        {"citizenship", citizenship_.isEmpty() ? QVariant() : QVariant(citizenship_)},
        {"docCode", docCode_.isEmpty() ? QVariant() : QVariant(docCode_)},
        {"docData", docData_.isEmpty() ? QVariant() : QVariant(docData_)},
        {"address", address_.isEmpty() ? QVariant() : QVariant(address_)}
    };
}

void BuyerInfo::parseMap(const QVariantMap &map)
{
    name_ = STR4FS_(map["name"].toString().trimmed());
    inn_ = map["inn"].toString().trimmed();
    birthDay_ = QDate::fromString(map["birthDay"].toString(), Qt::ISODate);
    citizenship_ = STR4FS_(map["citizenship"].toString().trimmed());
    docCode_ = STR4FS_(map["docCode"].toString().trimmed());
    docData_ = STR4FS_(map["docData"].toString().trimmed());
    address_ = STR4FS_(map["address"].toString().trimmed());
}

Tlv BuyerInfo::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv buf;
    Tlv::Stlv list;
    buf.setString(fdf::Tag::ClientName, name_);
    list << buf;
    Inn i(inn_);
    if(i.isValid())
    {
        buf.setInn(fdf::Tag::ClientInn, i);
        list << buf;
        if(!address_.isEmpty())
        {
            buf.setString(fdf::Tag::ClientAddress, address_);
            list << buf;
        }
    }
    else
    {
        buf.setString(fdf::Tag::ClientBirthDay, birthDay_.toString("dd.MM.yyyy"));
        list << buf;
        buf.setString(fdf::Tag::PersonDocTypeCode, docCode_);
        list << buf;
        buf.setString(fdf::Tag::PersonDocData, docData_);
        list << buf;
        if(FOREIGH_DOCS.contains(docCode_) && !citizenship_.isEmpty())
        {
            buf.setString(fdf::Tag::Citizenship, citizenship_);
            list << buf;
        }
    }
    buf.setStlv(fdf::Tag::ClientInfo, list);
    return buf;
}

void BuyerInfo::parseTlv(const Tlv &tlv)
{
    name_.clear();
    inn_.clear();
    birthDay_ = QDate();
    citizenship_.clear();
    docCode_.clear();
    docData_.clear();
    address_.clear();
    if(tlv.tag() == fdf::Tag::ClientInfo)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::ClientName)) name_ = map[fdf::Tag::ClientName].toString().trimmed();
        if(map.contains(fdf::Tag::ClientInn)) inn_ = map[fdf::Tag::ClientInn].toString().trimmed();
        if(map.contains(fdf::Tag::ClientBirthDay))
        {
            birthDay_ = QDate::fromString(
                        map[fdf::Tag::ClientBirthDay].toString().trimmed(),
                    "dd.MM.yyyy");
        }
        if(map.contains(fdf::Tag::Citizenship)) citizenship_ = map[fdf::Tag::Citizenship].toString().trimmed();
        if(map.contains(fdf::Tag::PersonDocTypeCode))
        {
            docCode_ = map[fdf::Tag::PersonDocTypeCode].toString().trimmed();
        }
        if(map.contains(fdf::Tag::PersonDocData)) docData_ = map[fdf::Tag::PersonDocData].toString().trimmed();
        if(map.contains(fdf::Tag::ClientAddress)) address_ = map[fdf::Tag::ClientAddress].toString().trimmed();
    }
}

BuyerInfo &BuyerInfo::operator =(const BuyerInfo &other) noexcept
{
    name_ = other.name_;
    inn_ = other.inn_;
    birthDay_ = other.birthDay_;
    citizenship_ = other.citizenship_;
    docCode_ = other.docCode_;
    docData_ = other.docData_;
    address_ = other.address_;
    return *this;
}
BuyerInfo &BuyerInfo::operator =(BuyerInfo &&other) noexcept
{
    name_.swap(other.name_);
    inn_.swap(other.inn_);
    birthDay_ = other.birthDay_;
    citizenship_.swap(other.citizenship_);
    docCode_.swap(other.docCode_);
    docData_.swap(other.docData_);
    address_.swap(other.address_);
    return *this;
}
bool BuyerInfo::operator ==(const BuyerInfo &other) const noexcept
{
    return name_ == other.name_ &&
            inn_ == other.inn_ &&
            birthDay_ == other.birthDay_ &&
            citizenship_ == other.citizenship_ &&
            docCode_ == other.docCode_ &&
            docData_ == other.docData_ &&
            address_ == other.address_;
}

bool BuyerInfo::operator !=(const BuyerInfo &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

IndustryProperty::IndustryProperty() noexcept
    : fedId_()
    , docDt_()
    , docNumber_()
    , value_()
{

}

IndustryProperty::IndustryProperty(const IndustryProperty &other) noexcept
    : fedId_(other.fedId_)
    , docDt_(other.docDt_)
    , docNumber_(other.docNumber_)
    , value_(other.value_)
{

}

IndustryProperty::IndustryProperty(IndustryProperty &&other) noexcept
    : fedId_()
    , docDt_(other.docDt_)
    , docNumber_()
    , value_()
{
    fedId_.swap(other.fedId_);
    docNumber_.swap(other.docNumber_);
    value_.swap(other.value_);
}

IndustryProperty::IndustryProperty(const QVariantMap &map) noexcept
    : fedId_()
    , docDt_()
    , docNumber_()
    , value_()
{
    parseMap(map);
}

IndustryProperty::~IndustryProperty()
{

}

bool IndustryProperty::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    bool res = isValid(false, err, msg);
    return  res;
}

bool IndustryProperty::isEmpty() const
{
    return fedId_.isEmpty() || value_.isEmpty();
}

bool IndustryProperty::isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(canBeEmpty && isEmpty()) return true;
    bool ok = false;
    qint32 f = fedId_.toInt(&ok);
    if(!ok || f < 1 || f > MAX_FED_ID || fedId_.size() > 3)
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1262 (идентификатор ФОИВ) имеет некорректное значение " + fedId_;
        return false;
    }
    if(!docDt_.isValid())
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1263 (дата документа основания) имеет некорректное значение ";
        return false;
    }
    if(!FormatUtils::checkString(docNumber_, 32, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1264 (номер документа основания) " + msg;
        return false;
    }
    if(!FormatUtils::checkString(value_, 256, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1265 (значение отраслевого реквизита) " + msg;
        return false;
    }

    return true;
}

const QString &IndustryProperty::fedId() const
{
    return fedId_;
}

void IndustryProperty::setFedId(const QString &newFedId)
{
    fedId_ = STR4FS_(newFedId);
}

const QDate &IndustryProperty::docDt() const
{
    return docDt_;
}

void IndustryProperty::setDocDt(const QDate &newDocDt)
{
    docDt_ = newDocDt;
}

const QString &IndustryProperty::docNumber() const
{
    return docNumber_;
}

void IndustryProperty::setDocNumber(const QString &newDocNumber)
{
    docNumber_ = STR4FS_(newDocNumber);
}

const QString &IndustryProperty::value() const
{
    return value_;
}

void IndustryProperty::setValue(const QString &newValue)
{
    value_ = STR4FS_(newValue);
}

QVariantMap IndustryProperty::toMap() const
{
    if(isEmpty()) return QVariantMap();

    return {
        {"fedId", fedId_},
        {"date", docDt_.toString(Qt::ISODate)},
        {"docNumber",  docNumber_},
        {"value", value_}
    };

}

void IndustryProperty::parseMap(const QVariantMap &map)
{
    fedId_ = STR4FS_(map["fedId"].toString().trimmed());
    docDt_ = QDate::fromString(map["date"].toString().trimmed(), Qt::ISODate);
    docNumber_ = STR4FS_(map["docNumber"].toString().trimmed());
    value_ = STR4FS_(map["value"].toString().trimmed());
}

Tlv IndustryProperty::toTlv(fdf::Tag tag) const
{
    if(!isValid()) return Tlv();
    Tlv buf;
    Tlv::Stlv list;

    QString id = fedId_;
    if(id.size() < 3) id = QString(3 - id.size(), QLatin1Char('0')) + id;
    buf.setString(fdf::Tag::FEDID, id);
    list << buf;
    buf.setString(fdf::Tag::ReasonDocDt, docDt_.toString("dd.MM.yyyy"));
    list << buf;
    buf.setString(fdf::Tag::ReasonDocNumber, docNumber_);
    list << buf;
    buf.setString(fdf::Tag::IndustryParamValue, value_);
    list << buf;

    buf.setStlv(tag, list);
    return buf;
}

/*
        IndustryItemParam                       = 1260, //отраслевой реквизит предмета расчета
        IndustryReceiptParam                    = 1261, //отраслевой реквизит чека
        FEDID                                   = 1262, //идентификатор ФОИВ
        ReasonDocDt                             = 1263, //дата документа основания
        ReasonDocNumber                         = 1264, //номер документа основания
        IndustryParamValue                      = 1265, //*/

void IndustryProperty::parseTlv(const Tlv &tlv)
{
    fedId_.clear();
    docDt_ = QDate();
    docNumber_.clear();
    value_.clear();
    if(tlv.tag() == fdf::Tag::IndustryItemParam ||
            tlv.tag() == fdf::Tag::IndustryReceiptParam)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::FEDID)) fedId_ = map[fdf::Tag::FEDID].toString().trimmed();
        if(map.contains(fdf::Tag::ReasonDocDt))
        {
            docDt_ = QDate::fromString(map[fdf::Tag::ReasonDocDt].toString().trimmed(), "dd.MM.yyyy");
        }
        if(map.contains(fdf::Tag::ReasonDocNumber))
        {
            docNumber_ = map[fdf::Tag::ReasonDocNumber].toString().trimmed();
        }
        if(map.contains(fdf::Tag::IndustryParamValue))
        {
            value_ = map[fdf::Tag::IndustryParamValue].toString().trimmed();
        }
    }
}

IndustryProperty &IndustryProperty::operator =(const IndustryProperty &other) noexcept
{
    fedId_ = other.fedId_;
    docDt_ = other.docDt_;
    docNumber_ = other.docNumber_;
    value_ = other.value_;
    return *this;
}
IndustryProperty &IndustryProperty::operator =(IndustryProperty &&other) noexcept
{
    fedId_.swap(other.fedId_);
    docDt_ = other.docDt_;
    docNumber_.swap(other.docNumber_);
    value_.swap(other.value_);
    return *this;
}
bool IndustryProperty::operator ==(const IndustryProperty &other) const noexcept
{
    return fedId_ == other.fedId_ &&
            docDt_ == other.docDt_ &&
            docNumber_ == other.docNumber_ &&
            value_ == other.value_;
}

bool IndustryProperty::operator !=(const IndustryProperty &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

ProviderData::ProviderData() noexcept
    : name_()
    , phones_()
{

}

ProviderData::ProviderData(const ProviderData &other) noexcept
    : name_(other.name_)
    , phones_(other.phones_)

{

}

ProviderData::ProviderData(ProviderData &&other) noexcept
    : name_()
    , phones_()

{
    name_.swap(other.name_);
    phones_.swap(other.phones_);
}

ProviderData::ProviderData(const QVariantMap &map)
    : name_()
    , phones_()

{
    parseMap(map);
}

ProviderData::~ProviderData()
{

}

bool ProviderData::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(false, err, msg);
}

bool ProviderData::isEmpty() const
{
    return phones_.isEmpty() || name_.isEmpty();
}

bool ProviderData::isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(canBeEmpty && isEmpty()) return true;
    if(!FormatUtils::checkString(name_, 256, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1225 (наименование поставщика) " + msg;
        return false;
    }
    if(phones_.isEmpty() ||
            !std::any_of(phones_.constBegin(), phones_.constEnd(),
                         [](const QString &p)->bool{return !p.trimmed().isEmpty();}))
    {
#if CASHBOX_MODEL != 2000
        err = CoreApiConst::ErrorCode::NoAnyParameter;
        msg = "Не хватает реквизита 1171 (телефон поставщика) ";
        return false;
#endif
    }
    for(const QString &p :phones_)
    {
        if(p.trimmed().isEmpty()) continue;
        if(!FormatUtils::checkPhone(p, 19, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = "Реквизит 1171 (телефон поставщика) " + msg;
            return false;
        }
    }
    return true;
}

const QString &ProviderData::name() const
{
    return name_;
}

void ProviderData::setName(const QString &newName)
{
    name_ = STR4FS_(newName);
}

const QStringList &ProviderData::phones() const
{
    return phones_;
}

void ProviderData::setPhones(const QStringList &newPhone)
{
    phones_ = newPhone;
}

QVariantMap ProviderData::toMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"name", name_},
        {"phones", phonesList()}
    };
}

void ProviderData::parseMap(const QVariantMap &map)
{
    name_ = STR4FS_(map["name"].toString().trimmed());
    phones_.clear();
    parsePhoneList(map["phones"].toList());
}

Tlv ProviderData::toTlv() const
{
    Tlv buf;
    if(!isValid()) return Tlv();
    Tlv::Stlv list;
    buf.setString(fdf::Tag::ProviderName, name_);
    list << buf;
    std::for_each(phones_.constBegin(), phones_.constEnd(),
                  [&list, &buf](const QString &p) {
        if(!p.trimmed().isEmpty())
        {
            buf.setString(fdf::Tag::ProviderPhone, p);
            list << buf;
        }
    });
    buf.setStlv(fdf::Tag::ProviderData, list);
    return buf;
}

void ProviderData::parseTlv(const Tlv &tlv)
{
    name_.clear();
    phones_.clear();
    if(tlv.tag() == fdf::Tag::ProviderData)
    {
        Tlv::MultiStlv map = tlv.toMultiStlv();
        if(map.contains(fdf::Tag::ProviderName))
        {
            name_ = map.value(fdf::Tag::ProviderName).toString().trimmed();
        }
        Tlv::Stlv list = map.values(fdf::Tag::ProviderPhone);
        for(const Tlv &t: list)phones_ << t.toString().trimmed();
    }
}

ProviderData &ProviderData::operator =(const ProviderData &other) noexcept
{
    name_ = other.name_;
    phones_ = other.phones_;
    return *this;
}

ProviderData &ProviderData::operator =( ProviderData &&other) noexcept
{
    name_.swap(other.name_);
    phones_.swap(other.phones_);
    return *this;
}

bool ProviderData::operator ==(const ProviderData &other) const noexcept
{
    return name_ == other.name_ &&
            phones_ == other.phones_;
}

bool ProviderData::operator !=(const ProviderData &other) const noexcept
{
    return !(*this == other);
}

QVariantList ProviderData::phonesList() const
{
    QVariantList res;
    std::for_each(phones_.constBegin(), phones_.constEnd(),
                  [&res](const QString &p){if(!p.trimmed().isEmpty()) res << p.trimmed();});
    return res;
}

void ProviderData::parsePhoneList(const QVariantList &list)
{
    phones_.clear();
    std::for_each(list.constBegin(), list.constEnd(),
                  [this](const QVariant &p){
        if(!p.toString().trimmed().isEmpty())
        {
            phones_ << p.toString().trimmed();
        }
    });
}
//--------------------------------------------------------------------------------------------------

AgentData::AgentData() noexcept
    : operatorName_()
    , operatorInn_()
    , operatorAddress_()
    , bankOperation_()
    , agentPhones_()
    , paysOperatorPhones_()
    , operatorPhones_()
{

}
AgentData::AgentData(const AgentData &other) noexcept
    : operatorName_(other.operatorName_)
    , operatorInn_(other.operatorInn_)
    , operatorAddress_(other.operatorAddress_)
    , bankOperation_(other.bankOperation_)
    , agentPhones_(other.agentPhones_)
    , paysOperatorPhones_(other.paysOperatorPhones_)
    , operatorPhones_(other.operatorPhones_)
{

}

AgentData::AgentData(AgentData &&other) noexcept
    : operatorName_()
    , operatorInn_()
    , operatorAddress_()
    , bankOperation_()
    , agentPhones_()
    , paysOperatorPhones_()
    , operatorPhones_()

{
    operatorName_.swap(other.operatorName_);
    operatorInn_.swap(other.operatorInn_);
    operatorAddress_.swap(other.operatorAddress_);
    bankOperation_.swap(other.bankOperation_);
    agentPhones_.swap(other.agentPhones_);
    paysOperatorPhones_.swap(other.paysOperatorPhones_);
    operatorPhones_.swap(other.operatorPhones_);
}
AgentData::AgentData(const QVariantMap &map) noexcept
    : operatorName_()
    , operatorInn_()
    , operatorAddress_()
    , bankOperation_()
    , agentPhones_()
    , paysOperatorPhones_()
    , operatorPhones_()

{
    parseMap(map);
}

AgentData::~AgentData()
{

}

bool AgentData::isValid(fdf::ItemPayAgentFlag agentFlag) const
{
     CoreApiConst::ErrorCode err;
     QString msg;
     return isValid(agentFlag,err, msg);
}

bool AgentData::isValid(fdf::ItemPayAgentFlag agentFlag, CoreApiConst::ErrorCode &err, QString &msg) const
{
//    lmWarning() << logvariant(toMap());
    if(!fdf::itemPayAgentFlagIsValid(agentFlag)) return true;
    switch (agentFlag)
    {
    case fdf::ItemPayAgentFlag::BankAgent   : return isValidBankAgent   (err, msg);
    case fdf::ItemPayAgentFlag::BankSubAgent: return isValidBankSubAgent(err, msg);
    case fdf::ItemPayAgentFlag::PayAgent    : return isValidPayAgent    (err, msg);
    case fdf::ItemPayAgentFlag::PaySubAgent : return isValidPaySubAgent (err, msg);
    case fdf::ItemPayAgentFlag::Attorney    : return isValidAttorney    (err, msg);
    case fdf::ItemPayAgentFlag::Broker      : return isValidBroker      (err, msg);
    case fdf::ItemPayAgentFlag::Agent       : return isValidAgent       (err, msg);
    default: return false;
    }
}

bool AgentData::isEmpty() const
{
    return operatorName_.isEmpty() &&
            operatorInn_.isEmpty() &&
            operatorAddress_.isEmpty() &&
            bankOperation_.isEmpty() &&
            agentPhones_.isEmpty() &&
            paysOperatorPhones_.isEmpty() &&
            operatorPhones_.isEmpty();
}

const QString &AgentData::operatorName() const
{
    return operatorName_;
}

void AgentData::setOperatorName(const QString &newOperatorName)
{
    operatorName_ = STR4FS_(newOperatorName);
}

const QString &AgentData::operatorInn() const
{
    return operatorInn_;
}

void AgentData::setOperatorInn(const QString &newOperatorInn)
{
    operatorInn_ = newOperatorInn;
}

const QString &AgentData::operatorAddress() const
{
    return operatorAddress_;
}

void AgentData::setOperatorAddress(const QString &newOperatorAddress)
{
    operatorAddress_ = STR4FS_(newOperatorAddress);
}

const QString &AgentData::bankOperation() const
{
    return bankOperation_;
}

void AgentData::setBankOperation(const QString &newBankOperation)
{
    bankOperation_ = STR4FS_(newBankOperation);
}

const QStringList &AgentData::agentPhones() const
{
    return agentPhones_;
}

void AgentData::setAgentPhones(const QStringList &newAgentPhones)
{
    agentPhones_ = newAgentPhones;
}

const QStringList &AgentData::paysOperatorPhones() const
{
    return paysOperatorPhones_;
}

void AgentData::setPaysOperatorPhones(const QStringList &newPaysOperatorPhones)
{
    paysOperatorPhones_ = newPaysOperatorPhones;
}

const QStringList &AgentData::operatorPhones() const
{
    return operatorPhones_;
}

void AgentData::setOperatorPhones(const QStringList &newOperatorPhones)
{
    operatorPhones_ = newOperatorPhones;
}

QVariantMap AgentData::toMap() const
{
    QVariantMap res;
    if(!operatorName_   .isEmpty())res["operatorName"] = operatorName_;
    if(!operatorInn_    .isEmpty())res["operatorInn"] = operatorInn_;
    if(!operatorAddress_.isEmpty())res["operatorAddress"] = operatorAddress_;
    if(!bankOperation_  .isEmpty())res["bankOperation"] = bankOperation_;

    if(!agentPhones_.isEmpty()) res["agentPhones"] = phonesList(agentPhones_);
    if(!paysOperatorPhones_.isEmpty()) res["paysOperatorPhones"] = phonesList(paysOperatorPhones_);
    if(!operatorPhones_.isEmpty()) res["operatorPhones"] = phonesList(operatorPhones_);
    return res;
}

void AgentData::parseMap(const QVariantMap &map)
{
    operatorName_    = STR4FS_(map["operatorName"].toString().trimmed());
    operatorInn_     = map["operatorInn"].toString().trimmed();
    operatorAddress_ = STR4FS_(map["operatorAddress"].toString().trimmed());
    bankOperation_   = STR4FS_(map["bankOperation"].toString().trimmed());
    agentPhones_ = parsePhoneList(map["agentPhones"].toList());
    paysOperatorPhones_ = parsePhoneList(map["paysOperatorPhones"].toList());
    operatorPhones_ = parsePhoneList(map["operatorPhones"].toList());
}

Tlv AgentData::toTlv(fdf::ItemPayAgentFlag agentFlag) const
{
    if(!fdf::itemPayAgentFlagIsValid(agentFlag)) return Tlv();
    switch (agentFlag)
    {
    case fdf::ItemPayAgentFlag::BankAgent   : return toTlvBankAgent   ();
    case fdf::ItemPayAgentFlag::BankSubAgent: return toTlvBankSubAgent();
    case fdf::ItemPayAgentFlag::PayAgent    : return toTlvPayAgent    ();
    case fdf::ItemPayAgentFlag::PaySubAgent : return toTlvPaySubAgent ();
    case fdf::ItemPayAgentFlag::Attorney    : return toTlvAttorney    ();
    case fdf::ItemPayAgentFlag::Broker      : return toTlvBroker      ();
    case fdf::ItemPayAgentFlag::Agent       : return toTlvAgent       ();
    default: return Tlv();
    }
}

void AgentData::parseTlv(const Tlv &tlv)
{
    operatorName_.clear();
    operatorInn_.clear();
    operatorAddress_.clear();
    bankOperation_.clear();
    agentPhones_.clear();
    paysOperatorPhones_.clear();
    operatorPhones_.clear();
    if(tlv.tag() == fdf::Tag::ItemAgentData)
    {
        Tlv::MultiStlv map = tlv.toMultiStlv();
        if(map.contains(fdf::Tag::PayingAgentPhone))
        {
            agentPhones_ = tlvToPhones(map.values(fdf::Tag::PayingAgentPhone));
        }
        if(map.contains(fdf::Tag::TransferOperatorPhone))
        {
            operatorPhones_ = tlvToPhones(map.values(fdf::Tag::TransferOperatorPhone));
        }
        if(map.contains(fdf::Tag::PayingOperatorPhone))
        {
            paysOperatorPhones_ = tlvToPhones(map.values(fdf::Tag::PayingOperatorPhone));
        }
        if(map.contains(fdf::Tag::TransferOperatorAddress))
        {
            operatorAddress_ = map.value(fdf::Tag::TransferOperatorAddress).toString().trimmed();
        }
        if(map.contains(fdf::Tag::TransferOperatorInn))
        {
            operatorInn_ = map.value(fdf::Tag::TransferOperatorInn).toString().trimmed();
        }
        if(map.contains(fdf::Tag::TransferOperatorName))
        {
            operatorName_ = map.value(fdf::Tag::TransferOperatorName).toString().trimmed();
        }
        if(map.contains(fdf::Tag::BankPayingAgentOperation))
        {
            bankOperation_ = map.value(fdf::Tag::BankPayingAgentOperation).toString().trimmed();
        }
    }
}

AgentData &AgentData::operator =(const AgentData &other) noexcept
{
    operatorName_ = other.operatorName_;
    operatorInn_ = other.operatorInn_;
    operatorAddress_ = other.operatorAddress_;
    bankOperation_ = other.bankOperation_;
    agentPhones_ = other.agentPhones_;
    paysOperatorPhones_ = other.paysOperatorPhones_;
    operatorPhones_ = other.operatorPhones_;
    return *this;
}
AgentData &AgentData::operator =(AgentData &&other) noexcept
{
    operatorName_.swap(other.operatorName_);
    operatorInn_.swap(other.operatorInn_);
    operatorAddress_.swap(other.operatorAddress_);
    bankOperation_.swap(other.bankOperation_);
    agentPhones_.swap(other.agentPhones_);
    paysOperatorPhones_.swap(other.paysOperatorPhones_);
    operatorPhones_.swap(other.operatorPhones_);
    return *this;
}
bool AgentData::operator ==(const AgentData &other) const noexcept
{
    return operatorName_ == other.operatorName_ &&
            operatorInn_ == other.operatorInn_ &&
            operatorAddress_ == other.operatorAddress_ &&
            bankOperation_ == other.bankOperation_ &&
            agentPhones_ == other.agentPhones_ &&
            paysOperatorPhones_ == other.paysOperatorPhones_  &&
            operatorPhones_ == other.operatorPhones_;
}

bool AgentData::operator !=(const AgentData &other) const noexcept
{
    return !(*this == other);
}

QVariantList AgentData::phonesList(const QStringList &l) const
{
    QVariantList res;
    std::for_each(l.constBegin(), l.constEnd(),
                  [&res](const QString &p){if(!p.trimmed().isEmpty()) res << p.trimmed();});
    return res;
}

QStringList AgentData::parsePhoneList(const QVariantList &list) const
{
    QStringList phones;
    std::for_each(list.constBegin(), list.constEnd(),
                  [&phones](const QVariant &p){
        if(!p.toString().trimmed().isEmpty())
        {
            phones << p.toString().trimmed();
        }
    });
    return phones;
}

bool AgentData::checkPhones(fdf::Tag tag, const QString &tagName, const QStringList &phones,
                            CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(phones.isEmpty() ||
            !std::any_of(phones.constBegin(), phones.constEnd(),
                         [](const QString &p)->bool{return !p.trimmed().isEmpty();}))
    {
        err = CoreApiConst::ErrorCode::NoAnyParameter;
        msg = QStringLiteral("Не хватает реквизита %1 (%2) ")
                .arg(static_cast<qint32>(tag)).arg(tagName);
        return false;
    }
    for(const QString &p :phones)
    {
        if(p.trimmed().isEmpty()) continue;
        if(!FormatUtils::checkPhone(p, 19, false, msg))
        {
            err = CoreApiConst::ErrorCode::InvalidParameter;
            msg = QStringLiteral("Реквизит %1 (%2) ").arg(static_cast<qint32>(tag)).arg(tagName) + msg;
            return false;
        }
    }
    return true;
}

bool AgentData::isValidBankAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(!checkPhones(fdf::Tag::PayingAgentPhone, "телефон платежного агента",
                    agentPhones_, err, msg) ||
         !checkPhones(fdf::Tag::TransferOperatorPhone, "телефон оператора перевода",
                     operatorPhones_, err, msg))
    {
        return false;
    }
    if(!FormatUtils::checkString(operatorAddress_, 256, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1005 (адрес оператора перевода) " + msg;
        return false;
    }
    Inn i(operatorInn_);
    if(!i.isValid())
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1016 (ИНН оператора перевода) имеет некорректное значение " + operatorInn_;
        return false;
    }
    if(!FormatUtils::checkString(operatorName_, 64, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1026 (наименование оператора перевода) " + msg;
        return false;
    }
    if(!FormatUtils::checkString(bankOperation_, 24, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1044 (операция банковского платежного агента) " + msg;
        return false;
    }
    return true;
}

bool AgentData::isValidBankSubAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    return isValidBankAgent(err, msg);
}

bool AgentData::isValidPayAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    return checkPhones(fdf::Tag::PayingAgentPhone, "телефон платежного агента",
                       agentPhones_, err, msg) &&
            checkPhones(fdf::Tag::PayingOperatorPhone, "телефон оператора по приему платежей",
                        paysOperatorPhones_, err, msg);
}

bool AgentData::isValidPaySubAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    return checkPhones(fdf::Tag::PayingAgentPhone, "телефон платежного агента",
                       agentPhones_, err, msg) &&
            checkPhones(fdf::Tag::PayingOperatorPhone, "телефон оператора по приему платежей",
                        paysOperatorPhones_, err, msg);
}

bool AgentData::isValidAttorney(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::InvalidItemAgentData;
    msg = QObject::tr("Некорректные данные агента. Перечень реквизитов для разных признаков агента смотрите в описании ФФД");
    return false;
}

bool AgentData::isValidBroker(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::InvalidItemAgentData;
    msg = QObject::tr("Некорректные данные агента. Перечень реквизитов для разных признаков агента смотрите в описании ФФД");
    return false;

}

bool AgentData::isValidAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::InvalidItemAgentData;
    msg = QObject::tr("Некорректные данные агента. Перечень реквизитов для разных признаков агента смотрите в описании ФФД");
    return false;

}
/*
 * TransferOperatorAddress
 * TransferOperatorInn
 * TransferOperatorName
 * BankPayingAgentOperation
 */


Tlv AgentData::toTlvBankAgent() const
{
    Tlv::Stlv list = phonesToTlv(agentPhones_,fdf::Tag::PayingAgentPhone);
    list.append(phonesToTlv(operatorPhones_,fdf::Tag::TransferOperatorPhone));
    Tlv buf;
    buf.setString(fdf::Tag::TransferOperatorAddress, operatorAddress_);
    list << buf;
    buf.setInn(fdf::Tag::TransferOperatorInn, Inn(operatorInn_));
    list << buf;
    buf.setString(fdf::Tag::TransferOperatorName, operatorName_);
    list << buf;
    buf.setString(fdf::Tag::BankPayingAgentOperation, bankOperation_);
    list << buf;
    buf.setStlv(fdf::Tag::ItemAgentData, list);
    return buf;
}

Tlv AgentData::toTlvBankSubAgent() const
{
    Tlv::Stlv list = phonesToTlv(agentPhones_,fdf::Tag::PayingAgentPhone);
    list.append(phonesToTlv(operatorPhones_,fdf::Tag::TransferOperatorPhone));
    Tlv buf;
    buf.setString(fdf::Tag::TransferOperatorAddress, operatorAddress_);
    list << buf;
    buf.setInn(fdf::Tag::TransferOperatorInn, Inn(operatorInn_));
    list << buf;
    buf.setString(fdf::Tag::TransferOperatorName, operatorName_);
    list << buf;
    buf.setString(fdf::Tag::BankPayingAgentOperation, bankOperation_);
    list << buf;
    buf.setStlv(fdf::Tag::ItemAgentData, list);
    return buf;
}

Tlv AgentData::toTlvPayAgent() const
{
    Tlv buf;
    Tlv::Stlv list = phonesToTlv(agentPhones_,fdf::Tag::PayingAgentPhone);
    list.append(phonesToTlv(paysOperatorPhones_,fdf::Tag::PayingOperatorPhone));
    buf.setStlv(fdf::Tag::ItemAgentData, list);
    return buf;
}

Tlv AgentData::toTlvPaySubAgent() const
{
    Tlv buf;
    Tlv::Stlv list = phonesToTlv(agentPhones_,fdf::Tag::PayingAgentPhone);
    list.append(phonesToTlv(paysOperatorPhones_,fdf::Tag::PayingOperatorPhone));
    buf.setStlv(fdf::Tag::ItemAgentData, list);
    return buf;
}

Tlv AgentData::toTlvAttorney() const
{
    return Tlv();
}

Tlv AgentData::toTlvBroker() const
{
    return Tlv();
}

Tlv AgentData::toTlvAgent() const
{
    return Tlv();
}

Tlv::Stlv AgentData::phonesToTlv(const QStringList &phones, fdf::Tag tag) const
{
    Tlv::Stlv list;
    std::for_each(phones.constBegin(), phones.constEnd(),
                  [&list, tag](const QString &p) {
        if(!p.trimmed().isEmpty()) {
            Tlv buf;
            buf.setString(tag, p);
            list << buf;
        }
    });
    return list;
}

QStringList AgentData::tlvToPhones(const Tlv::Stlv &list) const
{
    QStringList res;
    std::for_each(list.constBegin(), list.constEnd(),
                  [&res](const Tlv&t){res << t.toString().trimmed();});
    return res;
}


//--------------------------------------------------------------------------------------------------

CorrectionReason::CorrectionReason() noexcept
    : date_()
    , docNumber_()
    , isIndependent_(false)
{

}

CorrectionReason::CorrectionReason(const CorrectionReason &other) noexcept
    : date_(other.date_)
    , docNumber_(other.docNumber_)
    , isIndependent_(other.isIndependent_)
{

}

CorrectionReason::CorrectionReason(CorrectionReason &&other) noexcept
    : date_(other.date_)
    , docNumber_()
    , isIndependent_(other.isIndependent_)

{
    docNumber_.swap(other.docNumber_);
}

CorrectionReason::CorrectionReason(const QVariantMap &map)
    : date_()
    , docNumber_()

{
    parseMap(map);
}

CorrectionReason::~CorrectionReason()
{

}

bool CorrectionReason::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(err, msg);
}

bool CorrectionReason::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(!date_.isValid())
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1178 (дата совершения корректируемого расчета) имеет некорректное значение " + date_.toString();
        return false;
    }

    if(!FormatUtils::checkString(docNumber_, 32, false, isIndependent_, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1179 (номер предписания налогового органа)"+ msg;
        return false;
    }
    return true;
}

const QDate &CorrectionReason::date() const
{
    return date_;
}

void CorrectionReason::setDate(const QDate &newDate)
{
    date_ = newDate;
}

const QString &CorrectionReason::docNumber() const
{
    return docNumber_;
}

void CorrectionReason::setDocNumber(const QString &newDocNumber)
{
    docNumber_ = STR4FS_(newDocNumber);
}

bool CorrectionReason::isIndependent() const
{
    return isIndependent_;
}

void CorrectionReason::setIsIndependent(bool newIsIndependent)
{
    isIndependent_ = newIsIndependent;
}
QVariantMap CorrectionReason::toMap() const
{
    QVariantMap res {
        {"date", date_.toString(Qt::ISODate)},
        {"isIndependent", isIndependent_}
    };
    if(!docNumber_.trimmed().isEmpty()) res["docNumber"] = docNumber_.trimmed();
    return res;
}

void CorrectionReason::parseMap(const QVariantMap &map)
{
    date_ = QDate::fromString(map["date"].toString().trimmed(), Qt::ISODate);
    docNumber_ = STR4FS_(map["docNumber"].toString().trimmed());
    isIndependent_ = map.contains("isIndependent") && map["isIndependent"].toBool();
}

Tlv CorrectionReason::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv buf;
    Tlv::Stlv list;
    buf.setDate(fdf::Tag::CorrecionDate, date_);
    list << buf;
    if(!isIndependent_ && !docNumber_.isEmpty())
    {
        buf.setString(fdf::Tag::FnsResolutionNumber, docNumber_);
        list << buf;
    }
    buf.setStlv(fdf::Tag::CorrectionReason, list);
    return buf;
}

void CorrectionReason::parseTlv(const Tlv &tlv)
{
    date_ = QDate();
    docNumber_.clear();
    if(tlv.tag() == fdf::Tag::CorrectionReason)
    {
        Tlv::MapStlv map = tlv.toMapStlv();
        if(map.contains(fdf::Tag::CorrecionDate)) date_ = map[fdf::Tag::CorrecionDate].toDate();
        if(map.contains(fdf::Tag::FnsResolutionNumber)) docNumber_ = map[fdf::Tag::FnsResolutionNumber].toString().trimmed();
    }
}

CorrectionReason &CorrectionReason::operator =(const CorrectionReason &other) noexcept
{
    date_ = other.date_;
    docNumber_ = other.docNumber_;
    isIndependent_ = other.isIndependent_;
    return *this;
}
CorrectionReason &CorrectionReason::operator =(CorrectionReason &&other) noexcept
{
    date_ = other.date_;
    docNumber_.swap(other.docNumber_);
    isIndependent_ = other.isIndependent_;
    return *this;

}

bool CorrectionReason::operator ==(const CorrectionReason &other) const noexcept
{
    return date_ == other.date_ &&
            docNumber_ == other.docNumber_ &&
            isIndependent_ == other.isIndependent_;
}

bool CorrectionReason::operator !=(const CorrectionReason &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------

ProductCode::ProductCode() noexcept
    : type_(Type::LabelUnknown)
    , data_()
{

}

ProductCode::ProductCode(const Type &t, const QString &data)
    : type_(t)
    , data_(data)
{

}

ProductCode::ProductCode(const ProductCode &other) noexcept
    : type_(other.type_)
    , data_(other.data_)
{

}

ProductCode::ProductCode(ProductCode &&other) noexcept
    : type_(other.type_)
    , data_()

{
    data_.swap(other.data_);
}

ProductCode::ProductCode(const QVariantMap &map)
    : type_(Type::LabelUnknown)
    , data_()

{
    parseMap(map);
}

ProductCode::~ProductCode()
{

}

bool ProductCode::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(err, msg);
}

bool ProductCode::isEmpty() const
{
    return data_.isEmpty();
}

bool ProductCode::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    static const QSet<quint16> SUPPORTED =  {
        1300, 1301, 1302, 1303, 1304, 1305, 1306, 1307, 1308,
        1309, 1320, 1321, 1322, 1323, 1324, 1325
    };
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(!SUPPORTED.contains(static_cast<quint16>(type_)))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректный тип кода товара";
        return false;
    }
    return true;

}

bool ProductCode::isGs() const
{
    return type_ == Type::LabelGs10 ||
            type_ == Type::LabelGs1M ||
            type_ == Type::LabelShortCode;
}
bool ProductCode::isGsM() const
{
    return type_ == Type::LabelGs1M ||
            type_ == Type::LabelShortCode;
}
ProductCode::Type ProductCode::type() const
{
    return type_;
}

void ProductCode::setType(Type newType)
{
    type_ = newType;
}

const QString &ProductCode::data() const
{
    return data_;
}

void ProductCode::setData(const QString &newData)
{
    data_ = STR4FS_(newData);
}

QVariantMap ProductCode::toMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"type", static_cast<qint32>(type_)},
        {"data", data_}
    };
}

QVariantMap ProductCode::toExternalMap() const
{
    if(isEmpty()) return QVariantMap();
    return {
        {"type", static_cast<qint32>(type_)},
        {"data", data_}
    };
}

void ProductCode::parseMap(const QVariantMap &map)
{
    clean();
    type_ = static_cast<Type>(map["type"].toInt());
    data_ = STR4FS_(map["data"].toString().trimmed());
}

Tlv ProductCode::toTlv() const
{
    Tlv res;
    res.setString(static_cast<fdf::Tag>(static_cast<quint16>(type_)), data_);
    Tlv::Stlv list;
    list << res;
    res.setStlv(fdf::Tag::ProductCode, list);
    return res;
}

void ProductCode::parseTlv(const Tlv &tlv)
{
    type_ = Type::LabelUnknown;
    data_.clear();
    if(tlv.tag() == fdf::Tag::ProductCode)
    {
        Tlv::Stlv list = tlv.toStlv();
        if(!list.isEmpty())
        {
            type_ = static_cast<Type>(static_cast<quint16>(list.first().tag()));
            data_ = list.first().toString().trimmed();
        }
    }
}

void ProductCode::clean()
{
    type_ = Type::LabelUnknown;
    data_.clear();
}

ProductCode &ProductCode::operator =(const ProductCode &other) noexcept
{
    type_ = other.type_;
    data_ = other.data_;
    return *this;
}
ProductCode &ProductCode::operator =(ProductCode &&other) noexcept
{
    type_ = other.type_;
    data_ .swap(other.data_);
    return *this;

}

bool ProductCode::operator ==(const ProductCode &other) const noexcept
{
    return type_ == other.type_ &&
            data_ == other.data_;
}

bool ProductCode::operator !=(const ProductCode &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------


ReceiptsVatAmount::ReceiptsVatAmount() noexcept
{

}

ReceiptsVatAmount::ReceiptsVatAmount(const ReceiptsVatAmount &other) noexcept
    : type_(other.type_)
    , amount_(other.amount_)
{

}

ReceiptsVatAmount::ReceiptsVatAmount(ReceiptsVatAmount &&other) noexcept
    : type_(other.type_)
    , amount_(other.amount_)
{

}

ReceiptsVatAmount::ReceiptsVatAmount(fdf::VatRate rate, const FixNumber &amount) noexcept
    : type_(rate)
    , amount_(amount)
{

}

ReceiptsVatAmount::ReceiptsVatAmount(const QString &rateName, const FixNumber &amount) noexcept
    : amount_(amount)
{
    setType(rateName);
}

ReceiptsVatAmount::~ReceiptsVatAmount()
{

}


bool ReceiptsVatAmount::isValid() const
{
    return type_ >= fdf::VatRate::Vat5 && type_ <= fdf::VatRate::Vat7_107 && amount_.value() >= 0ll;
}

bool ReceiptsVatAmount::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    msg.clear();
    err = CoreApiConst::ErrorCode::Ok;
    if(type_ < fdf::VatRate::Vat5 || type_ > fdf::VatRate::Vat7_107 )
    {
        err = CoreApiConst::ErrorCode::InvalidVatRate;
        msg = QObject::tr("Значение реквизита 1199 в составе 1119 может быть 7, 8, 9 или 10");
        return false;
    }
    if(amount_.value() < 0)
    {
        err = CoreApiConst::ErrorCode::InvalidSumFormat;
        msg = QObject::tr("Значение реквизита 1120 не может быть меньше 0");
        return false;
    }
    return true;
}

fdf::VatRate ReceiptsVatAmount::type() const
{
    return type_;
}

void ReceiptsVatAmount::setType(fdf::VatRate newType)
{
    type_ = newType;
}

void ReceiptsVatAmount::setType(const QString &name)
{
    type_ = fdf::VatRate::Invalid;
    if(name == "vat5") type_ = fdf::VatRate::Vat5;
    else if(name == "vat7") type_ = fdf::VatRate::Vat7;
    else if(name == "vat5_105") type_ = fdf::VatRate::Vat5_105;
    else if(name == "vat7_107") type_ = fdf::VatRate::Vat7_107;
}

void ReceiptsVatAmount::setType(quint8 type)
{
    type_ = static_cast<fdf::VatRate>(type);
    if(type_ < fdf::VatRate::Vat5 || type_ > fdf::VatRate::Vat7_107) type_ = fdf::VatRate::Invalid;
}

QString ReceiptsVatAmount::typeName() const
{
    switch (type_)
    {
    case fdf::VatRate::Vat5     : return "vat5"    ;
    case fdf::VatRate::Vat7     : return "vat7"    ;
    case fdf::VatRate::Vat5_105 : return "vat5_105" ;
    case fdf::VatRate::Vat7_107 : return "vat7_107";

    default: return QString();
    }
}

FixNumber ReceiptsVatAmount::amount() const
{
    return amount_;
}

void ReceiptsVatAmount::setAmount(const FixNumber &newAmount)
{
    amount_ = newAmount;
}

void ReceiptsVatAmount::addAmount(const FixNumber &a)
{
    if(amount_.value() <= 0ll) amount_ = a;
    else
    {
        amount_.setValue(a.value());
    }
}

Tlv ReceiptsVatAmount::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv::Stlv list;
    Tlv buf;
    buf.setByte(fdf::Tag::VatRate, static_cast<quint8>(type_));
    list << buf;
    buf.setVln(fdf::Tag::VatAmount, amount_.value());
    list << buf;
    buf.setStlv(fdf::Tag::ReceiptVatAmount, list);
    return buf;
}

void ReceiptsVatAmount::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::ReceiptVatAmount)
    {
        Tlv::Stlv list = tlv.toStlv();
        for(const Tlv &t: list)
        {
            if(t.tag() == fdf::Tag::VatRate) setType(t.toByte());
            else if(t.tag() == fdf::Tag::VatAmount) amount_ = FixNumber(DEFAULT_AMOUNT_MULT, t.toVln());
            else {
                //UNSUPPORTED TAG
            }
        }
    }
}

void ReceiptsVatAmount::clean()
{
    type_ = fdf::VatRate::Invalid;
    amount_ = FixNumber{DEFAULT_AMOUNT_MULT, 0ll};
}

ReceiptsVatAmount &ReceiptsVatAmount::operator =(const ReceiptsVatAmount &other) noexcept
{
    type_ = other.type_;
    amount_ = other.amount_;
    return *this;
}


ReceiptsVatAmount &ReceiptsVatAmount::operator =(ReceiptsVatAmount &&other) noexcept
{
    type_ = other.type_;
    amount_ = other.amount_;
    return *this;
}


bool ReceiptsVatAmount::operator ==(const ReceiptsVatAmount &other) const noexcept
{
    return type_ == other.type_ && amount_ == other.amount_;
}

bool ReceiptsVatAmount::operator !=(const ReceiptsVatAmount &other) const noexcept
{
    return (*this == other);
}

//--------------------------------------------------------------------------------------------------

ReceiptsVatAmounts::ReceiptsVatAmounts() noexcept
{

}

ReceiptsVatAmounts::ReceiptsVatAmounts(const ReceiptsVatAmounts &other) noexcept
    : vat5_(other.vat5_)
    , vat7_(other.vat7_)
    , vat5_105_(other.vat5_105_)
    , vat7_107_(other.vat7_107_)
{

}

ReceiptsVatAmounts::ReceiptsVatAmounts(ReceiptsVatAmounts &&other) noexcept
{
    vat5_.swap(other.vat5_);
    vat7_.swap(other.vat7_);
    vat5_105_.swap(other.vat5_105_);
    vat7_107_.swap(other.vat7_107_);
}

ReceiptsVatAmounts::~ReceiptsVatAmounts()
{

}

bool ReceiptsVatAmounts::isValid() const
{
    return (!vat5_.has_value() || vat5_->isValid()) &&
           (!vat7_.has_value() || vat7_->isValid()) &&
           (!vat5_105_.has_value() || vat5_105_->isValid()) &&
           (!vat7_107_.has_value() || vat7_107_->isValid()) && !isNull();
}

bool ReceiptsVatAmounts::isNull() const
{
    return !vat5_.has_value() && !vat7_.has_value() && !vat5_105_.has_value() && !vat7_107_.has_value();
}

std::optional<ReceiptsVatAmount> ReceiptsVatAmounts::vat5() const
{
    return vat5_;
}

void ReceiptsVatAmounts::setVat5(std::optional<ReceiptsVatAmount> newVat5)
{
    vat5_ = newVat5;
}

std::optional<ReceiptsVatAmount> ReceiptsVatAmounts::vat7() const
{
    return vat7_;
}

void ReceiptsVatAmounts::setVat7(std::optional<ReceiptsVatAmount> newVat7)
{
    vat7_ = newVat7;
}

std::optional<ReceiptsVatAmount> ReceiptsVatAmounts::vat5_105() const
{
    return vat5_105_;
}

void ReceiptsVatAmounts::setVat5_105(std::optional<ReceiptsVatAmount> newVat5_105)
{
    vat5_105_ = newVat5_105;
}

std::optional<ReceiptsVatAmount> ReceiptsVatAmounts::vat7_107() const
{
    return vat7_107_;
}

void ReceiptsVatAmounts::setVat7_107(std::optional<ReceiptsVatAmount> newVat7_107)
{
    vat7_107_ = newVat7_107;
}

void ReceiptsVatAmounts::appendVat(fdf::VatRate rate, const FixNumber &value)
{
    auto store = [](std::optional<ReceiptsVatAmount> &vat, fdf::VatRate rate, const FixNumber &value) -> void {
        if(vat.has_value())
        {
            vat->addAmount(value);
        }
        else
        {
            ReceiptsVatAmount a(rate, value);
            vat = a;
        }
    };
    switch(rate)
    {
    case fdf::VatRate::Vat5: store(vat5_, rate, value);break;
    case fdf::VatRate::Vat7: store(vat7_, rate, value);break;
    case fdf::VatRate::Vat5_105: store(vat5_105_, rate, value);break;
    case fdf::VatRate::Vat7_107: store(vat7_107_, rate, value);break;
    default: break;
    }
}

void ReceiptsVatAmounts::clean()
{
    vat5_.reset();
    vat7_.reset();
    vat5_105_.reset();
    vat7_107_.reset();
}

Tlv ReceiptsVatAmounts::toTlv() const
{
    if(!isValid()) return Tlv();
    Tlv::Stlv list;
    if(vat5_.has_value()) list << vat5_->toTlv();
    if(vat7_.has_value()) list << vat7_->toTlv();
    if(vat5_105_.has_value()) list << vat5_105_->toTlv();
    if(vat7_107_.has_value()) list << vat7_107_->toTlv();
    Tlv res;
    res.setStlv(fdf::Tag::VatAmounts, list);
    return res;
}

void ReceiptsVatAmounts::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::VatAmounts)
    {
        Tlv::Stlv list = tlv.toStlv();
        for(const Tlv &t :list)
        {
            ReceiptsVatAmount a;
            a.parseTlv(t);
            if(a.isValid())
            {
                switch (a.type()) {
                case fdf::VatRate::Vat5:
                    vat5_ = a;
                    break;
                case fdf::VatRate::Vat7:
                    vat7_ = a;
                    break;
                case fdf::VatRate::Vat5_105:
                    vat5_105_ = a;
                    break;
                case fdf::VatRate::Vat7_107:
                    vat7_107_ = a;
                    break;
                default:
                    break;
                }
            }
        }
    }
}

ReceiptsVatAmounts &ReceiptsVatAmounts::operator =(const ReceiptsVatAmounts &other) noexcept
{
    vat5_ = other.vat5_;
    vat7_ = other.vat7_;
    vat5_105_ = other.vat5_105_;
    vat7_107_ = other.vat7_107_;
    return *this;
}

ReceiptsVatAmounts &ReceiptsVatAmounts::operator =(ReceiptsVatAmounts &&other) noexcept
{
    vat5_.swap(other.vat5_);
    vat7_.swap(other.vat7_);
    vat5_105_.swap(other.vat5_105_);
    vat7_107_.swap(other.vat7_107_);
    return *this;
}
bool ReceiptsVatAmounts::operator ==(const ReceiptsVatAmounts &other) const noexcept
{
    return vat5_ == other.vat5_ &&
           vat7_ == other.vat7_ &&
           vat5_105_ == other.vat5_105_ &&
           vat7_107_ == other.vat7_107_;
}

bool ReceiptsVatAmounts::operator !=(const ReceiptsVatAmounts &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

CardPaymentInfo::CardPaymentInfo() noexcept
{

}

CardPaymentInfo::CardPaymentInfo(const CardPaymentInfo &other) noexcept
    : amount_(other.amount_)
    , way_(other.way_)
    , ids_(other.ids_)
    , additional_(other.additional_)
{

}

CardPaymentInfo::CardPaymentInfo(CardPaymentInfo &&other) noexcept
    : amount_(other.amount_)
    , way_(other.way_)
{
    ids_.swap(other.ids_);
    additional_.swap(other.additional_);
}

CardPaymentInfo::~CardPaymentInfo()
{

}

bool CardPaymentInfo::isValid(CoreApiConst::ErrorCode &error, QString &msg) const
{
    if(amount_.value() <= 0)
    {
        error = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректное значение суммы оплаты безналичными (1082)";
        return false;
    }
    if(ids_.size() > 256)
    {
        error = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректное значение идентификаторов безналичной оплаты (1237)";
        return false;
    }
    if(additional_.size() > 256)
    {
        error = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Некорректное значение дополнительных сведений о безналичной оплате (1238)";
        return false;
    }
    return true;
}

bool CardPaymentInfo::isValid() const
{
    return amount_.value() > 0 && ids_.size() < 256 && additional_.size() < 256;
}

const FixNumber &CardPaymentInfo::amount() const
{
    return amount_;
}

void CardPaymentInfo::setAmount(const FixNumber &newAmount)
{
    amount_ = newAmount;
}

quint8 CardPaymentInfo::way() const
{
    return way_;
}

void CardPaymentInfo::setWay(quint8 newWay)
{
    way_ = newWay;
}

const QString &CardPaymentInfo::ids() const
{
    return ids_;
}

void CardPaymentInfo::setIds(const QString &newIds)
{
    ids_ = newIds;
}

const QString &CardPaymentInfo::additional() const
{
    return additional_;
}

void CardPaymentInfo::setAdditional(const QString &newAdditional)
{
    additional_ = newAdditional;
}

void CardPaymentInfo::clean()
{
    amount_ = {DEFAULT_AMOUNT_MULT, 0ll};
    way_ = 0;
    ids_.clear();
    additional_.clear();
}

Tlv CardPaymentInfo::toTlv() const
{
//    if(!isValid()) return Tlv();
    Tlv::Stlv list;
    Tlv tlv;
    tlv.setVln(fdf::Tag::ReceiptCardByWay, amount_.value());
    list << tlv;
    tlv.setByte(fdf::Tag::CardPayWay, way_);
    list << tlv;
    if(!ids_.isEmpty())
    {
        tlv.setString(fdf::Tag::CardPayIds, ids_);
        list << tlv;
    }
    if(!additional_.isEmpty())
    {
        tlv.setString(fdf::Tag::AdditionalCardPayInfo, additional_);
        list << tlv;
    }
    tlv.setStlv(fdf::Tag::CardPayInfo, list);
    return tlv;
}

void CardPaymentInfo::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::CardPayInfo)
    {
        Tlv::Stlv list = tlv.toStlv();
        for(const Tlv &t :list)
        {
            switch (t.tag()) {
            case fdf::Tag::ReceiptCardByWay:
                amount_ = FixNumber(DEFAULT_AMOUNT_MULT, t.toVln());
                break;
            case fdf::Tag::CardPayWay:
                way_ = t.toByte();
                break;
            case fdf::Tag::CardPayIds:
                ids_ = t.toString();
                break;
            case fdf::Tag::AdditionalCardPayInfo:
                additional_ = t.toString();
                break;
            default:
                break;
            }
        }
    }
}

QVariantMap CardPaymentInfo::toMap() const
{
    return {
        {"amount", amount_.toMap()},
        {"wayType", way_},
        {"ids", ids_},
        {"additional", additional_}
    };
}

QVariantMap CardPaymentInfo::toExtMap() const
{
    return {
        {"amount", amount_.toString()},
        {"wayType", way_},
        {"ids", ids_},
        {"additional", additional_}
    };

}

void CardPaymentInfo::parseMap(const QVariantMap &map)
{
    clean();
    amount_ = FormatUtils::parseSumm("amount", map, DEFAULT_AMOUNT_DEC);
    way_ = map["wayType"].toUInt();
    ids_ = map["ids"].toString().trimmed();
    additional_ = map["additional"].toString().trimmed();
}

CardPaymentInfo &CardPaymentInfo::operator =(const CardPaymentInfo &other) noexcept
{
    amount_ = other.amount_;
    way_ = other.way_;
    ids_ = other.ids_;
    additional_= other.additional_;
    return *this;
}

CardPaymentInfo &CardPaymentInfo::operator =(CardPaymentInfo &&other) noexcept
{
    amount_ = other.amount_;
    way_ = other.way_;
    ids_.swap(other.ids_);
    additional_.swap(other.additional_);
    return *this;
}

bool CardPaymentInfo::operator ==(const CardPaymentInfo &other) const noexcept
{
    return amount_ == other.amount_ &&
           way_ == other.way_ &&
           ids_ == other.ids_ &&
           additional_== other.additional_;
}

bool CardPaymentInfo::operator !=(const CardPaymentInfo &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------
CardPaymentsInfo::CardPaymentsInfo() noexcept
{

}

CardPaymentsInfo::CardPaymentsInfo(const CardPaymentsInfo &other) noexcept
    : payments_(other.payments_)
{

}

CardPaymentsInfo::CardPaymentsInfo(CardPaymentsInfo &&other) noexcept
{
    payments_.swap(other.payments_);
}

CardPaymentsInfo::~CardPaymentsInfo()
{

}

bool CardPaymentsInfo::isEmpty() const
{
    return payments_.isEmpty();
}

bool CardPaymentsInfo::isValid() const
{
    return !isEmpty() && std::all_of(payments_.constBegin(), payments_.constEnd(),
                                     [](const CardPaymentInfo &i){return i.isValid();});
}

bool CardPaymentsInfo::isValid(CoreApiConst::ErrorCode &error, QString &msg) const
{
    return !isEmpty() && std::all_of(payments_.constBegin(), payments_.constEnd(),
                                     [&error, &msg](const CardPaymentInfo &i){return i.isValid(error, msg);});

}

const QVector<CardPaymentInfo> &CardPaymentsInfo::payments() const
{
    return payments_;
}

void CardPaymentsInfo::setPayments(const QVector<CardPaymentInfo> &newPayments)
{
    payments_ = newPayments;
}

void CardPaymentsInfo::clean()
{
    payments_.clear();
}

Tlv CardPaymentsInfo::toTlv() const
{
    if(payments_.isEmpty()) return {};
    Tlv::Stlv list;
    for(const CardPaymentInfo &p :payments_) list << p.toTlv();
    Tlv tlv;
    tlv.setStlv(fdf::Tag::AllCardPaysInfo, list);
    return tlv;
}

void CardPaymentsInfo::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::AllCardPaysInfo)
    {
        Tlv::Stlv list = tlv.toStlv();
        for(const Tlv &t :list)
        {
            switch (t.tag()) {
            case fdf::Tag::CardPayInfo:
            {
                CardPaymentInfo p;
                p.parseTlv(t);
                payments_.push_back(p);
            }break;
            default:
                break;
            }
        }
    }
}

QVariantList CardPaymentsInfo::toList() const
{
    QVariantList res;
    res = std::accumulate(payments_.constBegin(), payments_.constEnd(), res,
                          [](QVariantList &list, const CardPaymentInfo &info) -> QVariantList {
                              if(info.isValid()) list << info.toMap();
                              return list;
                          });
    return res;
}

QVariantList CardPaymentsInfo::toExtList() const
{
    QVariantList res;
    res = std::accumulate(payments_.constBegin(), payments_.constEnd(), res,
                          [](QVariantList &list, const CardPaymentInfo &info) -> QVariantList {
                              if(info.isValid()) list << info.toExtMap();
                              return list;
                          });
    return res;
}

void CardPaymentsInfo::parseList(const QVariantList &data)
{
    clean();
    payments_ = std::accumulate(data.constBegin(), data.constEnd(), payments_,
                                [](QVector<CardPaymentInfo> &list, const QVariant &v) -> QVector<CardPaymentInfo> {
                                    CardPaymentInfo i;
                                    i.parseMap(v.toMap());
                                    list << i;
                                    return list;
                                });
}


CardPaymentsInfo &CardPaymentsInfo::operator =(const CardPaymentsInfo &other) noexcept
{
    payments_ = other.payments_;
    return *this;
}

CardPaymentsInfo &CardPaymentsInfo::operator =(CardPaymentsInfo &&other) noexcept
{
    payments_.swap(other.payments_);
    return *this;
}

bool CardPaymentsInfo::operator ==(const CardPaymentsInfo &other) const noexcept
{
    return payments_ == other.payments_;
}

bool CardPaymentsInfo::operator !=(const CardPaymentsInfo &other) const noexcept
{
    return !(*this == other);
}
