#ifndef RECEIPTDATACLASSES_H
#define RECEIPTDATACLASSES_H

#include "cashboxbuildconfig.h"
#include "tlv.h"
#include "fixnumber.h"
#include "coreapiconst.h"

#include <QVariantMap>
#include <QDate>
#include <QSet>

class ReceiptAdditionalUserParam
{
public:
    ReceiptAdditionalUserParam() noexcept;
    ReceiptAdditionalUserParam(const ReceiptAdditionalUserParam &other) noexcept;
    ReceiptAdditionalUserParam(ReceiptAdditionalUserParam &&other) noexcept;
    explicit ReceiptAdditionalUserParam(const QVariantMap &map) noexcept;
    ~ReceiptAdditionalUserParam();

    bool isEmpty() const;

    bool isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const;

    const QString &name() const;
    void setName(const QString &newName);

    const QString &value() const;
    void setValue(const QString &newValue);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    ReceiptAdditionalUserParam &operator =(const ReceiptAdditionalUserParam &other) noexcept;
    ReceiptAdditionalUserParam &operator =(ReceiptAdditionalUserParam &&other) noexcept;
    bool operator ==(const ReceiptAdditionalUserParam &other) const noexcept;
    bool operator !=(const ReceiptAdditionalUserParam &other) const noexcept;

private:
    QString name_;//64
    QString value_;//256
};

//--------------------------------------------------------------------------------------------------

class QuantityFraction
{
public:
    QuantityFraction() noexcept;
    QuantityFraction(const QuantityFraction &other)  = default;
    QuantityFraction(QuantityFraction &&other) = default;
    explicit QuantityFraction(const QVariantMap &map);
    ~QuantityFraction();

    bool is1() const;
    bool isValid() const;
    bool isValid(CoreApiConst::ErrorCode &err, QString &msg);

    qint64 nominator() const;
    void setNominator(qint64 newNominator);

    qint64 denominator() const;
    void setDenominator(qint64 newDenominator);

    QString text() const;
    FixNumber fixNumber() const;

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);


    QuantityFraction &operator =(const QuantityFraction &other) = default;
    QuantityFraction &operator =(QuantityFraction &&other) = default;
    bool operator ==(const QuantityFraction &other) const noexcept;
    bool operator !=(const QuantityFraction &other) const noexcept;

private:
    qint64 nominator_;
    qint64 denominator_;
};
//--------------------------------------------------------------------------------------------------

class ReceiptOperationalParam
{
public:
    ReceiptOperationalParam()noexcept;
    ReceiptOperationalParam(const ReceiptOperationalParam &other) noexcept;
    ReceiptOperationalParam(ReceiptOperationalParam &&other) noexcept;
    explicit ReceiptOperationalParam(const QVariantMap &map);
    ~ReceiptOperationalParam();

    bool isValid() const;
    bool isEmpty() const;
    bool isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const;

    quint8 code() const;
    void setCode(quint8 newCode);

    const QDateTime &dt() const;
    void setDt(const QDateTime &newDt);

    const QString &data() const;
    void setData(const QString &newData);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);


    ReceiptOperationalParam &operator =(const ReceiptOperationalParam &other)noexcept;
    ReceiptOperationalParam &operator =(ReceiptOperationalParam &&other) noexcept;
    bool operator ==(const ReceiptOperationalParam &other) const noexcept;
    bool operator !=(const ReceiptOperationalParam &other) const noexcept;

private:
    quint8 code_;
    QDateTime dt_;
    QString data_; //64
};

//--------------------------------------------------------------------------------------------------

class BuyerInfo
{
public:
    static const QSet<QString> RUSSIAN_DOCS;
    static const QSet<QString> FOREIGH_DOCS;
    BuyerInfo() noexcept;
    BuyerInfo(const BuyerInfo &other) noexcept;
    BuyerInfo(BuyerInfo &&other) noexcept;
    explicit BuyerInfo(const QVariantMap &map) noexcept;
    ~BuyerInfo();

    bool isValid() const;
    bool isEmpty() const;
    bool isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const;

    const QString &name() const;
    void setName(const QString &newName);

    const QString &inn() const;
    void setInn(const QString &newInn);

    const QDate &birthDay() const;
    void setBirthDay(const QDate &d);

    const QString &citizenship() const;
    void setCitizenship(const QString &newCitizenship);

    const QString &docCode() const;
    void setDocCode(const QString &newDocCode);

    const QString &docData() const;
    void setDocData(const QString &newDocData);

    const QString &address() const;
    void setAddress(const QString &newAddress);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    BuyerInfo &operator =(const BuyerInfo &other)noexcept;
    BuyerInfo &operator =(BuyerInfo &&other) noexcept;
    bool operator ==(const BuyerInfo &other) const noexcept;
    bool operator !=(const BuyerInfo &other) const noexcept;

private:
    QString name_;
    QString inn_;
    QDate birthDay_;
    QString citizenship_;
    QString docCode_;
    QString docData_;
    QString address_;
};
//--------------------------------------------------------------------------------------------------
class IndustryProperty
{
    static const qint32 MAX_FED_ID = 72;
public:
    IndustryProperty() noexcept;
    IndustryProperty(const IndustryProperty &other) noexcept;
    IndustryProperty(IndustryProperty &&other) noexcept;
    explicit IndustryProperty(const QVariantMap &map) noexcept;
    ~IndustryProperty();

    bool isValid() const;
    bool isEmpty() const;
    bool isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const;

    const QString &fedId() const;
    void setFedId(const QString &newFedId);

    const QDate &docDt() const;
    void setDocDt(const QDate &newDocDt);

    const QString &docNumber() const;
    void setDocNumber(const QString &newDocNumber);

    const QString &value() const;
    void setValue(const QString &newValue);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv(fdf::Tag tag) const;
    void parseTlv(const Tlv &tlv);

    IndustryProperty &operator =(const IndustryProperty &other)noexcept;
    IndustryProperty &operator =(IndustryProperty &&other) noexcept;
    bool operator ==(const IndustryProperty &other) const noexcept;
    bool operator !=(const IndustryProperty &other) const noexcept;

private:
    QString fedId_;
    QDate docDt_;
    QString docNumber_;
    QString value_;
};

//--------------------------------------------------------------------------------------------------
class ProviderData
{
public:
    ProviderData() noexcept;
    ProviderData(const ProviderData &other) noexcept;
    ProviderData(ProviderData &&other)noexcept;
    explicit ProviderData(const QVariantMap &map);
    ~ProviderData();

    bool isValid() const;
    bool isEmpty() const;
    bool isValid(bool canBeEmpty, CoreApiConst::ErrorCode &err, QString &msg) const;

    const QString &name() const;
    void setName(const QString &newName);

    const QStringList &phones() const;
    void setPhones(const QStringList &newPhone);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    ProviderData &operator =(const ProviderData &other)noexcept;
    ProviderData &operator =(ProviderData &&other) noexcept;
    bool operator ==(const ProviderData &other) const noexcept;
    bool operator !=(const ProviderData &other) const noexcept;

private:
    QString name_;
    QStringList phones_;

    QVariantList phonesList() const;
    void parsePhoneList(const QVariantList &list);

};

//--------------------------------------------------------------------------------------------------

class AgentData
{
public:
    AgentData() noexcept;
    AgentData(const AgentData &other) noexcept;
    AgentData(AgentData &&other) noexcept;
    explicit AgentData(const QVariantMap &map) noexcept;
    ~AgentData();

    bool isValid(fdf::ItemPayAgentFlag agentFlag) const;
    bool isValid(fdf::ItemPayAgentFlag agentFlag, CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isEmpty() const;


    const QString &operatorName() const;
    void setOperatorName(const QString &newOperatorName);

    const QString &operatorInn() const;
    void setOperatorInn(const QString &newOperatorInn);

    const QString &operatorAddress() const;
    void setOperatorAddress(const QString &newOperatorAddress);

    const QString &bankOperation() const;
    void setBankOperation(const QString &newBankOperation);

    const QStringList &agentPhones() const;
    void setAgentPhones(const QStringList &newAgentPhones);

    const QStringList &paysOperatorPhones() const;
    void setPaysOperatorPhones(const QStringList &newPaysOperatorPhones);

    const QStringList &operatorPhones() const;
    void setOperatorPhones(const QStringList &newOperatorPhones);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv(fdf::ItemPayAgentFlag agentFlag) const;
    void parseTlv(const Tlv &tlv);

    AgentData &operator =(const AgentData &other)noexcept;
    AgentData &operator =(AgentData &&other) noexcept;
    bool operator ==(const AgentData &other) const noexcept;
    bool operator !=(const AgentData &other) const noexcept;

private:
    QString operatorName_;
    QString operatorInn_;
    QString operatorAddress_;
    QString bankOperation_;
    QStringList agentPhones_;
    QStringList paysOperatorPhones_;
    QStringList operatorPhones_;

    QVariantList phonesList(const QStringList &l) const;
    QStringList parsePhoneList(const QVariantList &list) const;

    bool checkPhones(fdf::Tag tag, const QString &tagName, const QStringList &phones,
                     CoreApiConst::ErrorCode &err, QString &msg) const;

    bool isValidBankAgent   (CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidBankSubAgent(CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidPayAgent    (CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidPaySubAgent (CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidAttorney    (CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidBroker      (CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isValidAgent       (CoreApiConst::ErrorCode &err, QString &msg) const;

    Tlv toTlvBankAgent      ()const;
    Tlv toTlvBankSubAgent   ()const;
    Tlv toTlvPayAgent       ()const;
    Tlv toTlvPaySubAgent    ()const;
    Tlv toTlvAttorney       ()const;
    Tlv toTlvBroker         ()const;
    Tlv toTlvAgent          ()const;

    Tlv::Stlv phonesToTlv(const QStringList &phones, fdf::Tag tag) const;
    QStringList tlvToPhones(const Tlv::Stlv &list) const;
};
//--------------------------------------------------------------------------------------------------

class CorrectionReason
{
public:
    CorrectionReason()noexcept;
    CorrectionReason(const CorrectionReason &other)noexcept;
    CorrectionReason(CorrectionReason &&other)noexcept;
    explicit CorrectionReason(const QVariantMap &map);
    ~CorrectionReason();

    bool isValid() const;
    bool isValid(CoreApiConst::ErrorCode &err, QString &msg) const;

    const QDate &date() const;
    void setDate(const QDate &newDate);

    const QString &docNumber() const;
    void setDocNumber(const QString &newDocNumber);


    bool isIndependent() const;
    void setIsIndependent(bool newIsIndependent);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    CorrectionReason &operator =(const CorrectionReason &other)noexcept;
    CorrectionReason &operator =(CorrectionReason &&other) noexcept;
    bool operator ==(const CorrectionReason &other) const noexcept;
    bool operator !=(const CorrectionReason &other) const noexcept;


private:
    QDate date_;
    QString docNumber_;
    bool isIndependent_;
};

//--------------------------------------------------------------------------------------------------

class ProductCode
{
public:
    enum class Type
    {
        LabelUnknown                            = 1300, //КТ Н
        LabelEan8                               = 1301, //КТ EAN-8
        LabelEan13                              = 1302, //КТ EAN-13
        LabelItf14                              = 1303, //КТ ITF-14
        LabelGs10                               = 1304, //КТ GS1.0
        LabelGs1M                               = 1305, //КТ GS1.M
        LabelShortCode                          = 1306, //КТ КМК
        LabelFur                                = 1307, //КТ МИ
        LabelEgais2                             = 1308, //КТ ЕГАИС-2.0
        LabelEGais3                             = 1309, //КТ ЕГАИС-3.0
        LabelF1                                 = 1320, //КТ Ф.1
        LabelF2                                 = 1321, //КТ Ф.2
        LabelF3                                 = 1322, //КТ Ф.3
        LabelF4                                 = 1323, //КТ Ф.4
        LabelF5                                 = 1324, //КТ Ф.5
        LabelF6                                 = 1325, //КТ Ф.6
    };

    ProductCode()noexcept;
    ProductCode(const Type &t, const QString &data);
    ProductCode(const ProductCode &other)noexcept;
    ProductCode(ProductCode &&other)noexcept;
    explicit ProductCode(const QVariantMap &map);
    ~ProductCode();

    bool isValid() const;
    bool isEmpty() const;
    bool isValid(CoreApiConst::ErrorCode &err, QString &msg) const;
    bool isGs() const;
    bool isGsM() const;

    Type type() const;
    void setType(Type newType);

    const QString &data() const;
    void setData(const QString &newData);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);
    void clean();

    ProductCode &operator =(const ProductCode &other)noexcept;
    ProductCode &operator =(ProductCode &&other) noexcept;
    bool operator ==(const ProductCode &other) const noexcept;
    bool operator !=(const ProductCode &other) const noexcept;

private:
    Type type_;
    QString data_;
};

//--------------------------------------------------------------------------------------------------
enum class ReceiptOption
{
    IsCorrection                = 0x00000001,
    IsFosa                      = 0x00000002,
    InternetCashbox             = 0x00000004,
    SupportLabledTrandes        = 0x00000008,
    SupportExciseTrades         = 0x00000010,
};
Q_DECLARE_FLAGS(ReceiptOptions, ReceiptOption)

Q_DECLARE_OPERATORS_FOR_FLAGS(ReceiptOptions)
Q_DECL_CONSTEXPR inline QFlags<ReceiptOption> operator~(ReceiptOption f) Q_DECL_NOTHROW
{ return QFlags<ReceiptOption>(~static_cast<qint8>(f)) ;}

//--------------------------------------------------------------------------------------------------

class ReceiptsVatAmount
{
public:
    ReceiptsVatAmount() noexcept;
    ReceiptsVatAmount(const ReceiptsVatAmount &other) noexcept;
    ReceiptsVatAmount(ReceiptsVatAmount &&other) noexcept;
    explicit ReceiptsVatAmount(fdf::VatRate rate, const FixNumber &amount) noexcept;
    explicit ReceiptsVatAmount(const QString &rateName, const FixNumber &amount) noexcept;
    ~ReceiptsVatAmount();

    bool isValid() const;
    bool isValid(CoreApiConst::ErrorCode &err, QString &msg) const;

    fdf::VatRate type() const;
    void setType(fdf::VatRate newType);
    void setType(const QString &name);
    void setType(quint8 type);
    QString typeName() const;

    FixNumber amount() const;
    void setAmount(const FixNumber &newAmount);
    void addAmount(const FixNumber &a);

    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);
    void clean();

    ReceiptsVatAmount &operator =(const ReceiptsVatAmount &other) noexcept;
    ReceiptsVatAmount &operator =(ReceiptsVatAmount &&other) noexcept;
    bool operator ==(const ReceiptsVatAmount &other) const noexcept;
    bool operator !=(const ReceiptsVatAmount &other) const noexcept;

private:
    fdf::VatRate type_ = fdf::VatRate::Invalid;
    FixNumber amount_ = {DEFAULT_AMOUNT_MULT, 0ll};
};

//--------------------------------------------------------------------------------------------------
class ReceiptsVatAmounts
{
public:
    ReceiptsVatAmounts() noexcept;
    ReceiptsVatAmounts(const ReceiptsVatAmounts &other) noexcept;
    ReceiptsVatAmounts(ReceiptsVatAmounts &&other)noexcept;
    ~ReceiptsVatAmounts();

    bool isValid() const;
    bool isNull() const;

    std::optional<ReceiptsVatAmount> vat5() const;
    void setVat5(std::optional<ReceiptsVatAmount> newVat5);

    std::optional<ReceiptsVatAmount> vat7() const;
    void setVat7(std::optional<ReceiptsVatAmount> newVat7);

    std::optional<ReceiptsVatAmount> vat5_105() const;
    void setVat5_105(std::optional<ReceiptsVatAmount> newVat5_105);

    std::optional<ReceiptsVatAmount> vat7_107() const;
    void setVat7_107(std::optional<ReceiptsVatAmount> newVat7_107);

    void appendVat(fdf::VatRate rate, const FixNumber &value);

    void clean();
    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    ReceiptsVatAmounts &operator =(const ReceiptsVatAmounts &other) noexcept;
    ReceiptsVatAmounts &operator =(ReceiptsVatAmounts &&other) noexcept;
    bool operator ==(const ReceiptsVatAmounts &other) const noexcept;
    bool operator !=(const ReceiptsVatAmounts &other) const noexcept;

private:
    std::optional<ReceiptsVatAmount> vat5_{};
    std::optional<ReceiptsVatAmount> vat7_{};
    std::optional<ReceiptsVatAmount> vat5_105_{};
    std::optional<ReceiptsVatAmount> vat7_107_{};
};


//--------------------------------------------------------------------------------------------------

class CardPaymentInfo
{
public:
    CardPaymentInfo() noexcept;
    CardPaymentInfo(const CardPaymentInfo &other) noexcept;
    CardPaymentInfo(CardPaymentInfo &&other) noexcept;
    ~CardPaymentInfo();

    bool isValid(CoreApiConst::ErrorCode &error, QString &msg) const;
    bool isValid() const;

    const FixNumber &amount() const;
    void setAmount(const FixNumber &newAmount);

    quint8 way() const;
    void setWay(quint8 newWay);

    const QString &ids() const;
    void setIds(const QString &newIds);

    const QString &additional() const;
    void setAdditional(const QString &newAdditional);

    void clean();
    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    QVariantMap toMap() const;
    QVariantMap toExtMap() const;
    void parseMap(const QVariantMap &map);

    CardPaymentInfo &operator =(const CardPaymentInfo &other) noexcept;
    CardPaymentInfo &operator =(CardPaymentInfo &&other) noexcept;
    bool operator ==(const CardPaymentInfo &other) const noexcept;
    bool operator !=(const CardPaymentInfo &other) const noexcept;

private:
    FixNumber amount_ = {DEFAULT_AMOUNT_MULT, 0ll};
    quint8 way_ = 0;
    QString ids_ = {};
    QString additional_= {};
};

//--------------------------------------------------------------------------------------------------
class CardPaymentsInfo
{
public:
    CardPaymentsInfo() noexcept;
    CardPaymentsInfo(const CardPaymentsInfo &other) noexcept;
    CardPaymentsInfo(CardPaymentsInfo &&other) noexcept;
    ~CardPaymentsInfo();

    bool isEmpty() const;
    bool isValid() const;
    bool isValid(CoreApiConst::ErrorCode &error, QString &msg) const;

    const QVector<CardPaymentInfo> &payments() const;
    void setPayments(const QVector<CardPaymentInfo> &newPayments);

    void clean();
    Tlv toTlv() const;
    void parseTlv(const Tlv &tlv);

    QVariantList toList() const;
    QVariantList toExtList() const;
    void parseList(const QVariantList &data);


    CardPaymentsInfo &operator =(const CardPaymentsInfo &other) noexcept;
    CardPaymentsInfo &operator =(CardPaymentsInfo &&other) noexcept;
    bool operator ==(const CardPaymentsInfo &other) const noexcept;
    bool operator !=(const CardPaymentsInfo &other) const noexcept;

private:
    QVector<CardPaymentInfo> payments_;
};
//--------------------------------------------------------------------------------------------------

#endif // RECEIPTDATACLASSES_H

