#include "receiptoperation.h"
#include "cashboxbuildconfig.h"
#include "formatutils.h"
#include "productcodecreator.h"
#include "tax.h"


ReceiptOperation::ReceiptOperation() noexcept
    : SimpleReceiptOperation()
    , priceForPack_()
    , priceForUnit_()
    , labelCheckResult_()
    , fraction_()
    , checkLabelFlags_()
    , controlCode_()
    , skipB_(false)
{

}

ReceiptOperation::ReceiptOperation(const ReceiptOperation &other) noexcept
    : SimpleReceiptOperation(other)
    , priceForPack_(other.priceForPack_)
    , priceForUnit_(other.priceForUnit_)
    , labelCheckResult_(other.labelCheckResult_)
    , fraction_(other.fraction_)
    , checkLabelFlags_(other.checkLabelFlags_)
    , controlCode_(other.controlCode_)
    , skipB_(other.skipB_)

{

}

ReceiptOperation::ReceiptOperation(ReceiptOperation &&other) noexcept
    : SimpleReceiptOperation(other)
    , priceForPack_()
    , priceForUnit_()
    , labelCheckResult_()
    , fraction_()
    , checkLabelFlags_(other.checkLabelFlags_)
    , controlCode_()
    , skipB_(other.skipB_)


{
    priceForPack_.swap(other.priceForPack_);
    priceForUnit_.swap(other.priceForUnit_);
    labelCheckResult_.swap(other.labelCheckResult_);
    fraction_.swap(other.fraction_);
    controlCode_.swap(other.controlCode_);
}

ReceiptOperation::ReceiptOperation(const QVariantMap &map) noexcept
    : SimpleReceiptOperation()
    , priceForPack_()
    , priceForUnit_()
    , labelCheckResult_()
    , fraction_()
    , checkLabelFlags_()
    , controlCode_()
    , skipB_(false)

{
    ReceiptOperation::parseMap(map);
}

ReceiptOperation::~ReceiptOperation()
{

}

bool ReceiptOperation::isValid() const
{
    return SimpleReceiptOperation::isValid();
}


bool ReceiptOperation::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(!SimpleReceiptOperation::isValid(err, msg) ||
        !checkFraction(err, msg) ||
        !checkLabled(err, msg))
    {
        return false;
    }
    if(!fdf::unitIsValid(unit()))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = QStringLiteral("Реквизит 2108 ( мера количества предмета расчета) отсутствует или имеет некорректное значение");
        return false;
    }
    return true;
}

bool ReceiptOperation::isSimpleLabled() const
{
    return isLabled() || (code_.has_value() && code_.value().isValid());
}

bool ReceiptOperation::isLabled() const
{
    return (labelCheckResult_.has_value()  && labelCheckResult_.value().labelCode().isValid());

}




bool ReceiptOperation::hasPrice() const
{
    return price_.has_value() || (fraction_.has_value() && fraction_.value().isValid() &&
                                  ( priceForPack_.has_value() ||priceForUnit_.has_value()));
}

FixNumber ReceiptOperation::price() const
{
    if(price_.has_value()) return price_.value();
    if(fraction_.has_value() && fraction_.value().isValid())
    {
        if(priceForPack_.has_value())
        {
            FixNumber res = priceForPack_.value();
            res.setRealMultiplier(DEFAULT_AMOUNT_MULT * 10);
            res.setValue(10 * res.value() *
                         fraction_.value().nominator() /
                         fraction_.value().denominator());
            res = res.round(DEFAULT_AMOUNT_MULT);
            return res;
        }
        if(priceForUnit_.has_value())
        {
            FixNumber res = priceForUnit().value();
            res.setValue(res.value() * fraction_.value().nominator());
            return res;
        }
    }
    return FixNumber(DEFAULT_AMOUNT_MULT, 0ll);
}


FixNumber ReceiptOperation::priceForPack() const
{
    return priceForPack_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll));
}

void ReceiptOperation::setPriceForPack(std::optional<FixNumber> newPriceForPack)
{
    priceForPack_ = newPriceForPack;
}

void ReceiptOperation::setPriceForPack(const FixNumber &newPriceForPack)
{
    if(newPriceForPack.realMultiplier() != DEFAULT_AMOUNT_MULT || newPriceForPack.value() < 0ll)
    {
        priceForPack_.reset();
    }
    else priceForPack_ = newPriceForPack;
}

std::optional<FixNumber> ReceiptOperation::priceForUnit() const
{
    return priceForUnit_;
}

void ReceiptOperation::setPriceForUnit(std::optional<FixNumber> newPriceForUnit)
{
    priceForUnit_ = newPriceForUnit;
}


std::optional<LabelCheckResult> ReceiptOperation::labelCheckResult() const
{
    return labelCheckResult_;
}

void ReceiptOperation::setLabelCheckResult(std::optional<LabelCheckResult> newLabelCheckResult)
{
    labelCheckResult_ = newLabelCheckResult;
    if(labelCheckResult_.has_value())
    {
        quantity_ = labelCheckResult_.value().quantity().has_value() ?
                    labelCheckResult_.value().quantity().value() :
                    FixNumber(DEFAULT_QUANTITY_MULT, static_cast<qint64>(DEFAULT_QUANTITY_MULT));
        unit_ = labelCheckResult_.value().unit().value_or(fdf::ItemUnit::Ut);
        fraction_ = labelCheckResult_.value().fraction();
        checkLabelFlags_  = labelCheckResult_.value().clFlags();
    }

//    std::optional<LabelCode> labelCode_;
//    fdf::LabelProductStatus actualStatus_;
//    quint8 mode_;
//    std::optional<FixNumber> quantity_;
//    std::optional<fdf::ItemUnit> unit_;
//    std::optional<QuantityFraction> fraction_;
//    std::optional<fdf::LabelCodeType> lct2100_;
//    std::optional<QString> productId_;//2101
//    fdf::CheckLabelFlags clFlags_;
//    QDateTime reqDt_;
//    Status2109 st2109_;
//    ReqCode2105 reqCode2105_;
//    ReqResult reqResult_;
}




std::optional<QuantityFraction> ReceiptOperation::fraction() const
{
    return fraction_;
}

void ReceiptOperation::setFraction(std::optional<QuantityFraction> newFraction)
{
    fraction_ = newFraction;
}

fdf::CheckLabelFlags ReceiptOperation::checkLabelFlags() const
{
    if(labelCheckResult_.has_value()) return labelCheckResult_.value().clFlags();
    return checkLabelFlags_;
}

void ReceiptOperation::setCheckLabelFlags(const fdf::CheckLabelFlags &newCheckLabelFlags)
{
    checkLabelFlags_ = newCheckLabelFlags;
}

fdf::ItemUnit ReceiptOperation::unit() const
{
    if(labelCheckResult_.has_value() &&
        labelCheckResult_.value().unit().has_value() &&
        fdf::unitIsValid(labelCheckResult_.value().unit().value()))
    {
        return labelCheckResult_.value().unit().value();
    }
    return unit_;
}

const QString &ReceiptOperation::controlCode() const
{
    return controlCode_;
}

void ReceiptOperation::setControlCode(const QString &newControlCode)
{
    controlCode_ = newControlCode;
}


bool ReceiptOperation::skipB() const
{
    return skipB_;
}

void ReceiptOperation::setSkipB(bool newSkipB)
{
    skipB_ = newSkipB;
}



QVariantMap ReceiptOperation::toMap() const
{
    QVariantMap res = SimpleReceiptOperation::toMap();
    res.remove("unit");
    if(priceForPack_.has_value()) res["priceForPack"] = priceForPack_.value().toMap();
    if(priceForUnit_.has_value()) res["priceForPart"] = priceForUnit_.value().toMap();
//    if(!rawLabel().isEmpty()) res["rawLabel"] = rawLabel();


    if(fraction_.has_value() && fraction_.value().isValid())
    {
        res["fraction"] = fraction_.value().toMap();
    }
    if(static_cast<qint32>(checkLabelFlags_) != 0) res["checkLabelFlags"] = static_cast<qint32>(checkLabelFlags_);
    if(fdf::unitIsValid(unit())) res["unit"] = static_cast<qint32>(unit());
    if(!controlCode_.isEmpty()) res["crc"] = controlCode_;
    if(skipB_) res["skipB"] = 1;
    if(labelCheckResult_.has_value()) res["labelCheckResult"] = labelCheckResult_.value().toMap();
    return res;
}

QVariantMap ReceiptOperation::toExternalMap() const
{
    QVariantMap res = SimpleReceiptOperation::toExternalMap();
    res.remove("unit");
    if(priceForPack_.has_value()) res["priceForPack"] = priceForPack_.value().toString();
    if(priceForUnit_.has_value()) res["priceForPart"] = priceForUnit_.value().toString();
//    if(!rawLabel().isEmpty()) res["rawLabel"] = rawLabel();


    if(fraction_.has_value() && fraction_.value().isValid())
    {
        res["fraction"] = fraction_.value().toMap();
    }
    if(static_cast<qint32>(checkLabelFlags_) != 0) res["checkLabelFlags"] = static_cast<qint32>(checkLabelFlags_);
    if(fdf::unitIsValid(unit())) res["unit"] = static_cast<qint32>(unit());
    if(!controlCode_.isEmpty()) res["crc"] = controlCode_;
    if(labelCheckResult_.has_value()) res["labelCheckResult"] = labelCheckResult_.value().toExternalMap();
    return res;
}

void ReceiptOperation::parseMap(const QVariantMap &map, CoreApiConst::ErrorCode *err)
{
    ReceiptOperation::clean();
    SimpleReceiptOperation::parseMap(map);
    if(map.contains("priceForPack")) priceForPack_ = FormatUtils::parseSumm("priceForPack", map, DEFAULT_AMOUNT_DEC, err);
    if(map.contains("priceForPart")) priceForUnit_ = FormatUtils::parseSumm("priceForPart", map, DEFAULT_AMOUNT_DEC, err);
//    setRawLabel(map["rawLabel"].toString().trimmed());
//    if(map.contains("productCode")) code_ = ProductCode(map["productCode"].toMap());
    if(map.contains("fraction")) fraction_ = QuantityFraction(map["fraction"].toMap());
    checkLabelFlags_ = fdf::CheckLabelFlags(map["checkLabelFlags"].toInt());
    if(map.contains("unit"))unit_ = static_cast<fdf::ItemUnit>(map["unit"].toInt());
    controlCode_ = map["crc"].toString();
    skipB_ = map.contains("skipB") && map["skipB"].toUInt();
    if(map.contains("labelCheckResult")) labelCheckResult_ = LabelCheckResult(map["labelCheckResult"].toMap());
}


Tlv ReceiptOperation::to1059() const
{
    Tlv buf;
    Tlv::Stlv list;

    buf.setByte(fdf::Tag::ItemType, static_cast<quint8>(type_));
    list << buf;
    if(paymentType_ != fdf::ItemPaymentType::Advance)
    {
        buf.setString(fdf::Tag::ItemName, name_);
        list << buf;
    }
    buf.setByte(fdf::Tag::ItemPaymentType, static_cast<quint8>(paymentType_));
    list << buf;

    buf.setByte(fdf::Tag::VatRate, static_cast<quint8>(vatRate_));
    list << buf;

    buf.setVln(fdf::Tag::ItemPrice, price().value());
    list << buf;

    buf.setVln(fdf::Tag::ItemCost, cost().value());
    list << buf;

    buf.setFvln(fdf::Tag::ItemQuantity, quantity());
    list << buf;


    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::ItemAdditionalParam, additionalParam_);
        list << buf;
    }
    if(fdf::itemPayAgentFlagIsValid(agentFlag_))
    {
        buf.setByte(fdf::Tag::ItemAgentFlag, static_cast<quint8>(agentFlag_));
        list << buf;

        buf.setInn(fdf::Tag::ProviderInn, Inn(providerInn_));
        list << buf;
       if(providerData_.has_value()) list << providerData_.value().toTlv();
       if(agentData_.has_value()) list << agentData_.value().toTlv(agentFlag_);
    }
    if(exciseTax_.has_value())
    {
        buf.setVln(fdf::Tag::ExciseTax, exciseTax_.value().value());
        list << buf;
    }
    if(!countryCode_.isEmpty() && !declarationNumber_.isEmpty())
    {
        buf.setString(fdf::Tag::ProductCountryCode, countryCode_);
        list << buf;
        buf.setString(fdf::Tag::ProductDeclarationNumber, declarationNumber_);
        list << buf;
    }


    ProductCode pc;
    bool needunit = true;

    if(labelCheckResult_.has_value())
    {
        pc = labelCheckResult_.value().labelCode().productCode();
        if(labelCheckResult_.value().unit().has_value())
        {
            buf.setByte(fdf::Tag::ItemQuantityUnitType, static_cast<quint8>(
                                                            labelCheckResult_.value().unit().value()));
            list << buf;
            needunit = false;
        }
        if(/*labelCheckResult_.value().unit() == fdf::ItemUnit::Ut &&*/
            labelCheckResult_.value().fraction().has_value()  && labelCheckResult_.value().fraction()->isValid())
        {
            list << labelCheckResult_.value().fraction().value().toTlv();
        }
        buf.setByte(fdf::Tag::LabelProcessMode, labelCheckResult_.value().mode());
        list << buf;
    }
    else if(code_.has_value()) pc = code_.value();

    list << pc.toTlv();
    if(needunit)
    {
        fdf::ItemUnit lunit = fdf::ItemUnit::Ut;
        if(fdf::unitIsValid(unit()))
        {
            lunit = unit();
        }
        buf.setByte(fdf::Tag::ItemQuantityUnitType, static_cast<quint8>(lunit));
        list << buf;
    }

    for(const IndustryProperty & p: industryProperties_)list << p.toTlv(fdf::Tag::IndustryItemParam);


    buf.setStlv(fdf::Tag::ReceiptItem, list);
    return buf;

}

Tlv ReceiptOperation::to2007() const
{
    if(!isLabled()) return Tlv();
    Tlv::Stlv list;
    Tlv buf;
    LabelCode lc = labelCheckResult_.value().labelCode();
    buf.setTag(fdf::Tag::Label);
    buf.setValue(lc.preparedLabel());
    list << buf;

    buf.setByte(fdf::Tag::LabelCodeType, static_cast<quint8>(labelCheckResult_.value().lct2100()));
    list << buf;

    buf.setString(fdf::Tag::ProductId, labelCheckResult_.value().productId());
    list << buf;

/**
//    buf.setByte(fdf::Tag::ProductInfoCheckResult, static_cast<quint8>(labelCheckResult_.value().clFlags()));
//    list << buf;

//    if(labelCheckResult_.value().actualStatus() == fdf::LabelProductStatus::Measured ||
//            labelCheckResult_.value().actualStatus() == fdf::LabelProductStatus::MeasuredRefund)
//    {
////        buf.setFvln(fdf::Tag::ItemQuantity,
////                    labelCheckResult_.value().quantity().value_or(FixNumber(DEFAULT_QUANTITY_MULT, 1000ll)));
////        list << buf;
//        if(labelCheckResult_.value().unit().has_value())
//        {
////            buf.setByte(fdf::Tag::ItemQuantityUnitType, static_cast<quint8>(
////                            labelCheckResult_.value().unit().value()));
////            list << buf;
////            if(labelCheckResult_.value().unit() == fdf::ItemUnit::Ut &&
////                    labelCheckResult_.value().fraction().has_value())
////            {
////                list << labelCheckResult_.value().fraction().value().toTlv();
////            }
//        }
//    }
//    buf.setByte(fdf::Tag::LabelProcessMode, labelCheckResult_.value().mode());
//    list << buf;
//    buf.setByte(fdf::Tag::ItemPaymentType, static_cast<quint8>(paymentType_));
//    list << buf;
   **/

    buf.setByte(fdf::Tag::ProductStatus, static_cast<quint8>(labelCheckResult_.value().actualStatus()));
    list << buf;

//    buf.setVln(fdf::Tag::ItemCost, cost().value());
//    list << buf;


    buf.setStlv(fdf::Tag::LabledProductInfo, list);
    return buf;
}



void ReceiptOperation::clean()
{
    SimpleReceiptOperation::clean();
    priceForPack_.reset();
    priceForUnit_.reset();
    labelCheckResult_.reset();
    code_.reset();
    fraction_.reset();
    checkLabelFlags_ = fdf::CheckLabelFlags();
    controlCode_.clear();
    skipB_ = false;
}

ReceiptOperation &ReceiptOperation::operator =(const ReceiptOperation &other) noexcept
{
    SimpleReceiptOperation::operator =(other);
    priceForPack_ = other.priceForPack_;
    priceForUnit_ = other.priceForUnit_;
    labelCheckResult_ = other.labelCheckResult_;
    code_ = other.code_;
    fraction_ = other.fraction_;
    checkLabelFlags_ = other.checkLabelFlags_;
    controlCode_ = other.controlCode_;
    skipB_ = other.skipB_;
    return *this;
}
ReceiptOperation &ReceiptOperation::operator =(ReceiptOperation &&other) noexcept
{
    SimpleReceiptOperation::operator =(other);
    priceForPack_.swap(other.priceForPack_);
    priceForUnit_.swap(other.priceForUnit_);
    labelCheckResult_.swap(other.labelCheckResult_);
    code_.swap(other.code_);
    fraction_.swap(other.fraction_);
    checkLabelFlags_ = other.checkLabelFlags_;
    controlCode_.swap(other.controlCode_);
    skipB_ = other.skipB_;
    return *this;
}

bool ReceiptOperation::operator ==(const ReceiptOperation &other) const noexcept
{
    return SimpleReceiptOperation::operator ==(other) &&
            priceForPack_ == other.priceForPack_ &&
            priceForUnit_ == other.priceForUnit_ &&
            labelCheckResult_ == other.labelCheckResult_ &&
            code_ == other.code_ &&
            fraction_ == other.fraction_ &&
            checkLabelFlags_ == other.checkLabelFlags_ &&
            controlCode_ == other.controlCode_;
}

bool ReceiptOperation::operator !=(const ReceiptOperation &other) const noexcept
{
    return !(*this == other);
}



bool ReceiptOperation::checkFraction(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(!isLabled()) return true;
    if(fraction().has_value() && fraction().value().isValid())
    {
        if(!fraction().value().is1())
        {
            if(unit() != fdf::ItemUnit::Ut)
            {
                err = CoreApiConst::ErrorCode::InvalidProductUnit;
                msg = "У казан некорректный код единицы измерения товара. для дробного количества должен быть 0 (штуки)";
                return false;
            }
        }
        if(quantity().value() != DEFAULT_QUANTITY_MULT)
        {
            err = CoreApiConst::ErrorCode::InvalidItemQuantity;
            msg = "Если задано дробное количетво, то количество должно быть равно 1";
            return false;
        }
    }
    return true;
}

bool ReceiptOperation::checkLabled(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(code_.has_value())
    {
        if(code_.value().isGsM() && !labelCheckResult_.has_value())
        {
            err = CoreApiConst::ErrorCode::NoAnyLabel;
            msg = QStringLiteral("Не задана марка для проверки");
            return false;
        }
        if(!code_.value().isValid(err, msg)) return false;
    }
    if(labelCheckResult_.has_value())
    {
        if(!labelCheckResult_.value().labelCode().isValid())
        {

            err = CoreApiConst::ErrorCode::NoAnyLabel;
            msg = QStringLiteral("Не задана марка для проверки или пришел некорректный результат проверки");
            return false;
        }
    }
    return true;
}

bool ReceiptOperation::checkPriceAndCost(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(!hasPrice())
    {
        err = CoreApiConst::ErrorCode::NoItemPrice;
        msg = "Не указана цена товара";
        return false;
    }
    if(price().value() < 0)
    {
        err = CoreApiConst::ErrorCode::InvalidItemPrice;
        msg = "Некорректное знаечение цены";
        return false;
    }
    if(cost_.has_value() && qAbs(cost_.value().value() - calcCost().value()) > 1ll)
    {
        err = CoreApiConst::ErrorCode::InvalidItemPrice;
        msg = "Некорректное стоимости предмета расчета ";
        return false;
    }
    return true;
}

void ReceiptOperation::parseChildTlv(const Tlv &tlv)
{
    switch (tlv.tag()) {
    case fdf::Tag::ProductCode:
    {
        ProductCode pc;
        pc.parseTlv(tlv);
        code_ = pc;
    }break;
    case fdf::Tag::FractionPart:
    {
        QuantityFraction qf;
        qf.parseTlv(tlv);
        fraction_ = qf;
    }break;
    case fdf::Tag::ProductInfoCheckResult: checkLabelFlags_ = fdf::CheckLabelFlags(tlv.toByte());break;
    case fdf::Tag::ItemQuantityUnitType: unit_ = static_cast<fdf::ItemUnit>(tlv.toByte());break;
    case fdf::Tag::LabelCrc: controlCode_ = tlv.toString();break;
    default:
        SimpleReceiptOperation::parseChildTlv(tlv);
        break;
    }
}
