#include "regdata.h"

#include "formatutils.h"

#include <QCryptographicHash>

CoreRegData::CoreRegData() noexcept
    : IncomingReg()
    , serial_()
    , fsFfd_(fs::FFD::Invalid)
    , ffd_(fs::FFD::Invalid)
    , cashboxFfd_(fs::FFD::Invalid)
    , cashboxVersion_()
    , keysResource_(0)
{

}

CoreRegData::CoreRegData(const CoreRegData &other) noexcept
    : IncomingReg(other)
    , serial_(other.serial_)
    , fsFfd_(other.fsFfd_)
    , ffd_(other.ffd_)
    , cashboxFfd_(other.cashboxFfd_)
    , cashboxVersion_(other.cashboxVersion_)
    , keysResource_(other.keysResource_)
{

}

CoreRegData::CoreRegData(const IncomingReg &other) noexcept
    : IncomingReg(other)
    , serial_()
    , fsFfd_(fs::FFD::Invalid)
    , ffd_(fs::FFD::Invalid)
    , cashboxFfd_(fs::FFD::Invalid)
    , cashboxVersion_()
    , keysResource_(0)
{

}

CoreRegData::CoreRegData(CoreRegData &&other) noexcept
    : IncomingReg(other)
    , serial_()
    , fsFfd_(other.fsFfd_)
    , ffd_(other.ffd_)
    , cashboxFfd_(other.cashboxFfd_)
    , cashboxVersion_()
    , keysResource_(other.keysResource_)
{
    serial_.swap(other.serial_);
    cashboxVersion_.swap(other.cashboxVersion_);
}

CoreRegData::CoreRegData(const QVariantMap &map) noexcept
    : IncomingReg()
    , serial_()
    , fsFfd_(fs::FFD::Invalid)
    , ffd_(fs::FFD::Invalid)
    , cashboxFfd_(fs::FFD::Invalid)
    , cashboxVersion_()
    , keysResource_(0)
{
    CoreRegData::parseMap(map);
}

CoreRegData::~CoreRegData()
{

}

bool CoreRegData::isValid() const
{
    return !serial_.isEmpty() && !ownerInn_.isEmpty() && !regNumber_.isEmpty() &&
            RegNumber(regNumber_.trimmed(), serial_.trimmed(), ownerInn_.trimmed()).isValid();
}



const QString &CoreRegData::serial() const
{
    return serial_;
}

void CoreRegData::setSerial(const QString &newSerial)
{
    serial_ = newSerial.trimmed();
}


fs::FFD CoreRegData::fsFfd() const
{
    return fsFfd_;
}

void CoreRegData::setFsFfd(fs::FFD newFsFfd)
{
    fsFfd_ = newFsFfd;
}

fs::FFD CoreRegData::ffd() const
{
    return ffd_;
}

void CoreRegData::setFfd(fs::FFD newFfd)
{
    ffd_ = newFfd;
}

fs::FFD CoreRegData::cashboxFfd() const
{
    return cashboxFfd_;
}

void CoreRegData::setCashboxFfd(fs::FFD newCashboxFfd)
{
    cashboxFfd_ = newCashboxFfd;
}

const QString &CoreRegData::cashboxVersion() const
{
    return cashboxVersion_;
}

void CoreRegData::setCashboxVersion(const QString &newCashboxVersion)
{
    cashboxVersion_ = newCashboxVersion;
}

quint16 CoreRegData::keysResource() const
{
    return keysResource_;
}

void CoreRegData::setKeysResource(quint16 newKeysResource)
{
    keysResource_ = newKeysResource;
}


QVariantMap CoreRegData::toMap() const
{
    QVariantMap res = IncomingReg::toMap();
    res["registeredWith"] = serial_;
    res["fsFfd"] = static_cast<quint32>(fsFfd_);
    res["ffd"] = static_cast<quint32>(ffd_);
    res["cashboxFfd"] = static_cast<quint32>(cashboxFfd_);
    res["registeredWithVersion"] = cashboxVersion_;
    res["keysResource"] = keysResource_;
    return res;

}

QVariantMap CoreRegData::toExternalMap() const
{
    QVariantMap res = IncomingReg::toExternalMap();
    res["registeredWith"] = serial_;
    res["fsFfd"] = fs::ffd2String(fsFfd_);
    res["ffd"] = fs::ffd2String(ffd_);
    res["cashboxFfd"] = fs::ffd2String(cashboxFfd_);
    res["registeredWithVersion"] = cashboxVersion_;
    res["keysResource"] = keysResource_;
    return res;
}

void CoreRegData::parseMap(const QVariantMap &map)
{
    clean();
    IncomingReg::parseMap(map);
    serial_ = map["registeredWith"].toString().trimmed();
    fsFfd_ = static_cast<fs::FFD>(map["fsFfd"].toUInt());
    fs::checkFfd(fsFfd_);
    ffd_ = static_cast<fs::FFD>(map["ffd"].toUInt());
    fs::checkFfd(ffd_);
    cashboxFfd_ = static_cast<fs::FFD>(map["cashboxFfd"].toUInt());
    fs::checkFfd(cashboxFfd_);
    cashboxVersion_ = map["registeredWithVersion"].toString();
    keysResource_ = static_cast<quint16>(map["keysResource"].toUInt());
}

void CoreRegData::clean()
{
    IncomingReg::clean();
    serial_.clear();
    fsFfd_ = fs::FFD::Invalid;
    ffd_= fs::FFD::Invalid;
    cashboxFfd_= fs::FFD::Invalid;
    cashboxVersion_.clear();
    keysResource_ = 0;
}

QString CoreRegData::toHtml() const
{
    QString res;

    return res;
}

CoreRegData &CoreRegData::operator =(const CoreRegData &other) noexcept
{
    IncomingReg::operator =(other);
    serial_ = other.serial_;
    fsFfd_ = other.fsFfd_;
    ffd_ = other.ffd_;
    cashboxFfd_ = other.cashboxFfd_;
    cashboxVersion_ = other.cashboxVersion_;
    keysResource_ = other.keysResource_;
    return *this;

}

CoreRegData &CoreRegData::operator =(CoreRegData &&other) noexcept
{
    IncomingReg::operator =(other);
    serial_.swap(other.serial_);
    fsFfd_ = other.fsFfd_;
    ffd_ = other.ffd_;
    cashboxFfd_ = other.cashboxFfd_;
    cashboxVersion_.swap(other.cashboxVersion_);
    keysResource_ = other.keysResource_;
    return *this;

}
bool CoreRegData::operator ==(const CoreRegData &other) const noexcept
{
    return IncomingReg::operator ==(other) &&
            serial_ == other.serial_ &&
            fsFfd_ == other.fsFfd_ &&
            ffd_ == other.ffd_ &&
            cashboxFfd_ == other.cashboxFfd_ &&
            cashboxVersion_ == other.cashboxVersion_ &&
            keysResource_ == other.keysResource_;
}

bool CoreRegData::operator !=(const CoreRegData &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------
RegData::RegData() noexcept
    : CoreRegData()
    , dt_()
    , docNumber_(0)
    , fsNumber_()
    , fiscalCode_(0)

{

}

RegData::RegData(const RegData &other) noexcept
    : CoreRegData(other)
    , dt_(other.dt_)
    , docNumber_(other.docNumber_)
    , fsNumber_(other.fsNumber_)
    , fiscalCode_(other.fiscalCode_)
{

}

RegData::RegData(const CoreRegData &other) noexcept
    : CoreRegData(other)
    , dt_()
    , docNumber_()
    , fsNumber_()
    , fiscalCode_()

{

}

RegData::RegData(RegData &&other) noexcept
    : CoreRegData(other)
    , dt_()
    , docNumber_(other.docNumber_)
    , fsNumber_()
    , fiscalCode_(other.fiscalCode_)

{
    dt_.swap(other.dt_);
    fsNumber_.swap(other.fsNumber_);

}

RegData::RegData(const QVariantMap &map) noexcept
    : CoreRegData()
    , dt_()
    , docNumber_(0)
    , fsNumber_()
    , fiscalCode_(0)
{
    RegData::parseMap(map);
}

RegData::~RegData()
{

}

bool RegData::isValid() const
{
    return CoreRegData::isValid() && !fsNumber_.isEmpty();
}

const QDateTime &RegData::dt() const
{
    return dt_;
}

void RegData::setDt(const QDateTime &newDt)
{
    dt_ = newDt;
}

quint32 RegData::docNumber() const
{
    return docNumber_;
}

void RegData::setDocNumber(quint32 newDocNumber)
{
    docNumber_ = newDocNumber;
}

const QString &RegData::fsNumber() const
{
    return fsNumber_;
}

void RegData::setFsNumber(const QString &newFsNumber)
{
    fsNumber_ = newFsNumber.trimmed();
}

quint32 RegData::fiscalCode() const
{
    return fiscalCode_;
}

void RegData::setFiscalCode(quint32 newFiscalCode)
{
    fiscalCode_ = newFiscalCode;
}

QVariantMap RegData::toMap() const
{
    QVariantMap res = CoreRegData::toMap();
    if(dt_.isValid())res["regDt"] = DT2STR_(dt_);
    res["regDocNumber"] = docNumber_;
    res["fsNumber"] = fsNumber_;
    res["fiscalCode"] = QString::number(fiscalCode_);
    return res;
}

QVariantMap RegData::toExternalMap() const
{
    QVariantMap res = CoreRegData::toExternalMap();
    if(dt_.isValid())res["regDt"] = DT2STR_(dt_);
    res["regDocNumber"] = docNumber_;
    res["fsNumber"] = fsNumber_;
    res["fiscalCode"] = QString::number(fiscalCode_);
    return res;
}

void RegData::parseMap(const QVariantMap &map)
{
    clean();
    CoreRegData::parseMap(map);
    if(map.contains("regDt")) dt_ = STR2DT_(map["regDt"].toString());
    docNumber_ = map["regDocNumber"].toUInt();
    fsNumber_ = map["fsNumber"].toString();
    fiscalCode_ = map["fiscalCode"].toString().toUInt();
}

void RegData::clean()
{
    CoreRegData::clean();
    dt_ = QDateTime();
    docNumber_ = 0;
    fsNumber_.clear();
    fiscalCode_ = 0;
}

QString RegData::toHtml() const
{
    QStringList res;
    QStringList flags;
    if(getAutoMode()) flags  << "АВТ.РЕЖ";
    if(getOfflineMode()) flags << "АВТОНОМНО";
    if(getEncriptionMode()) flags << "ШФД";
    flags.append(fdf::regFlag1290ToStrings(extFlags_.flags()));
    res << QStringLiteral("<p align=\"center\"><b>%1</b></p>").arg(ownerName_)
        << QStringLiteral("<p align=\"center\">ИНН<b>: %1</b></p>").arg(ownerInn_)
        << QStringLiteral("<p align=\"center\">%1</p>").arg(address_)
        << QStringLiteral("<p align=\"center\">%1</p>").arg(place_)
        << QStringLiteral("<p align=\"right\">%1</p>").arg(flags.join(", "))
        << "<table align=\"center\" width=100%><tbody>";

        res << QStringLiteral("<tr><td>РНМ:</td><td align=\"right\">%1</td></tr>")
               .arg(regNumber_);
        if(taxes_ != fdf::Taxes())
        {
            res << QStringLiteral("<tr><td>СНО:</td><td align=\"right\">%1</td></tr>")
                   .arg(fdf::taxesToString(taxes_));
        }
        if(!getOfflineMode())
        {
            res << QStringLiteral("<tr><td>ОФД:</td><td align=\"right\">%1</td></tr>").arg(ofdName_)
                << QStringLiteral("<tr><td>ОФД ИНН:</td><td align=\"right\">%1</td></tr>").arg(ofdInn_)
                << QStringLiteral("<tr><td>САЙТ ФНС:</td><td align=\"right\">%1</td></tr>").arg(ftsSite_)
                << QStringLiteral("<tr><td>Эл. почта</td><td align=\"right\">%1</td></tr>").arg(email_);
        }
        res << QStringLiteral("<tr><td>ФФД ФН:</td><td align=\"right\">%1</td></tr>")
                   .arg(fs::ffd2String(fsFfd_))
            << QStringLiteral("<tr><td>ФФД ККТ:</td><td align=\"right\">%1</td></tr>")
                   .arg(fs::ffd2String(cashboxFfd_))
            << QStringLiteral("<tr><td>ФФД :</td><td align=\"right\">%1</td></tr>")
                .arg(fs::ffd2String(ffd_));
        if(!boxId_.trimmed().isEmpty())
        {
            res << QStringLiteral("<tr><td>АВТОМАТ:</td><td align=\"right\">%1</td></tr>")
                       .arg(boxId_);
        }
        if(!cashier_.isEmpty())
        {
            res << QStringLiteral("<tr><td>Регистрировал:</td><td align=\"right\">%1</td></tr>")
                       .arg(cashier_);
        }
        res << QStringLiteral("<tr><td>Дата регистрации:</td><td align=\"right\">%1</td></tr>")
                   .arg(dt_.toString("dd.MM.yyyy"));
        if(reasons_ != RegReasons())
        {
            res << QStringLiteral("<tr><td>Причины перерегистрации:</td><td align=\"right\">%1</td></tr>")
                       .arg(fdf::reasonsToString(reasons_.flags()));

        }

        res << "</tbody></table><hr>";


    return res.join("\n");
}


RegData &RegData::operator =(const RegData &other) noexcept
{
    CoreRegData::operator =(other);
    dt_ = other.dt_;
    docNumber_ = other.docNumber_;
    fsNumber_ = other.fsNumber_;
    fiscalCode_ = other.fiscalCode_;
    return *this;

}

RegData &RegData::operator =(RegData &&other) noexcept
{
    CoreRegData::operator =(other);
    dt_.swap(other.dt_);
    docNumber_ = other.docNumber_;
    fsNumber_.swap(other.fsNumber_);
    fiscalCode_ = other.fiscalCode_;
    return *this;

}
bool RegData::operator ==(const RegData &other) const noexcept
{
    return CoreRegData::operator ==(other) &&
            dt_ == other.dt_ &&
            docNumber_ == other.docNumber_ &&
            fsNumber_ == other.fsNumber_ &&
            fiscalCode_ == other.fiscalCode_;
}

bool RegData::operator !=(const RegData &other) const noexcept
{
    return !(*this == other);
}
