#include "regnumber.h"


#include "crc.h"
#include <random>

RegNumber::RegNumber() noexcept
    : value_()
{

}

RegNumber::RegNumber(const QString &val, const QString &serial, const QString &inn) noexcept
    : value_()
    , inn_(inn)
    , serial_(serial)
{
    value_ = normalize(val);
}

RegNumber::RegNumber(const QString &serial, const QString &inn, qint64 number) noexcept
    : value_(create(serial, inn, number))
    , inn_(inn)
    , serial_(serial)
{

}

RegNumber::RegNumber(const QString &serial, const QString &inn) noexcept
    : value_(create(serial, inn, -1))
    , inn_(inn)
    , serial_(serial)
{

}

RegNumber::RegNumber(const RegNumber &other)noexcept
    : value_(other.value_)
    , inn_(other.inn_)
    , serial_(other.serial_)
{

}

RegNumber::RegNumber(RegNumber &&other)noexcept
    : value_()
    , inn_(other.inn_)
    , serial_()
{
    value_.swap(other.value_);
    serial_.swap(other.serial_);
}

bool RegNumber::isValid() const
{
//    lmWarning() << serial_ << inn_.value() << value_;
    return check(value_, serial_, inn_);
}

const QString &RegNumber::value() const
{
    return value_;
}

void RegNumber::setValue(const QString &newValue)
{
    value_ = newValue;
}

const Inn &RegNumber::inn() const
{
    return inn_;
}

void RegNumber::setInn(const Inn &newInn)
{
    inn_ = newInn;
}

const QString &RegNumber::serial() const
{
    return serial_;
}

void RegNumber::setSerial(const QString &newSerial)
{
    serial_ = newSerial;
}

QByteArray RegNumber::data() const
{
    if(!isValid()) return QByteArray();
    if(value_.size() == 16) return value_.toLatin1() + "    ";
    QString val = normalize(value_);
    return val.toLatin1() + "    ";
}

RegNumber &RegNumber::operator =(const RegNumber &other) noexcept
{
    value_ = other.value_;
    inn_ = other.inn_;
    serial_ = other.serial_;
    return *this;
}

RegNumber &RegNumber::operator =(RegNumber &&other) noexcept
{
    value_.swap(other.value_);
    inn_ = other.inn_;
    serial_.swap(other.serial_);
    return *this;
}


bool RegNumber::operator ==(const RegNumber &other) const noexcept
{
    return value_ == other.value_ &&
            inn_ == other.inn_ &&
            serial_ == other.serial_;
}

bool RegNumber::operator !=(const RegNumber &other) const noexcept
{
    return !(*this == other);
}

QString RegNumber::normalize(const QString &val) const
{
    if(!check(val, serial_, inn_))return QString();
    QString res = val;
    if(res.size() < 16) res = QString(16 - res.size(), QLatin1Char('0'));
    return res;
}

bool RegNumber::check(const QString &val, const QString &serial, const Inn &inn) const
{
    if(val.size() < 7) return false;
    if(!inn.isValid()) return false;
    if(val.size() > 16) return false;
    QByteArray rn = val.toLatin1();
//    lmWarning() << QString::fromLatin1(rn);
    if(val.size() < 16) rn = QByteArray(16 - val.size(), '0') + rn;
    QByteArray orderNum = rn.mid(0, 10);
//    lmWarning() << QString::fromLatin1(orderNum);
    bool ok = false;
    quint16 crc1 = static_cast<quint16>(QString::fromLatin1(rn.mid(10)).toUInt(&ok));
    if(!ok) return false;

    QByteArray sn = serial.toLatin1();
    if(sn.size() < 20) sn = QByteArray(20 - sn.size(), '0') + sn;
//    lmWarning() << QString::fromLatin1(sn);


    QByteArray innd = inn.value().toLatin1();

    if(innd.size() == 12) orderNum.append(innd);
    else orderNum.append(QByteArray(2, '0') + innd);
    orderNum.append(sn);
//    lmWarning() << QString::fromLatin1(orderNum);
    if(orderNum.size() != 42) return false;
    Crc16CITT crc;
//    lmWarning() << crc1 << crc(orderNum);
    return crc1 == crc(orderNum);
}

QString RegNumber::create(const QString &serial, const QString &inn, qint64 number) const
{
    std::random_device rd;
    while(number < 0ll || number > 9999999999ll)
    {
        uint32_t numbers[1];
        std::generate(numbers, std::end(numbers), std::ref(rd));
        number = static_cast<qint64>(numbers[0]);
    }
    Inn i(inn);
    if(!i.isValid()) return QString();
    QByteArray orderNum = QStringLiteral("%1").arg(number, 10, 10, QLatin1Char('0')).toLatin1();
    if(orderNum.size() > 10) return QString();
    QByteArray sn = serial.toLatin1();
    if(sn.size() < 20) sn = QByteArray(20 - sn.size(), '0') + sn;
    QByteArray on = orderNum;
    if(i.value().size() == 12) on.append(i.value().toLatin1());
    else on.append(QByteArray(2, '0') + i.value().toLatin1());
    on.append(sn);
    Crc16CITT crc;
    QByteArray tail = QString("%1").arg(crc(on), 6, 10, QLatin1Char('0')).toLatin1();
    return on.mid(0, 10) + tail;
}
