#include "simplereceiptoperation.h"
#include "cashboxbuildconfig.h"
#include "formatutils.h"


#include "tax.h"

SimpleReceiptOperation::SimpleReceiptOperation()noexcept
    : quantity_(DEFAULT_QUANTITY_MULT, static_cast<qint64>(DEFAULT_QUANTITY_MULT))
    , name_()
    , cost_()
    , price_()
    , additionalParam_()
    , priceVat_()
    , vatRate_(fdf::VatRate::Invalid)
    , costVat_()
    , type_(fdf::ReceiptOperationType::Invalid)
    , paymentType_(fdf::ItemPaymentType::FullPayment)
    , agentFlag_(fdf::ItemPayAgentFlag::NotAgent)
    , agentData_()
    , providerData_()
    , providerInn_()
    , exciseTax_()
    , countryCode_()
    , declarationNumber_()
    , unit_(fdf::ItemUnit::Ut)
    , code_()
    , industryProperties_()


{
}

SimpleReceiptOperation::SimpleReceiptOperation(const SimpleReceiptOperation &other) noexcept
    : quantity_(other.quantity_)
    , name_(other.name_)
    , cost_(other.cost_)
    , price_(other.price_)
    , additionalParam_(other.additionalParam_)
    , priceVat_(other.priceVat_)
    , vatRate_(other.vatRate_)
    , costVat_(other.cost_)
    , type_(other.type_)
    , paymentType_(other.paymentType_)
    , agentFlag_(other.agentFlag_)
    , agentData_(other.agentData_)
    , providerData_(other.providerData_)
    , providerInn_(other.providerInn_)
    , exciseTax_(other.exciseTax_)
    , countryCode_(other.countryCode_)
    , declarationNumber_(other.declarationNumber_)
    , unit_(other.unit_)
    , code_(other.code_)
    , industryProperties_(other.industryProperties_)



{

}

SimpleReceiptOperation::SimpleReceiptOperation(SimpleReceiptOperation &&other) noexcept
    : quantity_(other.quantity_)
    , name_()
    , cost_()
    , price_()
    , additionalParam_()
    , priceVat_()
    , vatRate_(other.vatRate_)
    , costVat_()
    , type_(other.type_)
    , paymentType_(other.paymentType_)
    , agentFlag_(other.agentFlag_)
    , agentData_()
    , providerData_()
    , providerInn_()
    , exciseTax_()
    , countryCode_()
    , declarationNumber_()
    , unit_(other.unit_)
    , code_()
    , industryProperties_()



{
    name_.swap(other.name_);
    cost_.swap(other.cost_);
    price_.swap(other.price_);
    additionalParam_.swap(other.additionalParam_);
    priceVat_.swap(other.priceVat_);
    costVat_.swap(other.cost_);
    agentData_.swap(other.agentData_);
    providerData_.swap(other.providerData_);
    providerInn_.swap(other.providerInn_);
    exciseTax_.swap(other.exciseTax_);
    countryCode_.swap(other.countryCode_);
    declarationNumber_.swap(other.declarationNumber_);
    code_.swap(other.code_);
    industryProperties_.swap(other.industryProperties_);

}

SimpleReceiptOperation::SimpleReceiptOperation(const QVariantMap &map) noexcept
    : quantity_(DEFAULT_QUANTITY_MULT, static_cast<qint64>(DEFAULT_QUANTITY_MULT))
    , name_()
    , cost_()
    , price_()
    , additionalParam_()
    , priceVat_()
    , vatRate_(fdf::VatRate::Invalid)
    , costVat_()
    , type_(fdf::ReceiptOperationType::Invalid)
    , paymentType_(fdf::ItemPaymentType::FullPayment)
    , agentFlag_(fdf::ItemPayAgentFlag::NotAgent)
    , agentData_()
    , providerData_()
    , providerInn_()
    , exciseTax_()
    , countryCode_()
    , declarationNumber_()
    , unit_(fdf::ItemUnit::Ut)
    , code_()
    , industryProperties_()
{
    SimpleReceiptOperation::parseMap(map);
}

SimpleReceiptOperation::~SimpleReceiptOperation()
{

}

bool SimpleReceiptOperation::isValid() const
{
    CoreApiConst::ErrorCode err;
    QString msg;
    return isValid(err, msg);

}

bool SimpleReceiptOperation::isValid(CoreApiConst::ErrorCode &err, QString &msg) const
{
    err = CoreApiConst::ErrorCode::Ok;
    msg.clear();
    if(!fdf::opTypeIsValid(type_))
    {
        err = CoreApiConst::ErrorCode::InvalidItemName;
        msg = "Реквизит 1212 (признак предмета расчета) отсутствует или содержит некорректное значение";
        return  false;
    }
    if(!fdf::itemPaymenTypeIsValid(paymentType_))
    {
        err = CoreApiConst::ErrorCode::InvalidItemName;
        msg = "Реквизит 1214 (признак способа расчета) отсутствует или содержит некорректное значение";
        return  false;

    }
    if(!checkPriceAndCost(err, msg)) return false;


    if(!FormatUtils::checkString(name_, 128, false, paymentType_ == fdf::ItemPaymentType::Advance, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidItemName;
        msg = "Реквизит 1030 (наименование предмета расчета)" + msg;
        return false;
    }
    if(!fdf::vatIsValid(vatRate_))
    {
        err = CoreApiConst::ErrorCode::InvalidVatRate;
        msg = "Не указан реквизит 1199 (ставка НДС) отсутствует или содержит некорректное значение";
        return false;

    }
    //Специально не return ... тут, возможно будет что-то добавлено, чтоб не забыть

    if(!checkAgent(err, msg) ||!checkCountry(err, msg) || !checkIndustryCode(err, msg))
    {
        return false;
    }
    return true;
}

const FixNumber &SimpleReceiptOperation::quantity() const
{
    return quantity_;
}

void SimpleReceiptOperation::setQuantity(const FixNumber &newQuantity)
{
    quantity_ = newQuantity;
}

const QString &SimpleReceiptOperation::name() const
{
    return name_;
}

void SimpleReceiptOperation::setName(const QString &newName)
{
    name_ = STR4FS_(newName);
}

FixNumber SimpleReceiptOperation::cost() const
{
    if(cost_.has_value()) return cost_.value();
    return calcCost();
}

FixNumber SimpleReceiptOperation::calcCost() const
{
    FixNumber res = price();
    res.setRealMultiplier(res.realMultiplier() * quantity().realMultiplier());
    res.setValue(res.value() * quantity().value());
    res = res.round(DEFAULT_AMOUNT_MULT);
    return res;
}

void SimpleReceiptOperation::setCost(std::optional<FixNumber> newCost)
{
    cost_ = newCost;
}

void SimpleReceiptOperation::setCost(const FixNumber &newCost)
{
    if(newCost.realMultiplier() == DEFAULT_QUANTITY_MULT && newCost.value() >= 0ll)
    {
        cost_ = newCost;
    }
    else
    {
        cost_.reset();
    }
}

bool SimpleReceiptOperation::hasPrice() const
{
    return price_.has_value();
}

FixNumber SimpleReceiptOperation::price() const
{
    return price_.value_or(FixNumber(DEFAULT_AMOUNT_MULT, 0ll));
}

void SimpleReceiptOperation::setPrice(std::optional<FixNumber> newPrice)
{
    price_ = newPrice;
}

void SimpleReceiptOperation::setPrice(const FixNumber &newPrice)
{
    if(newPrice.realMultiplier() == DEFAULT_AMOUNT_MULT && newPrice.value() >= 0ll)
    {
        price_ = newPrice;
    }
    else
    {
        price_.reset();
    }
}

const QString &SimpleReceiptOperation::additionalParam() const
{
    return additionalParam_;
}

void SimpleReceiptOperation::setAdditionalParam(const QString &newAdditionalParam)
{
    additionalParam_ = STR4FS_(newAdditionalParam);
}

FixNumber SimpleReceiptOperation::priceVat() const
{
    if(priceVat_.has_value()) return priceVat_.value();
    Tax tax(vatRate_);
    return tax(price());
}

void SimpleReceiptOperation::setPriceVat(std::optional<FixNumber> newPriceVat)
{
    priceVat_ = newPriceVat;
}

fdf::VatRate SimpleReceiptOperation::vatRate() const
{
    return vatRate_;

}

void SimpleReceiptOperation::setVatRate(fdf::VatRate newVatRate)
{
    vatRate_ = newVatRate;
}

FixNumber SimpleReceiptOperation::costVat() const
{
    if(costVat_.has_value()) return costVat_.value();
    Tax tax(vatRate_);
    return tax(cost());
}

void SimpleReceiptOperation::setCostVat(std::optional<FixNumber> newCostRate)
{
    costVat_ = newCostRate;
}

QString SimpleReceiptOperation::printedCostVat() const
{
    switch (vatRate_) {
    case fdf::VatRate::Vat0:
    case fdf::VatRate::VatNone: return QString();
    default: return costVat().toString(QLatin1Char('.'));
    }
}

fdf::ReceiptOperationType SimpleReceiptOperation::type() const
{
    return type_;
}

void SimpleReceiptOperation::setType(fdf::ReceiptOperationType newType)
{
    type_ = newType;
}

fdf::ItemPaymentType SimpleReceiptOperation::paymentType() const
{
    return paymentType_;
}

void SimpleReceiptOperation::setPaymentType(fdf::ItemPaymentType newPaymentType)
{
    paymentType_ = newPaymentType;
}

fdf::ItemPayAgentFlag SimpleReceiptOperation::agentFlag() const
{
    return agentFlag_;
}

void SimpleReceiptOperation::setAgentFlag(fdf::ItemPayAgentFlag newAgentFlag)
{
    agentFlag_ = newAgentFlag;
}

std::optional<AgentData> SimpleReceiptOperation::agentData() const
{
    return agentData_;
}

void SimpleReceiptOperation::setAgentData(std::optional<AgentData> newAgentData)
{
    agentData_ = newAgentData;
}

std::optional<ProviderData> SimpleReceiptOperation::providerData() const
{
    return providerData_;
}

void SimpleReceiptOperation::setProviderData(std::optional<ProviderData> newProviderData)
{
    providerData_ = newProviderData;
}

const QString &SimpleReceiptOperation::providerInn() const
{
    return providerInn_;
}

void SimpleReceiptOperation::setProviderInn(const QString &newProviderInn)
{
    providerInn_ = newProviderInn;
}

std::optional<FixNumber> SimpleReceiptOperation::exciseTax() const
{
    return exciseTax_;
}

void SimpleReceiptOperation::setExciseTax(std::optional<FixNumber> newExciseTax)
{
    exciseTax_ = newExciseTax;
}

const QString &SimpleReceiptOperation::countryCode() const
{
    return countryCode_;
}

void SimpleReceiptOperation::setCountryCode(const QString &newCountryCode)
{
    countryCode_ = STR4FS_(newCountryCode);
}

const QString &SimpleReceiptOperation::declarationNumber() const
{
    return declarationNumber_;
}

void SimpleReceiptOperation::setDeclarationNumber(const QString &newDeclarationNumber)
{
    declarationNumber_ = STR4FS_(newDeclarationNumber);
}
fdf::ItemUnit SimpleReceiptOperation::unit() const
{
    return fdf::unitIsValid(unit_) ? unit_ : fdf::ItemUnit::Ut;
}

void SimpleReceiptOperation::setUnit(fdf::ItemUnit newUnit)
{
    unit_ = newUnit;
}

std::optional<ProductCode> SimpleReceiptOperation::code() const
{
    return code_;
}

void SimpleReceiptOperation::setCode(std::optional<ProductCode> newCode)
{
    code_ = newCode;
}

QList<IndustryProperty> SimpleReceiptOperation::industryProperties() const
{
    return industryProperties_;
}

void SimpleReceiptOperation::setIndustryProperties(const QList<IndustryProperty> &newIndustryProperties)
{
    industryProperties_ = newIndustryProperties;
}
QVariantMap SimpleReceiptOperation::toMap() const
{
    QVariantMap res;
    res["quantity"] = quantity_.toMap();
    if(!name_.isEmpty()) res["name"] = name_;
    if(cost_.has_value()) res["cost"] = cost_.value().toMap();
    if(price_.has_value()) res["price"] = price_.value().toMap();
    if(!additionalParam_.isEmpty()) res["additionalParam"] = additionalParam_;
    if(priceVat_.has_value()) res["priceVat"] = priceVat_.value().toMap();
    if(fdf::vatIsValid(vatRate_)) res["vatRate"] = static_cast<qint32>(vatRate_);
    if(costVat_.has_value()) res["costVat"] = costVat_.value().toMap();
    if(fdf::opTypeIsValid(type_)) res["type"] = static_cast<qint32>(type_);
    if(fdf::itemPaymenTypeIsValid(paymentType_)) res["paymentType"] = static_cast<qint32>(paymentType_);
    if(fdf::itemPayAgentFlagIsValid(agentFlag_))
    {
        res["agentFlag"] = static_cast<qint32>(agentFlag_);
        if(agentData_.has_value()) res["agentData"] = agentData_.value().toMap();
        if(providerData_.has_value())res["providerData"] = providerData_.value().toMap();
        if(!providerInn_.isEmpty()) res["providerInn"] = providerInn_;
    }
    if(exciseTax_.has_value()) res["exciseTax"] = exciseTax_.value().toMap();
    if(!countryCode_.isEmpty()) res["countryCode"] = countryCode_;
    if(!declarationNumber_.isEmpty()) res["declarationNumber"] = declarationNumber_;
    if(fdf::unitIsValid(unit())) res["unit"] = static_cast<qint32>(unit());
    if(code_.has_value() && !code_.value().isEmpty()) res["productCode"] = code_.value().toMap();
    if(!industryProperties_.isEmpty())
    {
        QVariantList list;
        for(const IndustryProperty &p :industryProperties_)
        {
            if(!p.isEmpty()) list << p.toMap();
        }
        if(!list.isEmpty()) res["industryProperties"] = list;
    }
    return res;
}

QVariantMap SimpleReceiptOperation::toExternalMap() const
{
    QVariantMap res;
    res["quantity"] = quantity_.toString();
    if(!name_.isEmpty()) res["name"] = name_;
    else res["name"] = QVariant();
    res["cost"] = cost().toString();
    res["price"] = price().toString();
    if(!additionalParam_.isEmpty()) res["additionalParam"] = additionalParam_;
    res["priceVat"] = priceVat().toString();
    if(fdf::vatIsValid(vatRate_)) res["vatRate"] = static_cast<qint32>(vatRate_);
    res["costVat"] = costVat().toString();
    if(fdf::opTypeIsValid(type_)) res["type"] = static_cast<qint32>(type_);
    if(fdf::itemPaymenTypeIsValid(paymentType_)) res["paymentType"] = static_cast<qint32>(paymentType_);
    if(fdf::itemPayAgentFlagIsValid(agentFlag_))
    {
        res["agentFlag"] = static_cast<qint32>(agentFlag_);
        if(agentData_.has_value()) res["agentData"] = agentData_.value().toMap();
        if(providerData_.has_value())res["providerData"] = providerData_.value().toMap();
        if(!providerInn_.isEmpty()) res["providerInn"] = providerInn_;
    }
    if(exciseTax_.has_value()) res["exciseTax"] = exciseTax_.value().toString();
    if(!countryCode_.isEmpty()) res["countryCode"] = countryCode_;
    if(!declarationNumber_.isEmpty()) res["declarationNumber"] = declarationNumber_;
    if(fdf::unitIsValid(unit())) res["unit"] = static_cast<qint32>(unit());
    if(code_.has_value() && !code_.value().isEmpty()) res["productCode"] = code_.value().toMap();
    if(!industryProperties_.isEmpty())
    {
        QVariantList list;
        for(const IndustryProperty &p :industryProperties_)
        {
            if(!p.isEmpty()) list << p.toMap();
        }
        if(!list.isEmpty()) res["industryProperties"] = list;
    }
    return res;
}

void SimpleReceiptOperation::parseMap(const QVariantMap &map, CoreApiConst::ErrorCode *err)
{
    SimpleReceiptOperation::clean();
    quantity_ = FormatUtils::parseSumm("quantity", map, DEFAULT_QUANTITY_DEC, err);
    name_ = STR4FS_(map["name"].toString().trimmed());
    if(map.contains("cost")) cost_ = FormatUtils::parseSumm("cost", map, DEFAULT_AMOUNT_DEC, err);
    if(map.contains("price")) price_ = FormatUtils::parseSumm("price", map, DEFAULT_AMOUNT_DEC, err);
    additionalParam_ = STR4FS_(map["additionalParam"].toString().trimmed());
    if(map.contains("priceVat"))priceVat_ = FormatUtils::parseSumm("priceVat", map, DEFAULT_AMOUNT_DEC, err);
    vatRate_ = static_cast<fdf::VatRate>(map["vatRate"].toInt());
    if(map.contains("costVat"))costVat_ = FormatUtils::parseSumm("costVat", map, DEFAULT_AMOUNT_DEC, err);
    type_ = static_cast<fdf::ReceiptOperationType>(map["type"].toInt());
    paymentType_ = static_cast<fdf::ItemPaymentType>(map["paymentType"].toInt());
    agentFlag_ = static_cast<fdf::ItemPayAgentFlag>(map["agentFlag"].toInt());
    if(map.contains("agentData") && !map["agentData"].toMap().isEmpty())
    {
        agentData_.reset();
        AgentData ad = AgentData(map["agentData"].toMap());
        if(!ad.isEmpty()) agentData_ = ad;
    }
    if(map.contains("providerData") && !map["providerData"].toMap().isEmpty()) providerData_ = ProviderData(map["providerData"].toMap());
    providerInn_ = map["providerInn"].toString();
    if(map.contains("exciseTax")) exciseTax_ = FormatUtils::parseSumm("exciseTax", map, DEFAULT_AMOUNT_DEC, err);
    countryCode_ = STR4FS_(map["countryCode"].toString());
    declarationNumber_ = STR4FS_(map["declarationNumber"].toString());
    if(map.contains("unit"))unit_ = static_cast<fdf::ItemUnit>(map["unit"].toInt());
    if(!fdf::unitIsValid(unit())) unit_ = fdf::ItemUnit::Ut;
    if(map.contains("productCode") && !map["productCode"].toMap().isEmpty()) code_ = ProductCode(map["productCode"].toMap());
    industryProperties_.clear();
    if(map.contains("industryProperties"))
    {
        QVariantList list = map["industryProperties"].toList();
        for(const QVariant &v: list)
        {
            IndustryProperty p(v.toMap());
            QString msg;
            if(err && !p.isValid(false, *err, msg)) return;
            if(!p.isEmpty()) industryProperties_ << p;
        }
    }

}

Tlv SimpleReceiptOperation::to1059() const
{
    Tlv buf;
    Tlv::Stlv list;

    buf.setByte(fdf::Tag::ItemType, static_cast<quint8>(type_));
    list << buf;
    if(paymentType_ != fdf::ItemPaymentType::Advance)
    {
        buf.setString(fdf::Tag::ItemName, name_);
        list << buf;
    }
    buf.setByte(fdf::Tag::ItemPaymentType, static_cast<quint8>(paymentType_));
    list << buf;

    buf.setByte(fdf::Tag::VatRate, static_cast<quint8>(vatRate_));
    list << buf;

    buf.setVln(fdf::Tag::ItemPrice, price().value());
    list << buf;

    buf.setVln(fdf::Tag::ItemCost, cost().value());
    list << buf;

    buf.setFvln(fdf::Tag::ItemQuantity, quantity());
    list << buf;

    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::ItemAdditionalParam, additionalParam_);
        list << buf;
    }
    if(fdf::itemPayAgentFlagIsValid(agentFlag_))
    {
        buf.setByte(fdf::Tag::ItemAgentFlag, static_cast<quint8>(agentFlag_));
        list << buf;

        buf.setInn(fdf::Tag::ProviderInn, Inn(providerInn_));
        list << buf;
        if(providerData_.has_value() && providerData_->isValid()) list << providerData_.value().toTlv();
       if(agentData_.has_value()) list << agentData_.value().toTlv(agentFlag_);
    }
    if(exciseTax_.has_value())
    {
        buf.setVln(fdf::Tag::ExciseTax, exciseTax_.value().value());
        list << buf;
    }
    if(!countryCode_.isEmpty() && !declarationNumber_.isEmpty())
    {
        buf.setString(fdf::Tag::ProductCountryCode, countryCode_);
        list << buf;
        buf.setString(fdf::Tag::ProductDeclarationNumber, declarationNumber_);
        list << buf;
    }
    fdf::ItemUnit unit = fdf::ItemUnit::Ut;
    if(fdf::unitIsValid(unit_))
    {
        unit = unit_;
    }
    buf.setByte(fdf::Tag::ItemQuantityUnitType, static_cast<quint8>(unit));
    list << buf;
    if(code_.has_value()) list << code_.value().toTlv();
    for(const IndustryProperty & p: industryProperties_)list << p.toTlv(fdf::Tag::IndustryItemParam);

    buf.setStlv(fdf::Tag::ReceiptItem, list);


    return buf;
}

void SimpleReceiptOperation::parseTlv(const Tlv::Stlv &list)
{
    clean();

    QStringList sl;
    for(const Tlv &t :list)
    {
        sl <<( QString::number(static_cast<qint32>(t.tag())) + ": " + QString::fromLatin1(t.value().toHex()));
        parseChildTlv(t);
    }
//    lmWarning() << loglist(sl);
}

void SimpleReceiptOperation::clean()
{
    quantity_ = FixNumber(DEFAULT_QUANTITY_MULT, 0ll);
    name_.clear();
    cost_.reset();
    price_.reset();
    additionalParam_.clear();
    priceVat_.reset();
    vatRate_ = fdf::VatRate::Invalid;
    costVat_.reset();
    type_ = fdf::ReceiptOperationType::Invalid;
    paymentType_ = fdf::ItemPaymentType::FullPayment;
    agentFlag_ = fdf::ItemPayAgentFlag::NotAgent;
    agentData_.reset();
    providerData_.reset();
    providerInn_.clear();
    exciseTax_.reset();
    countryCode_.clear();
    declarationNumber_.clear();
    unit_ = fdf::ItemUnit::Ut;
    code_.reset();
    industryProperties_.clear();

}

void SimpleReceiptOperation::addQuantity()
{
    quantity_.setValue(quantity_.value() + quantity_.realMultiplier());
}

void SimpleReceiptOperation::subQuantity()
{
    if(quantity_.value() - quantity_.realMultiplier() > 0)
    {
        quantity_.setValue(quantity_.value() - quantity_.realMultiplier());
    }
}

SimpleReceiptOperation &SimpleReceiptOperation::operator =(const SimpleReceiptOperation &other) noexcept
{
    quantity_ = other.quantity_;
    name_ = other.name_;
    cost_ = other.cost_;
    price_ = other.price_;
    additionalParam_ = other.additionalParam_;
    priceVat_ = other.priceVat_;
    vatRate_ = other.vatRate_;
    costVat_ = other.cost_;
    type_ = other.type_;
    paymentType_ = other.paymentType_;
    agentFlag_ = other.agentFlag_;
    agentData_ = other.agentData_;
    providerData_ = other.providerData_;
    providerInn_ = other.providerInn_;
    exciseTax_ = other.exciseTax_;
    countryCode_ = other.countryCode_;
    declarationNumber_ = other.declarationNumber_;
    unit_ = other.unit_;
    code_ = other.code_;
    industryProperties_ = other.industryProperties_;

    return *this;
}
SimpleReceiptOperation &SimpleReceiptOperation::operator =(SimpleReceiptOperation &&other) noexcept
{
    quantity_ = other.quantity_;
    name_.swap(other.name_);
    cost_.swap(other.cost_);
    price_.swap(other.price_);
    additionalParam_.swap(other.additionalParam_);
    priceVat_.swap(other.priceVat_);
    vatRate_ = other.vatRate_;
    costVat_.swap(other.cost_);
    type_ = other.type_;
    paymentType_ = other.paymentType_;
    agentFlag_ = other.agentFlag_;
    agentData_.swap(other.agentData_);
    providerData_.swap(other.providerData_);
    providerInn_.swap(other.providerInn_);
    exciseTax_.swap(other.exciseTax_);
    countryCode_.swap(other.countryCode_);
    declarationNumber_.swap(other.declarationNumber_);
    unit_ = other.unit_;
    code_.swap(other.code_);
    industryProperties_.swap(other.industryProperties_);
    return *this;
}

bool SimpleReceiptOperation::operator ==(const SimpleReceiptOperation &other) const noexcept
{
    return quantity_ == other.quantity_ &&
            name_ == other.name_ &&
            cost_ == other.cost_ &&
            price_ == other.price_ &&
            additionalParam_ == other.additionalParam_ &&
            priceVat_ == other.priceVat_ &&
            vatRate_ == other.vatRate_ &&
            costVat_ == other.cost_ &&
            type_ == other.type_ &&
            paymentType_ == other.paymentType_ &&
            agentFlag_ == other.agentFlag_ &&
            agentData_ == other.agentData_ &&
            providerData_ == other.providerData_ &&
            providerInn_ == other.providerInn_ &&
            exciseTax_ == other.exciseTax_ &&
            countryCode_ == other.countryCode_ &&
            declarationNumber_ == other.declarationNumber_  &&
            unit_ == other.unit_  &&
            code_ == other.code_ &&
           industryProperties_ == other.industryProperties_;

}

bool SimpleReceiptOperation::operator !=(const SimpleReceiptOperation &other) const noexcept
{
    return !(*this == other);
}


bool SimpleReceiptOperation::checkPriceAndCost(CoreApiConst::ErrorCode &err, QString &msg) const
{

    //21) В случае если значение реквизита "количество предмета расчета" (тег 1023) равно "0", то реквизит "предмет расчета" (тег 1059) не включается в состав кассового чека.

    if(quantity_.realMultiplier() != DEFAULT_QUANTITY_MULT || quantity_.value() <=0ll)
    {
        err = CoreApiConst::ErrorCode::InvalidItemQuantity;
        msg = "Не указано количество товара";
        return false;
    }

    if(!hasPrice())
    {
        err = CoreApiConst::ErrorCode::NoItemPrice;
        msg = "Не указана цена товара";
        return false;
    }
    if(price().value() < 0)
    {
        err = CoreApiConst::ErrorCode::InvalidItemPrice;
        msg = "Некорректное знаечение цены";
        return false;
    }
    if(cost_.has_value() && qAbs(cost_.value().value() - calcCost().value()) > 1ll)
    {
        err = CoreApiConst::ErrorCode::InvalidItemPrice;
        msg = "Некорректное стоимости предмета расчета ";
        return false;
    }
    return true;
}

bool SimpleReceiptOperation::checkAgent(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(!fdf::itemPayAgentFlagIsValid(agentFlag_)) return true;
    if(agentFlag_ < fdf::ItemPayAgentFlag::Attorney && !agentData_.has_value())
    {
        err = CoreApiConst::ErrorCode::InvalidItemAgentData;
        msg = "Не хватает реквизита 1223 (данные агента)";
        return false;
    }
    if(agentData_.has_value() && !agentData_.value().isValid(agentFlag_, err, msg))
    {
        return false;
    }
    Inn i(providerInn_);
    if(!i.isValid(true))
    {
        err = CoreApiConst::ErrorCode::InvalidProviderInn;
        msg = "Не корректное значение реквизита 1226 (ИНН поставщика)";
        return false;
    }
    if(!providerData_.has_value())
    {
        err = CoreApiConst::ErrorCode::NoProviderData;
        msg = "Не хватает реквизита 1224 (данные поставщика)";
        return false;
    }
    if(!providerData_.value().isValid(false, err, msg))
    {
#if CASHBOX_MODEL != 2000
        return false;
#endif
    }
    return true;
}

bool SimpleReceiptOperation::checkCountry(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(countryCode_.trimmed().isEmpty() && declarationNumber_.trimmed().isEmpty()) return true;
    if(!FormatUtils::checkString(countryCode_, 3, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1230 (код страны происхождения товара) " + msg;
        return false;
    }
    if(!FormatUtils::checkString(declarationNumber_, 32, false, false, msg))
    {
        err = CoreApiConst::ErrorCode::InvalidParameter;
        msg = "Реквизит 1231 (номер декларации на товар) " + msg;
        return false;
    }
    return true;
}
bool SimpleReceiptOperation::checkIndustryCode(CoreApiConst::ErrorCode &err, QString &msg) const
{
    if(industryProperties_.isEmpty()) return true;
    return std::all_of(industryProperties_.constBegin(), industryProperties_.constEnd(),
                       [&err, &msg](const IndustryProperty &p) -> bool {
                           return p.isValid(false, err, msg);
                       });
}
void SimpleReceiptOperation::parseChildTlv(const Tlv &tlv)
{
    switch (tlv.tag()) {
    case fdf::Tag::ItemQuantity: quantity_ = tlv.toFvln(); break;
    case fdf::Tag::ItemName: name_ = tlv.toString();break;
    case fdf::Tag::ItemCost: cost_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv.toVln()); break;
    case fdf::Tag::ItemPrice: price_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv.toVln());break;
    case fdf::Tag::ItemAdditionalParam: additionalParam_ = tlv.toString();break;
    case fdf::Tag::VatForItemUnit: priceVat_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv.toVln());break;
    case fdf::Tag::VatForItem: costVat_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv.toVln());break;
    case fdf::Tag::VatRate: vatRate_ = static_cast<fdf::VatRate>(tlv.toByte());break;
    case fdf::Tag::ItemType: type_ = static_cast<fdf::ReceiptOperationType>(tlv.toByte());break;
    case fdf::Tag::ItemPaymentType: paymentType_ = static_cast<fdf::ItemPaymentType>(tlv.toByte());break;
    case fdf::Tag::ItemAgentFlag: agentFlag_ = static_cast<fdf::ItemPayAgentFlag>(tlv.toByte());break;
    case fdf::Tag::ItemAgentData:
    {
        AgentData ad;
        ad.parseTlv(tlv);
        agentData_ = ad;
    }break;
    case fdf::Tag::ProviderData:
    {
        ProviderData pd;
        pd.parseTlv(tlv);
        providerData_ = pd;
    }break;
    case fdf::Tag::ProviderInn: providerInn_ = tlv.toString().trimmed();break;
    case fdf::Tag::ExciseTax: exciseTax_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv.toVln());break;
    case fdf::Tag::ProductCountryCode: countryCode_ = tlv.toString().trimmed();break;
    case fdf::Tag::ProductDeclarationNumber: declarationNumber_ = tlv.toString().trimmed();break;
    case fdf::Tag::ItemQuantityUnitType: unit_ = static_cast<fdf::ItemUnit>(tlv.toByte());break;
    case fdf::Tag::ProductCode:
    {
        ProductCode pc;
        pc.parseTlv(tlv);
        code_ = pc;
    }break;
    case fdf::Tag::IndustryItemParam:
    {
        IndustryProperty ip;
        ip.parseTlv(tlv);
        industryProperties_ << ip;
    }break;

    default:
        break;
    }
}
