#include "tax.h"
#include "cashboxbuildconfig.h"

#include "fdfconstants.h"
#include "receiptdataclasses.h"

#include <QDate>


Tax::Tax(TaxRate rate) :
    rate_(fdf::vatIsValid(rate) ? rate : TaxRate::Invalid)
{

}

Tax::Tax(const Tlv &tag) :
    rate_(TaxRate::Invalid)
{
    switch (tag.tag())
    {
    case fdf::Tag::ReceiptVat20Amount:
    case fdf::Tag::Vat20Amount:
        rate_ = TaxRate::Vat20;
        break;
    case fdf::Tag::ReceiptVat10Amount:
    case fdf::Tag::Vat10Amount:
        rate_ = TaxRate::Vat10;
        break;
    case fdf::Tag::ReceiptVat20_120Amount:
    case fdf::Tag::Vat20_120Amount:
        rate_ = TaxRate::Vat20_120;
        break;
    case fdf::Tag::ReceiptVat10_110Amount:
    case fdf::Tag::Vat10_110Amount:
        rate_ = TaxRate::Vat10_110;
        break;
    case fdf::Tag::ReceiptVat0Amount:
    case fdf::Tag::Vat0Amount:
        rate_ = TaxRate::Vat0;
        break;
    case fdf::Tag::ReceiptVatNoneAmount:
    case fdf::Tag::VatNoneAmount:
        rate_ = TaxRate::VatNone;
        break;
    case fdf::Tag::ReceiptVatAmount:
    {
        ReceiptsVatAmount v;
        v.parseTlv(tag);
        rate_ = v.type();
    }break;
    default: rate_ = TaxRate::Invalid;
    }
}

Tax::Tax(const Tax &other) :
    rate_(other.rate_)
{

}

Tax::Tax(Tax &&other):
    rate_(other.rate_)
{

}

Tax::~Tax()
{

}

quint32 Tax::rateValue() const
{
    switch (rate_)
    {
    case TaxRate::VatNone: return 0u;
    case TaxRate::Vat0: return  0u;
    case TaxRate::Vat10_110: return 1000u;
    case TaxRate::Vat20_120: return  2000u;
    case TaxRate::Vat20: return  2000u;
    case TaxRate::Vat10: return  1000u;
    case TaxRate::Vat5: return 500u;
    case TaxRate::Vat7: return 700u;
    case TaxRate::Vat5_105: return 500u;
    case TaxRate::Vat7_107: return 700u;
    default: return 0u;
    }
}

Tax::TaxRate Tax::rate() const
{
    return rate_;
}

void Tax::setRate(const TaxRate &rate)
{
    rate_ = rate;
}

FixNumber Tax::operator()(const FixNumber &sum) const
{
    FixNumber res(DEFAULT_AMOUNT_MULT, 0ll);
    if(sum.value() == 0ll) return res;
    if(sum.value() < 0ll)
    {
        res.setValue(-1ll);
        return res;
    }
    switch (rate_)
    {
    case TaxRate::Vat20:
        res.setDouble(sum.toDouble() * (1 -  1.0/1.2));
        break;
    case TaxRate::Vat10:
            res.setDouble(sum.toDouble() * (1 -  1.0/1.10) );
        break;
    case TaxRate::Vat0:
    case TaxRate::VatNone:
            res = sum;
        break;
    case TaxRate::Vat20_120:
        res.setDouble(20.0 * sum.toDouble() /120.0 );
        break;
    case TaxRate::Vat10_110:
            res.setDouble(10.0 * sum.toDouble() /110.0 );
        break;
    case TaxRate::Vat5:
        res.setDouble(sum.toDouble() * (1 -  1.0/1.05) );
        break;
    case TaxRate::Vat7:
        res.setDouble(sum.toDouble() * (1 -  1.0/1.07) );
        break;
    case TaxRate::Vat5_105:
        res.setDouble(5.0 * sum.toDouble() /105.0 );
        break;
    case TaxRate::Vat7_107:
        res.setDouble(7.0 * sum.toDouble() /107.0 );
        break;

    default: res.setValue(-1ll);
    }
    return res;
}

fdf::Tag Tax::receiptTag() const
{
    switch (rate_)
    {
    case TaxRate::Vat20: return  fdf::Tag::ReceiptVat20Amount;
    case TaxRate::Vat10: return  fdf::Tag::ReceiptVat10Amount;
    case TaxRate::Vat0: return  fdf::Tag::ReceiptVat0Amount;
    case TaxRate::VatNone: return  fdf::Tag::ReceiptVatNoneAmount;
    case TaxRate::Vat20_120: return  fdf::Tag::ReceiptVat20_120Amount;
    case TaxRate::Vat10_110: return  fdf::Tag::ReceiptVat10_110Amount;
    case TaxRate::Vat5:
    case TaxRate::Vat7:
    case TaxRate::Vat5_105:
    case TaxRate::Vat7_107: return fdf::Tag::ReceiptVatAmount;

    default: return fdf::Tag::Invalid;
    }
}

fdf::Tag Tax::reportsTag() const
{
    switch (rate_)
    {
    case TaxRate::Vat20: return  fdf::Tag::Vat20Amount;
    case TaxRate::Vat10: return  fdf::Tag::Vat10Amount;
    case TaxRate::Vat0: return  fdf::Tag::Vat0Amount;
    case TaxRate::VatNone: return  fdf::Tag::VatNoneAmount;
    case TaxRate::Vat20_120: return  fdf::Tag::Vat20_120Amount;
    case TaxRate::Vat10_110: return  fdf::Tag::Vat10_110Amount;
    // case TaxRate::Vat5:
    // case TaxRate::Vat7:
    // case TaxRate::Vat5_105:
    // case TaxRate::Vat7_107:

    default: return fdf::Tag::Invalid;
    }
}



Tlv Tax::receiptProperty(FixNumber * const value) const
{
    fdf::Tag tag = receiptTag();
    if(tag == fdf::Tag::Invalid) return Tlv();
    if(tag == fdf::Tag::ReceiptVatAmount)
    {
        ReceiptsVatAmount a;
        a.setType(rate_);
        if(value)a.setAmount(*value);
        return a.toTlv();
    }
    Tlv res;
    res.setTag(tag);
    if(value)res.setVln(value->value());
    else res.setVln(0ll);
    return res;
}

Tlv Tax::reportsProperty(FixNumber * const value) const
{
    fdf::Tag tag = reportsTag();
    if(tag == fdf::Tag::Invalid) return Tlv();
    Tlv res;
    res.setTag(tag);
    if(value)res.setVln(value->value());
    else res.setVln(0ll);
    return res;
}


QString Tax::caption() const
{
    return fdf::vatPrintable(rate_);
}



Tax &Tax::operator =(const Tax &other)
{
    rate_ = other.rate_;
    return *this;
}

Tax &Tax::operator =(Tax &&other)
{
    rate_ = other.rate_;
    return *this;
}

bool Tax::operator ==(const Tax &other) const
{
    return rate_ == other.rate_ &&
            rate_ != TaxRate::Invalid;
}

bool Tax::operator !=(const Tax &other) const
{
    return  !(*this == other);
}


Tax::operator bool() const
{
    return fdf::vatIsValid(rate_);
}


