#include "umka365SessionData.h"



#include <QJsonParseError>
#include <QJsonDocument>

namespace umka365 {


SessionData::SessionData()
    : sessionData_()
    , hardwareData_()
    , models_()
{

}

SessionData::SessionData(const SessionData &other)
    : sessionData_(other.sessionData_)
    , hardwareData_(other.hardwareData_)
    , models_(other.models_)
{

}

SessionData::SessionData(SessionData &&other)
    : sessionData_()
    , hardwareData_()
    , models_(other.models_)

{
    sessionData_.swap(other.sessionData_);
    hardwareData_.swap(other.hardwareData_);
}


SessionData::~SessionData()
{

}

bool SessionData::isValid() const
{
    return !userIsBlocked() && !dealerIsBlocked() &&
            dealerId() > 0ll && userId() > 0ll &&
            !sessionId().isEmpty();
}

qint64 SessionData::dealerId() const
{
    qint64 res = sessionData_.contains("idDealer") ? sessionData_["idDealer"].toULongLong() : -1;
    return res;
}

QString SessionData::dealerName() const
{
    QString res = sessionData_.contains("nameDealer") ? sessionData_["nameDealer"].toString() : QString();
    return res;

}

bool SessionData::dealerIsBlocked() const
{
    bool res = !sessionData_.contains("blockedDealer") || sessionData_["blockedDealer"].toBool();
    return res;
}

qint16 SessionData::userId() const
{
    qint64 res = sessionData_.contains("idUser") ? sessionData_["idUser"].toULongLong() : -1;
    return res;
}

QString SessionData::userName() const
{
    QString res = sessionData_.contains("nameUser") ? sessionData_["nameUser"].toString() : QString();
    return res;
}

bool SessionData::userIsBlocked() const
{
    bool res = !sessionData_.contains("blockedUser") || sessionData_["blockedUser"].toBool();
    return res;
}

QString SessionData::sessionId() const
{
    QString res = sessionData_.contains("idSession") ? sessionData_["idSession"].toString() : QString();
    return res;
}

SupportedModels SessionData::models() const
{
    SupportedModels res = models_;
    return res;
}

QHash<quint16, QString> SessionData::supportedModels() const
{
    QHash<quint16, QString> res = models_.models();
    return res;
}

bool SessionData::parseData(const QByteArray &data)
{
    QJsonParseError jerr;
    QJsonDocument jdoc = QJsonDocument::fromJson(data, &jerr);
    if(jerr.error != QJsonParseError::NoError)
    {
        return false;
    }
    QVariantMap map = jdoc.toVariant().toMap();
    if(map.contains("idSession"))
    {
        sessionData_ = map;
        bool res = isValid();
        if(!res) clean();
        return res;
    }
    else if(map.contains("hardware"))
    {
        QVariantList hw = map["hardware"].toList();
        QVector<QVariant> sorted(hw.size());

        qint64 dealer = dealerId();

        QVector<QVariant>::iterator sortedend = std::copy_if(hw.constBegin(), hw.constEnd(), sorted.begin(), [dealer](const QVariant &v) -> bool {
            QVariantMap m = v.toMap();
            bool res = (m["idManufacturer"].toLongLong() == dealer || dealer == 2) && m["idHardwareType"].toInt() == 1;
            return res; });

        sorted.resize(sortedend - sorted.begin());
        hardwareData_ = QVariantList::fromVector(sorted);
        models_ = SupportedModels(hardwareData_);
        return true;
    }
    return false;
}

QByteArray SessionData::hardwareRequestBody() const
{
   QByteArray res;
   if(isValid())
   {
       QVariantMap session;
       session.insert("idSession", sessionId());
       session.insert("idDealer", dealerId());
       session.insert("idUser", userId());
       QVariantMap chid;
       chid.insert("hardware", "0");
       chid.insert("hardwareType", "0");
       QVariantMap web;
       web.insert("afterChId", chid);
       QVariantMap req;
       req.insert("web", web);
       req.insert("session", session);
       res = QJsonDocument::fromVariant(req).toJson(QJsonDocument::Compact);
   }
   return res;
}

QVariantMap SessionData::toMap() const
{
    return QVariantMap {
        {"session", sessionData_},
        {"hardware", hardwareData_}
    };
}

void SessionData::cleanSession()
{
    sessionData_.clear();
}

void SessionData::cleanHardware()
{
    hardwareData_.clear();
    models_ = SupportedModels(QVariantList());
}

void SessionData::clean()
{
    cleanSession();
    cleanHardware();
}

SessionData &SessionData::operator =(const SessionData &other) noexcept
{
    sessionData_ = other.sessionData_;
    hardwareData_ = other.hardwareData_;
    models_ = other.models_;
    return *this;
}

SessionData &SessionData::operator =(SessionData &&other) noexcept
{
    sessionData_.swap(other.sessionData_);
    hardwareData_.swap(other.hardwareData_);
    models_ = other.models_;
    return *this;

}


bool SessionData::operator ==(const SessionData &other) const noexcept
{
    return sessionData_ == other.sessionData_ &&
               hardwareData_ == other.hardwareData_ &&
               models_ == other.models_;
}

bool SessionData::operator !=(const SessionData &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------

SupportedModels::SupportedModels() noexcept
    : byModel_()
    , byName_()
{

}

SupportedModels::SupportedModels(const SupportedModels &other) noexcept
    : byModel_(other.byModel_)
    , byName_(other.byName_)
{

}

SupportedModels::SupportedModels(SupportedModels &&other) noexcept
    : byModel_()
    , byName_()
{
    byModel_.swap(other.byModel_);
    byName_.swap(other.byName_);
}

SupportedModels::~SupportedModels()
{

}

QString SupportedModels::operator[](quint16 model) const
{
    return byModel_.contains(model) ? byModel_[model] :QString();
}

quint16 SupportedModels::operator[](const QString &name) const
{
    return byName_.contains(name.toLower()) ? byName_[name.toLower()] : 0u;
}

bool SupportedModels::isSupported(quint16 model) const
{
    return byModel_.contains(model);
}

bool SupportedModels::isSupported(const QString &name) const
{
    return byName_.contains(name.toLower());
}

QHash<quint16, QString> SupportedModels::models() const
{
    return byModel_;
}

SupportedModels &SupportedModels::operator =(const SupportedModels &other) noexcept
{
    byModel_ = other.byModel_;
    byName_ = other.byName_;
    return *this;
}

SupportedModels &SupportedModels::operator =(SupportedModels &&other) noexcept
{
    byModel_.swap(other.byModel_);
    byName_.swap(other.byName_);
    return *this;
}

bool SupportedModels::operator ==(const SupportedModels &other) const noexcept
{
    return byModel_ == other.byModel_;
}

bool SupportedModels::operator !=(const SupportedModels &other) const noexcept
{
    return !(*this == other);
}

SupportedModels::SupportedModels(const QVariantList &data)
    : byModel_()
    , byName_()
{
    for(const QVariant &v: data)
    {
        const QVariantMap &m = v.toMap();
        quint16 model = static_cast<quint16>(m["officialModel"].toString().toUInt());
        QString name = m["officialName"].toString();
        byModel_.insert(model, name);
        byName_.insert(name.toLower(), model);
    }
}

}
