#include "basedoc.h"

#include "formatutils.h"


#include <QJsonDocument>
#include <QJsonParseError>

BaseDoc::~BaseDoc()
{

}




QString BaseDoc::docName() const
{
    switch (type_) {
    case fdf::DocType::Registration      : return tr("Отчет о регистрации");
    case fdf::DocType::ReRegistration    : return tr("Отчет об изменении параметров регистрации");
    case fdf::DocType::CycleOpen         : return tr("Отчет об открытии смены");
    case fdf::DocType::CalcReport        : return tr("Отчет о текущем состоянии расчетов");
    case fdf::DocType::Receipt           : return tr("Кассовый чек");
    case fdf::DocType::Correction        : return tr("Кассовый чек коррекции");
    case fdf::DocType::Fosa              : return tr("Бланк строгой отчетности");
    case fdf::DocType::FosaCorrection    : return tr("Бланк строгой отчетности коррекции");
    case fdf::DocType::CycleClose        : return tr("Отчет о закрытии смены");
    case fdf::DocType::FsClose           : return tr("Отчет о закрытии фискального накопителя");
    case fdf::DocType::OfdResponse       : return tr("Подтверждение оператора");
    case fdf::DocType::LabelRequest      : return tr("Запрос о коде маркировки");
    case fdf::DocType::LabelNotification : return tr("Уведомление о реализации маркированноготовара");
    case fdf::DocType::LabelResponse     : return tr("Ответ на запрос");
    case fdf::DocType::NotificationTicket: return tr("Квитанция на уведомление");
    case fdf::DocType::Invalid           : return QString();
    default: return QString();
    }

}

QString BaseDoc::docShortName() const
{
    QString res;
    switch (type_) {
    case fdf::DocType::Registration      : res = tr("Отчет о рег.");        break;
    case fdf::DocType::ReRegistration    : res = tr("Отчет о перерег.");    break;
    case fdf::DocType::CycleOpen         : res = tr("Отчет об откр. см.");  break;
    case fdf::DocType::CalcReport        : res = tr("Отчет о расч.");       break;
    case fdf::DocType::Receipt           : res = tr("Кассовый чек");        break;
    case fdf::DocType::Correction        : res = tr("Чек коррекции");       break;
    case fdf::DocType::Fosa              : res = tr("БСО");                 break;
    case fdf::DocType::FosaCorrection    : res = tr("БСО коррекции");       break;
    case fdf::DocType::CycleClose        : res = tr("Отч. о закр. см.");    break;
    case fdf::DocType::FsClose           : res = tr("Отч. о закр. ФН");     break;
    case fdf::DocType::OfdResponse       : res = tr("Подтверждение");       break;
    default: return tr("НЕ ДОКУМЕНТ");
    }
    if(isCopy_)res += "\n(КОПИЯ)";
    return res;
}

fdf::DocType BaseDoc::type() const
{
    return type_;
}

void BaseDoc::setType(fdf::DocType newType)
{
    type_ = newType;
}

qint32 BaseDoc::fd() const
{
    return fd_;
}

void BaseDoc::setFd(qint32 newFd)
{
    fd_ = newFd;
}

quint32 BaseDoc::fiscalCode() const
{
    return fiscalCode_;
}

void BaseDoc::setFiscalCode(quint32 newFiscalCode)
{
    fiscalCode_ = newFiscalCode;
}

const QDateTime &BaseDoc::dt() const
{
    return dt_;
}

void BaseDoc::setDt(const QDateTime &newDt)
{
    dt_ = newDt;
}

const QString &BaseDoc::fsNumber() const
{
    return fsNumber_;
}

void BaseDoc::setFsNumber(const QString &newFsNumber)
{
    fsNumber_ = newFsNumber;
}

bool BaseDoc::isCopy() const
{
    return isCopy_;
}

void BaseDoc::setIsCopy(bool newIsCopy)
{
    isCopy_ = newIsCopy;
}

const QString &BaseDoc::externalId() const
{
    return externalId_;
}

void BaseDoc::setExternalId(const QString &newExternalId)
{
    externalId_ = newExternalId;
}

const QString &BaseDoc::clientId() const
{
    return clientId_;
}

void BaseDoc::setClientId(const QString &newClientId)
{
    clientId_ = newClientId;
}


void BaseDoc::setDataFromTask(const CoreTransaction &task)
{
    clientId_= task.clientId();
    externalId_ = task.externalId();
}
void BaseDoc::parseFromTlvList(const Tlv::Stlv &list, bool clear)
{
    if(clear)clean();
    QStringList sl;
    for(const Tlv &t: list)
    {
        sl << QStringLiteral("%1 [%2]: %3")
              .arg(static_cast<qint32>(t.tag()))
              .arg(t.len())
              .arg(QString::fromLatin1(t.value().toHex()));
        parseFromTlv(t);
    }
}

void BaseDoc::clean()
{
    fd_ = 0;
    fiscalCode_ = 0;
    dt_ = QDateTime();
    fsNumber_.clear();
    externalId_.clear();
    clientId_.clear();
}

QVariantMap BaseDoc::toMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"docNumber", fd_},
        {"fiscalCode", QString::number(fiscalCode_)},
        {"dt", DT2STR_(dt_)},
        {"fsNumber", fsNumber_},
        {"externalId",     externalId_      },
        {"clientId",       clientId_        }
    };
}

QVariantMap BaseDoc::toExternalMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"docNumber", fd_},
        {"fiscalCode", QString::number(fiscalCode_)},
        {"dt", dt_.isValid() ? QVariant(DT2STR_(dt_)) : QVariant()},
        {"fsNumber", fsNumber_.isEmpty() ? QVariant() : QVariant(fsNumber_)},
        {"externalId",  externalId_ .isEmpty() ? QVariant() : QVariant(externalId_ ) },
        {"clientId",     clientId_    .isEmpty() ? QVariant() : QVariant(clientId_    ) }
    };

}

void BaseDoc::parseMap(const QVariantMap &map)
{

    fdf::DocType tp = static_cast<fdf::DocType>(map["type"].toInt());
    if(tp == type_)
    {
        fd_= map["docNumber"].toInt();
        fiscalCode_ = map["docNumber"].toString().toUInt();
        dt_ = STR2DT_(map["dt"].toString());
        fsNumber_ = map["fsNumber"].toString().trimmed();
        externalId_ = map["externalId"].toString().trimmed();
        clientId_ = map["clientId"].toString().trimmed();
    }
}



BaseDoc::BaseDoc(fdf::DocType docType, QObject *parent)
    : QObject(parent)
    , type_(docType)
    , fd_(0)
    , fiscalCode_(0)
    , dt_()
    , fsNumber_()
    , isCopy_(false)
    , externalId_()
    , clientId_()

{

}

void BaseDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag()) {
    case fdf::Tag::Fd: fd_ = tlv.toUInt32(); break;
    case fdf::Tag::FPD: fiscalCode_ = tlv.toUInt32(); break;
    case fdf::Tag::DateTime: dt_ = tlv.toDt(); break;
    case fdf::Tag::Fs: fsNumber_ = tlv.toString().trimmed();break;
    default:
        break;
    }
}

bool BaseDoc::jsonToVariant(const QString &json, QVariant &v)
{
    QJsonParseError jerr;
    QJsonDocument jdoc = QJsonDocument::fromJson(json.toUtf8(), &jerr);
    v = jdoc.toVariant();
    return jerr.error == QJsonParseError::NoError;
}



//--------------------------------------------------------------------------------------------------





BaseWorkDoc::~BaseWorkDoc()
{

}

fs::FFD BaseWorkDoc::ffd() const
{
    return ffd_;
}

void BaseWorkDoc::setFfd(fs::FFD newFfd)
{
    ffd_ = newFfd;
}

quint16 BaseWorkDoc::cycle() const
{
    return cycle_;
}

void BaseWorkDoc::setCycle(quint16 newCycle)
{
    cycle_ = newCycle;
}
const QString &BaseWorkDoc::address() const
{
    return address_;
}

void BaseWorkDoc::setAddress(const QString &newAddress)
{
    address_ = STR4FS_(newAddress);
}

const QString &BaseWorkDoc::ownerInn() const
{
    return ownerInn_;
}

void BaseWorkDoc::setOwnerInn(const QString &newUserInn)
{
    ownerInn_ = newUserInn;
}

const QString &BaseWorkDoc::regNumber() const
{
    return regNumber_;
}

void BaseWorkDoc::setRegNumber(const QString &newRegNumber)
{
    regNumber_ = newRegNumber;
}

const QString &BaseWorkDoc::ownerName() const
{
    return ownerName_;
}

void BaseWorkDoc::setOwnerName(const QString &newUserName)
{
    ownerName_ = newUserName;
}

const QString &BaseWorkDoc::place() const
{
    return place_;
}

void BaseWorkDoc::setPlace(const QString &newPlace)
{
    place_ = newPlace;
}


void BaseWorkDoc::clean()
{
    BaseDoc::clean();
    ffd_ = fs::FFD::FFD1_2;
    cycle_ = 0;
    address_.clear();
    ownerInn_.clear();
    regNumber_.clear();
    ownerName_.clear();
    place_.clear();

}

QVariantMap BaseWorkDoc::toMap() const
{
    QVariantMap res {
        {"ffd", static_cast<qint32>(ffd_)},
        {"cycle", cycle_},
        {"address",        address_         },
        {"ownerInn",       ownerInn_        },
        {"regNumber",      regNumber_       },
        {"ownerName",      ownerName_       },
        {"place",          place_           },
    };
    res = UNIT_MAPS(BaseDoc::toMap(), res);
    return res;
}

QVariantMap BaseWorkDoc::toExternalMap() const
{
    QVariantMap res {
        {"ffd", fs::ffd2String(ffd_)},
        {"cycle", cycle_},
        {"address",   address_  .isEmpty() ? QVariant() : QVariant(address_  ) },
        {"ownerInn",   ownerInn_  .isEmpty() ? QVariant() : QVariant(ownerInn_  ) },
        {"regNumber", regNumber_.isEmpty() ? QVariant() : QVariant(regNumber_) },
        {"ownerName",  ownerName_ .isEmpty() ? QVariant() : QVariant(ownerName_ ) },
        {"place",     place_    .isEmpty() ? QVariant() : QVariant(place_    ) },
    };
    res = UNIT_MAPS(BaseDoc::toMap(), res);
    return res;
}

void BaseWorkDoc::parseMap(const QVariantMap &map)
{
    BaseDoc::parseMap(map);
    ffd_ = static_cast<fs::FFD>(map["ffd"].toUInt());
    fs::checkFfd(ffd_);
    cycle_ = map["cycle"].toUInt();
    address_        = map["address"].toString();
    ownerInn_        = map["ownerInn"].toString();
    regNumber_      = map["regNumber"].toString();
    ownerName_       = map["ownerName"].toString();
    place_          = map["place"].toString();
}

void BaseWorkDoc::setRegData(const RegData &rd, bool replace)
{
    ffd_ = rd.ffd();
    if(address_.isEmpty() || replace)address_ = rd.address();
    ownerInn_ = rd.ownerInn();
    regNumber_ = rd.regNumber();
    if(ownerName_.isEmpty() || replace)ownerName_ = rd.ownerName();
    if(place_.isEmpty()||replace)place_ = rd.place();
}


BaseWorkDoc::BaseWorkDoc(fdf::DocType type, QObject *parent)
    : BaseDoc(type, parent)
    , ffd_(fs::FFD::FFD1_2)
    , cycle_(0)
    , address_()
    , ownerInn_()
    , regNumber_()
    , ownerName_()
    , place_()
{

}

void BaseWorkDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag())
    {
    case fdf::Tag::Cycle: cycle_ = tlv.toUInt32();break;
    case fdf::Tag::Address: address_ = tlv.toString().trimmed();break;
    case fdf::Tag::OwnerInn: ownerInn_ = tlv.toString().trimmed();break;
    case fdf::Tag::CashboxRegNum: regNumber_ = tlv.toString().trimmed();break;
    case fdf::Tag::OwnerName: ownerName_ = tlv.toString().trimmed();break;
    case fdf::Tag::Place: place_ = tlv.toString().trimmed();break;
    case fdf::Tag::FFD: ffd_ = static_cast<fs::FFD>(tlv.toByte());break;
    default: BaseDoc::parseFromTlv(tlv);
    }
}

//--------------------------------------------------------------------------------------------------
BaseWorkIncoming::BaseWorkIncoming() noexcept
    : cashier_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , ownerName_()
    , address_()
    , place_()
{

}

BaseWorkIncoming::BaseWorkIncoming(const QVariantMap &map)noexcept
    : cashier_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , ownerName_()
    , address_()
    , place_()

{
    parseMap(map);
}

BaseWorkIncoming::BaseWorkIncoming(const BaseWorkIncoming &other)noexcept
    : cashier_(other.cashier_)
    , cashierInn_(other.cashierInn_)
    , additionalParam_(other.additionalParam_)
    , additionalData_(other.additionalData_)
    , ownerName_(other.ownerName_)
    , address_(other.address_)
    , place_(other.place_)
{

}

BaseWorkIncoming::BaseWorkIncoming(BaseWorkIncoming &&other) noexcept
    : cashier_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , ownerName_()
    , address_()
    , place_()

{
    cashier_.swap(other.cashier_);
    cashierInn_.swap(other.cashierInn_);
    additionalParam_.swap(other.additionalParam_);
    additionalData_.swap(other.additionalData_);
    ownerName_.swap(other.ownerName_);
    address_.swap(other.address_);
    place_.swap(other.place_);
}

BaseWorkIncoming::~BaseWorkIncoming()
{

}

const QString &BaseWorkIncoming::cashier() const
{
    return cashier_;
}

void BaseWorkIncoming::setCashier(const QString &newCashier)
{
    cashier_ = newCashier;
}

const QString &BaseWorkIncoming::cashierInn() const
{
    return cashierInn_;
}

void BaseWorkIncoming::setCashierInn(const QString &newCashierInn)
{
    cashierInn_ = newCashierInn;
}

const QString &BaseWorkIncoming::additionalParam() const
{
    return additionalParam_;
}

void BaseWorkIncoming::setAdditionalParam(const QString &newAdditionalParam)
{
    additionalParam_ = STR4FS_(newAdditionalParam);
}

const QByteArray &BaseWorkIncoming::additionalData() const
{
    return additionalData_;
}

void BaseWorkIncoming::setAdditionalData(const QByteArray &newAdditionalData)
{
    additionalData_ = newAdditionalData;
}
const QString &BaseWorkIncoming::ownerName() const
{
    return ownerName_;
}

void BaseWorkIncoming::setOwnerName(const QString &newUserName)
{
    ownerName_ = STR4FS_(newUserName);
}

const QString &BaseWorkIncoming::address() const
{
    return address_;
}

void BaseWorkIncoming::setAddress(const QString &newAddress)
{
    address_ = newAddress;
}

const QString &BaseWorkIncoming::place() const
{
    return place_;
}

void BaseWorkIncoming::setPlace(const QString &newPlace)
{
    place_ = STR4FS_(newPlace);
}


QVariantMap BaseWorkIncoming::toMap() const
{
    return {
        {"cashier",        cashier_         },
        {"cashierInn",     cashierInn_      },
        {"additionalParam",additionalParam_ },
        {"additionalData", QString::fromLatin1(additionalData_.toHex()) },
        {"ownerName",      ownerName_        },
        {"address",        address_         },
        {"place",          place_           }
    };
}

void BaseWorkIncoming::parseMap(const QVariantMap &map)
{
    if(map.contains("document"))
    {
        parseMap(map["document"].toMap());
    }
    else
    {
        cashier_        = map["cashier"].toString().trimmed();
        cashierInn_     = map["cashierInn"].toString().trimmed();
        additionalParam_= STR4FS_(map["additionalParam"].toString().trimmed());
        additionalData_ = QByteArray::fromHex(map["additionalData"].toString().toLatin1());
        ownerName_       = STR4FS_(map["ownerName"].toString());
        address_        = STR4FS_(map["address"].toString());
        place_          = STR4FS_(map["place"].toString());
    }
}

void BaseWorkIncoming::clean()
{
    cashier_.clear();
    cashierInn_.clear();
    additionalParam_.clear();
    additionalData_.clear();
    ownerName_.clear();
    address_.clear();
    place_.clear();
}


BaseWorkIncoming &BaseWorkIncoming::operator =(const BaseWorkIncoming &other) noexcept
{
    cashier_ = other.cashier_;
    cashierInn_ = other.cashierInn_;
    additionalParam_ = other.additionalParam_;
    additionalData_ = other.additionalData_;
    ownerName_ = other.ownerName_;
    address_ = other.address_;
    place_ = other.place_;
    return *this;

}
BaseWorkIncoming &BaseWorkIncoming::operator =(BaseWorkIncoming &&other) noexcept
{
    cashier_.swap(other.cashier_);
    cashierInn_.swap(other.cashierInn_);
    additionalParam_.swap(other.additionalParam_);
    additionalData_.swap(other.additionalData_);
    ownerName_.swap(other.ownerName_);
    address_.swap(other.address_);
    place_.swap(other.place_);
    return *this;
}

bool BaseWorkIncoming::operator ==(const BaseWorkIncoming &other) const noexcept
{
    return cashier_ == other.cashier_ &&
           cashierInn_ == other.cashierInn_ &&
           additionalParam_ == other.additionalParam_ &&
           additionalData_ == other.additionalData_ &&
           ownerName_ == other.ownerName_ &&
           address_ == other.address_ &&
           place_ == other.place_;

}

bool BaseWorkIncoming::operator !=(const BaseWorkIncoming &other) const noexcept
{
    return !(*this == other);
}
