#include "cycleclosedoc.h"
#include "formatutils.h"

#include <QJsonDocument>

CycleCloseIncoming::CycleCloseIncoming() noexcept
    : BaseWorkIncoming()
{

}

CycleCloseIncoming::CycleCloseIncoming(const QVariantMap &map) noexcept
    : BaseWorkIncoming(map)
{

}

CycleCloseIncoming::CycleCloseIncoming(const CycleCloseIncoming &other) noexcept
    : BaseWorkIncoming(other)
{

}

CycleCloseIncoming::CycleCloseIncoming(CycleCloseIncoming &&other) noexcept
    : BaseWorkIncoming(other)
{

}

CycleCloseIncoming::~CycleCloseIncoming()
{

}

Tlv::Stlv CycleCloseIncoming::tlv(const RegData &rd, const fdf::IncorrectLabelsFlags &ilf,
                                  const fdf::IncorrectNotificationsFlags &inf) const
{
    Q_UNUSED(rd)
    Tlv::Stlv res;
    Tlv buf;
    if(!cashier_.isEmpty())
    {
        buf.setString(fdf::Tag::Cashier, cashier_);
        res << buf;
        Inn inn(cashierInn_);
        if(inn.isValid())
        {
            buf.setInn(fdf::Tag::CashierInn, inn);
            res << buf;
        }
    }
    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::CycleCloseAdditionalParam, additionalParam_);
        res << buf;
    }
    if(!additionalData_.isEmpty())
    {
        buf.setTag(fdf::Tag::CycleCloseAdditionalData);
        buf.setValue(additionalData_);
        res << buf;
    }
    if(!address_.isEmpty() && address_ != rd.address())
    {
        buf.setString(fdf::Tag::Address, address_);
        res << buf;
    }
    if(!place_.isEmpty() && place_ != rd.place())
    {
        buf.setString(fdf::Tag::Place, place_);
        res << buf;
    }
    if(static_cast<qint32>(ilf))
    {
        buf.setByte(fdf::Tag::IncorrectLabelsFlag, static_cast<quint8>(ilf));
        res << buf;
    }
    if(static_cast<qint32>(inf))
    {
        buf.setByte(fdf::Tag::IncorrectNotificationsFlag, static_cast<quint8>(inf));
        res << buf;
    }
    return res;
}

CycleCloseIncoming &CycleCloseIncoming::operator =(const CycleCloseIncoming &other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;
}
CycleCloseIncoming &CycleCloseIncoming::operator =(CycleCloseIncoming &&other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;
}
bool CycleCloseIncoming::operator ==(const CycleCloseIncoming &other) const noexcept
{
    return BaseWorkIncoming::operator ==(other);
}

bool CycleCloseIncoming::operator !=(const CycleCloseIncoming &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

CycleCloseDoc::CycleCloseDoc(QObject *parent)
    : BaseWorkDoc(fdf::DocType::CycleClose, parent)
    , warnings_()
    , ofdWarnings_()
    , incorrectLabelsFlags_()
    , incorrectNotificationsFlags_()
    , documentsCount_()
    , receiptsCount_()
    , notSendedDocs_()
    , notSendedNotifications_()
    , keysResource_()
    , firstNotSendedDocDt_()
    , cashier_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , fsCounters_()
    , cycleCounters_()
{

}
CycleCloseDoc::~CycleCloseDoc()
{

}

const fs::Warnings &CycleCloseDoc::warnings() const
{
    return warnings_;
}

void CycleCloseDoc::setWarnings(const fs::Warnings &newWarnings)
{
    warnings_ = newWarnings;
}

const fdf::OfdWarnings &CycleCloseDoc::ofdWarnings() const
{
    return ofdWarnings_;
}

void CycleCloseDoc::setOfdWarnings(const fdf::OfdWarnings &newOfdWarnings)
{
    ofdWarnings_ = newOfdWarnings;
}

const fdf::IncorrectLabelsFlags &CycleCloseDoc::incorrectLabelsFlags() const
{
    return incorrectLabelsFlags_;
}

void CycleCloseDoc::setIncorrectLabelsFlags(const fdf::IncorrectLabelsFlags &newIncorrectLabelsFlags)
{
    incorrectLabelsFlags_ = newIncorrectLabelsFlags;
}

const fdf::IncorrectNotificationsFlags &CycleCloseDoc::incorrectNotificationsFlags() const
{
    return incorrectNotificationsFlags_;
}

void CycleCloseDoc::setIncorrectNotificationsFlags(const fdf::IncorrectNotificationsFlags &newIncorrectNotificationsFlags)
{
    incorrectNotificationsFlags_ = newIncorrectNotificationsFlags;
}

quint32 CycleCloseDoc::documentsCount() const
{
    return documentsCount_;
}

void CycleCloseDoc::setDocumentsCount(quint32 newDoucumentsCount)
{
    documentsCount_ = newDoucumentsCount;
}

quint32 CycleCloseDoc::receiptsCount() const
{
    return receiptsCount_;
}

void CycleCloseDoc::setReceiptsCount(quint32 newReceiptsCount)
{
    receiptsCount_ = newReceiptsCount;
}

quint32 CycleCloseDoc::notSendedDocs() const
{
    return notSendedDocs_;
}

void CycleCloseDoc::setNotSendedDocs(quint32 newNotSendedDocs)
{
    notSendedDocs_ = newNotSendedDocs;
}

quint32 CycleCloseDoc::notSendedNotifications() const
{
    return notSendedNotifications_;
}

void CycleCloseDoc::setNotSendedNotifications(quint32 newNotSendedNotifications)
{
    notSendedNotifications_ = newNotSendedNotifications;
}

quint16 CycleCloseDoc::keysResource() const
{
    return keysResource_;
}

void CycleCloseDoc::setKeysResource(quint16 newKeysResource)
{
    keysResource_ = newKeysResource;
}

const QDate &CycleCloseDoc::firstNotSendedDocDt() const
{
    return firstNotSendedDocDt_;
}

void CycleCloseDoc::setFirstNotSendedDocDt(const QDate &newFirstNotSendedDocDt)
{
    firstNotSendedDocDt_ = newFirstNotSendedDocDt;
}

const QString &CycleCloseDoc::cashier() const
{
    return cashier_;
}

void CycleCloseDoc::setCashier(const QString &newCashier)
{
    cashier_ = newCashier;
}

const QString &CycleCloseDoc::cashierInn() const
{
    return cashierInn_;
}

void CycleCloseDoc::setCashierInn(const QString &newCashierInn)
{
    cashierInn_ = newCashierInn;
}

const QString &CycleCloseDoc::additionalParam() const
{
    return additionalParam_;
}

void CycleCloseDoc::setAdditionalParam(const QString &newAdditionalParam)
{
    additionalParam_ = newAdditionalParam;
}

const QByteArray &CycleCloseDoc::additionalData() const
{
    return additionalData_;
}

void CycleCloseDoc::setAdditionalData(const QByteArray &newAdditionalData)
{
    additionalData_ = newAdditionalData;
}

const TaggedFsCounters &CycleCloseDoc::fsCounters() const
{
    return fsCounters_;
}

void CycleCloseDoc::setFsCounters(const TaggedFsCounters &newFsCounters)
{
    fsCounters_ = newFsCounters;
}

const TaggedFsCounters &CycleCloseDoc::cycleCounters() const
{
    return cycleCounters_;
}

void CycleCloseDoc::setCycleCounters(const TaggedFsCounters &newCycleCounters)
{
    cycleCounters_ = newCycleCounters;
}

void CycleCloseDoc::setIncoming(const CycleCloseIncoming &inc, bool replace)
{

    if(replace || cashier_        .isEmpty()) cashier_            = inc.cashier();
    if(replace || cashierInn_     .isEmpty()) cashierInn_         = inc.cashierInn();
    if(replace || additionalParam_.isEmpty()) additionalParam_    =  inc.additionalParam();
    if(replace || additionalData_ .isEmpty()) additionalData_     = inc.additionalData();
    if(replace || ownerName_      .isEmpty()) ownerName_           = inc.ownerName();
    if(replace || address_        .isEmpty()) address_            = inc.address();
    if(replace || place_          .isEmpty()) place_              = inc.place();
}

void CycleCloseDoc::clean()
{
    warnings_ = fs::Warnings();
    ofdWarnings_ = fdf::OfdWarnings();
    incorrectLabelsFlags_ = fdf::IncorrectLabelsFlags();
    incorrectNotificationsFlags_ = fdf::IncorrectNotificationsFlags();
    documentsCount_ = 0;
    receiptsCount_ = 0;
    notSendedDocs_ = 0;
    notSendedNotifications_ = 0;
    keysResource_ = 0;
    firstNotSendedDocDt_ = QDate();
    cashier_.clear();
    cashierInn_.clear();
    additionalParam_.clear();
    additionalData_.clear();
    fsCounters_.clean();
    cycleCounters_.clean();
}

QVariantMap CycleCloseDoc::toMap() const
{
    QVariantMap res {
        {"warnings", static_cast<qint32>(warnings_)},
        {"ofdWarnings", static_cast<qint32>(ofdWarnings_)},
        {"incorrectLabelsFlags", static_cast<qint32>(incorrectLabelsFlags_)},
        {"incorrectNotificationsFlags", static_cast<qint32>(incorrectNotificationsFlags_)},
        {"documentsCount", documentsCount_},
        {"receiptsCount", receiptsCount_},
        {"notSendedDocs", notSendedDocs_},
        {"notSendedNotifications", notSendedNotifications_},
        {"keysResource",        keysResource_},
        {"firstNotSendedDocDt", firstNotSendedDocDt_.toString(Qt::ISODate)},
        {"cashier",        cashier_         },
        {"cashierInn",     cashierInn_      },
        {"additionalParam",additionalParam_ },
        {"additionalData", QString::fromLatin1(additionalData_.toHex())  },
        {"fsCounters", fsCounters_.toMap()},
        {"cycleCounters", cycleCounters_.toMap()},
    };
    res = UNIT_MAPS(BaseWorkDoc::toMap(), res);
    return res;
}

QVariantMap CycleCloseDoc::toExternalMap() const
{
    QVariantMap res {
        {"fsIsExpired", warnings_.testFlag(fs::Warning::NeedFsChange)},
        {"fsIsNearExpired", warnings_.testFlag(fs::Warning::NeedFsChange30)},
        {"fsIsOwerFlow", warnings_.testFlag(fs::Warning::FsMemoryOverflow)},
        {"ofdTimedOut", warnings_.testFlag(fs::Warning::OfdTimedOut)},
        {"ofdLogicalError", ofdWarnings_.testFlag(fdf::OfdWarning::LogicalError)},
        {"ofdCheckAccount", ofdWarnings_.testFlag(fdf::OfdWarning::CheckOfdAccount)},
        {"ofdUpdateFFD", ofdWarnings_.testFlag(fdf::OfdWarning::UpdateFFD)},
        {"ofdCashboxInCheckPlan", ofdWarnings_.testFlag(fdf::OfdWarning::CashboxCheckPlan)},
        {"ofdCallToOfd", ofdWarnings_.testFlag(fdf::OfdWarning::Call2Ofd)},
        {"ofdIsClosed", ofdWarnings_.testFlag(fdf::OfdWarning::OfdIsClosed)},
        {"incorrectLabelResponse", incorrectLabelsFlags_.testFlag(fdf::IncorrectLabelsFlag::IncorrectLabelResponse)},
        {"incorrectLabelTicket", incorrectLabelsFlags_.testFlag(fdf::IncorrectLabelsFlag::IncorrectLabelTicket)},
        {"incorrectRequest", incorrectNotificationsFlags_.testFlag(fdf::IncorrectNotificationsFlag::IcorrectRequest)},
        {"incorrectNotification", incorrectNotificationsFlags_.testFlag(fdf::IncorrectNotificationsFlag::IncorrectNotification)},
        {"documentsCount", documentsCount_},
        {"receiptsCount", receiptsCount_},
        {"notSendedDocs", notSendedDocs_},
        {"notSendedNotifications", notSendedNotifications_},
        {"keysResource",        keysResource_},
        {"firstNotSendedDocDt", firstNotSendedDocDt_.isValid() ?
                            QVariant( firstNotSendedDocDt_.toString(Qt::ISODate)) : QVariant()},
        {"cashier",        cashier_        .isEmpty() ? QVariant() : QVariant(cashier_        ) },
        {"cashierInn",     cashierInn_     .isEmpty() ? QVariant() : QVariant(cashierInn_     ) },
        {"additionalParam",additionalParam_.isEmpty() ? QVariant() : QVariant(additionalParam_) },
        {"additionalData", additionalData_ .isEmpty() ? QVariant() :
                                                        QVariant(QString::fromLatin1(additionalData_.toHex()) ) },
        {"fsCounters", fsCounters_.toExternalMap()},
        {"cycleCounters", cycleCounters_.toExternalMap()},

    };
    res = UNIT_MAPS(BaseWorkDoc::toExternalMap(), res);
    return res;
}

void CycleCloseDoc::parseMap(const QVariantMap &map)
{
    clean();

    BaseWorkDoc::parseMap(map);
    warnings_ = fs::Warnings(map["warnings"].toInt());
    ofdWarnings_ = fdf::OfdWarnings(map["ofdWarnings"].toUInt());
    incorrectLabelsFlags_ = fdf::IncorrectLabelsFlags(map["incorrectLabelsFlags"].toUInt());
    incorrectNotificationsFlags_ = fdf::IncorrectNotificationsFlags(map["incorrectNotificationsFlags"].toUInt());
    documentsCount_  = map["documentsCount"].toUInt();
    receiptsCount_  = map["receiptsCount"].toUInt();
    notSendedDocs_  = map["notSendedDocs"].toUInt();
    notSendedNotifications_  = map["notSendedNotifications"].toUInt();
    keysResource_  = map["keysResource"].toUInt();
    firstNotSendedDocDt_ = QDate::fromString(map["firstNotSendedDocDt"].toString(), Qt::ISODate);
    cashier_        = map["cashier"].toString();
    cashierInn_     = map["cashierInn"].toString();
    additionalParam_= map["additionalParam"].toString();
    additionalData_ = QByteArray::fromHex(map["additionalData"].toString().toLatin1());
    fsCounters_.parseMap(map["fsCounters"].toMap());
    cycleCounters_.parseMap(map["cycleCounters"].toMap());
}

void CycleCloseDoc::setRegData(const RegData &rd, bool replace)
{
    BaseWorkDoc::setRegData(rd, replace);
}

void CycleCloseDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag())
    {
    case fdf::Tag::Cashier: cashier_ = tlv.toString().trimmed();break;
    case fdf::Tag::FsResourceOff30: warnings_.setFlag(fs::Warning::NeedFsChange30, tlv.toBool());break;
    case fdf::Tag::FsChangeNecessity:
        warnings_.setFlag(fs::Warning::NeedFsChange, tlv.toBool());
        break;
    case fdf::Tag::FsOverMemory: warnings_.setFlag(fs::Warning::FsMemoryOverflow, tlv.toBool());break;
    case fdf::Tag::OfdTimedOut: warnings_.setFlag(fs::Warning::OfdTimedOut, tlv.toBool());break;
    case fdf::Tag::CashierInn: cashierInn_ = tlv.toString().trimmed();break;
    case fdf::Tag::OfdMessage: ofdWarnings_ = fdf::OfdWarnings(tlv.toByte());break;
    case fdf::Tag::IncorrectLabelsFlag: incorrectLabelsFlags_ = fdf::IncorrectLabelsFlags(tlv.toByte());break;
    case fdf::Tag::IncorrectNotificationsFlag:
        incorrectNotificationsFlags_= fdf::IncorrectNotificationsFlags(tlv.toByte());break;
    case fdf::Tag::DocsForCycle: documentsCount_ = tlv.toUInt32();break;
    case fdf::Tag::ReceiptsForCycle: receiptsCount_ = tlv.toUInt32();break;
    case fdf::Tag::NotSendedDocsCount: notSendedDocs_ = tlv.toUInt32();break;
    case fdf::Tag::NotSendedNotificationsCount: notSendedNotifications_ = tlv.toUInt32();break;
    case fdf::Tag::KeysResource: keysResource_ = tlv.toUInt16();break;
    case fdf::Tag::FirstNotSendedDt: firstNotSendedDocDt_ = tlv.toDate();break;
    case fdf::Tag::CycleCloseAdditionalParam: additionalParam_ = tlv.toString().trimmed();break;
    case fdf::Tag::CycleCloseAdditionalData: additionalData_ = tlv.value();break;
    case fdf::Tag::FsTotalsCounters: fsCounters_.parseTlv(tlv);break;
    case fdf::Tag::CycleTotalCounters: cycleCounters_.parseTlv(tlv);break;
    default: BaseWorkDoc::parseFromTlv(tlv);
    }
}
