#include "cycleopendoc.h"
#include "formatutils.h"


CycleOpenDoc::CycleOpenDoc(QObject *parent)
    : BaseWorkDoc(fdf::DocType::CycleOpen, parent)
    , warnings_()
    , cashboxFFD_(fs::FFD::FFD1_2)
    , ofdWarnings_()
    , cashier_()
    , cashboxVersion_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , updateKeysText_()
{

}

CycleOpenDoc::~CycleOpenDoc()
{

}

const fs::Warnings &CycleOpenDoc::warnings() const
{
    return warnings_;
}

void CycleOpenDoc::setWarnings(const fs::Warnings &newWarnings)
{
    warnings_ = newWarnings;
}

fs::FFD CycleOpenDoc::cashboxFFD() const
{
    return cashboxFFD_;
}

void CycleOpenDoc::setCashboxFFD(fs::FFD newCashboxFFD)
{
    cashboxFFD_ = newCashboxFFD;
}

const fdf::OfdWarnings &CycleOpenDoc::ofdWarnings() const
{
    return ofdWarnings_;
}

void CycleOpenDoc::setOfdWarnings(const fdf::OfdWarnings &newOfdWarnings)
{
    ofdWarnings_ = newOfdWarnings;
}

const QString &CycleOpenDoc::cashier() const
{
    return cashier_;
}

void CycleOpenDoc::setCashier(const QString &newCashier)
{
    cashier_ = newCashier;
}

const QString &CycleOpenDoc::cashboxVersion() const
{
    return cashboxVersion_;
}

void CycleOpenDoc::setCashboxVersion(const QString &newCashboxVersion)
{
    cashboxVersion_ = newCashboxVersion;
}

const QString &CycleOpenDoc::cashierInn() const
{
    return cashierInn_;
}

void CycleOpenDoc::setCashierInn(const QString &newCashierInn)
{
    cashierInn_ = newCashierInn;
}

const QString &CycleOpenDoc::additionalParam() const
{
    return additionalParam_;
}

void CycleOpenDoc::setAdditionalParam(const QString &newAdditionalParam)
{
    additionalParam_ = newAdditionalParam;
}

const QByteArray &CycleOpenDoc::additionalData() const
{
    return additionalData_;
}

void CycleOpenDoc::setAdditionalData(const QByteArray &newAdditionalData)
{
    additionalData_ = newAdditionalData;
}

QString CycleOpenDoc::updateKeysText() const
{
    return updateKeysText_;
}

void CycleOpenDoc::setUpdateKeysText(const QString &newUpdateKeysText)
{
    updateKeysText_ = newUpdateKeysText;
}

void CycleOpenDoc::setIncoming(const CycleOpenIncoming &inc, bool replace)
{
    if(replace || cashier_        .isEmpty()) cashier_            = inc.cashier();
    if(replace || cashierInn_     .isEmpty()) cashierInn_         = inc.cashierInn();
    if(replace || additionalParam_.isEmpty()) additionalParam_    =  inc.additionalParam();
    if(replace || additionalData_ .isEmpty()) additionalData_     = inc.additionalData();
    if(replace || ownerName_      .isEmpty()) ownerName_           = inc.ownerName();
    if(replace || address_        .isEmpty()) address_            = inc.address();
    if(replace || place_          .isEmpty()) place_              = inc.place();
}

void CycleOpenDoc::clean()
{
    BaseWorkDoc::clean();
    cycle_ = 0u;
    warnings_ = fs::Warnings();
    cashboxFFD_ = fs::FFD::FFD1_2;
    ofdWarnings_ = fdf::OfdWarnings();
    cashier_.clear();
    cashboxVersion_.clear();
    cashierInn_.clear();
    additionalParam_.clear();
    additionalData_.clear();
    updateKeysText_.clear();
}

QVariantMap CycleOpenDoc::toMap() const
{
    QVariantMap res {
        {"warnings", static_cast<qint32>(warnings_)},
        {"cashboxFfd", static_cast<qint32>(cashboxFFD_)},
        {"ofdWarnings", static_cast<qint32>(ofdWarnings_)},
        {"cashier",        cashier_         },
        {"cashboxVersion", cashboxVersion_  },
        {"cashierInn",     cashierInn_      },
        {"additionalParam",additionalParam_ },
        {"additionalData", QString::fromLatin1(additionalData_.toHex())  },
        {"updateKeysText", updateKeysText_}
    };
    res = UNIT_MAPS(BaseWorkDoc::toMap(), res);
    return res;
}


QVariantMap CycleOpenDoc::toExternalMap() const
{
    QVariantMap res {
        {"fsIsExpired", warnings_.testFlag(fs::Warning::NeedFsChange)},
        {"fsIsNearExpired", warnings_.testFlag(fs::Warning::NeedFsChange30)},
        {"fsIsOwerFlow", warnings_.testFlag(fs::Warning::FsMemoryOverflow)},
        {"ofdTimedOut", warnings_.testFlag(fs::Warning::OfdTimedOut)},
        {"cashboxFfd", fs::ffd2String(cashboxFFD_)},
        {"ofdLogicalError", ofdWarnings_.testFlag(fdf::OfdWarning::LogicalError)},
        {"ofdCheckAccount", ofdWarnings_.testFlag(fdf::OfdWarning::CheckOfdAccount)},
        {"ofdUpdateFFD", ofdWarnings_.testFlag(fdf::OfdWarning::UpdateFFD)},
        {"ofdCashboxInCheckPlan", ofdWarnings_.testFlag(fdf::OfdWarning::CashboxCheckPlan)},
        {"ofdCallToOfd", ofdWarnings_.testFlag(fdf::OfdWarning::Call2Ofd)},
        {"ofdIsClosed", ofdWarnings_.testFlag(fdf::OfdWarning::OfdIsClosed)},
        {"cashier",        cashier_        .isEmpty() ? QVariant() : QVariant(cashier_        ) },
        {"cashboxVersion", cashboxVersion_ .isEmpty() ? QVariant() : QVariant(cashboxVersion_ ) },
        {"cashierInn",     cashierInn_     .isEmpty() ? QVariant() : QVariant(cashierInn_     ) },
        {"additionalParam",additionalParam_.isEmpty() ? QVariant() : QVariant(additionalParam_) },
        {"additionalData", additionalData_ .isEmpty() ? QVariant() :
                                                        QVariant(QString::fromLatin1(additionalData_.toHex()) ) },
        {"updateKeysText", updateKeysText_}
    };
    res = UNIT_MAPS(BaseWorkDoc::toExternalMap(), res);
    return res;
}

void CycleOpenDoc::parseMap(const QVariantMap &map)
{
    clean();

    BaseWorkDoc::parseMap(map);
    warnings_ = fs::Warnings(map["warnings"].toInt());
    cashboxFFD_  = static_cast<fs::FFD>(map["cashboxFfd"].toUInt());
    fs::checkFfd(cashboxFFD_);
    ofdWarnings_ = fdf::OfdWarnings(map["ofdWarnings"].toUInt());
    fs::checkFfd(ffd_);
    cashier_        = map["cashier"].toString();
    cashboxVersion_ = map["cashboxVersion"].toString();
    cashierInn_     = map["cashierInn"].toString();
    additionalParam_= map["additionalParam"].toString();
    additionalData_ = QByteArray::fromHex(map["additionalData"].toString().toLatin1());
    updateKeysText_ = map["updateKeysText"].toString();
}

void CycleOpenDoc::setRegData(const RegData &rd, bool replace)
{
    BaseWorkDoc::setRegData(rd, replace);
    if(cashboxVersion_.isEmpty() || replace) cashboxVersion_ = rd.cashboxVersion();
    cashboxFFD_ = rd.cashboxFfd();
}

void CycleOpenDoc::parseFromTlv(const Tlv &tlv)
{
//    lmWarning() << cycle_;
    switch (tlv.tag())
    {
    case fdf::Tag::Cashier: cashier_ = tlv.toString().trimmed();break;
    case fdf::Tag::FsResourceOff30: warnings_.setFlag(fs::Warning::NeedFsChange30, tlv.toBool());break;
    case fdf::Tag::FsChangeNecessity:
        warnings_.setFlag(fs::Warning::NeedFsChange, tlv.toBool());
        break;
    case fdf::Tag::FsOverMemory: warnings_.setFlag(fs::Warning::FsMemoryOverflow, tlv.toBool());break;
    case fdf::Tag::OfdTimedOut: warnings_.setFlag(fs::Warning::OfdTimedOut, tlv.toBool());break;
    case fdf::Tag::CashboxVersion: cashboxVersion_ = tlv.toString().trimmed();break;
    case fdf::Tag::CashboxFFD: cashboxFFD_ = static_cast<fs::FFD>(tlv.toByte());break;
    case fdf::Tag::CashierInn: cashierInn_ = tlv.toString().trimmed();break;
    case fdf::Tag::OfdMessage: ofdWarnings_ = fdf::OfdWarnings(tlv.toByte());break;
    case fdf::Tag::CycleOpenAdditionalParam: additionalParam_ = tlv.toString().trimmed();break;
    case fdf::Tag::CycleOpenAdditionalData: additionalData_ = tlv.value();break;
    default: BaseWorkDoc::parseFromTlv(tlv);
    }
    //    lmWarning() << cycle_  << logtab << static_cast<qint32>(tlv.tag()) << tlv.toUInt32() << tlv.toUInt16();
}

//--------------------------------------------------------------------------------------------------
CycleOpenIncoming::CycleOpenIncoming() noexcept
    : BaseWorkIncoming()
{

}

CycleOpenIncoming::CycleOpenIncoming(const QVariantMap &map)noexcept
    : BaseWorkIncoming(map)

{
//    parseMap(map);
}

CycleOpenIncoming::CycleOpenIncoming(const CycleOpenIncoming &other)noexcept
    : BaseWorkIncoming(other)

{

}

CycleOpenIncoming::CycleOpenIncoming(CycleOpenIncoming &&other) noexcept
    : BaseWorkIncoming(other)

{
}

CycleOpenIncoming::~CycleOpenIncoming()
{

}


Tlv::Stlv CycleOpenIncoming::tlv(const RegData &rd) const
{
    Tlv::Stlv res;
    Tlv buf;
    if(!cashier_.isEmpty())
    {
        buf.setString(fdf::Tag::Cashier, cashier_);
        res << buf;
        Inn inn(cashierInn_);
        if(inn.isValid())
        {
            buf.setInn(fdf::Tag::CashierInn, inn);
            res << buf;
        }
    }
    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::CycleOpenAdditionalParam, additionalParam_);
        res << buf;
    }
    if(!additionalData_.isEmpty())
    {
        buf.setTag(fdf::Tag::CycleOpenAdditionalData);
        buf.setValue(additionalData_);
        res << buf;
    }
    if(!address_.isEmpty() && address_ != rd.address())
    {
        buf.setString(fdf::Tag::Address, address_);
        res << buf;
    }
    if(!place_.isEmpty() && place_ != rd.place())
    {
        buf.setString(fdf::Tag::Place, place_);
        res << buf;
    }
    buf.setString(fdf::Tag::CashboxVersion, rd.cashboxVersion());
    res << buf;
    buf.setByte(fdf::Tag::CashboxFFD, static_cast<quint8>(rd.cashboxFfd()));
    res << buf;
    return res;
}

CycleOpenIncoming &CycleOpenIncoming::operator =(const CycleOpenIncoming &other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;

}
CycleOpenIncoming &CycleOpenIncoming::operator =(CycleOpenIncoming &&other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;
}

bool CycleOpenIncoming::operator ==(const CycleOpenIncoming &other) const noexcept
{
    return BaseWorkIncoming::operator ==(other);

}

bool CycleOpenIncoming::operator !=(const CycleOpenIncoming &other) const noexcept
{
    return !(*this == other);
}

