#include "fsclosedoc.h"
#include "formatutils.h"

#include <QJsonDocument>

FsCloseIncoming::FsCloseIncoming() noexcept
    : BaseWorkIncoming()
{

}

FsCloseIncoming::FsCloseIncoming(const QVariantMap &map) noexcept
    : BaseWorkIncoming(map)
{

}

FsCloseIncoming::FsCloseIncoming(const FsCloseIncoming &other) noexcept
    : BaseWorkIncoming(other)
{

}

FsCloseIncoming::FsCloseIncoming(FsCloseIncoming &&other) noexcept
    : BaseWorkIncoming(other)
{

}

FsCloseIncoming::~FsCloseIncoming()
{

}

Tlv::Stlv FsCloseIncoming::tlv(const RegData &rd) const
{
    Q_UNUSED(rd)
    Tlv::Stlv res;
    Tlv buf;
    if(!additionalParam_.isEmpty())
    {
        buf.setString(fdf::Tag::FsCloseAdditionalParam, additionalParam_);
        res << buf;
    }
    if(!additionalData_.isEmpty())
    {
        buf.setTag(fdf::Tag::FsCloseAdditionalData);
        buf.setValue(additionalData_);
        res << buf;
    }
    if(!cashier_.isEmpty())
    {
        buf.setString(fdf::Tag::Cashier, cashier_);
        res << buf;
        Inn inn(cashierInn_);
        if(inn.isValid())
        {
            buf.setInn(fdf::Tag::CashierInn, inn);
            res << buf;
        }
    }
    QString oN = ownerName_.trimmed();
    if(oN.isEmpty()) oN = rd.ownerName();
    QString a = address_;
    if(a.isEmpty()) a = rd.address();
    QString p = place_;
    if(p.isEmpty()) p = rd.place();

    buf.setString(fdf::Tag::OwnerName, oN);
    res << buf;
    buf.setString(fdf::Tag::Address, a);
    res << buf;
    buf.setString(fdf::Tag::Place, p);
    res << buf;

    return res;
}

FsCloseIncoming &FsCloseIncoming::operator =(const FsCloseIncoming &other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;
}

FsCloseIncoming &FsCloseIncoming::operator =(FsCloseIncoming &&other) noexcept
{
    BaseWorkIncoming::operator =(other);
    return *this;
}

bool FsCloseIncoming::operator ==(const FsCloseIncoming &other) const noexcept
{
    return BaseWorkIncoming::operator ==(other);
}

bool FsCloseIncoming::operator !=(const FsCloseIncoming &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsCloseDoc::FsCloseDoc(QObject *parent)
    : BaseWorkDoc(fdf::DocType::FsClose, parent)
    , cashier_()
    , cashierInn_()
    , additionalParam_()
    , additionalData_()
    , fsCounters_()
{

}

FsCloseDoc::~FsCloseDoc()
{

}

const QString &FsCloseDoc::cashier() const
{
    return cashier_;
}

void FsCloseDoc::setCashier(const QString &newCashier)
{
    cashier_ = newCashier;
}

const QString &FsCloseDoc::cashierInn() const
{
    return cashierInn_;
}

void FsCloseDoc::setCashierInn(const QString &newCashierInn)
{
    cashierInn_ = newCashierInn;
}

const QString &FsCloseDoc::additionalParam() const
{
    return additionalParam_;
}

void FsCloseDoc::setAdditionalParam(const QString &newAdditionalParam)
{
    additionalParam_ = newAdditionalParam;
}

const QByteArray &FsCloseDoc::additionalData() const
{
    return additionalData_;
}

void FsCloseDoc::setAdditionalData(const QByteArray &newAdditionalData)
{
    additionalData_ = newAdditionalData;
}

const TaggedFsCounters &FsCloseDoc::fsCounters() const
{
    return fsCounters_;
}

void FsCloseDoc::setFsCounters(const TaggedFsCounters &newFsCounters)
{
    fsCounters_ = newFsCounters;
}

void FsCloseDoc::setIncoming(const FsCloseIncoming &inc, bool replace)
{
    if(replace || cashier_        .isEmpty()) cashier_            = inc.cashier();
    if(replace || cashierInn_     .isEmpty()) cashierInn_         = inc.cashierInn();
    if(replace || additionalParam_.isEmpty()) additionalParam_    =  inc.additionalParam();
    if(replace || additionalData_ .isEmpty()) additionalData_     = inc.additionalData();
    if(replace || ownerName_      .isEmpty()) ownerName_           = inc.ownerName();
    if(replace || address_        .isEmpty()) address_            = inc.address();
    if(replace || place_          .isEmpty()) place_              = inc.place();
}

void FsCloseDoc::clean()
{
    cashier_.clear();
    cashierInn_.clear();
    additionalParam_.clear();
    additionalData_.clear();
    fsCounters_.clean();
}

QVariantMap FsCloseDoc::toMap() const
{
    QVariantMap res {
        {"cashier",        cashier_         },
        {"cashierInn",     cashierInn_      },
        {"additionalParam",additionalParam_ },
        {"additionalData", QString::fromLatin1(additionalData_.toHex())  },
        {"fsCounters", fsCounters_.toMap()},
    };
    res = UNIT_MAPS(BaseWorkDoc::toMap(), res);
    return res;
}

QVariantMap FsCloseDoc::toExternalMap() const
{
    QVariantMap res {
        {"cashier",        cashier_        .isEmpty() ? QVariant() : QVariant(cashier_        ) },
        {"cashierInn",     cashierInn_     .isEmpty() ? QVariant() : QVariant(cashierInn_     ) },
        {"additionalParam",additionalParam_.isEmpty() ? QVariant() : QVariant(additionalParam_) },
        {"additionalData", additionalData_ .isEmpty() ? QVariant() :
                                                        QVariant(QString::fromLatin1(additionalData_.toHex()) ) },
        {"fsCounters", fsCounters_.toExternalMap()},

    };
    res = UNIT_MAPS(BaseWorkDoc::toExternalMap(), res);
    return res;
}

void FsCloseDoc::parseMap(const QVariantMap &map)
{
    clean();

    BaseWorkDoc::parseMap(map);
    cashier_        = map["cashier"].toString();
    cashierInn_     = map["cashierInn"].toString();
    additionalParam_= map["additionalParam"].toString();
    additionalData_ = QByteArray::fromHex(map["additionalData"].toString().toLatin1());
    fsCounters_.parseMap(map["fsCounters"].toMap());
}

void FsCloseDoc::setRegData(const RegData &rd, bool replace)
{
    BaseWorkDoc::setRegData(rd, replace);
}


void FsCloseDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag())
    {
    case fdf::Tag::Cashier: cashier_ = tlv.toString().trimmed();break;
    case fdf::Tag::CashierInn: cashierInn_ = tlv.toString().trimmed();break;
    case fdf::Tag::FsCloseAdditionalParam: additionalParam_ = tlv.toString().trimmed();break;
    case fdf::Tag::FsCloseAdditionalData: additionalData_ = tlv.value();break;
    case fdf::Tag::FsTotalsCounters: fsCounters_.parseTlv(tlv);break;
    default: BaseWorkDoc::parseFromTlv(tlv);
    }
}

