#include "regdoc.h"
#include "formatutils.h"

#include <QJsonDocument>

BaseRegDoc::~BaseRegDoc()
{

}

const CoreRegData &BaseRegDoc::data() const
{
    return data_;
}

void BaseRegDoc::setData(const CoreRegData &newData)
{
    data_ = newData;
}

RegData BaseRegDoc::regData() const
{
    RegData res(data_);
    res.setDt(dt_);
    res.setDocNumber(fd_);
    res.setFsNumber(fsNumber_);
    res.setFiscalCode(fiscalCode_);
    return res;
}

void BaseRegDoc::clean()
{
    BaseDoc::clean();
    data_.clean();
}

QVariantMap BaseRegDoc::toMap() const
{
    QVariantMap res = BaseDoc::toMap();
    res = UNIT_MAPS(res, data_.toMap());
    return res;
}

QVariantMap BaseRegDoc::toExternalMap() const
{
    QVariantMap res = BaseDoc::toExternalMap();
    res = UNIT_MAPS(res, data_.toExternalMap());
    return res;
}

void BaseRegDoc::parseMap(const QVariantMap &map)
{
    BaseDoc::parseMap(map);
    data_.parseMap(map);
}

BaseRegDoc::BaseRegDoc(fdf::DocType docType, QObject *parent)
    : BaseDoc(docType, parent)
    , data_()
{

}

void BaseRegDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag()) {
    case fdf::Tag::AutoMode: data_.setAutoMode(tlv.toBool());break;
    case fdf::Tag::OfflineMode: data_.setOfflineMode(tlv.toBool());break;
    case fdf::Tag::Address: data_.setAddress(tlv.toString());break;
    case fdf::Tag::CashboxSerial: data_.setSerial(tlv.toString());break;
    case fdf::Tag::OfdInn: data_.setOfdInn(tlv.toString().trimmed());break;
    case fdf::Tag::OwnerInn: data_.setOwnerInn(tlv.toString().trimmed());break;
    case fdf::Tag::Cashier: data_.setCashier(tlv.toString().trimmed());break;
    case fdf::Tag::BoxId: data_.setBoxId(tlv.toString().trimmed());break;
    case fdf::Tag::CashboxRegNum: data_.setRegNumber(tlv.toString().trimmed());break;
    case fdf::Tag::OfdName: data_.setOfdName(tlv.toString().trimmed());break;
    case fdf::Tag::OwnerName: data_.setOwnerName(tlv.toString().trimmed());break;
    case fdf::Tag::EncryptionMode: data_.setEncriptionMode(tlv.toBool());break;
    case fdf::Tag::FnsSite: data_.setFtsSite(tlv.toString().trimmed());break;
    case fdf::Tag::TaxSystems: data_.setTaxes(fdf::Taxes(tlv.toByte()));break;
    case fdf::Tag::OwnerEmail: data_.setEmail(tlv.toString().trimmed());break;
    case fdf::Tag::Place: data_.setPlace(tlv.toString().trimmed());break;
    case fdf::Tag::CashboxVersion: data_.setCashboxVersion(tlv.toString().trimmed());break;
    case fdf::Tag::CashboxFFD: data_.setCashboxFfd(static_cast<fs::FFD>(tlv.toByte()));break;
    case fdf::Tag::FsFFD: data_.setFsFfd(static_cast<fs::FFD>(tlv.toByte()));break;
    case fdf::Tag::CashierInn: data_.setCashierInn(tlv.toString().trimmed());break;
    case fdf::Tag::FFD: data_.setFfd(static_cast<fs::FFD>(tlv.toByte()));break;
    case fdf::Tag::KeysResource: data_.setKeysResource(tlv.toUInt16());break;
    case fdf::Tag::RegAdditionalParam: data_.setAdditionalParam(tlv.toString().trimmed());break;
    case fdf::Tag::RegAdditionalData: data_.setAdditionalParamData(tlv.value());break;
    case fdf::Tag::CashboxUsageFlags: data_.setExtFlags(fdf::RegFlags1290(tlv.toUInt32()));break;
    case fdf::Tag::CashboxDataChangeReason: data_.setReasons(fdf::Reasons(tlv.toUInt32()));break;
    default: BaseDoc::parseFromTlv(tlv);
        break;
    }
}

//--------------------------------------------------------------------------------------------------

RegDoc::RegDoc(QObject *parent)
    :BaseRegDoc(fdf::DocType::Registration, parent)
{

}

RegDoc::~RegDoc()
{

}
//--------------------------------------------------------------------------------------------------

ReRegDoc::ReRegDoc(QObject *parent)
    : BaseRegDoc(fdf::DocType::ReRegistration, parent)
    , fsCounters_()
{

}

ReRegDoc::~ReRegDoc()
{

}

const TaggedFsCounters &ReRegDoc::fsCounters() const
{
    return fsCounters_;
}

void ReRegDoc::setFsCounters(const TaggedFsCounters &newFsCounters)
{
    fsCounters_ = newFsCounters;
}

void ReRegDoc::clean()
{
    BaseRegDoc::clean();
    fsCounters_.clean();
}

QVariantMap ReRegDoc::toMap() const
{
    QVariantMap res = BaseRegDoc::toMap();
    res.insert("fsCounters", fsCounters_.toMap());
    return res;
}

QVariantMap ReRegDoc::toExternalMap() const
{
    QVariantMap res = BaseRegDoc::toExternalMap();
    res.insert("fsCounters", fsCounters_.toExternalMap());
    return res;
}

void ReRegDoc::parseMap(const QVariantMap &map)
{
    BaseRegDoc::parseMap(map);
    fsCounters_.parseMap(map["fsCounters"].toMap());
}


void ReRegDoc::parseFromTlv(const Tlv &tlv)
{
    switch (tlv.tag())
    {
    case fdf::Tag::FsTotalsCounters: fsCounters_.parseTlv(tlv);break;

    default: BaseRegDoc::parseFromTlv(tlv);
    }
}
