#include "regdatavalidator.h"
#include "fdfconstants.h"

RegDataValidator::RegDataValidator(QObject *parent)
    : BaseDataValidator(parent)
    , fs_()
    , oldReg_()
    , model_()

{

}

RegDataValidator::~RegDataValidator()
{

}

const FsFullStatus &RegDataValidator::fs() const
{
    return fs_;
}

void RegDataValidator::setFs(const FsFullStatus &newFs)
{
    fs_ = newFs;
}

const RegData &RegDataValidator::oldReg() const
{
    return oldReg_;
}

void RegDataValidator::setOldReg(const RegData &newOldReg)
{
    oldReg_ = newOldReg;
}

const ModelData &RegDataValidator::model() const
{
    return model_;
}

void RegDataValidator::setModel(const ModelData &newModel)
{
    model_ = newModel;
}

bool RegDataValidator::validate(const IncomingReg &data)
{
    if(!model_.isValid())
    {
        lastError_ = CoreApiConst::ErrorCode::NoCashboxSerial;
        lastErrorMsg_ = model_.serial().isEmpty() ? tr("Нет заводского номера"): tr("Некорректные данные модели ККТ");
        return false;
    }
    if(!checkDataFormats(data)) return false;
    if(!checkCashboxState(data)) return false;
    return !fs_.isFiscalized() || checkReregistration(data);
}

void RegDataValidator::test(const IncomingReg &data)
{
    if(validate(data)) emit success();
    else
    {
        emit error(lastError_, lastErrorMsg_);
        emit errorText(lastErrorMsg_);
    }
}

bool RegDataValidator::checkDataFormats(const IncomingReg &data)
{
    if(!checkRegNumber(data.regNumber(), model_.serial(), data.ownerInn()))
    {
        return false;
    }
    QString errSuffix;
    if(!checkString(data.address(), 256, false, false, errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Адрес расчетов (%1)").arg(static_cast<qint32>(fdf::Tag::Address)) + errSuffix;
        return false;
    }

    if(!data.getOfflineMode() && !checkInn(data.ofdInn(), data.getOfflineMode()))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Некорректный ИНН ОФД (%1)").arg(static_cast<qint32>(fdf::Tag::OfdInn));
        return false;
    }
    if(!checkString(data.cashier(), 64, false, data.getAutoMode(), errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Кассир (%1)").arg(static_cast<qint32>(fdf::Tag::Cashier)) + errSuffix;
        return false;
    }
    if(!checkString(data.boxId(), 20, false, true, errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Номер автомата (%1)").arg(static_cast<qint32>(fdf::Tag::BoxId)) + errSuffix;
        return false;
    }
    if(!data.getOfflineMode() && !checkString(data.ofdName(), 256, false, data.getOfflineMode(), errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Наименование ОФД (%1)").arg(static_cast<qint32>(fdf::Tag::OfdName)) + errSuffix;
        return false;
    }
    if(!checkString(data.ownerName(), 256, false, false, errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Наименование пользователя (%1)").arg(static_cast<qint32>(fdf::Tag::OwnerName)) + errSuffix;
        return false;
    }
    if(!data.getOfflineMode() && !checkString(data.ftsSite(), 256, false, data.getOfflineMode(), errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Адрес сайта ФНС (%1)").arg(static_cast<qint32>(fdf::Tag::FnsSite)) + errSuffix;
        return false;
    }
    if(!data.getOfflineMode() && data.email() != "none" && !checkEmail(data.email(), 64, data.getOfflineMode(), errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Адрес электронной почты отправителя чека (%1)").arg(static_cast<qint32>(fdf::Tag::OwnerEmail)) + errSuffix;
        return false;
    }
    if(!checkString(data.place(), 256, false, false, errSuffix))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Место расчетов (%1)").arg(static_cast<qint32>(fdf::Tag::Place)) + errSuffix;
        return false;
    }
    if(!checkInn(data.cashierInn(), true))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Некорректный ИНН кассира");
        return false;
    }
    if(data.additionalParam().size() > 32 || data.additionalParamData().size() > 32)
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Некорректный дополнительный параметр ОР");
        return false;
    }
    if(data.getOfflineMode() && data.getEncriptionMode())
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("В Автономном режиме ККТ не может использовать шифрование");
        return false;
    }
    if(!fdf::taxesIsValid(data.taxes()))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Заданы некорректные налоговые системы");
        return false;
    }

    if(data.flags().testFlag(fdf::RegFlag::OfflineMode) && data.flags().testFlag(fdf::RegFlag::EncriptionMode))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Нельзя шифровать в автономном режиме");
        return false;
    }
    if(data.flags().testFlag(fdf::RegFlag::OfflineMode) && data.extFlags().flags().testFlag(fdf::RegFlag1290::InternetOnly))
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Нельзя ККТ в автономном режиме использовать для расчетов всети интернет");
        return false;
    }

    return true;
}

bool RegDataValidator::checkCashboxState(const IncomingReg &data)
{
    if(data.reasons().toUInt()== 0u && fs_.isFiscalized())
    {
        lastError_ = CoreApiConst::ErrorCode::InvalidParamValue;
        lastErrorMsg_ = tr("Укажите причины перерегистрации");
        return false;
    }
    return true;
}

bool RegDataValidator::checkReregistration(const IncomingReg &data)
{
    if(fs_.isFiscalized())
    {
        if(oldReg_.regNumber() != data.regNumber())
        {
            lastError_ = CoreApiConst::ErrorCode::InvalidRegNumber;
            lastErrorMsg_ = tr("Некорректный РНМ. ККТ зарегистрирована с ") + oldReg_.regNumber();
            return false;
        }
        if(oldReg_.serial() != model_.serial())
        {
            lastError_ = CoreApiConst::ErrorCode::AlienFs;
            lastErrorMsg_ = tr("Чужой фн. ФН был зарегистрирован на ККТ ") + oldReg_.serial();
            return false;

        }
    }
    return true;
}
