#include "fsapi.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/i2c-dev.h>
#include <linux/i2c.h>


#include <QCryptographicHash>

 // #include <QSerialPort>
#include <QDataStream>
#include <QFileInfo>
#include <QElapsedTimer>


#define I2C_BUS "/dev/i2c-0"
#define I2C_ADDR 0x02
#define TIMEOUT_MS 50
#define DELAY_AFTER_WRITE 10000

bool i2c_detect_device(int fd, uint8_t addr) {
 struct i2c_rdwr_ioctl_data ioctl_data;
 struct i2c_msg msg;
 unsigned char dummy;

 msg.addr = addr;
 msg.flags = I2C_M_RD;    // Флаг чтения
 msg.len = 0;             // Чтение 0 байт - только проверка ACK
 msg.buf = &dummy;        // Должен быть валидный указатель

 ioctl_data.msgs = &msg;
 ioctl_data.nmsgs = 1;

 if (ioctl(fd, I2C_RDWR, &ioctl_data) < 0) {
     return false; // Устройство не ответило
 }
 return true; // Устройство ответило
}

int i2c_write(int fd, uint8_t addr, const char *data, size_t len) {
    // int res = 0;
    // if ((res = ::write(fd, data, len)) != len)
    // {
    //     return -1;
    // }
    // return 0;
    struct i2c_rdwr_ioctl_data ioctl_data;
    struct i2c_msg msg;

    msg.addr = addr;
    msg.flags = 0;          // Флаг записи
    msg.len = len;
    msg.buf = (uint8_t *)data;

    ioctl_data.msgs = &msg;
    ioctl_data.nmsgs = 1;

    if (ioctl(fd, I2C_RDWR, &ioctl_data) < 0) {
        perror("I2C write failed");
        return -1;
    }
    return 0;
}

int i2c_read(int fd, uint8_t addr, uint8_t *buffer, size_t len) {
    struct i2c_rdwr_ioctl_data ioctl_data;
    struct i2c_msg msg;

    msg.addr = addr;
    msg.flags = I2C_M_RD;   // Флаг чтения
    msg.len = len;
    msg.buf = buffer;

    ioctl_data.msgs = &msg;
    ioctl_data.nmsgs = 1;

    if (ioctl(fd, I2C_RDWR, &ioctl_data) < 0) {
        // perror("I2C read failed");
        return -1;
    }
    return 0;
}


FsApi::FsApi(QObject *parent)
    : QObject(parent)
{

}

FsApi::~FsApi()
{
    close();
}

bool FsApi::open()
{
    close();
    if ((i2cFd_ = ::open(I2C_BUS, O_RDWR)) < 0)
    {
        // qCritical() << "OPEN ERROR";
        return false;
    }

    if (::ioctl(i2cFd_, I2C_SLAVE_FORCE, I2C_ADDR) < 0) {
        ::close(i2cFd_);
        // qCritical() << "Failed to set I2C address";
        i2cFd_ = -1;
        return false;
    }
    return true;
}

bool FsApi::isOpen() const
{
    return i2cFd_ > 0;
}

void FsApi::close()
{
    ::close(i2cFd_);
    i2cFd_ = -1;
}

bool FsApi::exec(const FsReq &req, FsRep &rep, FsRwTimeout timeout)
{
    rep = FsRep();
    if(req.isValid() && (isOpen() || open()))
    {
        QByteArray reqBuf = req.rawData();


        if (i2c_write(i2cFd_, I2C_ADDR, reqBuf.constData(), reqBuf.size()))
        {
            close();
            return false;
        }

        usleep(DELAY_AFTER_WRITE);
        QElapsedTimer t;
        t.start();
        while(i2c_read(i2cFd_, I2C_ADDR, readBuf_, 3) && t.elapsed() < timeout.first)
        {
            usleep(DELAY_AFTER_WRITE);
        }


        quint16 len = 0;

        if(readBuf_[0] != 0x04u)
        {
            // qCritical() << "Invalid start byte";
            close();
            return false;
        }
        len = static_cast<quint16>(readBuf_[2]) << 8 | static_cast<quint16>(readBuf_[1]);
        if(len + 5 > 4096)
        {
            // qCritical() << "Invalid length: " << len;
            close();
            return false;
        }

        while(i2c_read(i2cFd_, I2C_ADDR, readBuf_, len + 5) && t.elapsed() < timeout.first)
        {
            usleep(DELAY_AFTER_WRITE);
        }
        close();
        return rep.setRawData(QByteArray(reinterpret_cast<const char*>(readBuf_), len + 5)) == fs::RepParsingRes::Ok;
    }
    close();
    return false;

}
