#include "fsarchivereportbase.h"
#include "fdfconstants.h"
#include "fsfullstatus.h"
#include "cashboxbuildconfig.h"
#include "formatutils.h"
#include "registrationflags1290.h"
#include "regreasons.h"


#include <QDataStream>

class FsArchiveRegReport: public FsArchiveReportBase
{
    friend class FsArchiveReportBase;
public:
    FsArchiveRegReport(const FsArchiveRegReport &other) noexcept
        : FsArchiveReportBase(other)
        , inn_(other.inn_)
        , regNumber_(other.regNumber_)
        , tax_(other.tax_)
        , mode_(other.mode_)
        , extFlags_(other.extFlags_)
        , ofdInn_(other.ofdInn_){
    }
    FsArchiveRegReport(FsArchiveRegReport &&other) noexcept
        : FsArchiveReportBase(other)
        , inn_()
        , regNumber_()
        , tax_(other.tax_)
        , mode_(other.mode_)
        , extFlags_(other.extFlags_)
        , ofdInn_() {
        inn_.swap(other.inn_);
        regNumber_.swap(other.regNumber_);
        ofdInn_.swap(other.ofdInn_);
    }
    virtual ~FsArchiveRegReport(){}

    FsArchiveRegReport &operator =(const FsArchiveRegReport &other)	{
        FsArchiveReportBase::operator =((other));
        inn_ = other.inn_;
        regNumber_ = other.regNumber_;
        tax_ = other.tax_;
        mode_ = other.mode_;
        extFlags_ = other.extFlags_;
        ofdInn_ = other.ofdInn_;
        return *this;
    }

    FsArchiveRegReport &operator =(FsArchiveRegReport &&other)	{
        FsArchiveReportBase::operator =((other));
        inn_.swap(other.inn_);
        regNumber_.swap(other.regNumber_);
        tax_ = other.tax_;
        mode_ = other.mode_;
        extFlags_ = other.extFlags_;
        ofdInn_.swap(other.ofdInn_);
        return *this;
    }
    bool operator ==(const FsArchiveRegReport &other) const {
        return FsArchiveReportBase::operator ==((other)) &&
                inn_ == other.inn_ &&
                regNumber_ == other.regNumber_ &&
                tax_ == other.tax_ &&
                mode_ == other.mode_ &&
                extFlags_ == other.extFlags_ &&
                ofdInn_ == other.ofdInn_;
    }
    bool operator !=(const FsArchiveRegReport &other) const {

        return !(*this == other);
    }

    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveReportBase::toMap();
        res["ownerInn"] = inn_;
        res["regNumber"] = regNumber_;
        res["taxes"] = static_cast<qint32>(tax_);
        res["mode"] = static_cast<qint32>(mode_);
        if(ffd_ == fs::FFD::FFD1_2)
        {
            res["extFlags"] = extFlags_;
            res["ofdInn"] = ofdInn_;
        }
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveReportBase::toExternalMap();
        res["ownerInn"] = inn_;
        res["regNumber"] = regNumber_;
        res["taxes"] = static_cast<qint32>(tax_);

        fs::RegFlags flags (mode_);

        res["autoMode"] = flags.testFlag(fs::RegFlag::AutoMode);
        res["offlineMode"] = flags.testFlag(fs::RegFlag::OfflineMode);
        res["useEncription"] = flags.testFlag(fs::RegFlag::Enctiption);


        if(ffd_ == fs::FFD::FFD1_2)
        {
            fs::ExtRegFlags extFlags(extFlags_);
            RegistrationFlags1290 ext;
            ext.setFlags(flags, extFlags);
            UNIT_MAPS(res, ext.toExternalMap());
            if(!flags.testFlag(fs::RegFlag::OfflineMode))res["ofdInn"] = ofdInn_;
            else res["ofdInn"] = QVariant();
        }
        return res;
    }

protected:
    QString inn_;
    QString regNumber_;
    quint8 tax_;
    quint8 mode_;
    quint8 extFlags_;
    QString ofdInn_;
    FsArchiveRegReport()
        : FsArchiveReportBase()
        , inn_()
        , regNumber_()
        , tax_(0)
        , mode_(0)
        , extFlags_(0)
        , ofdInn_() {
    }
    virtual bool parse(const QByteArray &data)  override {
        if(data.size() == 60) return parseFfd12(data);
        if(data.size() != 47) return false;
        ffd_ = fs::FFD::FFD1_05;
        QByteArray dat;
        if(!alternative_) dat = data;
        else
        {
            dat = data.mid(0, 5) + data.mid(data.size() - 8, 8) + data.mid(5, 34);
        }

        dt_ = QDateTime();
        dt_.setDate(QDate(dat[0] + 2000, dat[1], dat[2]));
        dt_.setTime(QTime(dat[3], dat[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(dat);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_;
        inn_ = QString::fromLatin1(dat.mid(13, 12)).trimmed();
        regNumber_ = QString::fromLatin1(dat.mid(25, 20)).trimmed();
        tax_ = dat[45];
        mode_ = dat[46];
        return !inn_.isEmpty() && !regNumber_.isEmpty();
    }

    virtual bool parseFfd12(const QByteArray &data){
        if(data.size() != 60) return false;
        ffd_ = fs::FFD::FFD1_2;
        QByteArray dat;
        if(!alternative_) dat = data;
        else
        {
            dat = data.mid(0, 5) + data.mid(data.size() - 8, 8) + data.mid(5, 47);
        }

        dt_ = QDateTime();
        dt_.setDate(QDate(dat[0] + 2000, dat[1], dat[2]));
        dt_.setTime(QTime(dat[3], dat[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(dat);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_;
        inn_ = QString::fromLatin1(dat.mid(13, 12)).trimmed();
        regNumber_ = QString::fromLatin1(dat.mid(25, 20)).trimmed();
        tax_ = dat[45];
        mode_ = dat[46];
        extFlags_ = dat[47];
        ofdInn_ = QString::fromLatin1(dat.mid(48)).trimmed();
        if(ofdInn_.startsWith("00000")) ofdInn_.clear();
        return !inn_.isEmpty() && !regNumber_.isEmpty();
    }

};


//--------------------------------------------------------------------------------------------------
class FsArchiveReRegReport: public FsArchiveRegReport
{
    friend class FsArchiveReportBase;
public:
    FsArchiveReRegReport(const FsArchiveReRegReport &other)
        : FsArchiveRegReport(other)
        , reason_(other.reason_)
        , extReason_(other.extReason_) {
    }
    FsArchiveReRegReport(FsArchiveReRegReport &&other)
        : FsArchiveRegReport(other)
        , reason_(other.reason_)
        , extReason_(other.extReason_) {
    }
    virtual ~FsArchiveReRegReport(){}

    FsArchiveReRegReport &operator =(const FsArchiveReRegReport &other)	{
        FsArchiveRegReport::operator =(other);
        reason_ = other.reason_;
        extReason_ = other.extReason_;
        return *this;
    }

    FsArchiveReRegReport &operator =(FsArchiveReRegReport &&other)	{
        FsArchiveRegReport::operator =(other);
        reason_ = other.reason_;
        extReason_ = other.extReason_;
        return *this;
    }
    bool operator ==(const FsArchiveReRegReport &other) const {
        return FsArchiveRegReport::operator ==(other) &&
                reason_ == other.reason_ &&
                extReason_ == other.extReason_;
    }
    bool operator !=(const FsArchiveReRegReport &other) const {

        return !(*this == other);
    }
    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveRegReport::toMap();
        if(ffd_ == fs::FFD::FFD1_2) res["reason"] = QString::number(extReason_);
        else res["reason"] = QString::number(reason_);
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveRegReport::toMap();
        if(ffd_ == fs::FFD::FFD1_2)
        {
            RegReasons r;
            r.setFlags(fdf::Reasons(extReason_));
            UNIT_MAPS(res, r.toExternalMap());
        }
        else res["reason"] = QString::number(reason_);
        return res;
    }

private:
    quint8 reason_;
    quint32 extReason_;
    FsArchiveReRegReport()
        : FsArchiveRegReport()
        , reason_(0)
        , extReason_(0){
    }
    virtual bool parse(const QByteArray &data)  override {
        if(data.size() == 60) return FsArchiveRegReport::parseFfd12(data);
        if(data.size() == 64) return parseFfd12(data);
        if(data.size() != 48) return false;
        if(alternative_)
        {
            if(!FsArchiveRegReport::parse(data.mid(0, 39) + data.mid(40))) return false;
            reason_ = data[39];
        }
        else
        {
            if(!FsArchiveRegReport::parse(data.mid(0, 47))) return false;
            reason_ = data[47];
        }
        return true;
    }
    bool parseFfd12(const QByteArray &data) override{
        if(data.size() == 60) return FsArchiveRegReport::parseFfd12(data);
        if(data.size() != 64) return false;
        QByteArray dat = data;
        if(alternative_)
        {
            dat = data.mid(0, 52) + data.mid(56) + data.mid(52, 4);
        }
        if(!FsArchiveRegReport::parse(dat.mid(0, 60))) return false;

        QDataStream ds(dat.mid(60));
        ds.setByteOrder(QDataStream::LittleEndian);
        ds >> extReason_;
        return true;

    }
};
//--------------------------------------------------------------------------------
class FsArchiveCloseReport: public FsArchiveRegReport
{
    friend class FsArchiveReportBase;
public:
    FsArchiveCloseReport(const FsArchiveCloseReport &other)
        : FsArchiveRegReport(other){
    }
    FsArchiveCloseReport(FsArchiveCloseReport &&other)
        : FsArchiveRegReport(other){
    }
    virtual ~FsArchiveCloseReport(){}

    FsArchiveCloseReport &operator =(const FsArchiveCloseReport &other)	{
        FsArchiveRegReport::operator =(other);
        return *this;
    }

    FsArchiveCloseReport &operator =(FsArchiveCloseReport &&other)	{
        FsArchiveRegReport::operator =(other);
        return *this;
    }
    bool operator ==(const FsArchiveCloseReport &other) const {
        return FsArchiveRegReport::operator ==(other);
    }
    bool operator !=(const FsArchiveCloseReport &other) const {
        return !(*this == other);
    }

    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveReportBase::toMap();
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveReportBase::toExternalMap();
        return res;
    }

private:
    FsArchiveCloseReport() :
        FsArchiveRegReport(){
    }
    virtual bool parse(const QByteArray &data)  override {
        if(data.size() != 45) return false;
        dt_ = QDateTime();
        dt_.setDate(QDate(data[0] + 2000, data[1], data[2]));
        dt_.setTime(QTime(data[3], data[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(data);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_;
        inn_ = QString::fromLatin1(data.mid(13, 12)).trimmed();
        regNumber_ = QString::fromLatin1(data.mid(25, 20)).trimmed();
        return !inn_.isEmpty() && !regNumber_.isEmpty();
    }    
};
//--------------------------------------------------------------------------------
class FsArchiveReportCycleOpen :public FsArchiveReportBase
{
    friend class FsArchiveReportBase;
public:
    FsArchiveReportCycleOpen(const FsArchiveReportCycleOpen &other)
        : FsArchiveReportBase(other)
        , cycleNumber_(other.cycleNumber_){
    }
    FsArchiveReportCycleOpen(FsArchiveReportCycleOpen &&other)
        : FsArchiveReportBase(other)
        , cycleNumber_(other.cycleNumber_){
    }
    virtual ~FsArchiveReportCycleOpen(){}
    virtual quint16 cycleNumber() const override {return cycleNumber_;}
    FsArchiveReportCycleOpen &operator =(const FsArchiveReportCycleOpen &other)	{
        FsArchiveReportBase::operator =((other));
        cycleNumber_ = other.cycleNumber_;
        return *this;
    }

    FsArchiveReportCycleOpen &operator =(FsArchiveReportCycleOpen &&other)	{
        FsArchiveReportBase::operator =((other));
        cycleNumber_ = other.cycleNumber_;
        return *this;
    }
    bool operator ==(const FsArchiveReportCycleOpen &other) const {
        return FsArchiveReportBase::operator ==((other)) &&
                cycleNumber_ == other.cycleNumber_;
    }
    bool operator !=(const FsArchiveReportCycleOpen &other) const {

        return !(*this == other);
    }


    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveReportBase::toMap();
        res["cycle"] = cycleNumber_;
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveReportBase::toExternalMap();
        res["cycle"] = cycleNumber_;
        return res;
    }

protected:
    quint16 cycleNumber_;
    FsArchiveReportCycleOpen() :
        FsArchiveReportBase(),
        cycleNumber_(0){
    }
    virtual bool parse(const QByteArray &data)  override {
        if(data.size() != 15) return false;
        dt_ = QDateTime();
        dt_.setDate(QDate(data[0] + 2000, data[1], data[2]));
        dt_.setTime(QTime(data[3], data[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(data);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_ >>cycleNumber_;
        return true;
    }


};
//--------------------------------------------------------------------------------
class FsArchiveReportCycleClose :public FsArchiveReportCycleOpen
{
    friend class FsArchiveReportBase;
public:
    FsArchiveReportCycleClose(const FsArchiveReportCycleClose &other)
        : FsArchiveReportCycleOpen(other){
    }
    FsArchiveReportCycleClose(FsArchiveReportCycleClose &&other)
        : FsArchiveReportCycleOpen(other){
    }
    virtual ~FsArchiveReportCycleClose(){}

    FsArchiveReportCycleClose &operator =(const FsArchiveReportCycleClose &other)	{
        FsArchiveReportCycleOpen::operator =(other);
        return *this;
    }

    FsArchiveReportCycleClose &operator =(FsArchiveReportCycleClose &&other)	{
        FsArchiveReportCycleOpen::operator =(other);
        return *this;
    }
    bool operator ==(const FsArchiveReportCycleClose &other) const {
        return FsArchiveReportCycleOpen::operator ==(other);
    }
    bool operator !=(const FsArchiveReportCycleClose &other) const {

        return !(*this == other);
    }



protected:
    FsArchiveReportCycleClose() :
        FsArchiveReportCycleOpen(){
    }


};
//--------------------------------------------------------------------------------
class FsArchiveReportCheck :public FsArchiveReportBase
{
    friend class FsArchiveReportBase;
public:
    FsArchiveReportCheck(const FsArchiveReportCheck &other)
        : FsArchiveReportBase(other)
        , isCorrection_(other.isCorrection_)
        , isBso_(other.isBso_)
        , recType_(other.type_)
        , sum_(other.sum_){
    }
    FsArchiveReportCheck(FsArchiveReportCheck &&other)
        : FsArchiveReportBase(other)
        , isCorrection_(other.isCorrection_)
        , isBso_(other.isBso_)
        , recType_(other.type_)
        , sum_(other.sum_){
    }
    virtual ~FsArchiveReportCheck(){}

    FsArchiveReportCheck &operator =(const FsArchiveReportCheck &other)	{
        FsArchiveReportBase::operator =((other));
        isCorrection_ = other.isCorrection_;
        isBso_ = other.isBso_;
        recType_ = other.type_;
        sum_ = other.sum_;
        return *this;
    }

    FsArchiveReportCheck &operator =(FsArchiveReportCheck &&other)	{
        FsArchiveReportBase::operator =((other));
        isCorrection_ = other.isCorrection_;
        isBso_ = other.isBso_;
        recType_ = other.type_;
        sum_ = other.sum_;
        return *this;
    }
    bool operator ==(const FsArchiveReportCheck &other) const {
        return FsArchiveReportBase::operator ==((other)) &&
                type_ == other.type_ &&
                sum_ == other.sum_;
    }
    bool operator !=(const FsArchiveReportCheck &other) const {

        return !(*this == other);
    }

    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveReportBase::toMap();
        res["receiptType"] = recType_;
        FixNumber sum(DEFAULT_AMOUNT_MULT, sum_);
        res["amount"] = sum.toMap();
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveReportBase::toExternalMap();
        res["receiptType"] = recType_;
        FixNumber sum(DEFAULT_AMOUNT_MULT, sum_);
        res["amount"] = sum.toString();
        return res;
    }
    FixNumber amount() const override{
        return FixNumber(DEFAULT_AMOUNT_MULT, sum_);
    }
protected:
    bool isCorrection_;
    bool isBso_;
    quint8 recType_;
    qint64 sum_;
    FsArchiveReportCheck(bool isCorrection, bool isBso) :
        FsArchiveReportBase(),
        isCorrection_(isCorrection),
        isBso_(isBso),
        recType_(0),
        sum_(0)	{
    }
    virtual bool parse(const QByteArray &data)  override {
        if(data.size() != 19) return false;
        dt_ = QDateTime();
        dt_.setDate(QDate(data[0] + 2000, data[1], data[2]));
        dt_.setTime(QTime(data[3], data[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(data + QByteArray(3, '\x00'));
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_ >>recType_ >> sum_;
        return true;
    }


};

//--------------------------------------------------------------------------------
class FsArchiveReportCycleState :public FsArchiveReportBase
{
    friend class FsArchiveReportBase;
public:
    FsArchiveReportCycleState(const FsArchiveReportCycleState &other)
        : FsArchiveReportBase(other),
        offLineDocs_(other.offLineDocs_),
        offlineDocDt_(other.offlineDocDt_) {
    }
    FsArchiveReportCycleState(FsArchiveReportCycleState &&other) :
        FsArchiveReportBase(other),
        offLineDocs_(other.offLineDocs_),
        offlineDocDt_(other.offlineDocDt_){
    }
    virtual ~FsArchiveReportCycleState(){}

    FsArchiveReportCycleState &operator =(const FsArchiveReportCycleState &other)	{
        FsArchiveReportBase::operator =((other));
        offLineDocs_ = other.offLineDocs_;
        offlineDocDt_ = other.offlineDocDt_;
        return *this;
    }

    FsArchiveReportCycleState &operator =(FsArchiveReportCycleState &&other)	{
        FsArchiveReportBase::operator =((other));
        offLineDocs_ = other.offLineDocs_;
        offlineDocDt_ = other.offlineDocDt_;
        return *this;
    }
    bool operator ==(const FsArchiveReportCycleState &other) const {
        return FsArchiveReportBase::operator ==((other)) &&
                offLineDocs_ == other.offLineDocs_ &&
                offlineDocDt_ == other.offlineDocDt_;
    }
    bool operator !=(const FsArchiveReportCycleState &other) const {

        return !(*this == other);
    }


    QVariantMap toMap() const override {
        QVariantMap res = FsArchiveReportBase::toMap();
        res["notSendedDocs"] = offLineDocs_;
        res["firstNotSendedDocDt"] = DT2STR_(offlineDocDt_);
        return res;
    }
    QVariantMap toExternalMap() const override {
        QVariantMap res = FsArchiveReportBase::toExternalMap();
        res["notSendedDocs"] = offLineDocs_;
        res["firstNotSendedDocDt"] = offlineDocDt_.isValid() ? QVariant(DT2STR_(offlineDocDt_) ) : QVariant();
        return res;
    }
protected:
    quint32 offLineDocs_;
    QDateTime offlineDocDt_;

    FsArchiveReportCycleState() :
        FsArchiveReportBase(),
        offLineDocs_(0),
        offlineDocDt_(){
    }
    bool parse(const QByteArray &data)  override {
        if(data.size() != 20) return false;
        dt_ = QDateTime();
        dt_.setDate(QDate(data[0] + 2000, data[1], data[2]));
        dt_.setTime(QTime(data[3], data[4]));
        if(!dt_.isValid()) return false;
        QDataStream ds(data);
        ds.setByteOrder(QDataStream::LittleEndian);
        ds.skipRawData(5);
        ds >> docNumber_ >> fiscalCode_ >>offLineDocs_;
        offlineDocDt_ = QDateTime();
        offlineDocDt_.setDate(QDate(data[17] + 2000, data[18], data[19]));
        offlineDocDt_.setTime(QTime(0, 0));
        return true;
    }

};
//--------------------------------------------------------------------------------------------------


FsArchiveReportBase::FsArchiveReportBase(const FsArchiveReportBase &other) noexcept
    : data_(other.data_)
    , type_(other.type_)
    , hasTicket_(other.hasTicket_)
    , dt_(other.dt_)
    , docNumber_(other.docNumber_)
    , fiscalCode_(other.fiscalCode_)
    , header_(other.header_)
    , alternative_(other.alternative_)
    , ffd_(other.ffd_)
    , ticket_(other.ticket_)
    , ticketDt_(other.ticketDt_)
{

}

FsArchiveReportBase::FsArchiveReportBase(FsArchiveReportBase &&other) noexcept
    : data_()
    , type_(other.type_)
    , hasTicket_(other.hasTicket_)
    , dt_()
    , docNumber_(other.docNumber_)
    , fiscalCode_(other.fiscalCode_)
    , header_()
    , alternative_(other.alternative_)
    , ffd_(other.ffd_)
    , ticket_()
    , ticketDt_()
{
    data_.swap(other.data_);
    dt_.swap(other.dt_);
    header_.swap(other.header_);
    ticket_.swap(other.ticket_);
    ticketDt_.swap(other.ticketDt_);
}

FsArchiveReportBase::~FsArchiveReportBase()
{

}

QByteArray FsArchiveReportBase::data() const
{
    return data_;
}

void FsArchiveReportBase::setData(const QByteArray &data)
{
    data_ = data;
}

quint8 FsArchiveReportBase::type() const
{
    return type_;
}

void FsArchiveReportBase::setType(const quint8 &type)
{
    type_ = type;
}

bool FsArchiveReportBase::hasTicket() const
{
    return hasTicket_;
}

void FsArchiveReportBase::setHasTicket(bool hasTicket)
{
    hasTicket_ = hasTicket;
}

QDateTime FsArchiveReportBase::dt() const
{
    return dt_;
}

void FsArchiveReportBase::setDt(const QDateTime &dt)
{
    dt_ = dt;
}

quint32 FsArchiveReportBase::docNumber() const
{
    return docNumber_;
}

void FsArchiveReportBase::setDocNumber(const quint32 &docNumber)
{
    docNumber_ = docNumber;
}
quint32 FsArchiveReportBase::fiscalCode() const
{
    return fiscalCode_;
}

void FsArchiveReportBase::setFiscalCode(quint32 newFiscalCode)
{
    fiscalCode_ = newFiscalCode;
}


QStringList FsArchiveReportBase::header() const
{
    return header_;
}

void FsArchiveReportBase::setHeader(const QStringList &header)
{
    header_ = header;
}

fs::FFD FsArchiveReportBase::ffd() const
{
    return ffd_;
}

void FsArchiveReportBase::setFfd(fs::FFD newFfd)
{
    ffd_ = newFfd;
}


quint16 FsArchiveReportBase::cycleNumber() const
{
    return 0u;
}

FixNumber FsArchiveReportBase::amount() const
{
    return FixNumber(DEFAULT_AMOUNT_MULT);
}

FsArchiveReportBase &FsArchiveReportBase::operator =(const FsArchiveReportBase &other) noexcept
{
    data_ = other.data_;
    type_ = other.type_;
    hasTicket_ = other.hasTicket_;
    dt_ = other.dt_;
    docNumber_ = other.docNumber_;
    fiscalCode_ = other.fiscalCode_;
    header_ = other.header_;
    alternative_ = other.alternative_;
    ffd_ = other.ffd_;
    ticket_ = other.ticket_;
    ticketDt_ = other.ticketDt_;
    return *this;
}

FsArchiveReportBase &FsArchiveReportBase::operator =(FsArchiveReportBase &&other) noexcept
{
    data_.swap(other.data_);
    type_ = other.type_;
    hasTicket_ = other.hasTicket_;
    dt_.swap(other.dt_);
    docNumber_ = other.docNumber_;
    fiscalCode_ = other.fiscalCode_;
    header_.swap(other.header_);
    alternative_ = other.alternative_;
    ffd_ = other.ffd_;
    ticket_.swap(other.ticket_);
    ticketDt_.swap(other.ticketDt_);
    return *this;

}

bool FsArchiveReportBase::operator ==(const FsArchiveReportBase &other) const noexcept
{
    return data_ == other.data_ &&
            type_ == other.type_ &&
            hasTicket_ == other.hasTicket_ &&
            dt_ == other.dt_ &&
            docNumber_ == other.docNumber_ &&
            fiscalCode_ == other.fiscalCode_ &&
            header_ == other.header_ &&
            alternative_ == other.alternative_ &&
            ffd_ == other.ffd_ &&
            ticket_ == other.ticket_ &&
            ticketDt_ == other.ticketDt_;
}

bool FsArchiveReportBase::operator !=(const FsArchiveReportBase &other) const noexcept
{
    return !(*this == other);
}

QVariantMap FsArchiveReportBase::toMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"hasTicket", hasTicket_ ? 1 : 0},
        {"docNumber", docNumber_},
        {"fiscalCode", QString::number(fiscalCode_)},
        {"dt", DT2STR_(dt_)},
        {"ffd", static_cast<qint32>(ffd_)},
        {"ticket", QString::fromLatin1(ticket_.toHex())},
        {"ticketDt", DT2STR_(ticketDt_)}
    };
}

QVariantMap FsArchiveReportBase::toExternalMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"hasTicket", hasTicket_},
        {"docNumber", docNumber_},
        {"fiscalCode", fiscalCode_ > 0 ? QVariant(QString::number(fiscalCode_)): QVariant()},
        {"dt", DT2STR_(dt_)},
        {"ffd", fs::ffd2String(ffd_)},
        {"ticket", ticket_.isEmpty() ? QVariant() : QVariant(QString::fromLatin1(ticket_.toHex()))},
        {"ticketDt", ticketDt_.isValid() ? QVariant(DT2STR_(ticketDt_)) : QVariant()}
    };

}

QSharedPointer<FsArchiveReportBase> FsArchiveReportBase::createDocument(quint8 type, bool hasTicket,
                                                                        const QByteArray &data,
                                                                        const QStringList &header)
{
    QSharedPointer<FsArchiveReportBase> res;
    if(data.size() < 3 || type != quint8(data[0])) return res;
    hasTicket = data[1];
    FsArchiveReportBase *doc = nullptr;
    switch (static_cast<fdf::DocType>(type))
    {
    case fdf::DocType::Registration:
        doc = new FsArchiveRegReport;
        doc->alternative_ = false;
        break;
    case fdf::DocType::ReRegistration:
        doc = new FsArchiveReRegReport;
        doc->alternative_ = false;
        break;
    case fdf::DocType::CycleOpen: doc = new FsArchiveReportCycleOpen;break;
    case fdf::DocType::CalcReport: doc = new FsArchiveReportCycleState;break;
    case fdf::DocType::Receipt: doc = new FsArchiveReportCheck(false, false); break;
    case fdf::DocType::Correction: doc = new FsArchiveReportCheck(true, false); break;
    case fdf::DocType::Fosa:doc = new FsArchiveReportCheck(false, true); break;
    case fdf::DocType::FosaCorrection: doc = new FsArchiveReportCheck(true, true); break;
    case fdf::DocType::CycleClose: doc = new FsArchiveReportCycleClose; break;
    case fdf::DocType::FsClose: doc = new FsArchiveCloseReport; break;
    default:;
    }
    if(doc && !doc->parse(data.mid(2)))
    {
        delete doc;
        doc = nullptr;
    }
    if(doc)
    {
        doc->header_ = header;
        doc->type_ = type;
        doc->hasTicket_ = hasTicket;
        doc->data_ = data;
        res = QSharedPointer<FsArchiveReportBase>(doc);
    }
    return res;
}

QSharedPointer<FsArchiveReportBase> FsArchiveReportBase::createRegDocument(const QByteArray &data,
                                                                           const QStringList &header)
{
    QSharedPointer<FsArchiveReportBase> res;
    FsArchiveReportBase *doc = nullptr;
    fdf::DocType type = fdf::DocType::Registration;
    if(data.size() == 64 || data.size() == 48) type = fdf::DocType::ReRegistration;
    switch (type)
    {
    case fdf::DocType::Registration:
        doc = new FsArchiveRegReport;
        doc->alternative_ = true;
        break;
    case fdf::DocType::ReRegistration:
        doc = new FsArchiveReRegReport;
        doc->alternative_ = true;
        break;
    default:
    {
        return res;
    }
    }

    if(doc && !doc->parse(data))
    {
        delete doc;
        doc = nullptr;
    }
    if(doc)
    {
        doc->header_ = header;
        doc->type_ = static_cast<quint8>(type);
        doc->hasTicket_ = true;
        doc->data_ = data;
        res = QSharedPointer<FsArchiveReportBase>(doc);
    }
    return res;
}

const QByteArray &FsArchiveReportBase::ticket() const
{
    return ticket_;
}

void FsArchiveReportBase::setTicket(const QByteArray &newTicket)
{
    ticket_ = newTicket;
}

const QDateTime &FsArchiveReportBase::ticketDt() const
{
    return ticketDt_;
}

void FsArchiveReportBase::setTicketDt(const QDateTime &newTicketDt)
{
    ticketDt_ = newTicketDt;
}


FsArchiveReportBase::FsArchiveReportBase()
    : data_()
    , type_(0)
    , hasTicket_(false)
    , dt_()
    , docNumber_()
    , fiscalCode_(0)
    , header_()
    , alternative_(false)
    , ffd_(fs::FFD::FFD1_2)
{

}


//--------------------------------------------------------------------------------------------------


FsArchiveOfdTicket::FsArchiveOfdTicket()noexcept
    : fd_(0)
    , dt_()
    , sign_()
{

}

FsArchiveOfdTicket::FsArchiveOfdTicket(const FsArchiveOfdTicket &other)noexcept
    : fd_(other.fd_)
    , dt_(other.dt_)
    , sign_(other.sign_)
{

}

FsArchiveOfdTicket::FsArchiveOfdTicket(FsArchiveOfdTicket &&other)noexcept
    : fd_(other.fd_)
    , dt_()
    , sign_()

{
    dt_.swap(other.dt_);
    sign_.swap(other.sign_);
}

FsArchiveOfdTicket::~FsArchiveOfdTicket()
{

}

qint32 FsArchiveOfdTicket::fd() const
{
    return fd_;
}

void FsArchiveOfdTicket::setFd(qint32 newFd)
{
    fd_ = newFd;
}

const QDateTime &FsArchiveOfdTicket::dt() const
{
    return dt_;
}

void FsArchiveOfdTicket::setDt(const QDateTime &newDt)
{
    dt_ = newDt;
}

const QByteArray &FsArchiveOfdTicket::sign() const
{
    return sign_;
}

void FsArchiveOfdTicket::setSign(const QByteArray &newSign)
{
    sign_ = newSign;
}

QVariantMap FsArchiveOfdTicket::toMap() const
{
    return {
        {"docNumber", fd_},
        {"dt", DT2STR_(dt_)},
        {"sign", QString::fromLatin1(sign_.toHex())}
    };
}

QVariantMap FsArchiveOfdTicket::toExternalMap() const
{
    return {
        {"docNumber", fd_},
        {"dt", DT2STR_(dt_)},
        {"sign", QString::fromLatin1(sign_.toHex())}
    };
}


FsArchiveOfdTicket &FsArchiveOfdTicket::operator =(const FsArchiveOfdTicket &other) noexcept
{
    fd_ = other.fd_;
    dt_ = other.dt_;
    sign_ = other.sign_;
    return *this;
}

FsArchiveOfdTicket &FsArchiveOfdTicket::operator =(FsArchiveOfdTicket &&other) noexcept
{
    fd_ = other.fd_;
    dt_.swap(other.dt_);
    sign_.swap(other.sign_);
    return *this;

}

bool FsArchiveOfdTicket::operator ==(const FsArchiveOfdTicket &other) const noexcept
{
    return fd_ == other.fd_ &&
            dt_ == other.dt_ &&
            sign_ == other.sign_;
}

bool FsArchiveOfdTicket::operator !=(const FsArchiveOfdTicket &other) const noexcept
{
    return !(*this == other);
}
