#include "fscounters.h"
#include "cashboxbuildconfig.h"
#include "formatutils.h"


#include <QDataStream>


FsShortOperationCounters::FsShortOperationCounters() noexcept
    : count_(0)
    , totalAmount_{DEFAULT_AMOUNT_MULT, 0ll}
    , cash_{DEFAULT_AMOUNT_MULT, 0ll}
    , card_{DEFAULT_AMOUNT_MULT, 0ll}
    , prepay_{DEFAULT_AMOUNT_MULT, 0ll}
    , postpay_{DEFAULT_AMOUNT_MULT, 0ll}
    , barter_{DEFAULT_AMOUNT_MULT, 0ll}

{

}

FsShortOperationCounters::FsShortOperationCounters(const FsShortOperationCounters &other) noexcept
    : count_(other.count_)
    , totalAmount_(other.totalAmount_)
    , cash_(other.cash_)
    , card_(other.card_)
    , prepay_(other.prepay_)
    , postpay_(other.postpay_)
    , barter_(other.barter_)

{

}

FsShortOperationCounters::~FsShortOperationCounters()
{

}

quint32 FsShortOperationCounters::count() const
{
    return count_;
}

void FsShortOperationCounters::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FixNumber &FsShortOperationCounters::totalAmount() const
{
    return totalAmount_;
}

void FsShortOperationCounters::setTotalAmount(const FixNumber &newTotalAmount)
{
    totalAmount_ = newTotalAmount;
}

const FixNumber &FsShortOperationCounters::cash() const
{
    return cash_;
}

void FsShortOperationCounters::setCash(const FixNumber &newCash)
{
    cash_ = newCash;
}

const FixNumber &FsShortOperationCounters::card() const
{
    return card_;
}

void FsShortOperationCounters::setCard(const FixNumber &newCard)
{
    card_ = newCard;
}

const FixNumber &FsShortOperationCounters::prepay() const
{
    return prepay_;
}

void FsShortOperationCounters::setPrepay(const FixNumber &newPrepay)
{
    prepay_ = newPrepay;
}

const FixNumber &FsShortOperationCounters::postpay() const
{
    return postpay_;
}

void FsShortOperationCounters::setPostpay(const FixNumber &newPostpay)
{
    postpay_ = newPostpay;
}

const FixNumber &FsShortOperationCounters::barter() const
{
    return barter_;
}

void FsShortOperationCounters::setBarter(const FixNumber &newBarter)
{
    barter_ = newBarter;
}

void FsShortOperationCounters::clean()
{
    count_ = 0;
    totalAmount_.setValue(0ll);
    cash_.setValue(0ll);
    card_.setValue(0ll);
    prepay_.setValue(0ll);
    postpay_.setValue(0ll);
    barter_.setValue(0ll);
}

QVariantMap FsShortOperationCounters::toMap() const
{
    return {
        {"count", count_},
        {"totalAmount", totalAmount_.toMap()},
        {"cash", cash_.toMap()},
        {"card", card_.toMap()},
        {"prepay", prepay_.toMap()},
        {"postpay", postpay_.toMap()},
        {"barter", barter_.toMap()},
    };
}

QVariantMap FsShortOperationCounters::toExternalMap() const
{
    return {
        {"count", count_},
        {"totalAmount", totalAmount_.toString()},
        {"cash", cash_.toString()},
        {"card", card_.toString()},
        {"prepay", prepay_.toString()},
        {"postpay", postpay_.toString()},
        {"barter", barter_.toString()},
        };
}

QString FsShortOperationCounters::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<tr><td align=\"left\">ЧЕКОВ</td><td align=\"right\">%1</td></tr>").arg(count_);
    if(count_ > 0)
    {
        res << QStringLiteral("<tr><td align=\"left\">НА СУММУ</td><td align=\"right\">%1</td></tr>").arg(totalAmount_.toString())
            << QStringLiteral("<tr><td align=\"left\">НАЛИЧНЫМИ</td><td align=\"right\">%1</td></tr>").arg(cash_.toString())
            << QStringLiteral("<tr><td align=\"left\">БЕЗНАЛИЧНЫМИ</td><td align=\"right\">%1</td></tr>").arg(card_.toString())
            << QStringLiteral("<tr><td align=\"left\">АВАНСОМ</td><td align=\"right\">%1</td></tr>").arg(prepay_.toString())
            << QStringLiteral("<tr><td align=\"left\">КРЕДИТОМ</td><td align=\"right\">%1</td></tr>").arg(postpay_.toString())
            << QStringLiteral("<tr><td align=\"left\">ИНЫМИ ОПЛАТАМИ</td><td align=\"right\">%1</td></tr>").arg(barter_.toString());
    }
    return res.join("\n");
}

void FsShortOperationCounters::parseMap(const QVariantMap &map)
{
    clean();

    count_ = map["count"].toUInt();

    totalAmount_ = FixNumber(map["totalAmount"].toMap());
    cash_    = FixNumber(map["cash"].toMap());
    card_    = FixNumber(map["card"].toMap());
    prepay_  = FixNumber(map["prepay"].toMap());
    postpay_ = FixNumber(map["postpay"].toMap());
    barter_  = FixNumber(map["barter"].toMap());
}

bool FsShortOperationCounters::parseTlv(const Tlv::MapStlv &tlv)
{
    clean();
    if(tlv.contains(fdf::Tag::ReceiptsCountByType))
    {
        count_ = tlv[fdf::Tag::ReceiptsCountByType].toUInt32();
    }
    if(tlv.contains(fdf::Tag::ReceiptsCashTotal))
    {
        cash_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::ReceiptsCashTotal].toVln());
    }

    if(tlv.contains(fdf::Tag::ReceiptsCardTotal))
    {
        card_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::ReceiptsCardTotal].toVln());
    }
    if(tlv.contains(fdf::Tag::ReceiptsTotalPrepaymentAmount))
    {
        prepay_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::ReceiptsTotalPrepaymentAmount].toVln());
    }
    if(tlv.contains(fdf::Tag::ReceiptsTotalPostpaymentAmount))
    {
        postpay_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::ReceiptsTotalPostpaymentAmount].toVln());
    }
    if(tlv.contains(fdf::Tag::ReceiptsTOtalBarterAmount))
    {
        barter_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::ReceiptsTOtalBarterAmount].toVln());
    }
    if(tlv.contains(fdf::Tag::TotalReceiptsAMount))
    {
        totalAmount_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::TotalReceiptsAMount].toVln());
    }
    return true;
}

FsShortOperationCounters &FsShortOperationCounters::operator =(const FsShortOperationCounters &other) noexcept
{
    count_ = other.count_;
    totalAmount_ = other.totalAmount_;
    cash_ = other.cash_;
    card_ = other.card_;
    prepay_ = other.prepay_;
    postpay_ = other.postpay_;
    barter_ = other.barter_;
    return *this;

}

bool FsShortOperationCounters::operator ==(const FsShortOperationCounters &other) const noexcept
{
    return count_ == other.count_ &&
            totalAmount_ == other.totalAmount_ &&
            cash_ == other.cash_ &&
            card_ == other.card_ &&
            prepay_ == other.prepay_ &&
            postpay_ == other.postpay_ &&
            barter_ == other.barter_;
}

bool FsShortOperationCounters::operator !=(const FsShortOperationCounters &other) const noexcept
{
 return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsOperationCounters::FsOperationCounters() noexcept
    : FsShortOperationCounters()
    , vat20_{DEFAULT_AMOUNT_MULT, 0ll}
    , vat10_{DEFAULT_AMOUNT_MULT, 0ll}
    , vat0_{DEFAULT_AMOUNT_MULT, 0ll}
    , vatNone_{DEFAULT_AMOUNT_MULT, 0ll}
    , vat20_120_{DEFAULT_AMOUNT_MULT, 0ll}
    , vat10_110_{DEFAULT_AMOUNT_MULT, 0ll}
{

}

FsOperationCounters::FsOperationCounters(const FsOperationCounters &other) noexcept
    : FsShortOperationCounters(other)
    , vat20_(other.vat20_)
    , vat10_(other.vat10_)
    , vat0_(other.vat0_)
    , vatNone_(other.vatNone_)
    , vat20_120_(other.vat20_120_)
    , vat10_110_(other.vat10_110_)
{

}

FsOperationCounters::~FsOperationCounters()
{

}

const FixNumber &FsOperationCounters::vat20() const
{
    return vat20_;
}

void FsOperationCounters::setVat20(const FixNumber &newVat20)
{
    vat20_ = newVat20;
}

const FixNumber &FsOperationCounters::vat10() const
{
    return vat10_;
}

void FsOperationCounters::setVat10(const FixNumber &newVat10)
{
    vat10_ = newVat10;
}

const FixNumber &FsOperationCounters::vat0() const
{
    return vat0_;
}

void FsOperationCounters::setVat0(const FixNumber &newVat0)
{
    vat0_ = newVat0;
}

const FixNumber &FsOperationCounters::vatNone() const
{
    return vatNone_;
}

void FsOperationCounters::setVatNone(const FixNumber &newVatNone)
{
    vatNone_ = newVatNone;
}

const FixNumber &FsOperationCounters::vat20_120() const
{
    return vat20_120_;
}

void FsOperationCounters::setVat20_120(const FixNumber &newVat20_120)
{
    vat20_120_ = newVat20_120;
}

const FixNumber &FsOperationCounters::vat10_110() const
{
    return vat10_110_;
}

void FsOperationCounters::setVat10_110(const FixNumber &newVat10_110)
{
    vat10_110_ = newVat10_110;
}

bool FsOperationCounters::parseFromFs(const QByteArray &data)
{
    if(data.size() != 76)
    {
        return false;
    }
    count_ = FormatUtils::fsToUint32(data.mid(0, 4));
    QByteArray buf = data.mid(4);
    FixNumber * cts [12];
    cts[0] = &totalAmount_;
    cts[1] = &cash_;
    cts[2] = &card_;
    cts[3] = &prepay_;
    cts[4] = &postpay_;
    cts[5] = &barter_;
    cts[6] = &vat20_;
    cts[7] = &vat10_;
    cts[8] = &vat0_;
    cts[9] = &vatNone_;
    cts[10] = &vat20_120_;
    cts[11] = &vat10_110_;

    for(int i = 0; i < buf.size(); i+= 6)
    {
        bool ok = false;
        *(cts[i / 6]) = FROMUINTOK48_(buf.mid(i, 6), DEFAULT_AMOUNT_MULT, &ok);
        if(!ok) return false;
    }
    return true;
}

void FsOperationCounters::clean()
{

    vat20_.setValue(0ll);
    vat10_.setValue(0ll);
    vat0_.setValue(0ll);
    vatNone_.setValue(0ll);
    vat20_120_.setValue(0ll);
    vat10_110_.setValue(0ll);
}

QVariantMap FsOperationCounters::toMap() const
{
    QVariantMap res {
        {"vat20", vat20_.toMap()},
        {"vat10", vat10_.toMap()},
        {"vat0", vat0_.toMap()},
        {"vatNone", vatNone_.toMap()},
        {"vat20_120", vat20_120_.toMap()},
        {"vat10_110", vat10_110_.toMap()}
    };
    return UNIT_MAPS(res, FsShortOperationCounters::toMap());
}

QVariantMap FsOperationCounters::toExternalMap() const
{
    QVariantMap res  {
        {"vat20", vat20_.toString()},
        {"vat10", vat10_.toString()},
        {"vat0", vat0_.toString()},
        {"vatNone", vatNone_.toString()},
        {"vat20_120", vat20_120_.toString()},
        {"vat10_110", vat10_110_.toString()}
    };
    return UNIT_MAPS(res, FsShortOperationCounters::toExternalMap());
}

QString FsOperationCounters::toHtml() const
{
    QStringList res;
    res.append(FsShortOperationCounters::toHtml());
    if(count_ > 0)
    {
        res << QStringLiteral("<tr><td align=\"left\">НДС 20%</td><td align=\"right\">%1</td></tr>").arg(vat20_.toString())
            << QStringLiteral("<tr><td align=\"left\">НДС 10%</td><td align=\"right\">%1</td></tr>").arg(vat10_.toString())
            << QStringLiteral("<tr><td align=\"left\">НДС 0%</td><td align=\"right\">%1</td></tr>").arg(vat0_.toString())
            << QStringLiteral("<tr><td align=\"left\">БЕЗ НДС</td><td align=\"right\">%1</td></tr>").arg(vatNone_.toString())
            << QStringLiteral("<tr><td align=\"left\">НДС 20/120</td><td align=\"right\">%1</td></tr>").arg(vat20_120_.toString())
            << QStringLiteral("<tr><td align=\"left\">НДС 10/110</td><td align=\"right\">%1</td></tr>").arg(vat10_110_.toString());
    }
    return res.join("\n");
}

void FsOperationCounters::parseMap(const QVariantMap &map)
{
    clean();
    FsShortOperationCounters::parseMap(map);
    vat20_ = FixNumber(map["vat20"].toMap());
    vat10_    = FixNumber(map["vat10"].toMap());
    vat0_    = FixNumber(map["vat0"].toMap());
    vatNone_  = FixNumber(map["vatNone"].toMap());
    vat20_120_ = FixNumber(map["vat20_120"].toMap());
    vat10_110_  = FixNumber(map["vat10_110"].toMap());
}

bool FsOperationCounters::parseTlv(const Tlv::MapStlv &tlv)
{
    clean();
    FsShortOperationCounters::parseTlv(tlv);
    QStringList sl;
    for(auto it = tlv.constBegin(); it != tlv.constEnd(); ++it)
    {
        sl <<  (QString::number(static_cast<qint32>(it.value().tag())) + ": " +
                QString::fromLatin1(it.value().value().toHex()));
    }
//    lmWarning() << loglist(sl);

    if(tlv.contains(fdf::Tag::Vat20Amount))
    {
        vat20_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::Vat20Amount].toVln());
    }

    if(tlv.contains(fdf::Tag::Vat10Amount))
    {
        vat10_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::Vat10Amount].toVln());
    }
    if(tlv.contains(fdf::Tag::Vat20_120Amount))
    {
        vat20_120_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::Vat20_120Amount].toVln());
    }
    if(tlv.contains(fdf::Tag::Vat10_110Amount))
    {
        vat10_110_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::Vat10_110Amount].toVln());
    }
    if(tlv.contains(fdf::Tag::Vat0Amount))
    {
        vat0_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::Vat0Amount].toVln());
    }
    if(tlv.contains(fdf::Tag::VatNoneAmount))
    {
        vatNone_ = FixNumber(DEFAULT_AMOUNT_MULT, tlv[fdf::Tag::VatNoneAmount].toVln());
    }
    return true;
}

FsOperationCounters &FsOperationCounters::operator =(const FsOperationCounters &other) noexcept
{
    FsShortOperationCounters::operator =(other);
    vat20_ = other.vat20_;
    vat10_ = other.vat10_;
    vat0_ = other.vat0_;
    vatNone_ = other.vatNone_;
    vat20_120_ = other.vat20_120_;
    vat10_110_ = other.vat10_110_;
    return *this;
}

bool FsOperationCounters::operator ==(const FsOperationCounters &other) const noexcept
{
    return FsShortOperationCounters::operator ==(other) &&
            vat20_ == other.vat20_ &&
            vat10_ == other.vat10_ &&
            vat0_ == other.vat0_ &&
            vatNone_ == other.vatNone_ &&
            vat20_120_ == other.vat20_120_ &&
            vat10_110_ == other.vat10_110_;
}

bool FsOperationCounters::operator !=(const FsOperationCounters &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------

FsCorrectionCounter::FsCorrectionCounter() noexcept
    : count_(0)
    , amount_{DEFAULT_AMOUNT_MULT, 0ll}
{

}

FsCorrectionCounter::FsCorrectionCounter(const FsCorrectionCounter &other) noexcept
    : count_(other.count_)
    , amount_(other.amount_)
{

}

FsCorrectionCounter::~FsCorrectionCounter()
{

}

quint32 FsCorrectionCounter::count() const
{
    return count_;
}

void FsCorrectionCounter::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FixNumber &FsCorrectionCounter::amount() const
{
    return amount_;
}

void FsCorrectionCounter::setAmount(const FixNumber &newAmount)
{
    amount_ = newAmount;
}

bool FsCorrectionCounter::parseFromFs(const QByteArray &data)
{
    clean();
    if(data.size() != 10) return false;
    count_ = FormatUtils::fsToUint32(data.mid(0, 4));
    amount_ = FROMUINT48_(data.mid(4), DEFAULT_AMOUNT_MULT);
    return true;
}

void FsCorrectionCounter::clean()
{
    count_ = 0;
    amount_.setValue(0ll);
}

QVariantMap FsCorrectionCounter::toMap() const
{
    return {
        {"count", count_},
        {"amount", amount_.toMap()}
    };
}

void FsCorrectionCounter::parseMap(const QVariantMap &map)
{
    count_ = map["count"].toUInt();
    amount_ = FormatUtils::parseSumm("amount", map, DEFAULT_AMOUNT_DEC, nullptr);
}

QVariantMap FsCorrectionCounter::toExternalMap() const
{
    return {
        {"count", count_},
        {"amount", amount_.toString()}
    };

}

QString FsCorrectionCounter::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<tr><td align=\"left\">ЧЕКОВ</td><td align=\"right\">%1</td></tr>").arg(count_);
    if(count_ > 0)
    {
        res << QStringLiteral("<tr><td align=\"left\">НА СУММУ</td><td align=\"right\">%1</td></tr>").arg(amount_.toString());
    }
    return res.join("\n");
}

FsCorrectionCounter &FsCorrectionCounter::operator =(const FsCorrectionCounter &other) noexcept
{
    count_ = other.count_;
    amount_ = other.amount_;
    return *this;
}

bool FsCorrectionCounter::operator ==(const FsCorrectionCounter &other) const noexcept
{
    return count_ == other.count_ &&
            amount_ == other.amount_;
}

bool FsCorrectionCounter::operator !=(const FsCorrectionCounter &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------
FsCorrectionCounters::FsCorrectionCounters() noexcept
    : count_(0)
    , debit_()
    , debitRefund_()
    , credit_()
    , creditRefund_()
{

}

FsCorrectionCounters::FsCorrectionCounters(const FsCorrectionCounters &other) noexcept
    : count_(other.count_)
    , debit_(other.debit_)
    , debitRefund_(other.debitRefund_)
    , credit_(other.credit_)
    , creditRefund_(other.creditRefund_)
{

}

FsCorrectionCounters::~FsCorrectionCounters()
{

}

quint32 FsCorrectionCounters::count() const
{
    return count_;
}

void FsCorrectionCounters::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FsCorrectionCounter &FsCorrectionCounters::debit() const
{
    return debit_;
}

void FsCorrectionCounters::setDebit(const FsCorrectionCounter &newDebit)
{
    debit_ = newDebit;
}

const FsCorrectionCounter &FsCorrectionCounters::debitRefund() const
{
    return debitRefund_;
}

void FsCorrectionCounters::setDebitRefund(const FsCorrectionCounter &newDebitRefund)
{
    debitRefund_ = newDebitRefund;
}

const FsCorrectionCounter &FsCorrectionCounters::credit() const
{
    return credit_;
}

void FsCorrectionCounters::setCredit(const FsCorrectionCounter &newCredit)
{
    credit_ = newCredit;
}

const FsCorrectionCounter &FsCorrectionCounters::creditRefund() const
{
    return creditRefund_;
}

void FsCorrectionCounters::setCreditRefund(const FsCorrectionCounter &newCreditRefund)
{
    creditRefund_ = newCreditRefund;
}

bool FsCorrectionCounters::parseFromFs(const QByteArray &data)
{
    clean();
    if(data.size() != 44) return false;
    count_ = FormatUtils::fsToUint32(data.mid(0, 4));
    debit_.parseFromFs(data.mid(4, 10));
    debitRefund_.parseFromFs(data.mid(14, 10));
    credit_.parseFromFs(data.mid(24, 10));
    creditRefund_.parseFromFs(data.mid(34, 10));
    return true;
}

void FsCorrectionCounters::clean()
{
    count_ = 0;
    debit_.clean();
    debitRefund_.clean();
    credit_.clean();
    creditRefund_.clean();
}

QVariantMap FsCorrectionCounters::toMap() const
{
    return {
        {"count", count_},
        {"debit", debit_.toMap()},
        {"debitRefund", debitRefund_.toMap()},
        {"credit", credit_.toMap()},
        {"creditRefund", creditRefund_.toMap()}
    };
}

void FsCorrectionCounters::parseMap(const QVariantMap &map)
{
    count_ = map["count"].toUInt();
    debit_.parseMap(map["debit"].toMap());
    debitRefund_.parseMap(map["debitRefund"].toMap());
    credit_.parseMap(map["credit"].toMap());
    creditRefund_.parseMap(map["creditRefund"].toMap());
}

QVariantMap FsCorrectionCounters::toExternalMap() const
{
    return {
        {"count", count_},
        {"debit", debit_.toExternalMap()},
        {"debitRefund", debitRefund_.toExternalMap()},
        {"credit", credit_.toExternalMap()},
        {"creditRefund", creditRefund_.toExternalMap()}
    };
}

QString FsCorrectionCounters::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<tr><td align=\"center\" colspan=2>ЧЕКИ КОРРЕКЦИИ</td></tr>")
        << QStringLiteral("<tr><td align=\"left\">ВСЕГО ЧЕКОВ</td><td align=\"right\">%1</td></tr>").arg(count_);
    if(count_ > 0)
    {
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ПРИХОД</td></tr>");
        res.append(debit_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ВОЗВРАТ ПРИХОДА</td></tr>");
        res.append(debitRefund_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>РАСХОД</td></tr>");
        res.append(credit_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ВОЗВРАТ ВОЗВРАТ РАСХОДА</td></tr>");
        res.append(creditRefund_.toHtml());
    }

    return res.join("\n");
}

FsCorrectionCounters &FsCorrectionCounters::operator =(const FsCorrectionCounters &other) noexcept
{
    count_ = other.count_;
    debit_ = other.debit_;
    debitRefund_ = other.debitRefund_;
    credit_ = other.credit_;
    creditRefund_ = other.creditRefund_;
    return *this;
}

bool FsCorrectionCounters::operator ==(const FsCorrectionCounters &other) const noexcept
{
    return count_ == other.count_ &&
            debit_ == other.debit_ &&
            debitRefund_ == other.debitRefund_ &&
            credit_ == other.credit_ &&
            creditRefund_ == other.creditRefund_;

}

bool FsCorrectionCounters::operator !=(const FsCorrectionCounters &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsCounters::FsCounters() noexcept
    : type_(FsTotalType::CycleTotal)
    , cycle_(0)
    , count_(0)
    , debit_()
    , debitRefund_()
    , credit_()
    , creditRefund_()
    , corrections_()
{

}

FsCounters::FsCounters(const FsCounters &other) noexcept
    : type_(other.type_)
    , cycle_(other.cycle_)
    , count_(other.count_)
    , debit_(other.debit_)
    , debitRefund_(other.debitRefund_)
    , credit_(other.credit_)
    , creditRefund_(other.creditRefund_)
    , corrections_(other.corrections_)

{

}


FsCounters::~FsCounters()
{

}

FsTotalType FsCounters::type() const
{
    return type_;
}

void FsCounters::setType(FsTotalType newType)
{
    type_ = newType;
}

QString FsCounters::title() const
{
    if(type_ == FsTotalType::CycleTotal) return QStringLiteral("Счетчики итогов смены");
    return QStringLiteral("Счетчики итогов ФН");
}

quint16 FsCounters::cycle() const
{
    return cycle_;
}

void FsCounters::setCycle(quint16 newCycle)
{
    cycle_ = newCycle;
}

quint32 FsCounters::count() const
{
    return count_;
}

void FsCounters::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FsOperationCounters &FsCounters::debit() const
{
    return debit_;
}

void FsCounters::setDebit(const FsOperationCounters &newDebit)
{
    debit_ = newDebit;
}

const FsOperationCounters &FsCounters::debitRefund() const
{
    return debitRefund_;
}

void FsCounters::setDebitRefund(const FsOperationCounters &newDebitRefund)
{
    debitRefund_ = newDebitRefund;
}

const FsOperationCounters &FsCounters::credit() const
{
    return credit_;
}

void FsCounters::setCredit(const FsOperationCounters &newCredit)
{
    credit_ = newCredit;
}

const FsOperationCounters &FsCounters::creditRefund() const
{
    return creditRefund_;
}

void FsCounters::setCreditRefund(const FsOperationCounters &newCreditRefund)
{
    creditRefund_ = newCreditRefund;
}

const FsCorrectionCounters &FsCounters::corrections() const
{
    return corrections_;
}

void FsCounters::setCorrections(const FsCorrectionCounters &newCorrections)
{
    corrections_ = newCorrections;
}

FixNumber FsCounters::cash() const
{
    FixNumber res = debit_.cash();
    res.setValue(res.value() + creditRefund_.cash().value() - credit_.cash().value() + debitRefund_.cash().value());
    return res;
}

bool FsCounters::parseFromFs(const QByteArray &data, FsTotalType type)
{
    type_ = type;
    if(data.size() != 354) return false;
    cycle_ = FormatUtils::fsToUint16(data.mid(0, 2));
    count_ = FormatUtils::fsToUint32(data.mid(2, 4));
    QByteArray buf = data.mid(6);
    debit_.parseFromFs(buf.mid(0, 76));
    buf = buf.mid(76);
    debitRefund_.parseFromFs(buf.mid(0, 76));
    buf = buf.mid(76);
    credit_.parseFromFs(buf.mid(0, 76));
    buf = buf.mid(76);
    creditRefund_.parseFromFs(buf.mid(0, 76));
    buf = buf.mid(76);
    corrections_.parseFromFs(buf);
    return true;
}

void FsCounters::clean()
{
    type_ = FsTotalType::CycleTotal;
    cycle_ = 0;
    count_ = 0;
    debit_.clean();
    debitRefund_.clean();
    credit_.clean();
    creditRefund_.clean();
    corrections_.clean();
}

QVariantMap FsCounters::toMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"cycle", cycle_},
        {"count", count_},
        {"debit", debit_.toMap()},
        {"debitRefund", debitRefund_.toMap()},
        {"credit", credit_.toMap()},
        {"creditRefund", creditRefund_.toMap()},
        {"corrections", corrections_.toMap()}

    };
}

void FsCounters::parseMap(const QVariantMap &map)
{
    type_ = static_cast<FsTotalType>(map["type"].toInt());
    cycle_ = map["cycle"].toUInt();
    count_ = map["count"].toUInt();
    debit_.parseMap(map["debit"].toMap());
    debitRefund_.parseMap(map["debitRefund"].toMap());
    credit_.parseMap(map["credit"].toMap());
    creditRefund_.parseMap(map["creditRefund"].toMap());
    corrections_.parseMap(map["corrections"].toMap());
}

QVariantMap FsCounters::toExternalMap() const
{
    return {
        {"type", type_ == FsTotalType::CycleTotal ? QStringLiteral("cycle") : QStringLiteral("fs")},
        {"cycle", cycle_},
        {"count", count_},
        {"debit", debit_.toExternalMap()},
        {"debitRefund", debitRefund_.toExternalMap()},
        {"credit", credit_.toExternalMap()},
        {"creditRefund", creditRefund_.toExternalMap()},
        {"corrections", corrections_.toExternalMap()}

    };
}

QString FsCounters::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<p align=\"center\"><i><b>%1</b></i></p>").arg(title())
        << "<table align=\"center\" width=100%><tbody>"
        << QStringLiteral("<tr><td align=\"left\">%1</td><td align=\"right\">%2</td></tr>")
               .arg(type_ == FsTotalType::CycleTotal ? QStringLiteral("СМЕНА") : QStringLiteral("ВСЕГО СМЕН"))
               .arg(cycle_)
        << QStringLiteral("<tr><td align=\"left\">ВСЕГО ЧЕКОВ</td><td align=\"right\">%1</td></tr>").arg(count_);

    if(count_ > 0)
    {
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ПРИХОД</td></tr>");
        res.append(debit_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ВОЗВРАТ ПРИХОДА</td></tr>");
        res.append(debitRefund_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>РАСХОД</td></tr>");
        res.append(credit_.toHtml());
        res << QStringLiteral("<tr><td align=\"center\" colspan=2>ВОЗВРАТ ВОЗВРАТ РАСХОДА</td></tr>");
        res.append(creditRefund_.toHtml());
    }
    res.append(corrections_.toHtml());

    return res.join("\n");
}

FsCounters &FsCounters::operator =(const FsCounters &other) noexcept
{
    type_ = other.type_;
    cycle_ = other.cycle_;
    count_ = other.count_;
    debit_ = other.debit_;
    debitRefund_ = other.debitRefund_;
    credit_ = other.credit_;
    creditRefund_ = other.creditRefund_;
    corrections_ = other.corrections_;
    return *this;
}

bool FsCounters::operator ==(const FsCounters &other) const noexcept
{
    return type_ == other.type_ &&
            cycle_ == other.cycle_ &&
            count_ == other.count_ &&
            debit_ == other.debit_ &&
            debitRefund_ == other.debitRefund_ &&
            credit_ == other.credit_ &&
            creditRefund_ == other.creditRefund_ &&
            corrections_ == other.corrections_;
}

bool FsCounters::operator !=(const FsCounters &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------
ShortTaggedFsCounters::ShortTaggedFsCounters() noexcept
    : type_(Type::Invalid)
    , count_(0)
    , debit_()
    , debitRefund_()
    , credit_()
    , creditRefund_()
{

}

ShortTaggedFsCounters::ShortTaggedFsCounters(const ShortTaggedFsCounters &other) noexcept
    : type_(other.type_)
    , count_(other.count_)
    , debit_(other.debit_)
    , debitRefund_(other.debitRefund_)
    , credit_(other.credit_)
    , creditRefund_(other.creditRefund_)
{
}

ShortTaggedFsCounters::~ShortTaggedFsCounters()
{

}

ShortTaggedFsCounters::Type ShortTaggedFsCounters::type() const
{
    return type_;
}

void ShortTaggedFsCounters::setType(Type newType)
{
    type_ = newType;
}

QString ShortTaggedFsCounters::title() const
{
    switch (type_) {
    case Type::NotSendedCounter: return QStringLiteral("НЕПЕРЕДАНО");
    case Type::CorrectionsCounter: return QStringLiteral("ЧЕКИ (БСО) КОРРЕКЦИИ");
    default: return QString();
    }
}

quint32 ShortTaggedFsCounters::count() const
{
    return count_;
}

void ShortTaggedFsCounters::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FsShortOperationCounters &ShortTaggedFsCounters::debit() const
{
    return debit_;
}

void ShortTaggedFsCounters::setDebit(const FsShortOperationCounters &newDebit)
{
    debit_ = newDebit;
}

const FsShortOperationCounters &ShortTaggedFsCounters::debitRefund() const
{
    return debitRefund_;
}

void ShortTaggedFsCounters::setDebitRefund(const FsShortOperationCounters &newDebitRefund)
{
    debitRefund_ = newDebitRefund;
}

const FsShortOperationCounters &ShortTaggedFsCounters::credit() const
{
    return credit_;
}

void ShortTaggedFsCounters::setCredit(const FsShortOperationCounters &newCredit)
{
    credit_ = newCredit;
}

const FsShortOperationCounters &ShortTaggedFsCounters::creditRefund() const
{
    return creditRefund_;
}

void ShortTaggedFsCounters::setCreditRefund(const FsShortOperationCounters &newCreditRefund)
{
    creditRefund_ = newCreditRefund;
}

void ShortTaggedFsCounters::clean()
{
    type_ = Type::Invalid;
    count_ = 0;
    debit_.clean();
    debitRefund_.clean();
    credit_.clean();
    creditRefund_.clean();
}

QVariantMap ShortTaggedFsCounters::toMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"count", count_},
        {"debit", debit_.toMap()},
        {"debitRefund", debitRefund_.toMap()},
        {"credit", credit_.toMap()},
        {"creditRefund", creditRefund_.toMap()},
    };
}

QVariantMap ShortTaggedFsCounters::toExternalMap() const
{
    return {
        {"count", count_},
        {"debit", debit_.toExternalMap()},
        {"debitRefund", debitRefund_.toExternalMap()},
        {"credit", credit_.toExternalMap()},
        {"creditRefund", creditRefund_.toExternalMap()},
    };
}

void ShortTaggedFsCounters::parseMap(const QVariantMap &map)
{
    clean();
    type_ = static_cast<Type>(map["type"].toInt());
    count_ = map["count"].toUInt();
    debit_.parseMap(map["debit"].toMap());
    debitRefund_.parseMap(map["debitRefund"].toMap());
    credit_.parseMap(map["credit"].toMap());
    creditRefund_.parseMap(map["creditRefund"].toMap());
}

bool ShortTaggedFsCounters::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::FsNotSendedTotalsCounters) type_ = Type::NotSendedCounter;
    else if(tlv.tag() == fdf::Tag::CorrectuinsCounters) type_ = Type::CorrectionsCounter;
    else
    {
        type_ = Type::Invalid;
        return false;
    }
    Tlv::MapStlv map = tlv.toMapStlv();
    if(map.contains(fdf::Tag::NotSendedReceiptAndCorrectionsCount))
    {
        count_ = map[fdf::Tag::NotSendedReceiptAndCorrectionsCount].toUInt32();
    }
    if(map.contains(fdf::Tag::DebitAmountCounter))
    {
        debit_.parseTlv(map[fdf::Tag::DebitAmountCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::CreditAmountCounter))
    {
        credit_.parseTlv(map[fdf::Tag::CreditAmountCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::DebitRefundAmountCounters))
    {
        debitRefund_.parseTlv(map[fdf::Tag::DebitRefundAmountCounters].toMapStlv());
    }
    if(map.contains(fdf::Tag::CreditRefundAmountCounters))
    {
        creditRefund_.parseTlv(map[fdf::Tag::CreditRefundAmountCounters].toMapStlv());
    }
    return true;

}

ShortTaggedFsCounters &ShortTaggedFsCounters::operator =(const ShortTaggedFsCounters &other) noexcept
{
    type_ = other.type_;
    count_ = other.count_;
    debit_ = other.debit_;
    debitRefund_ = other.debitRefund_;
    credit_ = other.credit_;
    creditRefund_ = other.creditRefund_;
    return *this;
}

bool ShortTaggedFsCounters::operator ==(const ShortTaggedFsCounters &other) const noexcept
{
    return type_ == other.type_ &&
            count_ == other.count_ &&
            debit_ == other.debit_ &&
            debitRefund_ == other.debitRefund_ &&
            credit_ == other.credit_ &&
            creditRefund_ == other.creditRefund_ ;
}

bool ShortTaggedFsCounters::operator !=(const ShortTaggedFsCounters &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------

TaggedFsCounters::TaggedFsCounters() noexcept
    : type_(FsTotalType::CycleTotal)
    , count_(0)
    , debit_()
    , debitRefund_()
    , credit_()
    , creditRefund_()
    , corrections_()
{

}

TaggedFsCounters::TaggedFsCounters(const TaggedFsCounters &other) noexcept
    : type_(other.type_)
    , count_(other.count_)
    , debit_(other.debit_)
    , debitRefund_(other.debitRefund_)
    , credit_(other.credit_)
    , creditRefund_(other.creditRefund_)
    , corrections_(other.corrections_)
{

}

TaggedFsCounters::~TaggedFsCounters()
{

}

FsTotalType TaggedFsCounters::type() const
{
    return type_;
}

void TaggedFsCounters::setType(FsTotalType newType)
{
    type_ = newType;
}

QString TaggedFsCounters::title() const
{
    if(type_ == FsTotalType::CycleTotal) return QStringLiteral("Счетчики итогов смены");
    return QStringLiteral("Счетчики итогов ФН");
}

quint32 TaggedFsCounters::count() const
{
    return count_;
}

void TaggedFsCounters::setCount(quint32 newCount)
{
    count_ = newCount;
}

const FsOperationCounters &TaggedFsCounters::debit() const
{
    return debit_;
}

void TaggedFsCounters::setDebit(const FsOperationCounters &newDebit)
{
    debit_ = newDebit;
}

const FsOperationCounters &TaggedFsCounters::debitRefund() const
{
    return debitRefund_;
}

void TaggedFsCounters::setDebitRefund(const FsOperationCounters &newDebitRefund)
{
    debitRefund_ = newDebitRefund;
}

const FsOperationCounters &TaggedFsCounters::credit() const
{
    return credit_;
}

void TaggedFsCounters::setCredit(const FsOperationCounters &newCredit)
{
    credit_ = newCredit;
}

const FsOperationCounters &TaggedFsCounters::creditRefund() const
{
    return creditRefund_;
}

void TaggedFsCounters::setCreditRefund(const FsOperationCounters &newCreditRefund)
{
    creditRefund_ = newCreditRefund;
}

const ShortTaggedFsCounters &TaggedFsCounters::corrections() const
{
    return corrections_;
}

void TaggedFsCounters::setCorrections(const ShortTaggedFsCounters &newCorrections)
{
    corrections_ = newCorrections;
}

void TaggedFsCounters::clean()
{
    type_ = FsTotalType::CycleTotal;
    count_ = 0;
    debit_.clean();
    debitRefund_.clean();
    credit_.clean();
    creditRefund_.clean();
    corrections_.clean();
}

QVariantMap TaggedFsCounters::toMap() const
{
    return {
        {"type", static_cast<qint32>(type_)},
        {"count", count_},
        {"debit", debit_.toMap()},
        {"debitRefund", debitRefund_.toMap()},
        {"credit", credit_.toMap()},
        {"creditRefund", creditRefund_.toMap()},
        {"corrections", corrections_.toMap()}

    };
}

QVariantMap TaggedFsCounters::toExternalMap() const
{
    return {
        {"count", count_},
        {"debit", debit_.toExternalMap()},
        {"debitRefund", debitRefund_.toExternalMap()},
        {"credit", credit_.toExternalMap()},
        {"creditRefund", creditRefund_.toExternalMap()},
        {"corrections", corrections_.toExternalMap()}
    };
}

void TaggedFsCounters::parseMap(const QVariantMap &map)
{
    clean();
    type_ = static_cast<FsTotalType>(map["type"].toInt());
    count_ = map["count"].toUInt();
    debit_.parseMap(map["debit"].toMap());
    debitRefund_.parseMap(map["debitRefund"].toMap());
    credit_.parseMap(map["credit"].toMap());
    creditRefund_.parseMap(map["creditRefund"].toMap());
    corrections_.parseMap(map["corrections"].toMap());
}

bool TaggedFsCounters::parseTlv(const Tlv &tlv)
{
    clean();
    if(tlv.tag() == fdf::Tag::FsTotalsCounters) type_ = FsTotalType::FsTotalType;
    else if(tlv.tag() == fdf::Tag::CycleTotalCounters) type_ = FsTotalType::CycleTotal;
    else
    {
        return false;
    }
    Tlv::MapStlv map = tlv.toMapStlv();
    if(map.contains(fdf::Tag::ReceiptsAndCorrections))
    {
        count_ = map[fdf::Tag::ReceiptsAndCorrections].toUInt32();
    }
    if(map.contains(fdf::Tag::DebitCounter))
    {
        debit_.parseTlv(map[fdf::Tag::DebitCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::CreditCounter))
    {
        credit_.parseTlv(map[fdf::Tag::CreditCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::DebitRefundCounter))
    {
        debitRefund_.parseTlv(map[fdf::Tag::DebitRefundCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::CreditRefundCounter))
    {
        creditRefund_.parseTlv(map[fdf::Tag::CreditRefundCounter].toMapStlv());
    }
    if(map.contains(fdf::Tag::CorrectuinsCounters))
    {
        corrections_.parseTlv(map[fdf::Tag::CorrectuinsCounters]);
    }
    return true;

}

TaggedFsCounters &TaggedFsCounters::operator =(const TaggedFsCounters &other) noexcept
{
    type_ = other.type_;
    count_ = other.count_;
    debit_ = other.debit_;
    debitRefund_ = other.debitRefund_;
    credit_ = other.credit_;
    creditRefund_ = other.creditRefund_;
    corrections_ = other.corrections_;
    return *this;
}

bool TaggedFsCounters::operator ==(const TaggedFsCounters &other) const noexcept
{
    return type_ == other.type_ &&
            count_ == other.count_ &&
            debit_ == other.debit_ &&
            debitRefund_ == other.debitRefund_ &&
            credit_ == other.credit_ &&
            creditRefund_ == other.creditRefund_ &&
            corrections_ == other.corrections_;
}

bool TaggedFsCounters::operator !=(const TaggedFsCounters &other) const noexcept
{
    return !(*this == other);
}
