#include "fsfullstatus.h"
#include "formatutils.h"


#include <QDataStream>

const QByteArray fs::CODEC_NAME  = "CP866";


FsStatus::FsStatus() noexcept
    : phase_(fs::Phase::Invalid)
    , currentDoc_(fs::CurrentDocument::None)
    , hasDocData_(false)
    , cycleIsOpen_(false)
    , warnings_()
    , lastDocDt_()
    , fsNumber_()
    , lastFd_(0)

{

}

FsStatus::FsStatus(const QVariantMap map) noexcept
    : phase_(fs::Phase::Invalid)
    , currentDoc_(fs::CurrentDocument::None)
    , hasDocData_(false)
    , cycleIsOpen_(false)
    , warnings_()
    , lastDocDt_()
    , fsNumber_()
    , lastFd_(0)
{
    parseMap(map);
}

FsStatus::FsStatus(const FsStatus &other) noexcept
    : phase_(other.phase_)
    , currentDoc_(other.currentDoc_)
    , hasDocData_(other.hasDocData_)
    , cycleIsOpen_(other.cycleIsOpen_)
    , warnings_(other.warnings_)
    , lastDocDt_(other.lastDocDt_)
    , fsNumber_(other.fsNumber_)
    , lastFd_(other.lastFd_)
{

}

FsStatus::FsStatus(FsStatus &&other) noexcept
    : phase_(other.phase_)
    , currentDoc_(other.currentDoc_)
    , hasDocData_(other.hasDocData_)
    , cycleIsOpen_(other.cycleIsOpen_)
    , warnings_(other.warnings_)
    , lastDocDt_()
    , fsNumber_()
    , lastFd_(other.lastFd_)
{
    lastDocDt_.swap(other.lastDocDt_);
    fsNumber_.swap(other.fsNumber_);
}

FsStatus::~FsStatus()
{

}

bool FsStatus::phaseIsValid() const
{
    return phase_ == fs::Phase::ArchiveClosed ||
            phase_ == fs::Phase::Empty ||
            phase_ == fs::Phase::Fiscalized ||
            phase_ == fs::Phase::Finished;
}

bool FsStatus::isValid() const
{
    return phaseIsValid() && !fsNumber_.isEmpty();
}

fs::Phase FsStatus::phase() const
{
    return phase_;
}

void FsStatus::setPhase(fs::Phase newPhase)
{
    phase_ = newPhase;
    if(!phaseIsValid())phase_ = fs::Phase::Invalid;
}

fs::CurrentDocument FsStatus::currentDoc() const
{
    return currentDoc_;
}

void FsStatus::setCurrentDoc(fs::CurrentDocument newCurrentDoc)
{
    currentDoc_ = newCurrentDoc;
}

bool FsStatus::hasDocData() const
{
    return hasDocData_;
}

void FsStatus::setHasDocData(bool newHasDocData)
{
    hasDocData_ = newHasDocData;
}

bool FsStatus::cycleIsOpen() const
{
    return cycleIsOpen_;
}

void FsStatus::setCycleIsOpen(bool newCycleIsOpen)
{
    cycleIsOpen_ = newCycleIsOpen;
}

const fs::Warnings &FsStatus::warnings() const
{
    return warnings_;
}

void FsStatus::setWarnings(const fs::Warnings &newWarnings)
{
    warnings_ = newWarnings;
}

const QDateTime &FsStatus::lastDocDt() const
{
    return lastDocDt_;
}

void FsStatus::setLastDocDt(const QDateTime &newLastDocDt)
{
    lastDocDt_ = newLastDocDt;
}

const QString &FsStatus::fsNumber() const
{
    return fsNumber_;
}

void FsStatus::setFsNumber(const QString &newFsNumber)
{
    fsNumber_ = newFsNumber.trimmed();
}

quint32 FsStatus::lastFd() const
{
    return lastFd_;
}

void FsStatus::setLastFd(quint32 newLastFd)
{
    lastFd_ = newLastFd;
}

QVariantMap FsStatus::toMap() const
{
    QVariantMap res;
    if(phaseIsValid()) res["phase"] = static_cast<qint32>(phase_);
    if(currentDoc_ != fs::CurrentDocument::None) res["currentDoc"] = static_cast<qint32>(currentDoc_);
    if(hasDocData_) res["hasDocData"] = 1;
    if(cycleIsOpen_) res["cycleIsOpen"] = 1;
    if(warnings_ != fs::Warnings()) res["warnings"] = static_cast<qint32>(warnings_);
    if(lastDocDt_.isValid()) res["lastDocDt"] = DT2STR_(lastDocDt_);
    if(!fsNumber_.trimmed().isEmpty()) res["serial"] = fsNumber_.trimmed();
    if(lastFd_ > 0) res["lastFd"] = lastFd_;
    return res;
}

QVariantMap FsStatus::toExternalMap() const
{
    QVariantMap res;
    if(phaseIsValid()) res["phase"] = static_cast<qint32>(phase_);
    if(currentDoc_ != fs::CurrentDocument::None) res["currentDoc"] = static_cast<qint32>(currentDoc_);
    if(warnings_ != fs::Warnings()) res["warnings"] = static_cast<qint32>(warnings_);
    if(lastDocDt_.isValid()) res["lastDocDt"] = DT2STR_(lastDocDt_);
    if(!fsNumber_.trimmed().isEmpty()) res["serial"] = fsNumber_.trimmed();
    if(lastFd_ > 0) res["lastFd"] = lastFd_;


    return res;
}

void FsStatus::parseMap(const QVariantMap &map)
{
    phase_ = static_cast<fs::Phase>(map["phase"].toInt());
    if(!phaseIsValid()) phase_ = fs::Phase::Invalid;
    currentDoc_ = static_cast<fs::CurrentDocument>(map["currentDoc"].toInt());
    hasDocData_ = map.contains("hasDocData") && map["hasDocData"].toInt();
    cycleIsOpen_ = map.contains("cycleIsOpen") && map["cycleIsOpen"].toInt();
    warnings_ = fs::Warnings(map["warnings"].toInt());
    lastDocDt_ = map.contains("lastDocDt") ? STR2DT_(map["lastDocDt"].toString()): QDateTime();
    fsNumber_ = map["serial"].toString().trimmed();
    lastFd_ = map["lastFd"].toUInt();
}

FsStatus &FsStatus::operator =(const FsStatus &other) noexcept
{
    phase_ = other.phase_;
    currentDoc_ = other.currentDoc_;
    hasDocData_ = other.hasDocData_;
    cycleIsOpen_ = other.cycleIsOpen_;
    warnings_ = other.warnings_;
    lastDocDt_ = other.lastDocDt_;
    fsNumber_ = other.fsNumber_;
    lastFd_ = other.lastFd_;
    return *this;
}
FsStatus &FsStatus::operator =(FsStatus &&other) noexcept
{
    phase_ = other.phase_;
    currentDoc_ = other.currentDoc_;
    hasDocData_ = other.hasDocData_;
    cycleIsOpen_ = other.cycleIsOpen_;
    warnings_ = other.warnings_;
    lastDocDt_.swap(other.lastDocDt_);
    fsNumber_.swap(other.fsNumber_);
    lastFd_ = other.lastFd_;
    return *this;
}

bool FsStatus::operator ==(const FsStatus &other) const noexcept
{
   return phase_ == other.phase_ &&
           currentDoc_ == other.currentDoc_ &&
           hasDocData_ == other.hasDocData_ &&
           cycleIsOpen_ == other.cycleIsOpen_ &&
           warnings_ == other.warnings_ &&
           lastDocDt_ == other.lastDocDt_ &&
           fsNumber_ == other.fsNumber_ &&
           lastFd_ == other.lastFd_;
}

bool FsStatus::operator !=(const FsStatus &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------
FsLifeTime::FsLifeTime() noexcept
    : expiredDt_()
    , availableRegs_(0)
    , regsCount_(0)
    , expiredDays_(-1)
{

}

FsLifeTime::FsLifeTime(const QVariantMap &map) noexcept
    : expiredDt_()
    , availableRegs_(0)
    , regsCount_(0)
    , expiredDays_(-1)
{
    parseMap(map);
}

FsLifeTime::FsLifeTime(const FsLifeTime &other) noexcept
    : expiredDt_(other.expiredDt_)
    , availableRegs_(other.availableRegs_)
    , regsCount_(other.regsCount_)
    , expiredDays_(other.expiredDays_)
{

}

FsLifeTime::FsLifeTime(FsLifeTime &&other) noexcept
    : expiredDt_(other.expiredDt_)
    , availableRegs_(other.availableRegs_)
    , regsCount_(other.regsCount_)
    , expiredDays_(other.expiredDays_)
{

}

FsLifeTime::~FsLifeTime()
{

}

bool FsLifeTime::isValid() const
{
    return expiredDt_.isValid() && availableRegs_ > 0;
}

const QDate &FsLifeTime::expiredDt() const
{
    return expiredDt_;
}

void FsLifeTime::setExpiredDt(const QDate &newExpiredDt)
{
    expiredDt_ = newExpiredDt;
}

qint32 FsLifeTime::availableRegs() const
{
    return availableRegs_;
}

void FsLifeTime::setAvailableRegs(qint32 newAvailableRegs)
{
    availableRegs_ = newAvailableRegs;
}

qint32 FsLifeTime::regsCount() const
{
    return regsCount_;
}

void FsLifeTime::setRegsCount(qint32 newRegsCount)
{
    regsCount_ = newRegsCount;
}

qint32 FsLifeTime::expiredDays() const
{
    return expiredDays_;
}

void FsLifeTime::setExpiredDays(qint32 newExpiredDays)
{
    expiredDays_ = newExpiredDays;
}

QVariantMap FsLifeTime::toMap() const
{
    QVariantMap res;
    if(isValid())
    {
        res = {
            {"expiredDt", expiredDt_.toString(Qt::ISODate)},
            {"availableRegs", availableRegs_},
            {"regsCount", regsCount_},
            {"expiredDays", expiredDays_}
        };
    }
    return res;
}

QVariantMap FsLifeTime::toExternalMap() const
{
    return {
        {"expiredDt", expiredDt_.isValid() ? expiredDt_.toString(Qt::ISODate): QVariant()},
        {"availableRegs", availableRegs_},
        {"regsCount", regsCount_},
        {"expiredDays", expiredDays_>= 0 ? QVariant(expiredDays_) : QVariant()}
    };
}

void FsLifeTime::parseMap(const QVariantMap &map)
{
    expiredDt_ = QDate::fromString(map["expiredDt"].toString(), Qt::ISODate);
    availableRegs_ = map["availableRegs"].toInt();
    regsCount_ = map["regsCount"].toInt();
    expiredDays_ = map.contains("expiredDays") ? map["expiredDays"].toInt() : -1;
}

FsLifeTime &FsLifeTime::operator =(const FsLifeTime &other) noexcept
{
    expiredDt_ = other.expiredDt_;
    availableRegs_ = other.availableRegs_;
    regsCount_ = other.regsCount_;
    expiredDays_ = other.expiredDays_;
    return *this;
}
FsLifeTime &FsLifeTime::operator =(FsLifeTime &&other) noexcept
{
    expiredDt_ = other.expiredDt_;
    availableRegs_ = other.availableRegs_;
    regsCount_ = other.regsCount_;
    expiredDays_ = other.expiredDays_;
    return *this;
}
bool FsLifeTime::operator ==(const FsLifeTime &other) const noexcept
{
    return expiredDt_ == other.expiredDt_  &&
            availableRegs_ == other.availableRegs_ &&
            regsCount_ == other.regsCount_ &&
            expiredDays_ == other.expiredDays_;
}

bool FsLifeTime::operator !=(const FsLifeTime &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------
FsMemResourceInfo::FsMemResourceInfo() noexcept
    : availableDocs_(0)
    , freeBufSize_(0)
    , labelBufFree_(-1)
{

}

FsMemResourceInfo::FsMemResourceInfo(const QVariantMap &map)
    : availableDocs_(0)
    , freeBufSize_(0)
    , labelBufFree_(-1)
{
    parseMap(map);
}

quint32 FsMemResourceInfo::availableDocs() const
{
    return availableDocs_;
}

void FsMemResourceInfo::setAvailableDocs(quint32 newAvailableDocs)
{
    availableDocs_ = newAvailableDocs;
}

quint32 FsMemResourceInfo::freeBufSize() const
{
    return freeBufSize_;
}

void FsMemResourceInfo::setFreeBufSize(quint32 newFreeBufSize)
{
    freeBufSize_ = newFreeBufSize;
}

qint32 FsMemResourceInfo::labelBufFree() const
{
    return labelBufFree_;
}

void FsMemResourceInfo::setLabelBufFree(qint32 newLabelBufFree)
{
    labelBufFree_ = newLabelBufFree;
}

QVariantMap FsMemResourceInfo::toMap() const
{
    return {
        {"availableDocs", availableDocs_ },
        {"freeBufSize", freeBufSize_ },
        {"labelBufFree", labelBufFree_ < 0 ? QVariant() : QVariant(labelBufFree_)}

    };
}

void FsMemResourceInfo::parseMap(const QVariantMap &map)
{
    availableDocs_ = map["availableDocs"].toUInt();
    freeBufSize_ = map["freeBufSize"].toUInt();
    labelBufFree_ = map.contains("labelBufFree") ? map["labelBufFree"].toInt() : -1;
}

bool FsMemResourceInfo::operator ==(const FsMemResourceInfo &other) const
{
    return availableDocs_ == other.availableDocs_ &&
            freeBufSize_ == other.freeBufSize_ &&
            labelBufFree_ == other.labelBufFree_;
}

bool FsMemResourceInfo::operator !=(const FsMemResourceInfo &other) const
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

CycleStatus::CycleStatus()
    : opened_(false)
    , cycle_(-1)
    , receipt_(-1)
{

}

CycleStatus::CycleStatus(const QVariantMap &map)
    : opened_(false)
    , cycle_(-1)
    , receipt_(-1)
{
    parseMap(map);
}

CycleStatus::~CycleStatus()
{

}

bool CycleStatus::opened() const
{
    return opened_;
}

void CycleStatus::setOpened(bool newOpened)
{
    opened_ = newOpened;
}

qint32 CycleStatus::cycle() const
{
    return cycle_;
}

void CycleStatus::setCycle(qint32 newCycle)
{
    cycle_ = newCycle;
}

qint32 CycleStatus::receipt() const
{
    return receipt_;
}

void CycleStatus::setReceipt(qint32 newReceipt)
{
    receipt_ = newReceipt;
}

QVariantMap CycleStatus::toMap() const
{
    return QVariantMap{
        {"opened", opened_ ? 1 : 0},
        {"cycle", cycle_ < 0 ? QVariant() : QVariant(cycle_)},
        {"receipt", cycle_ < 0 ? QVariant() : QVariant(receipt_)}
    };
}

QVariantMap CycleStatus::toExternalMap() const
{
    return QVariantMap{
        {"opened", opened_ ? 1 : 0},
        {"cycle", cycle_ < 0 ? QVariant() : QVariant(cycle_)},
        {"receipt", cycle_ < 0 ? QVariant() : QVariant(receipt_)}
    };
}

void CycleStatus::parseMap(const QVariantMap &map)
{
    opened_ = map.contains("opened") && map["opened"].toInt();
    cycle_ = map.contains("cycle") ? map["cycle"].toInt(): -1;
    receipt_ = map.contains("receipt") ? map["receipt"].toInt(): -1;
}

bool CycleStatus::operator ==(const CycleStatus &other) const
{
    return opened_ == other.opened_ &&
            cycle_ == other.cycle_ &&
            receipt_ == other.receipt_;
}

bool CycleStatus::operator !=(const CycleStatus &other) const
{
    return  !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsTransportState::FsTransportState() noexcept
    : status_()
    , ofdMessageReading_(false)
    , messagesCount_(0)
    , firstDocNumber_(0)
    , firstDocDt_()
{

}

FsTransportState::FsTransportState(const QVariantMap &map) noexcept
    : status_()
    , ofdMessageReading_(false)
    , messagesCount_(0)
    , firstDocNumber_(0)
    , firstDocDt_()
{
    parseMap(map);
}

FsTransportState::FsTransportState(const FsTransportState &other) noexcept
    : status_(other.status_)
    , ofdMessageReading_(other.ofdMessageReading_)
    , messagesCount_(other.messagesCount_)
    , firstDocNumber_(other.firstDocNumber_)
    , firstDocDt_(other.firstDocDt_)

{

}

FsTransportState::FsTransportState(FsTransportState &&other) noexcept
    : status_(other.status_)
    , ofdMessageReading_(other.ofdMessageReading_)
    , messagesCount_(other.messagesCount_)
    , firstDocNumber_(other.firstDocNumber_)
    , firstDocDt_()
{
    firstDocDt_.swap(other.firstDocDt_);
}

FsTransportState::~FsTransportState()
{

}

const FsTransportState::Status &FsTransportState::status() const
{
    return status_;
}

void FsTransportState::setStatus(const Status &newStatus)
{
    status_ = newStatus;
}

bool FsTransportState::ofdConnected() const
{
    return status_.testFlag(StatusFlag::HasOfdConnection);
}

bool FsTransportState::hasMessageForOfd() const
{
    return status_.testFlag(StatusFlag::HasMessageForOfd);
}

bool FsTransportState::ofdMessageReading() const
{
    return ofdMessageReading_;
}

void FsTransportState::setOfdMessageReading(bool newOfdMessageReading)
{
    ofdMessageReading_ = newOfdMessageReading;
}

quint16 FsTransportState::messagesCount() const
{
    return messagesCount_;
}

void FsTransportState::setMessagesCount(quint16 newMessagesCount)
{
    messagesCount_ = newMessagesCount;
}

quint32 FsTransportState::firstDocNumber() const
{
    return firstDocNumber_;
}

void FsTransportState::setFirstDocNumber(quint32 newFirstDocNumber)
{
    firstDocNumber_ = newFirstDocNumber;
}

const QDateTime &FsTransportState::firstDocDt() const
{
    return firstDocDt_;
}

void FsTransportState::setFirstDocDt(const QDateTime &newFirstDocDt)
{
    firstDocDt_ = newFirstDocDt;
}

QVariantMap FsTransportState::toMap() const
{
    return {
        {"status", static_cast<qint32>(status_)},
        {"ofdMessageReading", ofdMessageReading_ ? 1 : 0},
        {"messagesCount", messagesCount_},
        {"firstDocNumber", firstDocNumber_},
        {"firstDocDt", DT2STR_(firstDocDt_)}
    };
}

QVariantMap FsTransportState::toExternalMap() const
{
    return {
        {"status", static_cast<qint32>(status_)},
        {"ofdMessageReading", ofdMessageReading_ },
        {"messagesCount", messagesCount_},
        {"firstDocNumber", firstDocNumber_},
        {"firstDocDt", firstDocDt_.isValid() ? QVariant(DT2STR_(firstDocDt_)) : QVariant()}
    };
}

void FsTransportState::parseMap(const QVariantMap &map)
{
    status_ = Status(map["status"].toUInt());
    ofdMessageReading_ = map.contains("ofdMessageReading") && map["ofdMessageReading"].toBool();
    messagesCount_ = map["messagesCount"].toUInt();
    firstDocNumber_ = map["firstDocNumber"].toUInt();
    firstDocDt_ = STR2DT_(map["firstDocNumber"].toString());
}

FsTransportState &FsTransportState::operator =(const FsTransportState &other) noexcept
{
    status_ = other.status_;
    ofdMessageReading_ = other.ofdMessageReading_;
    messagesCount_ = other.messagesCount_;
    firstDocNumber_ = other.firstDocNumber_;
    firstDocDt_ = other.firstDocDt_;
    return *this;
}
FsTransportState &FsTransportState::operator =(FsTransportState &&other) noexcept
{
    status_ = other.status_;
    ofdMessageReading_ = other.ofdMessageReading_;
    messagesCount_ = other.messagesCount_;
    firstDocNumber_ = other.firstDocNumber_;
    firstDocDt_ = other.firstDocDt_;
    return *this;
}
bool FsTransportState::operator ==(const FsTransportState &other) const noexcept
{
    return status_ == other.status_ &&
            ofdMessageReading_ == other.ofdMessageReading_ &&
            messagesCount_ == other.messagesCount_ &&
            firstDocNumber_ == other.firstDocNumber_ &&
            firstDocDt_ == other.firstDocDt_;
}

bool FsTransportState::operator !=(const FsTransportState &other) const noexcept
{
    return !(*this == other);
}
//--------------------------------------------------------------------------------------------------
LabelsTransportState::LabelsTransportState() noexcept
    : status_(Status::NoExchange)
    , notificationsCount_(0)
    , firstNotificationNumber_(0)
    , firstNotificationDt_()
    , memFullPercent_(0)
{

}

LabelsTransportState::LabelsTransportState(const QVariantMap &map) noexcept
    : status_(Status::NoExchange)
    , notificationsCount_(0)
    , firstNotificationNumber_(0)
    , firstNotificationDt_()
    , memFullPercent_(0)
{
    parseMap(map);
}

LabelsTransportState::LabelsTransportState(const LabelsTransportState &other) noexcept
    : status_(other.status_)
    , notificationsCount_(other.notificationsCount_)
    , firstNotificationNumber_(other.firstNotificationNumber_)
    , firstNotificationDt_(other.firstNotificationDt_)
    , memFullPercent_(other.memFullPercent_)

{

}

LabelsTransportState::LabelsTransportState(LabelsTransportState &&other) noexcept
    : status_(other.status_)
    , notificationsCount_(other.notificationsCount_)
    , firstNotificationNumber_(other.firstNotificationNumber_)
    , firstNotificationDt_()
    , memFullPercent_(other.memFullPercent_)
{
    firstNotificationDt_.swap(other.firstNotificationDt_);
}

LabelsTransportState::~LabelsTransportState()
{

}

LabelsTransportState::Status LabelsTransportState::status() const
{
    return status_;
}

void LabelsTransportState::setStatus(const Status &newStatus)
{
    status_ = newStatus;
}

quint16 LabelsTransportState::notificationCount() const
{
    return notificationsCount_;
}

void LabelsTransportState::setNotificationCount(quint16 newMessagesCount)
{
    notificationsCount_ = newMessagesCount;
}

quint32 LabelsTransportState::firstNotificationNumber() const
{
    return firstNotificationNumber_;
}

void LabelsTransportState::setFirstNotificationNumber(quint32 newFirstDocNumber)
{
    firstNotificationNumber_ = newFirstDocNumber;
}

const QDateTime &LabelsTransportState::firstNotificationDt() const
{
    return firstNotificationDt_;
}

void LabelsTransportState::setFirstNotificationDt(const QDateTime &newFirstDocDt)
{
    firstNotificationDt_= newFirstDocDt;
}

quint8 LabelsTransportState::memFullPercent() const
{
    return memFullPercent_;
}

void LabelsTransportState::setMemFullPercent(quint8 newMemFullPercent)
{
    memFullPercent_ = newMemFullPercent;
}

QVariantMap LabelsTransportState::toMap() const
{
    return {
        {"status", static_cast<qint32>(status_)},
        {"notSendedCount", notificationsCount_},
        {"firstNotification", firstNotificationNumber_},
        {"firstNotificationDt", DT2STR_(firstNotificationDt_)},
        {"memFullPercent", static_cast<qint32>(memFullPercent_)}
    };
}

QVariantMap LabelsTransportState::toExternalMap() const
{
    return {
        {"status", static_cast<qint32>(status_)},
        {"notSendedCount", notificationsCount_},
        {"firstNotification", firstNotificationNumber_},
        {"firstNotificationDt", firstNotificationDt_.isValid() ?
                        QVariant(DT2STR_(firstNotificationDt_)): QVariant()},
        {"memFullPercent", static_cast<qint32>(memFullPercent_)}
    };

}

void LabelsTransportState::parseMap(const QVariantMap &map)
{
    status_ = static_cast<Status>(map["status"].toInt());
    notificationsCount_ = map["notSendedCount"].toUInt();
    firstNotificationNumber_ = map["firstNotification"].toUInt();
    firstNotificationDt_ = STR2DT_(map["firstNotificationDt"].toString());
    memFullPercent_ = map["memFullPercent"].toUInt();
}

LabelsTransportState &LabelsTransportState::operator =(const LabelsTransportState &other) noexcept
{
    status_ = other.status_;
    notificationsCount_ = other.notificationsCount_;
    firstNotificationNumber_ = other.firstNotificationNumber_;
    firstNotificationDt_ = other.firstNotificationDt_;
    memFullPercent_ = other.memFullPercent_;
    return *this;
}
LabelsTransportState &LabelsTransportState::operator =(LabelsTransportState &&other) noexcept
{
    status_ = other.status_;
    notificationsCount_ = other.notificationsCount_;
    firstNotificationNumber_ = other.firstNotificationNumber_;
    firstNotificationDt_.swap(other.firstNotificationDt_);
    memFullPercent_ = other.memFullPercent_;
    return *this;
}

bool LabelsTransportState::operator ==(const LabelsTransportState &other) const noexcept
{
    return status_ == other.status_ &&
            notificationsCount_ == other.notificationsCount_ &&
            firstNotificationNumber_ == other.firstNotificationNumber_ &&
            firstNotificationDt_ == other.firstNotificationDt_ &&
            memFullPercent_ == other.memFullPercent_;


}

bool LabelsTransportState::operator !=(const LabelsTransportState &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------
BStatus::BStatus() noexcept
    : checkState_(CheckState::NoAnyLablesForCheck)
    , notificationState_(NotificationState::NoAnyNotification)
    , flags_()
    , resultsCount_(0)
    , notificatedCount_()
    , warning_(OverflowWarning::Less50)
    , notications_(0)

{

}

BStatus::BStatus(const BStatus &other) noexcept
    : checkState_(other.checkState_)
    , notificationState_(other.notificationState_)
    , flags_(other.flags_)
    , resultsCount_(other.resultsCount_)
    , notificatedCount_(other.notificatedCount_)
    , warning_(other.warning_)
    , notications_(other.notications_)
{

}


BStatus::BStatus(const QVariantMap &map) noexcept
    : checkState_(CheckState::NoAnyLablesForCheck)
    , notificationState_(NotificationState::NoAnyNotification)
    , flags_()
    , resultsCount_(0)
    , notificatedCount_()
    , warning_(OverflowWarning::Less50)
    , notications_(0)

{
    parseMap(map);
}

BStatus::~BStatus()
{

}

BStatus::CheckState BStatus::checkState() const
{
    return checkState_;
}

void BStatus::setCheckState(CheckState newCheckState)
{
    checkState_ = newCheckState;
}

BStatus::NotificationState BStatus::notificationState() const
{
    return notificationState_;
}

void BStatus::setNotificationState(NotificationState newNotificationState)
{
    notificationState_ = newNotificationState;
}

const BStatus::Flags &BStatus::flags() const
{
    return flags_;
}

void BStatus::setFlags(const Flags &newFlags)
{
    flags_ = newFlags;
}

quint8 BStatus::resultsCount() const
{
    return resultsCount_;
}

void BStatus::setResultsCount(quint8 newResultsCount)
{
    resultsCount_ = newResultsCount;
}

quint8 BStatus::notificatedCount() const
{
    return notificatedCount_;
}

void BStatus::setNotificatedCount(quint8 newNotificatedCount)
{
    notificatedCount_ = newNotificatedCount;
}

BStatus::OverflowWarning BStatus::warning() const
{
    return warning_;
}

void BStatus::setWarning(OverflowWarning newWarning)
{
    warning_ = newWarning;
}

quint16 BStatus::notications() const
{
    return notications_;
}

void BStatus::setNotications(quint16 newNotications)
{
    notications_ = newNotications;
}

QVariantMap BStatus::toMap() const
{
    return {
        {"checkState", static_cast<qint32>(checkState_)},
        {"notificationState", static_cast<qint32>(notificationState_)},
        {"flags", static_cast<qint32>(flags_)},
        {"resultsCount", resultsCount_},
        {"notificatedCount", notificatedCount_},
        {"warning", static_cast<qint32>(warning_)},
        {"notications", notications_}
    };
}

QVariantMap BStatus::toExternalMap() const
{
    return toMap();
}

void BStatus::parseMap(const QVariantMap &map)
{
    checkState_        = static_cast<CheckState>(map["checkState"].toUInt());
    notificationState_ = static_cast<NotificationState>(map["notificationState"].toUInt());
    flags_             = Flags(map["flags"].toUInt());
    resultsCount_      = map["resultsCount"].toUInt();
    notificatedCount_  = map["notificatedCount"].toUInt();
    warning_           = static_cast<OverflowWarning>(map["warning"].toUInt());
    notications_       = map["notications"].toUInt();
}

BStatus &BStatus::operator =(const BStatus &other) noexcept
{
    checkState_         = other.checkState_;
    notificationState_  = other.notificationState_;
    flags_              = other.flags_;
    resultsCount_       = other.resultsCount_;
    notificatedCount_   = other.notificatedCount_;
    warning_            = other.warning_;
    notications_        = other.notications_;
    return *this;
}


bool BStatus::operator ==(const BStatus &other) const noexcept
{
    return checkState_ == other.checkState_ &&
            notificationState_ == other.notificationState_ &&
            flags_ == other.flags_ &&
            resultsCount_ == other.resultsCount_ &&
            notificatedCount_ == other.notificatedCount_ &&
            warning_ == other.warning_ &&
            notications_ == other.notications_;
}

bool BStatus::operator !=(const BStatus &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------
const FsFullStatus::CleanFlags FsFullStatus::CLEAN_ALL (
            FsFullStatus::CleanFlag::Status         |
            FsFullStatus::CleanFlag::LifeTime       |
            FsFullStatus::CleanFlag::Version        |
            FsFullStatus::CleanFlag::Release        |
            FsFullStatus::CleanFlag::Serial         |
            FsFullStatus::CleanFlag::Formats        |
            FsFullStatus::CleanFlag::Resource       |
            FsFullStatus::CleanFlag::Cycle          |
            FsFullStatus::CleanFlag::Transport      |
            FsFullStatus::CleanFlag::CycleDates     |
            FsFullStatus::CleanFlag::LabelsTransport|
            FsFullStatus::CleanFlag::Error);
const FsFullStatus::CleanFlags FsFullStatus::CLEAN_NONE;
const FsFullStatus::CleanFlags FsFullStatus::CLEAN_CYCLE (
            FsFullStatus::CleanFlag::Status         |
            FsFullStatus::CleanFlag::LifeTime       |
            FsFullStatus::CleanFlag::Resource       |
            FsFullStatus::CleanFlag::Cycle          |
            FsFullStatus::CleanFlag::CycleDates     |
            FsFullStatus::CleanFlag::LabelsTransport|
            FsFullStatus::CleanFlag::Transport      );
const FsFullStatus::CleanFlags FsFullStatus::CLEAN_TRANSPORT (
            FsFullStatus::CleanFlag::Status         |
            FsFullStatus::CleanFlag::Transport      );
const FsFullStatus::CleanFlags FsFullStatus::CLEAN_LABELS_TRANSPORT (
            FsFullStatus::CleanFlag::LabelsTransport      );

FsFullStatus::FsFullStatus() noexcept
    : status_()
    , lifeTime_()
    , version_()
    , isRelease_(true)
    , release_()
    , fsNumber_()
    , currentFfd_(fs::FFD::Invalid)
    , supportedFfd_(fs::FFD::Invalid)
    , memResource_()
    , cycle_()
    , transport_()
    , labelsTransport_()
    , cashboxSerial_()
    , cycleOpenDt_()
    , cycleCloseDt_()
    , error_(CoreApiConst::ErrorCode::EmptyFsStatus)
{

}

FsFullStatus::FsFullStatus(const QVariantMap &map) noexcept
    : status_()
    , lifeTime_()
    , version_()
    , isRelease_(true)
    , release_()
    , fsNumber_()
    , currentFfd_(fs::FFD::Invalid)
    , supportedFfd_(fs::FFD::Invalid)
    , memResource_()
    , cycle_()
    , transport_()
    , labelsTransport_()
    , cashboxSerial_()
    , cycleOpenDt_()
    , cycleCloseDt_()
    , error_(CoreApiConst::ErrorCode::EmptyFsStatus)
{
    parseMap(map);
}

FsFullStatus::FsFullStatus(const FsFullStatus &other) noexcept
    : status_(other.status_)
    , lifeTime_(other.lifeTime_)
    , version_(other.version_)
    , isRelease_(other.isRelease_)
    , release_(other.release_)
    , fsNumber_(other.fsNumber_)
    , currentFfd_(other.currentFfd_)
    , supportedFfd_(other.supportedFfd_)
    , memResource_(other.memResource_)
    , cycle_(other.cycle_)
    , transport_(other.transport_)
    , labelsTransport_(other.labelsTransport_)
    , cashboxSerial_(other.cashboxSerial_)
    , cycleOpenDt_(other.cycleOpenDt_)
    , cycleCloseDt_(other.cycleCloseDt_)
    , error_(other.error_)

{

}

FsFullStatus::FsFullStatus(FsFullStatus &&other) noexcept
    : status_()
    , lifeTime_()
    , version_()
    , isRelease_(other.isRelease_)
    , release_()
    , fsNumber_()
    , currentFfd_(other.currentFfd_)
    , supportedFfd_(other.supportedFfd_)
    , memResource_()
    , cycle_()
    , transport_()
    , labelsTransport_()
    , cashboxSerial_()
    , cycleOpenDt_()
    , cycleCloseDt_()
    , error_(other.error_)


{
    status_.swap(other.status_);
    lifeTime_.swap(other.lifeTime_);
    version_.swap(other.version_);
    release_.swap(other.release_);
    fsNumber_.swap(other.fsNumber_);
    memResource_.swap(other.memResource_);
    cycle_.swap(other.cycle_);
    transport_.swap(other.transport_);
    labelsTransport_.swap(other.labelsTransport_);
    cashboxSerial_.swap(other.cashboxSerial_);
    cycleOpenDt_.swap(other.cycleOpenDt_);
    cycleCloseDt_.swap(other.cycleCloseDt_);
}

FsFullStatus::~FsFullStatus()
{

}

bool FsFullStatus::isFiscalized() const
{
    return status_.has_value() && status_.value().phase() >= fs::Phase::Fiscalized;
}

bool FsFullStatus::archiveIsClosed() const
{
    return status_.has_value() && status_.value().phase() > fs::Phase::Fiscalized;
}

bool FsFullStatus::cycleIsOpen() const
{
    return status_.has_value() && status_.value().cycleIsOpen();
}

bool FsFullStatus::cycleIsOver() const
{
    return cycleIsOpen() && cycleOpenDt_.secsTo(QDateTime::currentDateTime())
            > (23 * 3600 + 55 * 60);
}

bool FsFullStatus::hasStatus() const
{
    return status_.has_value();
}

FsStatus FsFullStatus::status() const
{
    return status_.value_or(FsStatus());
}

void FsFullStatus::setStatus(const FsStatus &status)
{
    if(status.isValid()) status_ = status;
    else status_.reset();
}

void FsFullStatus::setStatus(const std::optional<FsStatus> &status)
{
    status_ = status;
}

void FsFullStatus::cleanStatus()
{
    status_.reset();
}

bool FsFullStatus::hasLifeTime() const
{
    return lifeTime_.has_value();
}

FsLifeTime FsFullStatus::lifeTime() const
{
    return lifeTime_.value_or(FsLifeTime());
}

void FsFullStatus::setLifeTime(const FsLifeTime &status)
{
    if(status.isValid()) lifeTime_ = status;
    else lifeTime_.reset();
}

void FsFullStatus::setLifeTime(const std::optional<FsLifeTime> &status)
{
    lifeTime_ = status;
}

void FsFullStatus::cleanLifeTime()
{
    lifeTime_.reset();
}

const QString &FsFullStatus::version() const
{
    return version_;
}

void FsFullStatus::setVersion(const QString &newVersion)
{
    version_ = newVersion.trimmed();
}

bool FsFullStatus::isRelease() const
{
    return isRelease_;
}

void FsFullStatus::setIsRelease(bool newIsRelease)
{
    isRelease_ = newIsRelease;
}

QString FsFullStatus::release() const
{
    return release_;
}

void FsFullStatus::setRelease(const QString &newRelease)
{
    release_ = newRelease.trimmed();
}

QString FsFullStatus::fsNumber() const
{
    return fsNumber_.isEmpty() ? status().fsNumber() : fsNumber_;
}

void FsFullStatus::setFsNumber(const QString &newFsNumber)
{
    fsNumber_ = newFsNumber;
}

fs::FFD FsFullStatus::currentFfd() const
{
    return currentFfd_;
}

void FsFullStatus::setCurrentFfd(fs::FFD newCurrentFfd)
{
    currentFfd_ = newCurrentFfd;
}

fs::FFD FsFullStatus::supportedFfd() const
{
    return supportedFfd_;
}

void FsFullStatus::setSupportedFfd(fs::FFD newSupportedFfd)
{
    supportedFfd_ = newSupportedFfd;
}

bool FsFullStatus::hasMemResource() const
{
    return memResource_.has_value();
}

FsMemResourceInfo FsFullStatus::memResource() const
{
    return memResource_.value_or(FsMemResourceInfo());
}

void FsFullStatus::setMemResource(const FsMemResourceInfo &res)
{
    memResource_ = res;

}

void FsFullStatus::setMemResource(const std::optional<FsMemResourceInfo> &res)
{
    memResource_ = res;
}

void FsFullStatus::cleanMemResource()
{
    memResource_.reset();
}

bool FsFullStatus::hasCycle() const
{
    return cycle_.has_value();
}

CycleStatus FsFullStatus::cycle() const
{
    return cycle_.value_or(CycleStatus());
}

void FsFullStatus::setCycle(const CycleStatus &res)
{
    cycle_.reset();
    if(res.cycle() >= 0) cycle_ = res;
}

void FsFullStatus::setCycle(const std::optional<CycleStatus> &res)
{
    cycle_ = res;
}

void FsFullStatus::cleanCycle()
{
    cycle_.reset();
}

bool FsFullStatus::hasTranspot() const
{
    return transport_.has_value();
}

FsTransportState FsFullStatus::transport() const
{
    return transport_.value_or(FsTransportState());
}

void FsFullStatus::setTransport(const FsTransportState &res)
{
    transport_ = res;
}

void FsFullStatus::setTransport(const std::optional<FsTransportState> &res)
{
    transport_ = res;
}

void FsFullStatus::cleanTransport()
{
    transport_.reset();
}

bool FsFullStatus::hasLabelsTranspot() const
{
    return labelsTransport_.has_value();
}

LabelsTransportState FsFullStatus::labelsTransport() const
{
    return labelsTransport_.value_or(LabelsTransportState());
}

void FsFullStatus::setLabelsTransport(std::optional<LabelsTransportState> newLabelsTransport)
{
    labelsTransport_ = newLabelsTransport;
}

void FsFullStatus::setLabelsTransport(const LabelsTransportState &state)
{
    labelsTransport_ = state;
}

void FsFullStatus::cleanLabelsTransport()
{
    labelsTransport_.reset();
}

const QString &FsFullStatus::cashboxSerial() const
{
    return cashboxSerial_;
}

void FsFullStatus::setCashboxSerial(const QString &newCashboxSerial)
{
    cashboxSerial_ = newCashboxSerial;
}

const QDateTime &FsFullStatus::cycleOpenDt() const
{
    return cycleOpenDt_;
}

void FsFullStatus::setCycleOpenDt(const QDateTime &newCycleOpenDt)
{
    cycleOpenDt_ = newCycleOpenDt;
}

const QDateTime &FsFullStatus::cycleCloseDt() const
{
    return cycleCloseDt_;
}

void FsFullStatus::setCycleCloseDt(const QDateTime &newCycleCloseDt)
{
    cycleCloseDt_ = newCycleCloseDt;
}

CoreApiConst::ErrorCode FsFullStatus::error() const
{
    return error_;
}

void FsFullStatus::setError(const CoreApiConst::ErrorCode &err)
{
    error_ = err;
}

bool FsFullStatus::hasError() const
{
    return error_ != CoreApiConst::ErrorCode::Ok;
}



QVariantMap FsFullStatus::toMap() const
{
    QVariantMap res;
    if(status_.has_value()) res["status"] = status_.value().toMap();
    if(lifeTime_.has_value()) res["lifeTime"] = lifeTime_.value().toMap();
    if(!version_.isEmpty())
    {
        res["version"] = version_;
        res["isDebug"] = isRelease_ ? 0 : 1;
    }
    if(!release_.isEmpty()) res["release"] = release_;
    if(!fsNumber_.isEmpty()) res["serial"] = fsNumber_;
    res["currentFfd"] = static_cast<qint32>(currentFfd_);
    res["supportedFfd"] = static_cast<qint32>(supportedFfd_);
    if(memResource_.has_value())
    {
        res["memResource"] = memResource_.value().toMap();
    }
    if(cycle_.has_value())
    {
        res["cycle"] = cycle_.value().toMap();
    }
    if(transport_.has_value())
    {
        res["transport"] = transport_.value().toMap();
    }
    if(labelsTransport_.has_value())
    {
        res["notifications"] = labelsTransport_.value().toMap();
    }
    if(!cashboxSerial_.isEmpty())
    {
        res["cashboxSerial"] = cashboxSerial_;
    }
    if(error_ != CoreApiConst::ErrorCode::Ok)
    {
        res["error"] = static_cast<qint32>(error_);
    }
    if(cycleOpenDt_.isValid()) res["cycleOpenDt"] = DT2STR_(cycleOpenDt_);
    if(cycleCloseDt_.isValid()) res["cycleCloseDt"] = DT2STR_(cycleCloseDt_);
    return res;
}

QVariantMap FsFullStatus::toExternalMap() const
{
    return QVariantMap {
        {"status", status().toExternalMap()},
        {"lifeTime", lifeTime().toExternalMap()},
        {"version", version_.isEmpty() ? QVariant() : QVariant(version_)},
        {"isDebug", version_.isEmpty() ? QVariant() : QVariant(!isRelease_)},
        {"release", release_.isEmpty() ? QVariant() : QVariant(release_)},
        {"currentFfd", currentFfd_ == fs::FFD::Invalid ?
                                    QVariant() : QVariant(fs::ffd2String(currentFfd_))},
        {"supportedFfd", supportedFfd_ == fs::FFD::Invalid ?
                                    QVariant() :
                                    QVariant(fs::ffd2String(supportedFfd_))},
        {"availableDocs", memResource_.has_value() ?
                                            QVariant(memResource_.value().availableDocs()) :
                                            QVariant()},
        {"freeBufSize", memResource_.has_value() ?
                                                QVariant(memResource_.value().freeBufSize()) :
                                                QVariant()},
        {"labelBufFree", memResource_.has_value()  && memResource_.value().labelBufFree() >= 0 ?
                                                    QVariant(memResource_.value().labelBufFree()) :
                                                    QVariant()},

        {"cycle", cycle_.has_value() && cycle_.value().cycle() >= 0 ?
                                                        QVariant(cycle_.value().cycle()) :
                                                        QVariant()
                                        },
        {"transport", transport_.has_value() ? QVariant(transport_.value().toExternalMap()) :
                                                        QVariant() },
        {"notifications", labelsTransport_.has_value() ? QVariant(labelsTransport_.value().toExternalMap()) :
                                                                                                QVariant() },
        {"receipt", cycle_.has_value()  && cycle_.value().receipt() >= 0 ?
                                                    QVariant(cycle_.value().receipt()) :
                                                    QVariant()},
        {"error", error_ == CoreApiConst::ErrorCode::Ok ? QVariant() :
                                                          QVariant(static_cast<qint32>(error_))},
        {"cycleIsOpen", cycleIsOpen()},
        {"cycleOpenDt", cycleOpenDt_.isValid() ? QVariant(DT2STR_(cycleOpenDt_)): QVariant()},
        {"cycleCloseDt", cycleCloseDt_.isValid() ? QVariant(DT2STR_(cycleCloseDt_)): QVariant()}
    };
}

void FsFullStatus::parseMap(const QVariantMap &map)
{
    clean(CLEAN_ALL);
    if(map.contains("status"))
    {
        status_ = FsStatus(map["status"].toMap());
    }
    if(map.contains("lifeTime"))
    {
        lifeTime_ = FsLifeTime(map["lifeTime"].toMap());
    }
    if(map.contains("version")) version_ = map["version"].toString().trimmed();
    isRelease_ = !map.contains("isDebug") || !map["isDebug"].toInt();
    if(map.contains("release"))release_ = map["release"].toString().trimmed();
    currentFfd_ = static_cast<fs::FFD>(map["currentFfd"].toInt());
    fs::checkFfd(currentFfd_);
    supportedFfd_ = static_cast<fs::FFD>(map["supportedFfd"].toInt());
    fs::checkFfd(supportedFfd_);
    if(map.contains("memResource"))
    {
        memResource_ = FsMemResourceInfo(map["memResource"].toMap());
    }
    if(map.contains("cycle"))cycle_ = CycleStatus(map["cycle"].toMap());
    if(map.contains("transport"))transport_ = FsTransportState(map["transport"].toMap());
    if(map.contains("notifications")) labelsTransport_ = LabelsTransportState(map["notifications"].toMap());
    cashboxSerial_ = map["cashboxSerial"].toString().trimmed();
    if(map.contains("error"))error_ = static_cast<CoreApiConst::ErrorCode>(map["error"].toInt());
    else error_= CoreApiConst::ErrorCode::Ok;

    cycleOpenDt_ = STR2DT_(map["cycleOpenDt"].toString());
    cycleCloseDt_ = STR2DT_(map["cycleCloseDt"].toString());
}

void FsFullStatus::clean(CleanFlags flags)
{
    if(flags.testFlag(CleanFlag::Status)) status_.reset();
    if(flags.testFlag(CleanFlag::LifeTime))lifeTime_.reset();
    if(flags.testFlag(CleanFlag::Version))
    {
        version_.clear();
        isRelease_ = true;
    }
    if(flags.testFlag(CleanFlag::Release)) release_.clear();
    if(flags.testFlag(CleanFlag::Serial))
    {
        fsNumber_.clear();
        cashboxSerial_.clear();
    }
    if(flags.testFlag(CleanFlag::Formats))
    {
        currentFfd_ = fs::FFD::Invalid;
        supportedFfd_ = fs::FFD::Invalid;
    }
    if(flags.testFlag(CleanFlag::Resource))memResource_.reset();
    if(flags.testFlag(CleanFlag::Cycle))cycle_.reset();
    if(flags.testFlag(CleanFlag::Transport)) transport_.reset();
    if(flags.testFlag(CleanFlag::LabelsTransport)) labelsTransport_.reset();
    if(flags.testFlag(CleanFlag::Error))error_ = CoreApiConst::ErrorCode::EmptyFsStatus;
    if(flags.testFlag(CleanFlag::CycleDates))
    {
        cycleOpenDt_ = QDateTime();
        cycleCloseDt_ = QDateTime();
    }
}

QStringList FsFullStatus::toHtml() const
{
    QStringList res;
    res << QStringLiteral("<p align=\"center\"><b>Состояние ФН</b></p>");
    res << QStringLiteral("<table align=\"center\" width=100%><tbody>")
        << QStringLiteral("<tr><td>ФН:</td><td align=\"right\">%1</td></tr>")
           .arg(status().fsNumber())
        << QStringLiteral("<tr><td>Версия:</td><td align=\"right\">%1%2</td></tr>")
               .arg(version_).arg(isRelease_ ? QString() : QStringLiteral(" (МГМ)"))
        << QStringLiteral("<tr><td>Исполнение:</td><td align=\"right\">%1</td></tr>")
               .arg(release_)
        << QStringLiteral("<tr><td>Поддерживает ФФД:</td><td align=\"right\">%1</td></tr>")
               .arg(fs::ffd2String(supportedFfd_));

    if(status_.has_value())
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>Фаза:</td><td align=\"right\">%1</td></tr>")
               .arg(static_cast<qint32>(status().phase()));

        QStringList ws = fs::warningsToString(status_.value().warnings());
        if(!ws.isEmpty())
        {
            for(const QString &s: ws)
            {
                res << QStringLiteral("<tr><td></td><td align=\"right\">%1</td></tr>")
                       .arg(s);
            }
        }
        if(status_.value().lastFd() > 0)
        {
            res << QStringLiteral("<tr><td>Посл. док:</td><td align=\"right\">%1 от %2</td></tr>")
                   .arg(status_.value().lastFd())
                   .arg(status_.value().lastDocDt().toString("dd.MM.yy hh:mm"));

        }
    }
    if(lifeTime_.has_value() && isFiscalized())
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>ФН действ. до</td><td align=\"right\">%1 (%2 дней)</td></tr>")
               .arg(lifeTime_.value().expiredDt().toString("dd.MM.yyyy"))
               .arg(lifeTime_.value().expiredDays())
            << QStringLiteral("<tr><td>Регистраций:</td><td align=\"right\">%1</td></tr>")
               .arg(lifeTime_.value().regsCount())
            << QStringLiteral("<tr><td>Доступно:</td><td align=\"right\">%1</td></tr>")
               .arg(lifeTime_.value().availableRegs());
    }
    if(memResource_.has_value() && isFiscalized())
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>Доступно документов:</td><td align=\"right\">%1</td></tr>")
                   .arg(memResource_.value().availableDocs())
            << QStringLiteral("<tr><td>Ресурс 30-дн. хранения:</td><td align=\"right\">%1 Кб</td></tr>")
                   .arg(memResource_.value().freeBufSize())
            << QStringLiteral("<tr><td>Ресурс уведомлений:</td><td align=\"right\">%1%</td></tr>")
                   .arg(memResource_.value().labelBufFree());

    }
    if(cycle_.has_value() && isFiscalized() && cycle_.value().cycle() > 0)
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>Смена:</td><td align=\"right\">%1</td></tr>")
                   .arg(cycle_.value().cycle())
            << QStringLiteral("<tr><td>Открыта:</td><td align=\"right\">%1</td></tr>")
                   .arg(cycleOpenDt_.toString("dd.MM.yyyy hh:mm"));
        if(!cycle_.value().opened())
        {
            res << QStringLiteral("<tr><td>Закрыта:</td><td align=\"right\">%1</td></tr>")
                   .arg(cycleCloseDt_.toString("dd.MM.yyyy hh:mm"));
        }

    }
    if(transport_.has_value() && isFiscalized())
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>Не отправлено документов:</td><td align=\"right\">%1</td></tr>")
                   .arg(transport_.value().messagesCount());
        if(transport_.value().messagesCount() > 0)
        {
            res << QStringLiteral("<tr><td>С ФД:</td><td align=\"right\">%1</td></tr>")
                       .arg(transport_.value().firstDocNumber())
                << QStringLiteral("<tr><td>От:</td><td align=\"right\">%1</td></tr>")
                       .arg(transport_.value().firstDocDt().toString("dd.MM.yyyy hh:mm"));
        }
    }
    if(labelsTransport_.has_value() && isFiscalized())
    {
        res << QStringLiteral("<tr><td> </td><td align=\"right\"> </td></tr>")
            << QStringLiteral("<tr><td>Не отправлено уведомлений:</td><td align=\"right\">%1</td></tr>")
                   .arg(labelsTransport_.value().notificationCount());
        if(labelsTransport_.value().notificationCount() > 0)
        {
            res << QStringLiteral("<tr><td>С №:</td><td align=\"right\">%1</td></tr>")
                       .arg(labelsTransport_.value().firstNotificationNumber())
                << QStringLiteral("<tr><td>От:</td><td align=\"right\">%1</td></tr>")
                       .arg(labelsTransport_.value().firstNotificationDt().toString("dd.MM.yyyy hh:mm"));
        }
    }
    if(error_ != CoreApiConst::ErrorCode::Ok)
    {
        res << QStringLiteral("<tr><td><font color=\"red\">Ошибка получения статуса:</font></td>"
                              "<td align=\"right\"><font color=\"red\">%1</font></td></tr>")
                   .arg(CoreApiConst::defaultErrorMsg(error_));
    }
    res << "</tbody></table><hr>";


    return res;
}

FsFullStatus &FsFullStatus::operator =(const FsFullStatus &other) noexcept
{
    status_ = other.status_;
    lifeTime_ = other.lifeTime_;
    version_ = other.version_;
    isRelease_ = other.isRelease_;
    release_ = other.release_;
    fsNumber_ = other.fsNumber_;
    currentFfd_ = other.currentFfd_;
    supportedFfd_ = other.supportedFfd_;
    memResource_ = other.memResource_;
    cycle_ = other.cycle_;
    transport_ = other.transport_;
    labelsTransport_ = other.labelsTransport_;
    cashboxSerial_ = other.cashboxSerial_;
    cycleOpenDt_ = other.cycleOpenDt_;
    cycleCloseDt_ = other.cycleCloseDt_;
    error_ = other.error_;
    return *this;
}

FsFullStatus &FsFullStatus::operator =(FsFullStatus &&other) noexcept
{
    status_.swap(other.status_);
    lifeTime_.swap(other.lifeTime_);
    version_.swap(other.version_);
    isRelease_ = other.isRelease_;
    release_.swap(other.release_);
    fsNumber_.swap(other.fsNumber_);
    currentFfd_ = other.currentFfd_;
    supportedFfd_ = other.supportedFfd_;
    memResource_.swap(other.memResource_);
    cycle_.swap(other.cycle_);
    transport_.swap(other.transport_);
    labelsTransport_.swap(other.labelsTransport_);
    cashboxSerial_.swap(other.cashboxSerial_);
    cycleOpenDt_.swap(other.cycleOpenDt_);
    cycleCloseDt_.swap(other.cycleCloseDt_);

    error_ = other.error_;
    return *this;
}

bool FsFullStatus::operator ==(const FsFullStatus &other) const noexcept
{
    return status_ == other.status_ &&
            lifeTime_ == other.lifeTime_ &&
            version_ == other.version_ &&
            isRelease_ == other.isRelease_ &&
            release_ == other.release_ &&
            fsNumber_ == other.fsNumber_ &&
            currentFfd_ == other.currentFfd_ &&
            supportedFfd_ == other.supportedFfd_ &&
            memResource_ == other.memResource_ &&
            cycle_ == other.cycle_ &&
            transport_ == other.transport_ &&
            labelsTransport_ == other.labelsTransport_ &&
            cashboxSerial_ == other.cashboxSerial_ &&
            cycleOpenDt_ == other.cycleOpenDt_ &&
            cycleCloseDt_ == other.cycleCloseDt_ &&
            error_ == other.error_;

}

bool FsFullStatus::operator !=(const FsFullStatus &other) const noexcept
{
    return !(*this == other);
}


//--------------------------------------------------------------------------------------------------

OfflineDSessionStatus::OfflineDSessionStatus() noexcept
    : notSendedCount_(0)
    , fistNotSended_(0)
    , sessionNotSendedCount_(0)
    , fistSessionNotSended_(0)
{

}

OfflineDSessionStatus::OfflineDSessionStatus(const QVariantMap &map) noexcept
    : notSendedCount_(0)
    , fistNotSended_(0)
    , sessionNotSendedCount_(0)
    , fistSessionNotSended_(0)
{
    parseMap(map);
}

OfflineDSessionStatus::~OfflineDSessionStatus()
{

}

quint16 OfflineDSessionStatus::notSendedCount() const
{
    return notSendedCount_;
}

void OfflineDSessionStatus::setNotSendedCount(quint16 newNotSendedCount)
{
    notSendedCount_ = newNotSendedCount;
}

quint32 OfflineDSessionStatus::fistNotSended() const
{
    return fistNotSended_;
}

void OfflineDSessionStatus::setFistNotSended(quint32 newFistNotSended)
{
    fistNotSended_ = newFistNotSended;
}

quint16 OfflineDSessionStatus::sessionNotSendedCount() const
{
    return sessionNotSendedCount_;
}

void OfflineDSessionStatus::setSessionNotSendedCount(quint16 newSessionNotSendedCount)
{
    sessionNotSendedCount_ = newSessionNotSendedCount;
}

quint32 OfflineDSessionStatus::fistSessionNotSended() const
{
    return fistSessionNotSended_;
}

void OfflineDSessionStatus::setFistSessionNotSended(quint32 newFistSessionNotSended)
{
    fistSessionNotSended_ = newFistSessionNotSended;
}

QVariantMap OfflineDSessionStatus::toMap() const
{
    return {
        {"notSendedCount", notSendedCount_},
        {"fistNotSended", fistNotSended_ },
        {"sessionNotSendedCount", sessionNotSendedCount_},
        {"fistSessionNotSended", fistSessionNotSended_}
    };
}

void OfflineDSessionStatus::parseMap(const QVariantMap &map)
{
    notSendedCount_          = map["notSendedCount"].toUInt();
    fistNotSended_           = map["fistNotSended"].toUInt();
    sessionNotSendedCount_   = map["sessionNotSendedCount"].toUInt();
    fistSessionNotSended_    = map["fistSessionNotSended"].toUInt();
}

bool OfflineDSessionStatus::parseFs(const QByteArray &data)
{
    if(data.size() != 12) return false;
    QDataStream ds(data);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> notSendedCount_
            >> fistNotSended_
            >> sessionNotSendedCount_
            >> fistSessionNotSended_ ;
    return true;
}

bool OfflineDSessionStatus::operator ==(const OfflineDSessionStatus &other) const noexcept
{
    return  notSendedCount_ == other.notSendedCount_ &&
            fistNotSended_ == other.fistNotSended_ &&
            sessionNotSendedCount_ == other.sessionNotSendedCount_ &&
            fistSessionNotSended_ == other.fistSessionNotSended_;

}

bool OfflineDSessionStatus::operator !=(const OfflineDSessionStatus &other) const noexcept
{
    return !(*this == other);
}
