#ifndef FSFULLSTATUS_H
#define FSFULLSTATUS_H

#include "coreapiconst.h"

#include <QVariant>
#include <QVariantMap>
#include <QString>
#include <QFlags>
#include <QDateTime>
#include <optional>

namespace fs {
    extern const QByteArray CODEC_NAME;
    enum class Phase: quint8
    {
        Invalid                         = 0x7F,
        Empty                           = 0x01,
        Fiscalized                      = 0x03,
        ArchiveClosed                   = 0x07,
        Finished                        = 0x0F
    };
    enum class CurrentDocument: quint8
    {
        None                            = 0x00u,//h – Нет открытого документа
        RegDoc                          = 0x01u,//h – Отчёт о регистрации
        OpenCycleDoc                    = 0x02u,//h – Отчёт об открытии смены
        Receipt                         = 0x04u,//h – Кассовый чек
        CloseCycleDoc                   = 0x08u,//h – Отчёт о закрытии смены
        ColseFsDoc                      = 0x10u,//h – Отчёт о закрытии фискального накопителя
        FOSADoc                         = 0x11u,//h – Бланк строкой отчетности
        NewFsRegDoc                     = 0x12u,//h - Отчет об изменении параметров регистрации в связи с заменой ФН
        ReRegDoc                        = 0x13u,//h – Отчет об изменении параметров регистрации
        CorrectionReceipt               = 0x14u,//h – Кассовый чек коррекции
        FOSACorrectionDoc               = 0x15u,//h – БСО коррекции
        CalcReport                      = 0x17u //h – Отчет о текущем состояниирасчетов
    };

    enum class Warning: quint8
    {
        NeedFsChange                    = 0x01u,
        NeedFsChange30                  = 0x02u,
        FsMemoryOverflow                = 0x04u,
        OfdTimedOut                     = 0x08u,
        OfdDocumentsError               = 0x10u,
        NeedKktConfiguration            = 0x20u,
        OfdIsClosed                     = 0x40u
    };
    Q_DECLARE_FLAGS(Warnings, Warning)

    inline QStringList warningsToString(Warnings w) {
        QStringList sl;
        if(w.testFlag(Warning::NeedFsChange        )) sl << "РЕСУРС ФН МЕНЕЕ 3 ДНЕЙ";
        if(w.testFlag(Warning::NeedFsChange30      )) sl << "РЕСУРС ФН МЕНЕЕ 30 ДНЕЙ";
        if(w.testFlag(Warning::FsMemoryOverflow    )) sl << "ПАМЯТЬ ФН ЗАПОЛНЕНА";
        if(w.testFlag(Warning::OfdTimedOut         )) sl << "ОФД НЕ ОТВЕЧАЕТ";
//        if(w.testFlag(Warning::OfdDocumentsError   )) sl << "";
//        if(w.testFlag(Warning::NeedKktConfiguration)) sl << "";
//        if(w.testFlag(Warning::OfdIsClosed         )) sl << "";
        return sl;
    }

    enum class FFD:quint8
    {
        Invalid                         = 0x00,
        FFD1_0                          = 0x01,
        FFD1_05                         = 0x02,
        FFD1_1                          = 0x03,
        FFD1_2                          = 0x04
    };
    inline QString ffd2String(FFD ffd) {
        switch(ffd)
        {
        case FFD::FFD1_0 : return QStringLiteral("1.05");
        case FFD::FFD1_1 : return QStringLiteral("1.1");
        case FFD::FFD1_2 : return QStringLiteral("1.2");
        default: return QString();
        }
    }
    inline void checkFfd(FFD &ffd) {
        switch(ffd)
        {
        case FFD::FFD1_0 :
        case FFD::FFD1_1 :
        case FFD::FFD1_2 : return;
        default: ffd = FFD::Invalid;
        }
    }
    //Для команды регистрации
    enum class RegFlag:quint8
    {
        Enctiption                  = 0x01u,
        OfflineMode                 = 0x02u,
        AutoMode                    = 0x04u,
        Services                    = 0x08u,
        Fosa                        = 0x10u,
        ForInternet                 = 0x20u,
        Restaurant                  = 0x40u,
        Wholesale                   = 0x80u
    };
    Q_DECLARE_FLAGS(RegFlags, RegFlag)
    enum class ExtRegFlag:quint8
    {
        ExcisableProducts           = 0x01u,
        GamesCashbox                = 0x02u,
        LotteryCashbox              = 0x04u,
        ExternalPrinter             = 0x08u,
        LabledProducts              = 0x10u,
        Pawnshop                    = 0x20u,
        Insurance                   = 0x40u,
        AutoCashboxForLabled        = 0x80u

    };
    Q_DECLARE_FLAGS(ExtRegFlags, ExtRegFlag)
}

Q_DECLARE_OPERATORS_FOR_FLAGS(fs::Warnings)
Q_DECL_CONSTEXPR inline QFlags<fs::Warning> operator~(fs::Warning f) Q_DECL_NOTHROW{
    return QFlags<fs::Warning>(~static_cast<quint8>(f)) ;
}

Q_DECLARE_OPERATORS_FOR_FLAGS(fs::RegFlags)
Q_DECL_CONSTEXPR inline QFlags<fs::RegFlag> operator~(fs::RegFlag f) Q_DECL_NOTHROW{
    return QFlags<fs::RegFlag>(~static_cast<quint8>(f)) ;
}
Q_DECLARE_OPERATORS_FOR_FLAGS(fs::ExtRegFlags)
Q_DECL_CONSTEXPR inline QFlags<fs::ExtRegFlag> operator~(fs::ExtRegFlag f) Q_DECL_NOTHROW{
    return QFlags<fs::ExtRegFlag>(~static_cast<quint8>(f)) ;
}

class FsStatus
{
public:
    FsStatus() noexcept;
    explicit FsStatus(const QVariantMap map) noexcept;
    FsStatus(const FsStatus &other) noexcept;
    FsStatus(FsStatus &&other) noexcept;
    ~FsStatus();


    bool phaseIsValid() const;
    bool isValid() const;

    fs::Phase phase() const;
    void setPhase(fs::Phase newPhase);

    fs::CurrentDocument currentDoc() const;
    void setCurrentDoc(fs::CurrentDocument newCurrentDoc);

    bool hasDocData() const;
    void setHasDocData(bool newHasDocData);

    bool cycleIsOpen() const;
    void setCycleIsOpen(bool newCycleIsOpen);

    const fs::Warnings &warnings() const;
    void setWarnings(const fs::Warnings &newWarnings);

    const QDateTime &lastDocDt() const;
    void setLastDocDt(const QDateTime &newLastDocDt);

    const QString &fsNumber() const;
    void setFsNumber(const QString &newFsNumber);

    quint32 lastFd() const;
    void setLastFd(quint32 newLastFd);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    FsStatus &operator =(const FsStatus &other) noexcept;
    FsStatus &operator =(FsStatus &&other) noexcept;
    bool operator ==(const FsStatus &other) const  noexcept;
    bool operator !=(const FsStatus &other) const  noexcept;

private:
    fs::Phase phase_;
    fs::CurrentDocument currentDoc_;
    bool hasDocData_;
    bool cycleIsOpen_;
    fs::Warnings warnings_;
    QDateTime lastDocDt_;
    QString fsNumber_;
    quint32 lastFd_;

};

//--------------------------------------------------------------------------------------------------
class FsLifeTime
{
public:
    FsLifeTime() noexcept;
    explicit FsLifeTime(const QVariantMap &map) noexcept;
    FsLifeTime(const FsLifeTime &other) noexcept;
    FsLifeTime(FsLifeTime &&other) noexcept;
    ~FsLifeTime();

    bool isValid() const;

    const QDate &expiredDt() const;
    void setExpiredDt(const QDate &newExpiredDt);

    qint32 availableRegs() const;
    void setAvailableRegs(qint32 newAvailableRegs);

    qint32 regsCount() const;
    void setRegsCount(qint32 newRegsCount);

    qint32 expiredDays() const;
    void setExpiredDays(qint32 newExpiredDays);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);


    FsLifeTime &operator =(const FsLifeTime &other) noexcept;
    FsLifeTime &operator =(FsLifeTime &&other) noexcept;
    bool operator ==(const FsLifeTime &other) const noexcept;
    bool operator !=(const FsLifeTime &other) const noexcept;



private:
    QDate expiredDt_;
    qint32 availableRegs_;
    qint32 regsCount_;
    qint32 expiredDays_;
};

//--------------------------------------------------------------------------------------------------


class FsMemResourceInfo
{
public:
    FsMemResourceInfo() noexcept;
    explicit FsMemResourceInfo(const QVariantMap &map);
    FsMemResourceInfo(const FsMemResourceInfo &other) = default;
    FsMemResourceInfo(FsMemResourceInfo &other) = default;

    quint32 availableDocs() const;
    void setAvailableDocs(quint32 newAvailableDocs);

    quint32 freeBufSize() const;
    void setFreeBufSize(quint32 newFreeBufSize);

    qint32 labelBufFree() const;
    void setLabelBufFree(qint32 newLabelBufFree);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);

    FsMemResourceInfo &operator =(const FsMemResourceInfo &other) = default;
    FsMemResourceInfo &operator =(FsMemResourceInfo &&other) = default;
    bool operator ==(const FsMemResourceInfo &other) const;
    bool operator !=(const FsMemResourceInfo &other) const;

private:
    quint32 availableDocs_;
    quint32 freeBufSize_;
    qint32 labelBufFree_;

};

//--------------------------------------------------------------------------------------------------
class CycleStatus
{
public:
    CycleStatus();
    explicit CycleStatus(const QVariantMap &map);
    CycleStatus(const CycleStatus &other) = default;
    CycleStatus(CycleStatus &&other) = default;
    ~CycleStatus();
    bool opened() const;
    void setOpened(bool newOpened);

    qint32 cycle() const;
    void setCycle(qint32 newCycle);

    qint32 receipt() const;
    void setReceipt(qint32 newReceipt);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    CycleStatus &operator =(const CycleStatus &other) = default;
    CycleStatus &operator =(CycleStatus &&other) = default;
    bool operator ==(const CycleStatus &other) const;
    bool operator !=(const CycleStatus &other) const;



private:
    bool opened_;
    qint32 cycle_;
    qint32 receipt_;
};


//--------------------------------------------------------------------------------------------------

class FsTransportState
{
public:
    enum class StatusFlag: quint8
    {
        HasOfdConnection                        = 0x01u,
        HasMessageForOfd                        = 0x02u,
        WaitTicket                              = 0x04u,
        HasOfdCommand                           = 0x08u,
        OfdConnectionCfgChanged                 = 0x10u,
        WaitOfdCommandResponse                  = 0x20u
    };
    Q_DECLARE_FLAGS(Status, StatusFlag)
    FsTransportState()noexcept;
    explicit FsTransportState(const QVariantMap &map)noexcept;
    FsTransportState(const FsTransportState &other)noexcept;
    FsTransportState(FsTransportState &&other)noexcept;
    ~FsTransportState();

    const Status &status() const;
    void setStatus(const Status &newStatus);

    bool ofdConnected() const;
    bool hasMessageForOfd() const;

    bool ofdMessageReading() const;
    void setOfdMessageReading(bool newOfdMessageReading);

    quint16 messagesCount() const;
    void setMessagesCount(quint16 newMessagesCount);

    quint32 firstDocNumber() const;
    void setFirstDocNumber(quint32 newFirstDocNumber);

    const QDateTime &firstDocDt() const;
    void setFirstDocDt(const QDateTime &newFirstDocDt);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    FsTransportState &operator =(const FsTransportState &other)noexcept;
    FsTransportState &operator =(FsTransportState &&other)noexcept;
    bool operator ==(const FsTransportState &other) const noexcept;
    bool operator !=(const FsTransportState &other) const noexcept;

private:
    Status status_;
    bool ofdMessageReading_;
    quint16 messagesCount_;
    quint32 firstDocNumber_;
    QDateTime firstDocDt_;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(FsTransportState::Status)

Q_DECL_CONSTEXPR inline QFlags<FsTransportState::StatusFlag> operator~(FsTransportState::StatusFlag f) Q_DECL_NOTHROW{
    return QFlags<FsTransportState::StatusFlag>(~static_cast<quint8>(f)) ;
}

//--------------------------------------------------------------------------------------------------

class LabelsTransportState
{
public:
    enum class Status: quint8
    {
        NoExchange                          = 0,
        ReadNotification                    = 1,
        WaitTicket                          = 2
    };

    LabelsTransportState()noexcept;
    explicit LabelsTransportState(const QVariantMap &map)noexcept;
    LabelsTransportState(const LabelsTransportState &other)noexcept;
    LabelsTransportState(LabelsTransportState &&other)noexcept;
    ~LabelsTransportState();

    Status status() const;
    void setStatus(const Status &newStatus);

    quint16 notificationCount() const;
    void setNotificationCount(quint16 newMessagesCount);

    quint32 firstNotificationNumber() const;
    void setFirstNotificationNumber(quint32 newFirstDocNumber);

    const QDateTime &firstNotificationDt() const;
    void setFirstNotificationDt(const QDateTime &newFirstDocDt);

    quint8 memFullPercent() const;
    void setMemFullPercent(quint8 newMemFullPercent);


    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    LabelsTransportState &operator =(const LabelsTransportState &other)noexcept;
    LabelsTransportState &operator =(LabelsTransportState &&other)noexcept;
    bool operator ==(const LabelsTransportState &other) const noexcept;
    bool operator !=(const LabelsTransportState &other) const noexcept;


private:
    Status status_;
    quint16 notificationsCount_;
    quint32 firstNotificationNumber_;
    QDateTime firstNotificationDt_;
    quint8 memFullPercent_;
};

//--------------------------------------------------------------------------------------------------
class BStatus
{
public:
    enum class CheckState :quint8
    {
        CheckTableOwerflow                  = 0x00,
        NoAnyLablesForCheck                 = 0x01,
        B1Sended                            = 0x02,
        B5Sended                            = 0x03,
        B6Sended                            = 0x04
    };
    enum class NotificationState : quint8
    {
        NoAnyNotification                   = 0x00,
        NotificationStarted                 = 0x01,
        NotificationBlocked                 = 0x02
    };
    enum class Flag: quint8
    {
        B1                                  = 0x01,
        B2                                  = 0x02,
        B3                                  = 0x04,
        B5                                  = 0x08,
        B6                                  = 0x10,
        B7_1                                = 0x20,
        B7_2                                = 0x40,
        B7_3                                = 0x80,
    };
    Q_DECLARE_FLAGS(Flags, Flag)
    enum class OverflowWarning:quint8
    {
        Less50                              = 0x00,
        From50To80                          = 0x01,
        From80To90                          = 0x02,
        Over90                              = 0x03,
        Overflow                            = 0x04
    };

    BStatus() noexcept;
    BStatus(const BStatus &other)noexcept;
    BStatus(BStatus &&other) = default;
    explicit BStatus(const QVariantMap &map) noexcept;
    ~BStatus();

    CheckState checkState() const;
    void setCheckState(CheckState newCheckState);

    NotificationState notificationState() const;
    void setNotificationState(NotificationState newNotificationState);

    const Flags &flags() const;
    void setFlags(const Flags &newFlags);

    quint8 resultsCount() const;
    void setResultsCount(quint8 newResultsCount);

    quint8 notificatedCount() const;
    void setNotificatedCount(quint8 newNotificatedCount);

    OverflowWarning warning() const;
    void setWarning(OverflowWarning newWarning);

    quint16 notications() const;
    void setNotications(quint16 newNotications);

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);

    BStatus &operator =(const BStatus &other) noexcept;
    BStatus &operator =(BStatus &&other) = default;
    bool operator ==(const BStatus &other) const noexcept;
    bool operator !=(const BStatus &other) const noexcept;

private:
    CheckState checkState_;
    NotificationState notificationState_;
    Flags flags_;
    quint8 resultsCount_;
    quint8 notificatedCount_;
    OverflowWarning warning_;
    quint16 notications_;

};

Q_DECLARE_OPERATORS_FOR_FLAGS(BStatus::Flags)

Q_DECL_CONSTEXPR inline QFlags<BStatus::Flag> operator~(BStatus::Flag f) Q_DECL_NOTHROW{
    return QFlags<BStatus::Flag>(~static_cast<quint8>(f)) ;
}

//--------------------------------------------------------------------------------------------------

class FsFullStatus
{
public:
    enum class CleanFlag :qint32
    {
        Status                  = 0x00000001,
        LifeTime                = 0x00000002,
        Version                 = 0x00000004,
        Release                 = 0x00000008,
        Serial                  = 0x00000010,
        Formats                 = 0x00000020,
        Resource                = 0x00000040,
        Cycle                   = 0x00000080,
        Error                   = 0x00000100,
        Transport               = 0x00000200,
        CycleDates              = 0x00000400,
        LabelsTransport         = 0x00000800,
    };
    Q_DECLARE_FLAGS(CleanFlags, CleanFlag)

    static const CleanFlags CLEAN_ALL;
    static const CleanFlags CLEAN_NONE;
    static const CleanFlags CLEAN_CYCLE;
    static const CleanFlags CLEAN_TRANSPORT;
    static const CleanFlags CLEAN_LABELS_TRANSPORT;

    FsFullStatus()  noexcept;
    explicit FsFullStatus(const QVariantMap &map) noexcept;
    FsFullStatus(const FsFullStatus &other) noexcept;
    FsFullStatus(FsFullStatus &&other) noexcept;
    ~FsFullStatus();

    bool isFiscalized() const;
    bool archiveIsClosed() const;
    bool cycleIsOpen()const;
    bool cycleIsOver() const;

    bool hasStatus() const;
    FsStatus status() const;
    void setStatus(const FsStatus &status);
    void setStatus(const std::optional<FsStatus> &status);
    void cleanStatus();

    bool hasLifeTime() const;
    FsLifeTime lifeTime() const;
    void setLifeTime(const FsLifeTime &status);
    void setLifeTime(const std::optional<FsLifeTime> &status);
    void cleanLifeTime();

    const QString &version() const;
    void setVersion(const QString &newVersion);

    bool isRelease() const;
    void setIsRelease(bool newIsRelease);

    QString release() const;
    void setRelease(const QString &newRelease);

    QString fsNumber() const;
    void setFsNumber(const QString &newFsNumber);

    fs::FFD currentFfd() const;
    void setCurrentFfd(fs::FFD newCurrentFfd);

    fs::FFD supportedFfd() const;
    void setSupportedFfd(fs::FFD newSupportedFfd);

    bool hasMemResource() const;
    FsMemResourceInfo memResource() const;
    void setMemResource(const FsMemResourceInfo &res);
    void setMemResource(const std::optional<FsMemResourceInfo> &res);
    void cleanMemResource();

    bool hasCycle() const;
    CycleStatus cycle() const;
    void setCycle(const CycleStatus &res);
    void setCycle(const std::optional<CycleStatus> &res);
    void cleanCycle();

    bool hasTranspot() const;
    FsTransportState transport() const;
    void setTransport(const FsTransportState &res);
    void setTransport(const std::optional<FsTransportState> &res);
    void cleanTransport();

    bool hasLabelsTranspot() const;
    LabelsTransportState labelsTransport() const;
    void setLabelsTransport(std::optional<LabelsTransportState> newLabelsTransport);
    void setLabelsTransport(const LabelsTransportState &state);
    void cleanLabelsTransport();


    const QString &cashboxSerial() const;
    void setCashboxSerial(const QString &newCashboxSerial);

    const QDateTime &cycleOpenDt() const;
    void setCycleOpenDt(const QDateTime &newCycleOpenDt);

    const QDateTime &cycleCloseDt() const;
    void setCycleCloseDt(const QDateTime &newCycleCloseDt);

    CoreApiConst::ErrorCode error() const;
    void setError(const CoreApiConst::ErrorCode &err);
    bool hasError() const;

    QVariantMap toMap() const;
    QVariantMap toExternalMap() const;
    void parseMap(const QVariantMap &map);
    void clean(CleanFlags flags);

    QStringList toHtml() const;


    FsFullStatus &operator = (const FsFullStatus &other) noexcept;
    FsFullStatus &operator = (FsFullStatus &&other) noexcept;
    bool operator == (const FsFullStatus &other) const noexcept;
    bool operator != (const FsFullStatus &other) const noexcept;



private:
    std::optional<FsStatus> status_;
    std::optional<FsLifeTime> lifeTime_;
    QString version_;
    bool isRelease_;
    QString release_;
    QString fsNumber_;
    fs::FFD currentFfd_;
    fs::FFD supportedFfd_;
    std::optional<FsMemResourceInfo> memResource_;
    std::optional<CycleStatus> cycle_;
    std::optional<FsTransportState> transport_;
    std::optional<LabelsTransportState> labelsTransport_;
    QString cashboxSerial_;
    QDateTime cycleOpenDt_;
    QDateTime cycleCloseDt_;
    CoreApiConst::ErrorCode error_;
};
Q_DECLARE_OPERATORS_FOR_FLAGS(FsFullStatus::CleanFlags)
Q_DECL_CONSTEXPR inline QFlags<FsFullStatus::CleanFlag> operator~(FsFullStatus::CleanFlag f) Q_DECL_NOTHROW{
    return QFlags<FsFullStatus::CleanFlag>(~static_cast<quint32>(f)) ;
}

//--------------------------------------------------------------------------------------------------

class OfflineDSessionStatus
{
public:
    OfflineDSessionStatus()noexcept;
    OfflineDSessionStatus(const OfflineDSessionStatus &other) = default;
    OfflineDSessionStatus(OfflineDSessionStatus && other) = default;
    explicit OfflineDSessionStatus(const QVariantMap &map)noexcept;
    ~OfflineDSessionStatus();

    quint16 notSendedCount() const;
    void setNotSendedCount(quint16 newNotSendedCount);

    quint32 fistNotSended() const;
    void setFistNotSended(quint32 newFistNotSended);

    quint16 sessionNotSendedCount() const;
    void setSessionNotSendedCount(quint16 newSessionNotSendedCount);

    quint32 fistSessionNotSended() const;
    void setFistSessionNotSended(quint32 newFistSessionNotSended);

    QVariantMap toMap() const;
    void parseMap(const QVariantMap &map);
    bool parseFs(const QByteArray &data);

    OfflineDSessionStatus &operator =(const OfflineDSessionStatus &other) = default;
    OfflineDSessionStatus &operator =(OfflineDSessionStatus &&other) = default;
    bool operator ==(const OfflineDSessionStatus &other) const noexcept;
    bool operator !=(const OfflineDSessionStatus &other) const noexcept;
private:
    quint16 notSendedCount_;
    quint32 fistNotSended_;
    quint16 sessionNotSendedCount_;
    quint32 fistSessionNotSended_;
};

#endif // FSFULLSTATUS_H
