#include "fsmsg.h"

#include "crc.h"


#include <QDataStream>
#include <QTextCodec>
#include <QSet>

quint8 FsMsg::code() const
{
    return code_;
}

QByteArray FsMsg::data() const
{
    return data_;
}

QByteArray FsMsg::rawData() const
{
    QByteArray res;
    QDataStream ds(&res, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << MSG_START << static_cast<quint16>(data_.size() + 1)
       << code_;
    ds.writeRawData(data_.constData(), data_.size());
    quint16 crc = Crc16CITT()(res.mid(1));
    ds << crc;
    return res;
}

fs::RepParsingRes FsMsg::setRawData(const QByteArray &data)
{
    data_.clear();
    if(data.size() < 6)
    {
        return fs::RepParsingRes::InvalidPackgSize;
    }
    if(static_cast<quint8>(data[0]) != MSG_START)
    {
        return fs::RepParsingRes::NoMsgStart;
    }
    QDataStream ds(data);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds.skipRawData(1);
    quint16 len = 0;
    ds >> len;
    if(len != data.size() - 5)
    {
        return fs::RepParsingRes::InvalidLen;
    }
    quint16 crc1 = Crc16CITT()(data.mid(1, len + 2)), crc2 = 0;
    ds >> code_;
    ds.skipRawData(len - 1);
    if(len > 1)data_ = data.mid(4, len - 1);
    ds >> crc2;

    if(crc1 != crc2)
    {
        return fs::RepParsingRes::InvalidCrc;
    }
    return fs::RepParsingRes::Ok;
}

QByteArray FsMsg::serializeDt(const QDateTime &dt)
{
    QByteArray res;
    if(dt.isValid() && dt.date().year() > 2000)
    {
        res.push_back(dt.date().year() - 2000);
        res.push_back(dt.date().month());
        res.push_back(dt.date().day());
        res.push_back(dt.time().hour());
        res.push_back(dt.time().minute());
    }
    return res;
}

QDateTime FsMsg::parseDt(const QByteArray &raw)
{
    QDateTime res;
    if(raw.size() == 5)
    {
        res.setDate(QDate(2000 + raw[0], raw[1], raw[2]));
        res.setTime(QTime(raw[3], raw[4]));
    }
    return res;
}

QByteArray FsMsg::serializeDate(const QDate &date, bool zeroTime)
{
    if(zeroTime)
    {
        QDateTime dt(date, QTime(0, 0));
        return serializeDt(dt);
    }
    QByteArray res;
    if(date.isValid() && date.year() > 2000)
    {
        res.push_back(date.year() - 2000);
        res.push_back(date.month());
        res.push_back(date.day());
    }
    return res;
}

QDate FsMsg::parseDate(const QByteArray &raw)
{
    QDate res;
    if(raw.size() == 3)
    {
        res = QDate(2000 + raw[0], raw[1], raw[2]);
    }
    return res;
}

FsMsg::FsMsg(quint8 code, const QByteArray &data) noexcept
    : code_(code)
    , data_(data)
{

}

FsMsg::FsMsg(const FsMsg &other) noexcept
    : code_(other.code_)
    , data_(other.data_)
{

}

FsMsg::FsMsg(FsMsg &&other) noexcept
    : code_(other.code_)
    , data_()
{
    data_.swap(other.data_);
}

FsMsg::~FsMsg()
{

}

FsMsg &FsMsg::operator=(const FsMsg &other) noexcept
{
    code_ = other.code_;
    data_ = other.data_;
    return *this;
}
FsMsg &FsMsg::operator=(FsMsg &&other) noexcept
{
    code_ = other.code_;
    data_.swap(other.data_);
    return *this;
}
bool FsMsg::operator==(const FsMsg &other) const noexcept
{
    return code_ == other.code_ &&
            data_ == other.data_;
}

bool FsMsg::operator!=(const FsMsg &other) const noexcept
{
    return !(*this == other);
}

bool FsMsg::checkTlvStructure() const
{
    //TODO: Сделать полную проверку TLV структуры для команд и ответов
    return !data_.isEmpty();
}

//--------------------------------------------------------------------------------------------------

constexpr quint16 repSizeForREq(fs::ReqCode req) {
    switch (req) {
    case fs::ReqCode::FsState: return 30;
    case fs::ReqCode::FsNumber: return 16;
    case fs::ReqCode::FsLifetime: return 5;
    case fs::ReqCode::FsVersion: return 17;
    case fs::ReqCode::FsRelease: return 48;
    case fs::ReqCode::FsFormat: return 2;
    case fs::ReqCode::FsLifeDays: return 2;
    case fs::ReqCode::CurrentCycleData: return 5;
    case fs::ReqCode::CompleteRegistrationDoc2: return 8;
    case fs::ReqCode::CompletePaymentStateReport: return 15;
    case fs::ReqCode::TransfereStatus: return 13;
    default: return 0;
    }
}

FsReq::FsReq()
    : FsMsg(INVALID_CODE, QByteArray())
    , expRepSz_(0u)
{

}

FsReq::FsReq(fs::ReqCode code)
    : FsMsg(static_cast<quint8>(code), QByteArray())
    , expRepSz_(repSizeForREq(code))
{

}

FsReq::FsReq(fs::RegType regType, fs::FFD ffd)
    : FsMsg(static_cast<quint8>(fs::ReqCode::BeginRegistrationDoc2),
                 QByteArray(2, '\x00'))
    , expRepSz_(repSizeForREq(fs::ReqCode::BeginRegistrationDoc2))
{
    data_[0] = static_cast<char>(regType);
    data_[1] = static_cast<char>(ffd);
}

FsReq::FsReq(fs::ReqCode code, const QByteArray &docData)
    : FsMsg(static_cast<quint8>(code), docData)
    , expRepSz_(repSizeForREq(code))
{

}

FsReq::FsReq(const FsReq &other) noexcept
    : FsMsg(other)
    , expRepSz_(other.expRepSz_)
{

}

FsReq::FsReq(FsReq &&other) noexcept
    : FsMsg(other)
    , expRepSz_(other.expRepSz_)
{

}

FsReq::~FsReq()
{

}

bool FsReq::isValid() const
{
    switch (reqCode()) {
    case fs::ReqCode::FsState:
    case fs::ReqCode::FsNumber:
    case fs::ReqCode::FsLifetime:
    case fs::ReqCode::FsVersion:
    case fs::ReqCode::FsErrors:
    case fs::ReqCode::CancelDocument:
    case fs::ReqCode::TransfereStatus:
    case fs::ReqCode::BeginOfdMsgReading:
    case fs::ReqCode::CancelOfdMsgReading:
    case fs::ReqCode::FinishOfdMsgReading:
    case fs::ReqCode::CompleteCycleOpening:
    case fs::ReqCode::CloseCycle:
    case fs::ReqCode::CurrentCycleData:
    case fs::ReqCode::CompletePaymentStateReport:
    case fs::ReqCode::FsDocumentTLVReading:
    case fs::ReqCode::StartFsClosing:
    case fs::ReqCode::OfflineDocsCount:
    case fs::ReqCode::FiscalisationSummary:
    case fs::ReqCode::FiscalisationParameter:
    case fs::ReqCode::FsRelease:
    case fs::ReqCode::FsFormat:
    case fs::ReqCode::FsFreeMem:
    case fs::ReqCode::CleanLabelCheckResult:
    case fs::ReqCode::GetNotificationState:
    case fs::ReqCode::StartNotificationReading:
    case fs::ReqCode::FinishNotificationReading:
    case fs::ReqCode::CancelNotificationReading:
    case fs::ReqCode::FsForLabelsStatus:
        return true;
    case fs::ReqCode::SendDocumentData:
        return checkTlvStructure();
    case fs::ReqCode::ResetFsState:
    case fs::ReqCode::SendOfdConnectionState:
        case fs::ReqCode::FsCounters:
    case fs::ReqCode::GetIngoForKeysUpdate:
    case fs::ReqCode::SaveLabelCheckResult:
    case fs::ReqCode::StartUploadSession:
    case fs::ReqCode::ReadNextNotification:
        return data_.size() == 1;
    case fs::ReqCode::ReadOfdMsgBlock:
    case fs::ReqCode::ReadNotificationBlock:
    case fs::ReqCode::FsDocumentTLV:
    case fs::ReqCode::FsDocByNumber:
    case fs::ReqCode::FsDocReceiptByNumber:
    case fs::ReqCode::ReadCurrentNotificationBlock:
        return data_.size() == 4;
    case fs::ReqCode::SendOfdReceipt:
    case fs::ReqCode::SendReplyForKeysUpdate:
    case fs::ReqCode::SendLabelForFsCheck:
    case fs::ReqCode::CreateLabelRequest:
    case fs::ReqCode::SendReplyAboutLabel:
    case fs::ReqCode::SendFiscalDataForLabels:
    case fs::ReqCode::SendNotificationTicket:
    case fs::ReqCode::CommitNotificationUpload:
        return !data_.isEmpty();
    case fs::ReqCode::FsLifeDays:
    case fs::ReqCode::BeginCycleOpening:
    case fs::ReqCode::BeginCycleClosing:
    case fs::ReqCode::BeginCheckCreating:
    case fs::ReqCode::BeginCorrectionCheckCreating:
    case fs::ReqCode::BeginPaymentStateReport:
    case fs::ReqCode::GetRequestForKeysUpdate:
        return data_.size() == 5;
    case fs::ReqCode::CloseFsFiscalMode:
        return data_.size() == 25;
    case fs::ReqCode::CompleteCheckCreating:
        return data_.size() == 11;
    case fs::ReqCode::BeginRegistrationDoc2:
        return data_.size() == 2;
    case fs::ReqCode::CompleteRegistrationDoc2:
        return data_.size() == 52 || data_.size() == 56;
    default: return false;
    }
    return false;
}

fs::ReqCode FsReq::reqCode() const
{
    return static_cast<fs::ReqCode>(code_);
}

quint16 FsReq::expRepSz() const
{
    return expRepSz_;
}

FsReq &FsReq::operator=(const FsReq &other) noexcept
{
    FsMsg::operator =(other);
    expRepSz_ = other.expRepSz_;
    return *this;
}
FsReq &FsReq::operator=(FsReq &&other) noexcept
{
    FsMsg::operator =(other);
    expRepSz_ = other.expRepSz_;
    return *this;
}

bool FsReq::operator==(const FsReq &other) const noexcept
{
    return FsMsg::operator ==(other) &&
            expRepSz_ == other.expRepSz_;
}

bool FsReq::operator!=(const FsReq &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsRep::FsRep()
    : FsMsg(INVALID_CODE, QByteArray())
    , parsingResult_(fs::RepParsingRes::NoParsing)
{

}

FsRep::FsRep(const QByteArray &rawData)
    : FsMsg(INVALID_CODE, QByteArray())
    , parsingResult_(fs::RepParsingRes::NoParsing)
{
    parsingResult_ = FsMsg::setRawData(rawData);
}

FsRep::FsRep(const FsRep &other)
    : FsMsg(other)
    , parsingResult_(other.parsingResult_)
{

}

FsRep::FsRep(FsRep &&other)
    : FsMsg(other)
    , parsingResult_(other.parsingResult_)
{

}

FsRep::~FsRep()
{

}

bool FsRep::isValid() const
{
    if(parsingResult_ != fs::RepParsingRes::Ok) return false;
    fs::RepCode c = repCode();
    return c != fs::RepCode::InvalidCode;
}

fs::RepParsingRes FsRep::setRawData(const QByteArray &data)
{
    parsingResult_ = FsMsg::setRawData(data);

    return parsingResult_;
}

fs::RepParsingRes FsRep::parsingResult() const
{
    return parsingResult_;
}

fs::RepCode FsRep::repCode() const
{
    static const QSet<quint8> VALID_CODES = {
        static_cast<quint8>(fs::RepCode::Ok                             ),
        static_cast<quint8>(fs::RepCode::UnknownCmd                     ),
        static_cast<quint8>(fs::RepCode::IncorrectFsState               ),
        static_cast<quint8>(fs::RepCode::FsError                        ),
        static_cast<quint8>(fs::RepCode::CcError                        ),
        static_cast<quint8>(fs::RepCode::FsOverLifetime                 ),
        static_cast<quint8>(fs::RepCode::FsOverflow                     ),
        static_cast<quint8>(fs::RepCode::IncorrectDt                    ),
        static_cast<quint8>(fs::RepCode::NoDataRequested                ),
        static_cast<quint8>(fs::RepCode::InvalidCmdParams               ),
        static_cast<quint8>(fs::RepCode::InvlidCmd                      ),
        static_cast<quint8>(fs::RepCode::DeniedParams                   ),
        static_cast<quint8>(fs::RepCode::DubleData                      ),
        static_cast<quint8>(fs::RepCode::NotEnophData                   ),
        static_cast<quint8>(fs::RepCode::TooManyOperations              ),
        static_cast<quint8>(fs::RepCode::InvalidTLVSize                 ),
        static_cast<quint8>(fs::RepCode::OfdTransportMissing            ),
        static_cast<quint8>(fs::RepCode::CcOverLifetime                 ),
        static_cast<quint8>(fs::RepCode::ExhaustedStorageResource       ),
        static_cast<quint8>(fs::RepCode::DocumentTransferTimeout        ),
        static_cast<quint8>(fs::RepCode::CycleOver24h                   ),
        static_cast<quint8>(fs::RepCode::TimePeriodError                ),
        static_cast<quint8>(fs::RepCode::InvalidParameter               ),
        static_cast<quint8>(fs::RepCode::ParameterBannedByReg           ),
        static_cast<quint8>(fs::RepCode::OfdMsgError                    ),
        static_cast<quint8>(fs::RepCode::KeysServerFailure              ),
        static_cast<quint8>(fs::RepCode::KeysServerUnknownError         ),
        static_cast<quint8>(fs::RepCode::RetryKeysUpdate                ),
        static_cast<quint8>(fs::RepCode::LabledProductsNotSupported     ),
        static_cast<quint8>(fs::RepCode::BxRequestsSequenceError        ),
        static_cast<quint8>(fs::RepCode::LabledProductsTemporarryBlocked),
        static_cast<quint8>(fs::RepCode::LabelsCheckTableOverflow       ),
        static_cast<quint8>(fs::RepCode::KeyCheckPeriod90Off            ),
        static_cast<quint8>(fs::RepCode::TlvStructureParametersNotEnoph ),
        static_cast<quint8>(fs::RepCode::Parameter2007HasNotCheckedLabel)
    };
    if(!VALID_CODES.contains(static_cast<quint8>(code_ & 0x7Fu))) return fs::RepCode::InvalidCode;
    else
    {
        fs::RepCode c = static_cast<fs::RepCode>(code_ & 0x7Fu);
        return c;
    }
//    bool res = (c >= fs::RepCode::Ok && c <= fs::RepCode::CcOverLifetime) ||
//            (c >= fs::RepCode::ExhaustedStorageResource && c <= fs::RepCode::TimePeriodError) ||
//            c == fs::RepCode::OfdMsgError;
//    return res ? c : fs::RepCode::InvalidCode;
}

CoreApiConst::ErrorCode FsRep::repError() const
{
    return fs::fsCodeToError(repCode());
}

void FsRep::setRepCode(fs::RepCode code)
{
    code_ = static_cast<quint8>(code);
}

bool FsRep::hasWarning() const
{
    return static_cast<bool>(code_ & 0x80u);
}

fs::RepParsingRes FsRep::getFsStatus(std::optional<FsStatus> &status) const
{
    status.reset();
    FsStatus st;
    if(parsingResult_ !=  fs::RepParsingRes ::Ok)
    {
        return parsingResult_;
    }
    QByteArray data = data_;
    if(data.size() != repSizeForREq(fs::ReqCode::FsState))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    st.setPhase(static_cast<fs::Phase>(data.at(0)));
    st.setCurrentDoc(static_cast<fs::CurrentDocument>(data.at(1)));
    st.setHasDocData(static_cast<quint8>(data[2]) == 1u);
    st.setCycleIsOpen(static_cast<quint8>(data[3]) == 1u);
    st.setWarnings(fs::Warnings(static_cast<quint8>(data.at(4))));
    QByteArray buf = data.mid(5, 5);
    st.setLastDocDt(parseDt(buf));

    buf = data.mid(10, 16);
    QTextDecoder decoder(QTextCodec::codecForName(fs::CODEC_NAME));
    st.setFsNumber(decoder.toUnicode(buf).trimmed());
    QDataStream ds(data.mid(26));
    ds.setByteOrder(QDataStream::LittleEndian);
    quint32 fd = 0;
    ds >> fd;
    st.setLastFd(fd);
    if(!st.lastDocDt().isValid() && st.lastFd() > 0) return fs::RepParsingRes::InvalidData;
    if(st.isValid())
    {
        status = st;
        return fs::RepParsingRes::Ok;
    }
    return fs::RepParsingRes::InvalidData;
}

fs::RepParsingRes FsRep::getFsNumber(QString &serial) const
{
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;
    serial.clear();
    if(data_.size() != repSizeForREq(fs::ReqCode::FsNumber)) return fs::RepParsingRes::InvalidLen;
    QTextDecoder decoder(QTextCodec::codecForName(fs::CODEC_NAME));
    serial = decoder.toUnicode(data_).trimmed();
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getFsLifeTime(std::optional<FsLifeTime> &lifeTime) const
{
    lifeTime.reset();
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() != repSizeForREq(fs::ReqCode::FsLifetime)) return fs::RepParsingRes::InvalidLen;
    FsLifeTime lt;
    lt.setExpiredDt(parseDate(data_.mid(0, 3)));
    lt.setAvailableRegs(static_cast<qint32>(data_[3]));
    lt.setRegsCount(static_cast<qint32>(data_[4]));
    lifeTime = lt;
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getFsVersion(QString &version, bool &isRelease) const
{
    version.clear();
    isRelease  = true;
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() != repSizeForREq(fs::ReqCode::FsVersion)) return fs::RepParsingRes::InvalidLen;
    QByteArray data = data_.mid(0, 16);
    data = data.replace('\x00',' ');

    QTextDecoder decoder(QTextCodec::codecForName(fs::CODEC_NAME));
    version = decoder.toUnicode(data).trimmed();
    isRelease = static_cast<qint32>(data_[16]);

    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getLastFsErrors(QByteArray &dump) const
{
    dump.clear();
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;
    dump = data_;
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getFsRelease(QString &release) const
{

    if(parsingResult_ != fs::RepParsingRes::Ok)
    {
        return parsingResult_;
    }
    release.clear();
    if(data_.size() != repSizeForREq(fs::ReqCode::FsRelease))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    QByteArray data = data_;
    data = data.replace('\x00',' ');
    QTextDecoder decoder(QTextCodec::codecForName(fs::CODEC_NAME));
    release = decoder.toUnicode(data).trimmed();
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getFsFormat(fs::FFD &curFfd, fs::FFD &supportedFFd) const
{
    curFfd = fs::FFD::Invalid;
    supportedFFd = fs::FFD::Invalid;
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() != repSizeForREq(fs::ReqCode::FsFormat))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    curFfd = static_cast<fs::FFD>(data_[0]);
    supportedFFd= static_cast<fs::FFD>(data_[1]);
    fs::checkFfd(curFfd);
    fs::checkFfd(supportedFFd);

    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getExpiredDays(qint32 &days) const
{
    days = -1;
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() != repSizeForREq(fs::ReqCode::FsLifeDays))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    quint16 buf = 0;
    QDataStream ds(data_);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> buf;
    days = buf;
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getMemResource(std::optional<FsMemResourceInfo> &resource) const
{
    resource.reset();
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() < 8)
    {
        return fs::RepParsingRes::InvalidLen;
    }
    FsMemResourceInfo r;
    QDataStream ds(data_);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint32 a = 0, f = 0;
    ds >> a >> f;
    r.setAvailableDocs(a);
    r.setFreeBufSize(f);
    if(data_.size() == 9)
    {
        r.setLabelBufFree(static_cast<qint32>(static_cast<quint8>(data_[8])));
    }

    resource = r;
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getFsCycle(std::optional<CycleStatus> &cycle) const
{
    cycle.reset();
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() < repSizeForREq(fs::ReqCode::CurrentCycleData))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    CycleStatus cs;
    QDataStream ds(data_);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint16 c = 0, r = 0;
    quint8 f = 0;
    ds >> f >> c >> r;
    cs.setCycle(c);
    cs.setOpened(f);
    cs.setReceipt(r);

    cycle = cs;
    return fs::RepParsingRes::Ok;

}

fs::RepParsingRes FsRep::getFsTransport(std::optional<FsTransportState> &tranport) const
{
    tranport.reset();
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() < repSizeForREq(fs::ReqCode::TransfereStatus))
    {
        return fs::RepParsingRes::InvalidLen;
    }
    FsTransportState s;
    QDataStream ds(data_);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint8 b1 = 0, b2 = 0;
    quint16 c = 0;
    quint32 fd = 0;
    ds >> b1 >> b2 >> c >> fd;
    s.setStatus(FsTransportState::Status(b1));
    s.setOfdMessageReading(b2);
    s.setMessagesCount(c);
    s.setFirstDocNumber(fd);
    s.setFirstDocDt(parseDt(data_.mid(8)));

    tranport = s;
    return fs::RepParsingRes::Ok;
}

fs::RepParsingRes FsRep::getDocResult(qint32 &docNumber, quint32 &fiscalCode) const
{
    docNumber = 0;
    fiscalCode = 0;
    if(parsingResult_ != fs::RepParsingRes::Ok) return parsingResult_;

    if(data_.size() < 8)
    {
        return fs::RepParsingRes::InvalidLen;
    }
    QDataStream ds(data_);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> docNumber >> fiscalCode;
    return fs::RepParsingRes::Ok;
}

FsRep &FsRep::operator=(const FsRep &other)
{
    FsMsg::operator =(other);
    parsingResult_ = other.parsingResult_;
    return *this;
}
FsRep &FsRep::operator=(FsRep &&other)
{
    FsMsg::operator =(other);
    parsingResult_ = other.parsingResult_;
    return *this;
}

bool FsRep::operator==(const FsRep &other) const
{
    return FsMsg::operator ==(other) && parsingResult_ == other.parsingResult_;
}

bool FsRep::operator!=(const FsRep &other) const
{
    return !(*this == other);
}
