#ifndef FSMSG_H
#define FSMSG_H

#include "fsfullstatus.h"
#include "coreapiconst.h"


#include <QString>

#include <QByteArray>
#include <QDate>
#include <QDateTime>
#include <QMap>

namespace  fs {

    enum class RepCode: quint8
    {
        Ok                                  = 0x00u,
        UnknownCmd                          = 0x01u,
        IncorrectFsState                    = 0x02u,
        FsError                             = 0x03u,
        CcError                             = 0x04u,
        FsOverLifetime                      = 0x05u,
        FsOverflow                          = 0x06u,
        IncorrectDt                         = 0x07u,
        NoDataRequested                     = 0x08u,
        InvalidCmdParams                    = 0x09u,
        InvlidCmd                           = 0x0Au,
        DeniedParams                        = 0x0Bu,
        DubleData                           = 0x0Cu,
        NotEnophData                        = 0x0Du,
        TooManyOperations                   = 0x0Eu,
        InvalidTLVSize                      = 0x10u,
        OfdTransportMissing                 = 0x11u,
        CcOverLifetime                      = 0x12u,
        ExhaustedStorageResource            = 0x14u,
        DocumentTransferTimeout             = 0x15u,
        CycleOver24h                        = 0x16u,
        TimePeriodError                     = 0x17u,
        InvalidParameter                    = 0x18u,
        ParameterBannedByReg                = 0x19u,
        OfdMsgError                         = 0x20u,
        KeysServerFailure                   = 0x23u,
        KeysServerUnknownError              = 0x24u,
        RetryKeysUpdate                     = 0x30u,
        LabledProductsNotSupported          = 0x32u,
        BxRequestsSequenceError             = 0x33u,
        LabledProductsTemporarryBlocked     = 0x34u,
        LabelsCheckTableOverflow            = 0x35u,
        KeyCheckPeriod90Off                 = 0x36u,
        TlvStructureParametersNotEnoph      = 0x3Cu,
        Parameter2007HasNotCheckedLabel     = 0x3Eu,
        MaxCodeNumber                       = 0x3Eu,
        InvalidCode                         = 0x7Fu,
    };


    inline CoreApiConst::ErrorCode fsCodeToError(RepCode code) {
        if(code == RepCode::Ok) return CoreApiConst::ErrorCode::Ok;
        if(static_cast<qint32>(code) >= static_cast<qint32>(RepCode::MaxCodeNumber))
        {
            return CoreApiConst::ErrorCode::UnknownFsError;
        }
        return static_cast<CoreApiConst::ErrorCode>(0x00000500u | static_cast<quint32>(code));
    }

    enum class ReqCode: quint8
    {
    //Служебные команды ФН
        FsState                          = 0x30u,
        FsNumber                         = 0x31u,
        FsLifetime                       = 0x32u,
        FsVersion                        = 0x33u,
        FsErrors                         = 0x35u,
        FsCounters                       = 0x36u,
        FsOpCounters                     = 0x37u,
        FsMoneyCounters                  = 0x38u,
        FsNotSendedCounters              = 0x39u,
        FsFormat                         = 0x3Au,
        FsLifeDays                       = 0x3Bu,
        FsFreeMem                        = 0x3Du,
        FsRelease                        = 0x3Fu,
    //Общие команды для формирования фискальных документов
        CancelDocument                   = 0x06u,
        SendDocumentData                 = 0x07u,
    //Команды фискальзации и завершения фискального режима
//        BeginRegistrationDoc             = 0x02u, //ФФД < 1.1. Не буду реализовывать
//        CompleteRegistrationDoc          = 0x03u, //ФФД < 1.1. Не буду реализовывать
        StartFsClosing                   = 0x04u,
        CloseFsFiscalMode                = 0x05u,
    //Команды формирования фискальных документов о расчетах
        CurrentCycleData                 = 0x10u,
        BeginCycleOpening                = 0x11u,
        CompleteCycleOpening             = 0x12u,
        BeginCycleClosing                = 0x13u,
        CloseCycle                       = 0x14u,
        BeginCheckCreating               = 0x15u,
        CompleteCheckCreating            = 0x16u,
        BeginCorrectionCheckCreating     = 0x17u,
        BeginPaymentStateReport          = 0x18u,
        CompletePaymentStateReport       = 0x19u,
    //Команды информационного обмена с сервером ОФД
        TransfereStatus                  = 0x20u,
        SendOfdConnectionState           = 0x21u,
        BeginOfdMsgReading               = 0x22u,
        ReadOfdMsgBlock                  = 0x23u,
        CancelOfdMsgReading              = 0x24u,
        FinishOfdMsgReading              = 0x25u,
        SendOfdReceipt                   = 0x26u,
    //Команды получения данных из архива ФН
        FsDocByNumber                    = 0x40u,
        FsDocReceiptByNumber             = 0x41u,
        OfflineDocsCount                 = 0x42u,
        FiscalisationSummary             = 0x43u,
        FiscalisationParameter           = 0x44u,
        FsDocumentTLV                    = 0x45u,
        FsDocumentTLVReading             = 0x46u,
        FsopenDocumentTLVReading         = 0x47u,
    // Команды получения данных из Архива ФН в режиме поддержки ФФД 1.1 и 1.2 .
        FsDocByNumber1_2                 = 0x50u,
    //Отладочные команды
        ResetFsState                     = 0x60u,
    //Команды формирования отчетов о регистрации/перерегистрации
        BeginRegistrationDoc2            = 0xA2u,
        CompleteRegistrationDoc2         = 0xA3u,
    //Служебные команды ФН
        TotalDataForB7                   = 0xA7u,
        IncreasedSpeed                   = 0xA8u,
    // Команды для работы с маркированными товарами Команда  – Запрос статуса ФН по работе с ко
        FsForLabelsStatus                = 0xB0u,
        SendLabelForFsCheck              = 0xB1u,
        SaveLabelCheckResult             = 0xB2u,
        CleanLabelCheckResult            = 0xB3u,
        CreateLabelRequest               = 0xB5u,
        SendReplyAboutLabel              = 0xB6u,
        SendFiscalDataForLabels          = 0xB7u,
    //Команды для работы с уведомлениями о реализации маркированного товара . Алгоритм передачи уведомлений
        GetNotificationState             = 0xBAu,
        StartNotificationReading         = 0xBBu,
        ReadNotificationBlock            = 0xBCu,
        CancelNotificationReading        = 0xBDu,
        FinishNotificationReading        = 0xBEu,
        SendNotificationTicket           = 0xBFu,
     // Команды обмена с сервером обновления ключей проверки КМ
        GetRequestForKeysUpdate          = 0xD0u,
        SendReplyForKeysUpdate           = 0xD1u,
        GetIngoForKeysUpdate             = 0xD7u,
    // Команды для выгрузки уведомлений для ККТ в автономном режиме.
        StartUploadSession               = 0xD3u,
        ReadNextNotification             = 0xD4u,
        ReadCurrentNotificationBlock     = 0xD5u,
        CommitNotificationUpload         = 0xD6u,


        InvalidCode                      = 0xFFu
    };

    enum class RepParsingRes: quint8
    {
        Ok                               = 0x00u,
        NoMsgStart                       = 0x01u,
        InvalidPackgSize                 = 0x02u,
        InvalidLen                       = 0x03u,
        InvalidCrc                       = 0x04u,
        InvalidData                      = 0x05u,
        NoParsing                        = 0xffu
    };
    enum class RegType: quint8
    {
        NewRegistration                  = 0x00, //Регистрация новой кассы
        FsChanging                       = 0x01, //Перерегистрация с заменой фн
        KktSettingsChanging              = 0x02 //Перерегистрация без замены фн
    };
    using FsRwTimeout = QPair<qint32, qint32> ;

    enum class KeysStatus: quint8
    {
        KeysUpdated             = 0,
        Keys15                  = 1,
        Keys60                  = 2
    };
}

class FsMsg
{
public:
    static const quint8 MSG_START       = 0x04u;
    static const quint8 INVALID_CODE    = 0x7Fu;
    virtual bool isValid() const = 0;

    quint8 code() const;
    QByteArray data() const;

    virtual QByteArray rawData() const;
    virtual fs::RepParsingRes setRawData(const QByteArray &data);


    static QByteArray serializeDt(const QDateTime &dt);
    static QDateTime parseDt(const QByteArray &raw);

    static QByteArray serializeDate(const QDate &date, bool zeroTime);
    static QDate parseDate(const QByteArray &raw);


    quint8 code_;
    QByteArray data_;

    FsMsg(quint8 code, const QByteArray &data)noexcept;
    FsMsg(const FsMsg &other)noexcept;
    FsMsg(FsMsg &&other)noexcept;
    virtual ~FsMsg();

    FsMsg &operator=(const FsMsg & other)noexcept;
    FsMsg &operator=(FsMsg &&other)noexcept;
    bool operator==(const FsMsg &other) const noexcept;
    bool operator!=(const FsMsg &other) const noexcept;

    virtual bool checkTlvStructure() const;
};


//--------------------------------------------------------------------------------------------------



class FsReq : public FsMsg
{
public:
    FsReq();
    explicit FsReq(fs::ReqCode code);
    FsReq(fs::RegType regType, fs::FFD ffd); //Передавать при regtype=1 причины перерегистрации! и в конструктор!

    FsReq(fs::ReqCode code, const QByteArray &docData);
    FsReq(const FsReq &other) noexcept;
    FsReq(FsReq &&other) noexcept;
    virtual ~FsReq() override;

    virtual bool isValid() const override;

    fs::ReqCode reqCode() const;

    quint16 expRepSz() const;

    FsReq &operator=(const FsReq &other) noexcept;
    FsReq &operator=(FsReq &&other) noexcept;
    bool operator==(const FsReq &other) const noexcept;
    bool operator!=(const FsReq &other) const noexcept;

private:
    quint16 expRepSz_;
};

//--------------------------------------------------------------------------------------------------

class FsRep : public FsMsg
{
public:
    FsRep();
    explicit FsRep(const QByteArray &rawData);
    FsRep(const FsRep &other);
    FsRep(FsRep &&other);
    virtual ~FsRep() override;

    virtual bool isValid() const override;
    virtual fs::RepParsingRes setRawData(const QByteArray &data) override;

    fs::RepParsingRes parsingResult() const;

    fs::RepCode repCode() const;
    CoreApiConst::ErrorCode repError() const;
    void setRepCode(fs::RepCode code);
    bool hasWarning() const;

    fs::RepParsingRes getFsStatus(std::optional<FsStatus> &status) const;
    fs::RepParsingRes getFsNumber(QString &serial) const;
    fs::RepParsingRes getFsLifeTime(std::optional<FsLifeTime> &lifeTime) const;
    fs::RepParsingRes getFsVersion(QString &version, bool &isRelease) const;

    fs::RepParsingRes getLastFsErrors(QByteArray &dump) const;
    fs::RepParsingRes getFsRelease(QString &release) const;
    fs::RepParsingRes getFsFormat(fs::FFD &curFfd, fs::FFD &supportedFFd) const;
    fs::RepParsingRes getExpiredDays(qint32 &days) const;
    fs::RepParsingRes getMemResource(std::optional<FsMemResourceInfo>& resource) const;
    fs::RepParsingRes getFsCycle(std::optional<CycleStatus>& cycle) const;
    fs::RepParsingRes getFsTransport(std::optional<FsTransportState>& tranport) const;
    fs::RepParsingRes getDocResult(qint32 &docNumber, quint32 &fiscalCode) const;
//    fs::RepParsingRes getFsOfdTransportState(FsOfdTransportState &state) const;
//    fs::RepParsingRes getOfdMsgSize(quint16 &sz) const;
//    fs::RepParsingRes getCurrentCycleData(bool &opened, quint16 &cycleNumber, quint16 &checkNumber) const;
//    fs::RepParsingRes getCheckData(quint16 &checkNumber,quint32 &docNumber, quint32 &fiscalCode) const;
//    fs::RepParsingRes getPsRepData(quint32 offLineDocs, QDate &offlineDocDate,
//                                    quint32 &docNumber, quint32 &fiscalCode) const;

    FsRep &operator=(const FsRep &other);
    FsRep &operator=(FsRep &&other);
    bool operator==(const FsRep &other) const;
    bool operator!=(const FsRep &other) const;


private:
    fs::RepParsingRes parsingResult_;
};
#endif // FSMSG_H
