#include "ofdfscmd.h"

OfdFsCmd::OfdFsCmd() noexcept
    : cmd_(Code::Invalid)
    , params_()
{

}

OfdFsCmd::OfdFsCmd(Code code) noexcept
    : cmd_(code)
    , params_()
{

}

OfdFsCmd::OfdFsCmd(const QVariantMap &map) noexcept
    : cmd_(Code::Invalid)
    , params_()
{
    parseMap(map);
}

OfdFsCmd::OfdFsCmd(const OfdFsCmd &other) noexcept
    : cmd_(other.cmd_)
    , params_(other.params_)

{

}

OfdFsCmd::OfdFsCmd(OfdFsCmd &&other) noexcept
    : cmd_(other.cmd_)
    , params_()

{
    params_.swap(other.params_);
}

OfdFsCmd::~OfdFsCmd()
{

}

bool OfdFsCmd::isValid() const
{
    return cmd_ >= Code::TransfereStatus && cmd_ <= Code::WriteNotificationTicket;
}

bool OfdFsCmd::cmdIsTransport() const
{
    return cmd_ >= Code::SendOfdConnectionState &&
            cmd_ <= Code::SendOfdReceipt;
}

bool OfdFsCmd::cmdIsLabelTransport() const
{
    return cmd_ >= Code::ReadNotification &&
            cmd_ <= Code::WriteNotificationTicket;
}

OfdFsCmd::Code OfdFsCmd::cmd() const
{
    return cmd_;
}

void OfdFsCmd::setCmd(Code newCmd)
{
    cmd_ = newCmd;
}

const QVariantMap &OfdFsCmd::params() const
{
    return params_;
}

void OfdFsCmd::setParams(const QVariantMap &newParams)
{
    params_ = newParams;
}

QVariantMap OfdFsCmd::toMap() const
{
    QVariantMap res;
    if(!isValid()) return res;
    res["cmd"] = static_cast<qint32>(cmd_);
    if(!params_.isEmpty()) res["params"] = params_;
    return res;
}

void OfdFsCmd::parseMap(const QVariantMap &map)
{
    cmd_ = static_cast<Code>(map["cmd"].toInt());
    params_ = map["params"].toMap();
}

CoreTransaction OfdFsCmd::transaction() const
{
    if(!isValid()) return CoreTransaction();
    CoreTransaction res;
    res.setOperation(CoreTransaction::Operation::OFD_2_FS_CMD);
    res.setParams(toMap());
    res.genUid();
    return res;
}

bool OfdFsCmd::ofdConnected() const
{
    return params_.contains("connected") && params_["connected"].toInt();
}

void OfdFsCmd::setOfdConnected(bool connected)
{
    params_["connected"] = (connected ? 1 : 0);
}

QByteArray OfdFsCmd::ofdTicket() const
{
    QByteArray res;
    if(params_.contains("ticket"))
    {
        res = QByteArray::fromHex(params_["ticket"].toString().toLatin1());
    }
    return res;
}

void OfdFsCmd::setOfdTicket(const QByteArray &data)
{
    params_.clear();
    if(!data.isEmpty())
    {
        params_["ticket"] = QString::fromLatin1(data.toHex());
    }
}

OfdFsCmd &OfdFsCmd::operator =(const OfdFsCmd &other) noexcept
{
    cmd_ = other.cmd_;
    params_ = other.params_;
    return *this;
}

OfdFsCmd &OfdFsCmd::operator =( OfdFsCmd &&other) noexcept
{
    cmd_ = other.cmd_;
    params_.swap(other.params_);
    return *this;
}


bool OfdFsCmd::operator ==(const OfdFsCmd &other) const noexcept
{
    return cmd_ == other.cmd_ &&
            params_ == other.params_;
}

bool OfdFsCmd::operator !=(const OfdFsCmd &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------
OfdTicketResult::OfdTicketResult() noexcept
    : fsCode_(fdf::FsTicketAnswerCode::Ok)
    , ofdCode_(fdf::OfdAnswerCode::Ok)
    , ofdMessage_()
    , oismProps_()
{

}

OfdTicketResult::OfdTicketResult(const QVariantMap &map) noexcept
    : fsCode_(fdf::FsTicketAnswerCode::Ok)
    , ofdCode_(fdf::OfdAnswerCode::Ok)
    , ofdMessage_()
    , oismProps_()
{
    parseMap(map);
}

OfdTicketResult::OfdTicketResult(const OfdTicketResult &other) noexcept
    : fsCode_(other.fsCode_)
    , ofdCode_(other.ofdCode_)
    , ofdMessage_(other.ofdMessage_)
    , oismProps_(other.oismProps_)


{

}

OfdTicketResult::OfdTicketResult(OfdTicketResult &&other) noexcept
    : fsCode_(other.fsCode_)
    , ofdCode_(other.ofdCode_)
    , ofdMessage_(other.ofdMessage_)
    , oismProps_()

{
    oismProps_.swap(other.oismProps_);
}

OfdTicketResult::~OfdTicketResult()
{

}

fdf::FsTicketAnswerCode OfdTicketResult::fsCode() const
{
    return fsCode_;
}

void OfdTicketResult::setFsCode(fdf::FsTicketAnswerCode newFsCode)
{
    fsCode_ = newFsCode;
}

fdf::OfdAnswerCode OfdTicketResult::ofdCode() const
{
    return ofdCode_;
}

void OfdTicketResult::setOfdCode(fdf::OfdAnswerCode newOfdCode)
{
    ofdCode_ = newOfdCode;
}

const Tlv &OfdTicketResult::ofdMessage() const
{
    return ofdMessage_;
}

void OfdTicketResult::setOfdMessage(const Tlv &newOfdMessage)
{
    ofdMessage_ = newOfdMessage;
}

const Tlv::Stlv &OfdTicketResult::oismProps() const
{
    return oismProps_;
}

void OfdTicketResult::setOismProps(const Tlv::Stlv &newOismProps)
{
    oismProps_ = newOismProps;
}


QVariantMap OfdTicketResult::toMap() const
{
    QVariantMap res;
    res["fsCode"] = static_cast<qint32>(fsCode_);
    res["ofdCode"] = static_cast<qint32>(ofdCode_);
    if(ofdMessage_.isValid())
    {
        res["ofdMessage"] = QString::fromLatin1(ofdMessage_.rawData().toHex());
    }
    if(!oismProps_.isEmpty())
    {
        QVariantList list;
        for(const Tlv &t: oismProps_)
        {
            QVariantMap m;
            m["tag"] = static_cast<quint32>(t.tag());
            m["value"] = QString::fromLatin1(t.value().toHex());
            list << m;
        }
        res["oismProps"] = list;
    }
    return res;
}

void OfdTicketResult::parseMap(const QVariantMap &map)
{
    fsCode_ = static_cast<fdf::FsTicketAnswerCode>(map["fsCode"].toInt());
    ofdCode_ = static_cast<fdf::OfdAnswerCode>(map["ofdCode"].toInt());
    ofdMessage_.setRawData(QByteArray::fromHex(map["ofdMessage"].toString().toLatin1()));
    oismProps_.clear();
    if(map.contains("oismProps"))
    {
        QVariantList list = map["oismProps"].toList();
        for(const QVariant &v :list)
        {
            Tlv t;
            t.setTag(static_cast<fdf::Tag>(v.toMap()["tag"].toUInt()));
            t.setValue(QByteArray::fromHex(v.toMap()["value"].toString().toLatin1()));
            oismProps_ << t;
        }
    }
}

OfdTicketResult &OfdTicketResult::operator =(const OfdTicketResult &other) noexcept
{
    fsCode_ = other.fsCode_;
    ofdCode_ = other.ofdCode_;
    ofdMessage_ = other.ofdMessage_;
    return *this;
}
OfdTicketResult &OfdTicketResult::operator =(OfdTicketResult &&other) noexcept
{
    fsCode_ = other.fsCode_;
    ofdCode_ = other.ofdCode_;
    ofdMessage_ = other.ofdMessage_;
    return *this;

}

bool OfdTicketResult::operator ==(const OfdTicketResult &other) const noexcept
{
    return fsCode_ == other.fsCode_ &&
            ofdCode_ == other.ofdCode_ &&
            ofdMessage_ == other.ofdMessage_ ;
}

bool OfdTicketResult::operator !=(const OfdTicketResult &other) const noexcept
{
    return !(*this == other);
}

//--------------------------------------------------------------------------------------------------

FsOfdResponse::FsOfdResponse(const OfdFsCmd &cmd) noexcept
    : cmd_(cmd)
    , error_(CoreApiConst::ErrorCode::Ok)
    , data_()
{

}

FsOfdResponse::FsOfdResponse(const QVariantMap &map) noexcept
    : cmd_()
    , error_(CoreApiConst::ErrorCode::Ok)
    , data_()
{
    parseMap(map);
}

FsOfdResponse::FsOfdResponse(const FsOfdResponse &other) noexcept
    : cmd_(other.cmd_)
    , error_(other.error_)
    , data_(other.data_)

{

}

FsOfdResponse::FsOfdResponse(FsOfdResponse &&other) noexcept
    : cmd_(other.cmd_)
    , error_(other.error_)
    , data_()

{
    data_.swap(other.data_);
}

FsOfdResponse::~FsOfdResponse()
{

}

const OfdFsCmd &FsOfdResponse::cmd() const
{
    return cmd_;
}

void FsOfdResponse::setCmd(const OfdFsCmd &newCmd)
{
    cmd_ = newCmd;
}

CoreApiConst::ErrorCode FsOfdResponse::error() const
{
    return error_;
}

void FsOfdResponse::setError(CoreApiConst::ErrorCode newError)
{
    error_ = newError;
}

const QVariantMap &FsOfdResponse::data() const
{
    return data_;
}

void FsOfdResponse::setData(const QVariantMap &newData)
{
    data_ = newData;
}

QVariantMap FsOfdResponse::toMap() const
{
    QVariantMap res;
    res["cmd"] = cmd_.toMap();
    res["error"] = static_cast<qint32>(error_);
    if(!data_.isEmpty()) res["data"] = data_;
    return res;
}

void FsOfdResponse::parseMap(const QVariantMap &map)
{
    cmd_.parseMap(map["cmd"].toMap());
    error_ = static_cast<CoreApiConst::ErrorCode>(map["error"].toInt());
    data_ = map["data"].toMap();
}

QByteArray FsOfdResponse::ofdMessage() const
{
    QByteArray res;
    if(data_.contains("ofdMessage"))
    {
        res = QByteArray::fromHex(data_["ofdMessage"].toString().toLatin1());
    }
    return res;
}

void FsOfdResponse::setOfdMessage(const QByteArray &msg)
{
    data_["ofdMessage"] = QString::fromLatin1(msg.toHex());
}

FsTransportState FsOfdResponse::transportState() const
{
    return FsTransportState(data_["transport"].toMap());
}

void FsOfdResponse::setTransportState(const FsTransportState &state)
{
    data_["transport"] = state.toMap();
}

OfdTicketResult FsOfdResponse::ticketResult() const
{
    return OfdTicketResult(data_["ticketResult"].toMap());
}

void FsOfdResponse::setTicketResult(const OfdTicketResult &res)
{
    data_["ticketResult"] = res.toMap();
}

QByteArray FsOfdResponse::errorDump() const
{
    QByteArray res;
    if(data_.contains("errorDump"))
    {
        res = QByteArray::fromHex(data_["errorDump"].toString().toLatin1());
    }
    return res;
}

void FsOfdResponse::setErrorDump(const QByteArray &dump)
{
    data_.remove("errorDump");
    if(!dump.isEmpty())
    {
        data_["errorDump"] = QString::fromLatin1(dump.toHex());
    }
}

CoreApiResult FsOfdResponse::toResult() const
{
    if(!cmd_.isValid()) return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsCmd);
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), toMap());
}

FsOfdResponse &FsOfdResponse::operator =(const FsOfdResponse &other) noexcept
{
    cmd_ = other.cmd_;
    error_ = other.error_;
    data_ = other.data_;
    return *this;
}
FsOfdResponse &FsOfdResponse::operator =(FsOfdResponse &&other) noexcept
{
    cmd_ = other.cmd_;
    error_ = other.error_;
    data_.swap(other.data_);
    return *this;
}
bool FsOfdResponse::operator ==(const FsOfdResponse &other) const noexcept
{
    return cmd_ == other.cmd_ &&
            error_ == other.error_ &&
            data_ == other.data_;
}

bool FsOfdResponse::operator !=(const FsOfdResponse &other) const noexcept
{
    return !(*this == other);
}

