#include "ofdmsgcontainer.h"
#include "crc.h"


#include <QDataStream>

BaseMsgContainer::~BaseMsgContainer()
{

}

const QString &BaseMsgContainer::fsNumber() const
{
    return fsNumber_;
}

const QByteArray &BaseMsgContainer::container() const
{
    return container_;
}

QByteArray BaseMsgContainer::serialize() const
{
    if(fsNumber_.size() != 16)
    {
        return QByteArray();
    }
    QByteArray res;
    QDataStream ds(&res, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds.writeRawData(sign_.data(), 4);
    ds.writeRawData(s_.data(), 2);
    ds.writeRawData(p_.data(), 2);
    ds.writeRawData(fsNumber_.toLatin1().constData(), 16);
    ds << static_cast<quint16>(container_.size()) << flags_;
    quint16 crc = 0u;
    if(flags_ & 0x0003)
    {
        Crc16CITT crc16;
        crc16.startCalculating();
        crc16.addData(res);
        if(flags_ & 0x0002) crc16.addData(container_);
        crc = crc16.finishCalculating();
    }
    ds << crc;
    return res + container_;
}

BaseMsgContainer::BaseMsgContainer()
    : sign_()
    , s_()
    , p_()
    , fsNumber_()
    , len_(0)
    , flags_(0)
    , crc_()
    , container_()
{

}

BaseMsgContainer::BaseMsgContainer(const Buf4 &sign, const Buf2 &s, const Buf2 &p, quint16 flags,
                                   const QString &fs, const QByteArray &container)
    : sign_(sign)
    , s_(s)
    , p_(p)
    , fsNumber_(fs)
    , len_(static_cast<quint16>(container.length()))
    , flags_(flags)
    , crc_(0)
    , container_(container)
{

}

BaseMsgContainer::BaseMsgContainer(const QString &fs, const QByteArray &raw)
    : sign_()
    , s_()
    , p_()
    , fsNumber_(fs)
    , len_(0)
    , flags_(0)
    , crc_()
    , container_()
{
    Q_UNUSED(raw)
}

bool BaseMsgContainer::parse(const QByteArray &raw)
{

    if(raw.size() < 30) return false;
    container_ = raw.mid(30);
    QByteArray header = raw.mid(0, 30);
    sign_ = header.mid(0, 2);
    s_ = header.mid(4, 2);
    p_ = header.mid(6, 2);

    QString fsNumber = QString::fromLatin1(header.mid(8, 16));
    if(!checkHeader()) return false;
    QDataStream ds(header);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds.skipRawData(24);
    ds >> len_ >> flags_ >> crc_;
    quint16 f = flags_ &  0x0003;
    switch (f) {
    case 0: return true;
    case 1:
    {
        Crc16CITT crc16;
//			lmWarning() << crc16(header.mid(0, 28)) << crc_ << logbinary(header.mid(0, 28));
        return crc16(header.mid(0, 28)) == crc_;
    }
    case 2:
    {
        Crc16CITT crc16;
        crc16.startCalculating();
        crc16.addData(header.mid(0, 28));
        crc16.addData(container_);
        quint16 crc = crc16.finishCalculating();
        return true;
    }
    default: return false;
    }
    return false;
}

//--------------------------------------------------------------------------------------------------
const QByteArray OfdMsgContainer::HEADER = "\x2A\x08\x41\x0A";
const QByteArray OfdMsgContainer::S_VERSION = "\x81\xA2";
const QByteArray OfdMsgContainer::P_VERSION = "\x01\x20";



OfdMsgContainer::OfdMsgContainer(quint16 flags, const QString &fs, const QByteArray &container)
    : BaseMsgContainer(HEADER, S_VERSION, P_VERSION, flags, fs, container)
{

}

OfdMsgContainer::OfdMsgContainer(const QString &fs, const QByteArray &raw)
    : BaseMsgContainer(fs, raw)
{
    parse(raw);
}

bool OfdMsgContainer::checkHeader() const
{
    return sign_ == HEADER &&
            s_ == S_VERSION;
}

//--------------------------------------------------------------------------------------------------
const QByteArray KeysMsgContainer::HEADER = "\xDD\x80\xCA\xA1";
const QByteArray KeysMsgContainer::S_VERSION = "\x82\xA1";
const QByteArray KeysMsgContainer::P_VERSION = QByteArray("\x00\x01", 2);

KeysMsgContainer::KeysMsgContainer(quint16 flags, const QString &fs, const QByteArray &container)
    : BaseMsgContainer(HEADER, S_VERSION, P_VERSION, flags, fs, container)
{

}

KeysMsgContainer::KeysMsgContainer(const QString &fs, const QByteArray &raw)
    : BaseMsgContainer(fs, raw)
{
    parse(raw);
}

bool KeysMsgContainer::checkHeader() const
{
    return sign_ == HEADER &&
            s_ == S_VERSION;
}

//--------------------------------------------------------------------------------------------------
const QByteArray LabelTestContainer::HEADER = "\xDD\x80\xCA\xA1";
const QByteArray LabelTestContainer::S_VERSION = "\x82\xA2";
const QByteArray LabelTestContainer::P_VERSION = QByteArray("\x00\x01", 2);

LabelTestContainer::LabelTestContainer(quint16 flags, const QString &fs, const QByteArray &container)
    : BaseMsgContainer(HEADER, S_VERSION, P_VERSION, flags, fs, container)
{

}

LabelTestContainer::LabelTestContainer(const QString &fs, const QByteArray &raw)
//    : BaseMsgContainer(fs, raw)
{
    Q_UNUSED(fs)
    parse(raw);
}

bool LabelTestContainer::checkHeader() const
{
    return sign_ == HEADER &&
            s_ == S_VERSION;
}

//--------------------------------------------------------------------------------------------------
const QByteArray LabelNotifyContainer::HEADER = "\xDD\x80\xCA\xA1";
const QByteArray LabelNotifyContainer::S_VERSION = "\x82\xA2";
const QByteArray LabelNotifyContainer::P_VERSION = QByteArray("\x00\x01", 2);

LabelNotifyContainer::LabelNotifyContainer(quint16 flags, const QString &fs, const QByteArray &container)
    : BaseMsgContainer(HEADER, S_VERSION, P_VERSION, flags, fs, container)
{

}

LabelNotifyContainer::LabelNotifyContainer(const QString &fs, const QByteArray &raw)
{
    Q_UNUSED(fs)
    parse(raw);
}

bool LabelNotifyContainer::checkHeader() const
{
    return sign_ == HEADER &&
            s_ == S_VERSION;
}
