#include "tlv.h"

#include "fsfullstatus.h"
#include "formatutils.h"

#include <QTextCodec>

Tlv::Tlv() noexcept
    : tag_(Tag::Invalid)
    , value_()
{

}

Tlv::Tlv(const Tlv &other) noexcept
    : tag_(other.tag_)
    , value_(other.value_)
{

}

Tlv::Tlv(Tlv &&other) noexcept
    : tag_(other.tag_)
    , value_()
{
    value_.swap(other.value_);
}

Tlv::~Tlv()
{

}

bool Tlv::isValid() const
{
    return tag_ != Tag::Invalid  && value_.size() < 65535;
}

quint16 Tlv::len() const
{
    return static_cast<quint16>(value_.size());
}

qint32 Tlv::rawLen() const
{
    return 4 + len();
}

fdf::Tag Tlv::tag() const
{
    return tag_;
}

void Tlv::setTag(fdf::Tag newTag)
{
    tag_ = newTag;
}

const QByteArray &Tlv::value() const
{
    return value_;
}

void Tlv::setValue(const QByteArray &newValue)
{
    value_ = newValue;
}

QByteArray Tlv::rawData() const
{
    if(!isValid()) return QByteArray();
    QByteArray buf;
    QDataStream ds(&buf, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << static_cast<quint16>(tag_) << len();
    ds.writeRawData(value_.constData(), value_.size());
    return buf;
}

bool Tlv::setRawData(const QByteArray &raw)
{
    tag_ = Tag::Invalid;
    value_.clear();
    if(raw.size() < 4)
    {
//        lmWarning() << logbinary(raw);
        return false;
    }
    QDataStream ds(raw);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint16 buf = 0, len = 0;
    ds >> buf >> len;
    if(raw.size() != 4 + len)
    {
        return false;
    }
    value_.clear();
    tag_ = static_cast<Tag>(buf);
    if(len == 0)return true;
    value_ = raw.mid(4, len);
    return true;
}

Tlv::Stlv Tlv::toStlv(bool *ok) const
{
    if(ok)*ok = false;
    Tlv::Stlv res;
    if(!isValid()) return res;
    QDataStream ds(value_);
    ds.setByteOrder(QDataStream::LittleEndian);
    while(!ds.atEnd())
    {
        Tlv tlv;
        ds >> tlv;
        if(tlv.isValid()) res << tlv;
        else continue;
    }
    if(ok)*ok = !res.isEmpty();
    return res;
}

void Tlv::setStlv(const Stlv &list)
{
    value_.clear();
    QDataStream ds(&value_, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    for(const Tlv &t: list) ds << t;
}

void Tlv::setStlv(const fdf::Tag &tag, const Stlv &list)
{
    setTag(tag);
    setStlv(list);
}

Tlv::MapStlv Tlv::toMapStlv(bool *ok) const
{
    if(ok)*ok = false;
    Tlv::MapStlv res;
    if(!isValid()) return res;
    QDataStream ds(value_);
    ds.setByteOrder(QDataStream::LittleEndian);
    while(!ds.atEnd())
    {
        Tlv tlv;
        ds >> tlv;
        if(tlv.isValid()) res.insert(tlv.tag(), tlv);
        else return MapStlv();
    }
    if(ok)*ok = !res.isEmpty();
    return res;
}

Tlv::MultiStlv Tlv::toMultiStlv(bool *ok) const
{
    if(ok)*ok = false;
    Tlv::MultiStlv res;
    if(!isValid()) return res;
    QDataStream ds(value_);
    ds.setByteOrder(QDataStream::LittleEndian);
    while(!ds.atEnd())
    {
        Tlv tlv;
        ds >> tlv;
        if(tlv.isValid()) res.insert(tlv.tag(), tlv);
        else return MapStlv();
    }
    if(ok)*ok = !res.isEmpty();
    return res;
}

quint8 Tlv::toByte(bool *ok) const
{
    if(ok)*ok = false;
    quint8 res = 0u;
    if(value_.size() != 1) return res;
    res = static_cast<quint8>(value_[0]);
    if(ok) *ok = true;
    return res;
}

void Tlv::setByte(quint8 byte)
{
    value_.resize(1);
    value_[0] = static_cast<char>(byte);
}

void Tlv::setByte(char byte)
{
    value_.resize(1);
    value_[0] = byte;
}

void Tlv::setByte(const fdf::Tag &tag, quint8 byte)
{
    tag_ = tag;
    setByte(byte);
}

void Tlv::setByte(const fdf::Tag &tag, char byte)
{
    tag_ = tag;
    setByte(byte);
}

bool Tlv::toBool(bool *ok) const
{
    quint8 byte = toByte(ok);
    return byte == 1;
}

void Tlv::setBool(bool byte)
{
    setByte(static_cast<quint8>(byte ? 1u: 0u));
}

void Tlv::setBool(const fdf::Tag &tag, bool byte)
{
    tag_ = tag;
    setBool(byte);
}

quint16 Tlv::toUInt16(bool *ok) const
{
    if(ok)*ok = false;
    quint16 res = 0u;
    if(value_.size() != 2) return res;
    QDataStream ds(value_);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> res;
    if(ok) *ok = true;
    return res;
}

void Tlv::setUInt16(quint16 val)
{
    value_.clear();
    QDataStream ds(&value_, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val;
}

void Tlv::setUInt16(const fdf::Tag &tag, quint16 val)
{
    tag_ = tag;
    setUInt16(val);
}

quint32 Tlv::toUInt32(bool *ok) const
{
    if(tag_ == Tag::FPD && value_.size() == 6)
    {
        Tlv t;
        QByteArray src = value_.mid(2);
        QByteArray dst = src;
        std::copy(src.rbegin(), src.rend(), dst.begin());
        t.setTag(tag_);
        t.setValue(dst);
        return t.toUInt32(ok);
    }
    if(ok)*ok = false;
    quint32 res = 0u;
    if(value_.size() != 4) return res;
    QDataStream ds(value_);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> res;
    if(ok) *ok = true;
    return res;
}

void Tlv::setUInt32(quint32 val)
{
    value_.clear();
    QDataStream ds(&value_, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << val;
}

void Tlv::setUInt32(const fdf::Tag &tag, quint32 val)
{
    tag_ = tag;
    setUInt32(val);
}

qint64 Tlv::toVln(bool *ok) const
{
    if(ok)*ok = false;
    qint64 res = -1;
    QByteArray ba = value_;
    if(ba.size() < 8)
    {
        ba.append(QByteArray(8 - ba.size(), '\x00'));
    }
    QDataStream ds(ba);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds >> res;
    if(ok)*ok = true;
    return res;
}

void Tlv::setVln(qint64 val)
{
    quint64 buf = static_cast<quint64>(val);
    QByteArray ba;
    QDataStream ds(&ba, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds<< buf;
    while(!ba.isEmpty())
    {
        if(ba[ba.size() - 1] == '\x00') ba = ba.mid(0, ba.size() - 1);
        else break;
    }
    if(ba.isEmpty()) ba.push_back('\x00');
    value_ = ba;
}

void Tlv::setVln(const fdf::Tag &tag, qint64 val)
{
    tag_ = tag;
    setVln(val);
}

FixNumber Tlv::toFvln(bool *ok) const
{
    if(ok)*ok = false;
    QByteArray ba = value_;
    if(ba.size() < 9)
    {
        ba.append(QByteArray(9 - ba.size(), '\x00'));
    }
    QDataStream ds(ba);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint8 dp = 0;
    qint64 res = 0;
    ds >> dp >> res;
    FixNumber fn;
    fn.setDecimal(dp);
    fn.setValue(res);
    return fn;
}

void Tlv::setFvln(const FixNumber &val)
{
    quint64 buf = val.value();
    QByteArray ba;
    QDataStream  ds(&ba, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << buf;
    while(!ba.isEmpty())
    {
        if(ba[ba.size() - 1] == '\x00') ba = ba.mid(0, ba.size() - 1);
        else break;
    }

    if(ba.isEmpty()) ba.push_back('\x00');
    ba.push_front((char)val.decimals());
    value_ = ba;
}

void Tlv::setFvln(const fdf::Tag &tag, const FixNumber &val)
{
    tag_ = tag;
    setFvln(val);
}

QDateTime Tlv::toDt(bool *ok) const
{
    if(ok) *ok =false;
    bool rdok = false;
    quint32 buf = toUInt32(&rdok);
    if(!rdok)
    {
        return QDateTime();
    }
    QDateTime dt = QDateTime::fromTime_t(buf).toUTC();
//    lmWarning() << DT2STR_(dt);
    dt.setTimeSpec(Qt::LocalTime);
//    lmWarning() << DT2STR_(dt);
    if(ok)*ok = dt.isValid();
    return dt;
}

void Tlv::setDt(const QDateTime &dt)
{
//    lmWarning() << DT2STR_(dt);
    QDateTime bdt = dt;
    //Дает правильный результат в электронной форме. видимо рассчитано на tm to time_t
    bdt.setTimeSpec(Qt::UTC);
    quint32 buf = bdt.toTime_t();
    setUInt32(buf);
}

void Tlv::setDt(const fdf::Tag &tag, const QDateTime &dt)
{
    tag_ = tag;
    setDt(dt);
}

QDate Tlv::toDate(bool *ok) const
{
    bool ready = false;
    if(ok)*ok = false;
    QDateTime dt = toDt(&ready);
    if(!ready || !dt.isValid())
    {
        return QDate();
    }
    if(ok)*ok = true;
    return dt.date();
}

void Tlv::setDate(const QDate &dt)
{
    setDt(QDateTime(dt, QTime(0, 0, 0)));
}

void Tlv::setDate(const fdf::Tag &tag, const QDate &dt)
{
    tag_ = tag;
    setDate(dt);;
}

QString Tlv::toString() const
{
    if(value_.isEmpty()) return QString();

    QTextCodec *codec = QTextCodec::codecForName(fs::CODEC_NAME);
    if(!codec) return QString();
    return codec->toUnicode(value_);
}

void Tlv::setString(const QString &str)
{
    value_.clear();
    if(!str.isEmpty())
    {
        QTextCodec *codec = QTextCodec::codecForName(fs::CODEC_NAME);
        if(codec)
        {
            value_ = codec->fromUnicode(str);
        }
    }
}

void Tlv::setString(const fdf::Tag &tag, const QString &str)
{
    tag_ = tag;
    setString(str);
}

void Tlv::clean()
{
    tag_ = Tag::Invalid;
    value_.clear();
}

Inn Tlv::toInn(bool *ok) const
{
    QString str = QString::fromLatin1(value_).trimmed();
    Inn res (str);
    if(ok)*ok = res.isValid();
    return res;
}

void Tlv::setInn(const Inn &inn)
{
    value_.clear();;
    if(inn.isValid()) value_ = inn.data();
    else value_ = "000000000000";
}

void Tlv::setInn(const QString &inn)
{
    setInn(Inn(inn));
}

void Tlv::setInn(const fdf::Tag &tag, const Inn &inn)
{
    tag_ = tag;
    setInn(inn);
}

void Tlv::setInn(const fdf::Tag &tag, const QString &inn)
{
    tag_ = tag;
    setInn(inn);
}

Tlv &Tlv::operator =(const Tlv &other) noexcept
{
    tag_ = other.tag_;
    value_ = other.value_;
    return *this;
}

Tlv &Tlv::operator =(Tlv &&other) noexcept
{
    tag_ = other.tag_;
    value_.swap(other.value_);
    return *this;
}


bool Tlv::operator ==(const Tlv &other) const noexcept
{
    return tag_ == other.tag_ &&
            value_ == other.value_;
}

bool Tlv::operator !=(const Tlv &other) const noexcept
{
    return tag_ != other.tag_ ||
            value_ != other.value_;
}



QDataStream &operator << (QDataStream &ds, const Tlv &tlv)
{
    if(tlv.isValid())
    {
        QByteArray buf = tlv.rawData();
        ds.writeRawData(buf.constData(), buf.size());
    }
    return ds;
}


QDataStream &operator >> (QDataStream &ds, Tlv &tlv)
{
    tlv.setRawData(QByteArray());

    quint16 buf = 0;
    quint16 len = 0;
    QByteArray ba;
    ds >> buf >> len;
    tlv.setTag(static_cast<fdf::Tag>(buf));

    if(ds.atEnd() || len == 0u || (len > (ds.device()->size() - ds.device()->pos())))
    {
        tlv.setTag(fdf::Tag::Invalid);
        return ds;
    }
    ba.resize(len);
    ds.readRawData(ba.data(), len);

    tlv.setValue(ba);
    return ds;

}
