#include "modeldatastorage.h"
#include "cashboxbuildconfig.h"
#include "appdirectories.h"

#include "dircreator.h"



#include <QJsonObject>
#include <QJsonParseError>
#include <QDir>

const QString ModelDataStorage::MAIN_FILE =
        QString("%1/s_main.enc").arg(APATH.modelDataDir());
const QString ModelDataStorage::BACK_FILE=
        QString("%1/s_back.enc").arg(APATH.modelDataDir());

QMutex ModelDataStorage::_dataMutex;


ModelDataStorage::ModelDataStorage(QObject *parent) :
    QObject (parent)

{
    DirCreator::checkAndCreate(QDir::fromNativeSeparators(APATH.modelDataDir()));
}

ModelDataStorage::~ModelDataStorage()
{

}

ModelData ModelDataStorage::getModelData()
{
    ModelData res1, res2;
    _dataMutex.lock();
    CryptedSettingsStorage main(MAIN_FILE, nullptr);
    CryptedSettingsStorage backup(BACK_FILE, nullptr);
    getModelData(&main, res1);
    getModelData(&backup, res2);
    if(res1 != res2)
    {
        if(res1.isValid())
        {
            setModelData(&backup, res1);
        }
        else if(res2.isValid())
        {
            setModelData(&main, res2);
            res1 = res2;
        }
    }
    _dataMutex.unlock();
    return res1;
}

void ModelDataStorage::setModelData(const ModelData &md)
{
    _dataMutex.lock();
    CryptedSettingsStorage main(MAIN_FILE, nullptr);
    CryptedSettingsStorage backup(BACK_FILE, nullptr);
    setModelData(&main, md);
    setModelData(&backup, md);
    _dataMutex.unlock();
}

void ModelDataStorage::cleanModelData()
{
    ModelData md = getModelData();
    md.setSerial(QString());
    md.setSign(QString());
    setModelData(md);
}

RegData ModelDataStorage::getRegData()
{
    RegData res1, res2;
    _dataMutex.lock();
    CryptedSettingsStorage main(MAIN_FILE, nullptr);
    CryptedSettingsStorage backup(BACK_FILE, nullptr);
    getRegData(&main, res1);
    getRegData(&backup, res2);
    if(res1 != res2)
    {
        if(res1.isValid())
        {
            setRegData(&backup, res1);
        }
        else if(res2.isValid())
        {
            setRegData(&main, res2);
            res1 = res2;
        }
    }
    _dataMutex.unlock();
    return res1;
}

void ModelDataStorage::setRegData(const RegData &rd)
{
    _dataMutex.lock();
    CryptedSettingsStorage main(MAIN_FILE, nullptr);
    CryptedSettingsStorage backup(BACK_FILE, nullptr);
    setRegData(&main, rd);
    setRegData(&backup, rd);
    _dataMutex.unlock();
}

void ModelDataStorage::cleanRegData()
{
    setRegData(RegData());
}

void ModelDataStorage::cleanAll()
{
    cleanRegData();
    cleanModelData();
}





bool ModelDataStorage::getModelData(CryptedSettingsStorage *storage, ModelData &data)
{
    data = ModelData();
    if(!storage) return false;
    storage->beginGroup("MODEL");
    QByteArray modelData = storage->value("modelData", QString()).toString().trimmed().toLatin1();
    storage->endGroup();
    if(modelData.isEmpty()) return false;
    modelData = QByteArray::fromBase64(modelData);
    QJsonParseError jerr;
    QJsonDocument jdoc = QJsonDocument::fromJson(modelData, &jerr);
    if(jerr.error != QJsonParseError::NoError)
    {
        return false;
    }
    data.parseMap(jdoc.toVariant().toMap());
    return data.isValid();
}

void ModelDataStorage::setModelData(CryptedSettingsStorage *storage, const ModelData &md)
{
    if(storage)
    {
        storage->beginGroup("MODEL");
        if(!md.isValid()) storage->remove("modelData");
        else
        {
            QJsonDocument jdoc = QJsonDocument::fromVariant(md.toMap());
            QByteArray modelData = jdoc.toJson(QJsonDocument::Compact);
            modelData = modelData.toBase64();
            storage->setValue("modelData", QString::fromLatin1(modelData));
        }
        storage->endGroup();
        storage->sync();
    }
}

bool ModelDataStorage::getRegData(CryptedSettingsStorage *storage, RegData &data)
{
    data = RegData();
    if(!storage) return false;
    storage->beginGroup("REG");
    QByteArray regData = storage->value("regData", QString()).toString().trimmed().toLatin1();
    storage->endGroup();
    if(regData.isEmpty()) return false;
    regData = QByteArray::fromBase64(regData);
    QJsonParseError jerr;
    QJsonDocument jdoc = QJsonDocument::fromJson(regData, &jerr);
    if(jerr.error != QJsonParseError::NoError)
    {
        return false;
    }
    data.parseMap(jdoc.toVariant().toMap());
    return data.isValid();
}

void ModelDataStorage::setRegData(CryptedSettingsStorage *storage, const RegData &rd)
{
    if(storage)
    {
        storage->beginGroup("REG");
        if(!rd.isValid()) storage->remove("regData");
        else
        {
            QJsonDocument jdoc = QJsonDocument::fromVariant(rd.toMap());
            QByteArray regData = jdoc.toJson(QJsonDocument::Compact);
            regData = regData.toBase64();
            storage->setValue("regData", QString::fromLatin1(regData));
        }
        storage->endGroup();
        storage->sync();
    }
}
