#include "fiscalizationcontroller.h"
#include "regdatavalidator.h"

#include "fswrapper.h"

#include "regdoc.h"
#include "modeldatastorage.h"
#include "internalobjectsorage.h"

#include "fsclosedoc.h"


#include <QElapsedTimer>
#include <QCoreApplication>

FiscalizationController::FiscalizationController(FsWrapper *fs, ProcessingDataWrapper *dw, QObject *parent)
    : BaseTaskProcessor(fs, dw, parent)
{

}

CoreApiResult FiscalizationController::fiscalize(const CoreTransaction &task)
{
    OfdSettings oldOfd = pdw_->getOfd();
    QElapsedTimer testTimer;
    testTimer.start();

    CoreApiResult result;
    if(!checkState(task, result)) return result;
    RegDataValidator validator;

    validator.setFs(pdw_->getFsFullStatus(FsFullStatus::CLEAN_ALL));
    validator.setModel(pdw_->getModelData());
    validator.setOldReg(pdw_->getRegData(true));
    IncomingReg newReg(task.params());
    if(!validator.validate(newReg))
    {
        return CoreApiResult(validator.lastError(), validator.lastErrorMsg());
    }
    if(!startFiscalization(newReg, result)) return result;
    if(!fillFiscParams(newReg, result)) return result;
    qint32 fd = 0;
    quint32 fiscCode = 0;
    QDateTime docDt;
    if(!commitFiscalization(newReg, result, fd, fiscCode, docDt)) return result;

    quint16 docType = 0;
    Tlv::Stlv list;
    CoreApiConst::ErrorCode err = fs_->readTlvDoc(fd, docType, list);
    BaseRegDoc *doc = nullptr;
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        if(newReg.forFiscalize()) doc = new RegDoc(this);
        else doc = new ReRegDoc(this);
        doc->setDt(docDt);
        doc->setFd(fd);
        doc->setFiscalCode(fiscCode);
        doc->setFsNumber(validator.fs().fsNumber());
        CoreRegData rd(newReg);
        rd.setSerial(validator.model().serial());
        rd.setFsFfd(fs::FFD::FFD1_2);
        rd.setCashboxFfd(fs::FFD::FFD1_2);
        rd.setFfd(fs::FFD::FFD1_2);
        rd.setCashboxVersion(validator.model().version());
        doc->setData(rd);
    }
    else
    {
        if(docType == static_cast<quint16>(fdf::DocType::Registration)) doc = new RegDoc(this);
        else doc = new ReRegDoc(this);
        doc->parseFromTlvList(list);
        doc->setFiscalCode(fiscCode);
        doc->setDt(docDt);
        doc->setFd(fd);
    }
    QVariantMap resdata = task.formatToExternal() ? doc->toExternalMap() : doc->toMap();
    RegData rd = doc->regData();
    ModelDataStorage md;
    md.setRegData(rd);
    IOSG::cleanRegData();
    IOSG::cleanFs();
    IOSG::cleanOfd();

    CoreConfigStore store;
    QMap<QString, OfdSettings>  supportefOfds = pdw_->getFsFullStatus().isRelease() ?
                                                   OfdSettings::SUPPORTED_OFD :
                                                   OfdSettings::SUPPORTED_DEBUG_OFD;
    OfdSettings newOfd = supportefOfds[rd.ofdInn().trimmed()];
    if(newOfd.isValid() && newOfd.inn().trimmed() !=oldOfd.inn().trimmed())
    {
        store.setOfdConfig(newOfd);
    }
    pdw_->reloadOfd();
    pdw_->getCashboxStatus();
    delete doc;
    return CoreApiResult(err, QString(), resdata);
}


CoreApiResult FiscalizationController::closefs(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    FsCloseIncoming incoming (task.params());

    RegData rd = pdw_->getRegData();
    FsFullStatus fs = pdw_->getFsFullStatus();
    QDateTime dt = QDateTime::currentDateTime();
    if(fs.lifeTime().expiredDt() <= QDate::currentDate())
    {
        dt = fs.status().lastDocDt().addSecs(1);
    }

    CoreApiConst::ErrorCode err = fs_->closeFsStart();
    if(err != CoreApiConst::ErrorCode::Ok) return CoreApiResult(err);
    err = fs_->sendDocData(incoming.tlv(rd));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    qint32 fd = 0;
    quint32 fiscCode = 0;
    RegNumber rnm(rd.regNumber().trimmed(), pdw_->getModelData().serial(), rd.ownerInn());
    err = fs_->closeFsCommit(dt, rnm, fd, fiscCode);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }


    quint16 docType = 0;
    Tlv::Stlv list;
    err = fs_->readTlvDoc(fd, docType, list);
    FsCloseDoc doc;

    if(err != CoreApiConst::ErrorCode::Ok)
    {
        doc.setIncoming(incoming, true);
        doc.setDt(dt);
        doc.setFd(fd);
        doc.setFiscalCode(fiscCode);
        doc.setFsNumber(fs.fsNumber());
        doc.setRegData(rd);
    }
    else
    {

        doc.parseFromTlvList(list);
        doc.setFiscalCode(fiscCode);
        doc.setDt(dt);
        doc.setFd(fd);
        doc.setIncoming(incoming, true);
        doc.setRegData(rd, false);
    }

    QVariantMap resdata = task.formatToExternal() ? doc.toExternalMap() : doc.toMap();


    pdw_->getFsFullStatus(FsFullStatus::CLEAN_ALL);
    return CoreApiResult(err, QString(), resdata);
}

CoreApiConst::ErrorCode FiscalizationController::syncRegData()
{
    FsFullStatus st = pdw_->getFsFullStatus();
    int count = 0;
    while(count < 3 && !st.isFiscalized() && st.error() != CoreApiConst::ErrorCode::Ok)
    {
        QCoreApplication::processEvents();
        ++count;
    }
    if(!st.isFiscalized()) return st.error();// CoreApiConst::ErrorCode::Ok;
    RegData reg;
    CoreApiConst::ErrorCode err = fs_->readRegData(st.lifeTime().regsCount(), reg);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    RegData oldReg = pdw_->getRegData(false);
    ModelData md = pdw_->getModelData();
    if(reg.serial() == md.serial() && (reg.fsNumber() == oldReg.fsNumber() || oldReg.fsNumber().trimmed().isEmpty())
            && reg.isValid())
    {
        IOSG::cleanRegData();
        ModelDataStorage mds;
        mds.setRegData(reg);
    }
    return CoreApiConst::ErrorCode::Ok;
}

bool FiscalizationController::checkState(const CoreTransaction &task, CoreApiResult &result)
{

    if(!BaseTaskProcessor::checkState(task, result)) return false;

    CashboxStatus st = pdw_->getCashboxStatus();
    if(st.isFiscalized() && st.cycleIsOpen())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::CycleIsOpened);
        return false;
    }
    if(task.operation() == CoreTransaction::Operation::FS_CLOSE)
    {
        if(st.fs().transport().messagesCount() > 0)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::HasNotSended);
            return false;
        }
        if(!st.isFiscalized())
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::NotFiscalized);
            return false;
        }
    }


    return true;
}


bool FiscalizationController::startFiscalization(const IncomingReg &reg, CoreApiResult &result)
{
    result = CoreApiResult();
    fs::RegType rt;
    if(reg.forFiscalize()) rt = fs::RegType::NewRegistration;
    else if(reg.forChangeFs()) rt = fs::RegType::FsChanging;
    else rt = fs::RegType::KktSettingsChanging;
    CoreApiConst::ErrorCode err = fs_->startFiscalization(rt, fs::FFD::FFD1_2);
    if(err == CoreApiConst::ErrorCode::Ok) return true;
    result = CoreApiResult(err);
    return false;
}

bool FiscalizationController::fillFiscParams(const IncomingReg &reg, CoreApiResult &result)
{
    ModelData md = pdw_->getModelData();
    Tlv::Stlv list;
    Tlv buf;
    buf.setString(fdf::Tag::Address, reg.address());
    list << buf;
    buf.setString(fdf::Tag::CashboxSerial, md.serial());
    list << buf;
    if(!reg.getAutoMode() || !reg.cashier().isEmpty())
    {
        buf.setString(fdf::Tag::Cashier, reg.cashier());
        list << buf;
        Inn inn(reg.cashierInn());
        if(inn.isValid())
        {
            buf.setInn(fdf::Tag::CashierInn, inn);
            list << buf;
        }
    }
    if(reg.getAutoMode() && !reg.boxId().isEmpty())
    {
        buf.setString(fdf::Tag::BoxId, reg.boxId());
        list << buf;
    }
    if(!reg.getOfflineMode())
    {
        buf.setString(fdf::Tag::OfdName, reg.ofdName());
        list << buf;
        buf.setString(fdf::Tag::FnsSite, reg.ftsSite());
        list << buf;
        buf.setString(fdf::Tag::OwnerEmail, reg.email());
        list << buf;

    }
    buf.setString(fdf::Tag::OwnerName, reg.ownerName());
    list << buf;

    buf.setString(fdf::Tag::Place, reg.place());
    list << buf;

    buf.setString(fdf::Tag::CashboxVersion, md.version());
    list << buf;
    buf.setByte(fdf::Tag::CashboxFFD, static_cast<quint8>(fs::FFD::FFD1_2));
    list << buf;

    if(!reg.additionalParam().isEmpty())
    {
        buf.setString(fdf::Tag::RegAdditionalParam, reg.additionalParam());
        list << buf;
    }
    if(!reg.additionalParamData().isEmpty())
    {
        buf.setTag(fdf::Tag::RegAdditionalData);
        buf.setValue(reg.additionalParamData());
        list << buf;
    }
    CoreApiConst::ErrorCode err = fs_->sendDocData(list);
    if(err == CoreApiConst::ErrorCode::Ok) return true;
    fs_->cancelDoc();
    result = CoreApiResult(err);
    return false;
}

bool FiscalizationController::commitFiscalization(const IncomingReg &reg, CoreApiResult &result,
                                                  qint32 &fd, quint32 &fiscCode, QDateTime &docDt)
{
    ModelData md = pdw_->getModelData();
    FsFullStatus status = pdw_->getFsFullStatus();
    docDt = QDateTime::currentDateTime();
    Inn userInn (reg.ownerInn());
    RegNumber rnm(reg.regNumber(), md.serial(), reg.ownerInn());
    fs::RegFlags flags;
    flags.setFlag(fs::RegFlag::Enctiption , reg.getEncriptionMode());
    flags.setFlag(fs::RegFlag::OfflineMode, reg.getOfflineMode());
    flags.setFlag(fs::RegFlag::AutoMode   , reg.getAutoMode());
    flags.setFlag(fs::RegFlag::Services   , reg.extFlags().getServicesOnly());
    flags.setFlag(fs::RegFlag::Fosa       , reg.extFlags().getFosaCashbox());
    flags.setFlag(fs::RegFlag::ForInternet, reg.extFlags().getInternetOnly());
    flags.setFlag(fs::RegFlag::Restaurant, reg.extFlags().getRestaurant());
    flags.setFlag(fs::RegFlag::Wholesale, reg.extFlags().getWholesale());
    fs::ExtRegFlags extFlags;
    extFlags.setFlag(fs::ExtRegFlag::ExcisableProducts, reg.extFlags().getExcisableProducts());
    extFlags.setFlag(fs::ExtRegFlag::GamesCashbox     , reg.extFlags().getGamesCashbox());
    extFlags.setFlag(fs::ExtRegFlag::LotteryCashbox   , reg.extFlags().getLotteryCashbox());
    extFlags.setFlag(fs::ExtRegFlag::ExternalPrinter  , reg.getAutoMode() && reg.extFlags().getExternalPrinter());
    extFlags.setFlag(fs::ExtRegFlag::LabledProducts   , reg.extFlags().getLabledProducts());
    extFlags.setFlag(fs::ExtRegFlag::Pawnshop         , reg.extFlags().getPawnshop());
    extFlags.setFlag(fs::ExtRegFlag::Insurance        , reg.extFlags().getInsurance());
    extFlags.setFlag(fs::ExtRegFlag::AutoCashboxForLabled, reg.extFlags().getAutoCashboxForLabled());

    Inn ofdInn(reg.ofdInn());
    QSharedPointer<fdf::Reasons> reasons;
    if(!reg.forFiscalize())
    {
        reasons = QSharedPointer<fdf::Reasons>(new::fdf::Reasons(reg.reasons().flags()));
        if(reg.forRefiscalize() && status.currentFfd() != fs::FFD::FFD1_2)
        {
            reasons->setFlag(fdf::Reason::FFDChange);
            reasons->setFlag(fdf::Reason::CashboxVersionChange);
        }
    }
    CoreApiConst::ErrorCode err = fs_->completeFiscalization(
                docDt, userInn, rnm, reg.taxes(),
                flags, extFlags, ofdInn, fd, fiscCode, reasons.data());
    if(err == CoreApiConst::ErrorCode::Ok) return true;
    result = CoreApiResult(err);
    fs_->cancelDoc();
    return false;

}
