#include "frworkingthread.h"


#include "abstractworkerthread.h"

#include "internalobjectsorage.h"
#include "devicehelper.h"
#include "formatutils.h"

#include "modeldatastorage.h"

#include "fiscalizationcontroller.h"
#include "reportscontroller.h"

#include "cashboxbuildconfig.h"

#include "appdirectories.h"
#include "cashiersstore.h"


#include <QTimer>
#include <QFile>
#include <QCoreApplication>
#include <QTextDecoder>
#include <QVariantList>

FrWorkingThread::FrWorkingThread()
    : AbstractWorker()
    , started_(false)
    , queue_(new ProcessingQueue())
    , loopTimer_(nullptr)
    , pdw_(nullptr)
    , fs_(nullptr)
    , receipts_(nullptr)


{
}

FrWorkingThread::~FrWorkingThread()
{
    delete queue_;
    queue_ = nullptr;
    doStop();
}

bool FrWorkingThread::doStart()
{
    started_ = false;
    doStop();


    fs_ = new FsWrapper(this);
    fs_->open();
    pdw_ = new ProcessingDataWrapper(fs_, this);
    connect(pdw_, &ProcessingDataWrapper::sendToOfd, this, &FrWorkingThread::sendToOfd);

    receipts_ = new ReceiptsController(fs_, pdw_, this);



    loopTimer_ =new QTimer(this);
    loopTimer_->setInterval(MAX_LOOP_INTERVAL);
    loopTimer_->setSingleShot(true);
    connect(loopTimer_, SIGNAL(timeout()), this, SLOT(execCmd()));
    loopTimer_->start();


    return true;
}

bool FrWorkingThread::doStop()
{
    if(loopTimer_)
    {
        loopTimer_->blockSignals(true);
        loopTimer_->stop();
        delete loopTimer_;
        loopTimer_ = nullptr;
    }
    if(receipts_)
    {
        receipts_->deleteLater();
        receipts_ = nullptr;
    }
    if(pdw_)
    {
        pdw_->deleteLater();
        pdw_ = nullptr;
    }
    if(fs_)
    {
        fs_->close();
        fs_->deleteLater();
        fs_ = nullptr;
    }
    return true;
}

void FrWorkingThread::processTask(const CoreTransaction &task, CoreApiResult &result)
{

    if(task.operation() == CoreApiConst::RemoteCommand::FULL_CLEAR)
    {
        result = fullClear(task);
        return;
    }
    Cashier cashier;
    if(!checkAccess(task, result, cashier)) return;


    if(task.clientDateTime().isValid() && qAbs(task.clientDateTime().secsTo(QDateTime::currentDateTime())) > 180)
    {
        DeviceHelper().setDateTime(task.clientDateTime());
    }

    if(!checkModelData(task, result)) return;
    else switch (task.operation())
    {
    case CoreApiConst::RemoteCommand::GET_CASHIER_NAMES: result = getCashierNames();break;
    case CoreApiConst::RemoteCommand::LOGIN: result = getLogin(task);break;
    case CoreApiConst::RemoteCommand::CASHBOX_STATUS: result = getCashboxStatus(task);break;
    case CoreApiConst::RemoteCommand::WRITE_SERIAL: result = writeSerial(task);break;
    case CoreApiConst::RemoteCommand::FISCALIZATION: result = fiscalize(task);break;
    case CoreApiConst::RemoteCommand::FS_CLEAN: result = fsClean(task);break;
    case CoreApiConst::RemoteCommand::FS_COUNTERS: result = fsCounters(task);break;
    case CoreApiConst::RemoteCommand::OPEN_CYCLE: result = receipts_->openCycle(task);break;
    case CoreApiConst::RemoteCommand::CALC_REPORT: result = receipts_->calcReport(task);break;
    case CoreApiConst::RemoteCommand::CLOSE_CYCLE: result = receipts_->closeCycle(task);break;
    case CoreApiConst::RemoteCommand::RECEIPT: result = receipts_->receipt(task, cashier);break;
    case CoreApiConst::RemoteCommand::CORRECTION: result = receipts_->correction(task, cashier);break;
    case CoreApiConst::RemoteCommand::CHECK_LABEL: result = receipts_->checkLabel(task);break;
    case CoreApiConst::RemoteCommand::CLEAN_LABELS: result = receipts_->cleanLabels(task);break;
    case CoreApiConst::RemoteCommand::OFD_2_FS_CMD: result = ofdFsCommand(task);break;
    case CoreApiConst::RemoteCommand::FIND_FS_DOC:
        result = ReportsController(fs_, pdw_, this).findFsDocument(task);
        break;
    case CoreApiConst::RemoteCommand::FIND_OFD_TICKET:
        result = ReportsController(fs_, pdw_, this).findOfdTicket(task);
        break;
    case CoreApiConst::RemoteCommand::GET_TLV_DOC:
        result = ReportsController(fs_, pdw_, this).tlvDocument(task);
        break;
    case CoreApiConst::RemoteCommand::GET_REG_ARCHIVE_DOC:
        result = ReportsController(fs_, pdw_, this).regArchiveDocument(task);
        break;
    case CoreApiConst::RemoteCommand::GET_REG_TLV_DOC:
        result = ReportsController(fs_, pdw_, this).regArchiveTlvDocument(task);
        break;
    case CoreApiConst::RemoteCommand::FS_CLOSE: result = closefs(task);break;
    case CoreApiConst::RemoteCommand::LOAD_OFD_SETTINGS: result = downloadOfdSettings(task);break;
    case CoreApiConst::RemoteCommand::SAVE_OFD_SETTINGS: result = saveOfdSettings(task);break;
    case CoreApiConst::RemoteCommand::GET_OFFLINE_NOTIFICATIONS: result = getOfflineNotifications(task);break;
    case CoreApiConst::RemoteCommand::DOWNLOAD_CASHIERS: result = downloadCashiers(task);break;
    case CoreApiConst::RemoteCommand::SAVE_CASHIER: result = saveCashier(task);break;
    case CoreApiConst::RemoteCommand::REMOVE_CASHIER: result = removeCashier(task);break;
    case CoreApiConst::RemoteCommand::SELF_TEST: result = selfTest(task);break;
    case CoreApiConst::RemoteCommand::GET_CASHBOX_MODELDATA: result = getCashboxModelData(task);break;
    case CoreApiConst::RemoteCommand::LOAD_CORE_SETTINGS: result = loadConfig(task);break;
    case CoreApiConst::RemoteCommand::SAVE_CORE_SETTINGS: result = saveConfig(task);break;
    case CoreApiConst::RemoteCommand::SET_DATE_TIME: result = setDateTime(task);break;

   default: result = CoreApiResult(CoreApiConst::ErrorCode::UnsupportedApiCommand);break;
    }
    //    fs_->frcoreStatus();
}

bool FrWorkingThread::checkAccess(const CoreTransaction &task, CoreApiResult &result, Cashier &cashier)
{
    static const QSet<CoreApiConst::RemoteCommand> SKIP_ALL_CHECK = {
        CoreTransaction::Operation::GET_CASHIER_NAMES,
        CoreTransaction::Operation::CASHBOX_STATUS,
        CoreTransaction::Operation::OFD_2_FS_CMD,
        CoreTransaction::Operation::SELF_TEST,
        CoreTransaction::Operation::GET_CASHBOX_MODELDATA,
    };

    static const QSet<CoreApiConst::RemoteCommand> ANY_PERMISSION = {
        CoreTransaction::Operation::FS_COUNTERS,
        CoreTransaction::Operation::LOAD_OFD_SETTINGS,
        CoreTransaction::Operation::LOAD_CORE_SETTINGS,
        CoreTransaction::Operation::LOGIN
    };

    static const QMap<CoreApiConst::RemoteCommand, Cashier::Permission> PERMISSIONS = {
        std::make_pair(CoreApiConst::RemoteCommand::WRITE_SERIAL, Cashier::Permission::WriteSerial),
        std::make_pair(CoreApiConst::RemoteCommand::FISCALIZATION, Cashier::Permission::Fiscalization),
        std::make_pair(CoreApiConst::RemoteCommand::FS_CLEAN, Cashier::Permission::Fiscalization),
        std::make_pair(CoreApiConst::RemoteCommand::OPEN_CYCLE, Cashier::Permission::CycleOpen),
        std::make_pair(CoreApiConst::RemoteCommand::CALC_REPORT, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::CLOSE_CYCLE, Cashier::Permission::CycleClose),
        std::make_pair(CoreApiConst::RemoteCommand::FIND_FS_DOC, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::FIND_OFD_TICKET, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::GET_TLV_DOC, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::GET_REG_ARCHIVE_DOC, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::GET_REG_TLV_DOC, Cashier::Permission::Reports),
        std::make_pair(CoreApiConst::RemoteCommand::FS_CLOSE, Cashier::Permission::Fiscalization),
        std::make_pair(CoreApiConst::RemoteCommand::SAVE_OFD_SETTINGS, Cashier::Permission::Settings),
        std::make_pair(CoreApiConst::RemoteCommand::GET_OFFLINE_NOTIFICATIONS,
        Cashier::Permission::OfflineNotifications),
        std::make_pair(CoreApiConst::RemoteCommand::DOWNLOAD_CASHIERS,
        Cashier::Permission::Settings),
        std::make_pair(CoreApiConst::RemoteCommand::SAVE_CASHIER,
        Cashier::Permission::Settings),
        std::make_pair(CoreApiConst::RemoteCommand::REMOVE_CASHIER,
        Cashier::Permission::Settings),
        std::make_pair(CoreApiConst::RemoteCommand::SAVE_CORE_SETTINGS,
                       Cashier::Permission::Settings),
        std::make_pair(CoreApiConst::RemoteCommand::SET_DATE_TIME,
                       Cashier::Permission::Settings),
    };

    if(SKIP_ALL_CHECK.contains(task.operation())) return true;
    QString msg;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    if(!task.isValid(err, &msg))
    {
        result.setCode(err).setDescr(msg);
        return false;
    }
    Cashier c;
    if(!getCashier(task.cashierLogin(), task.cashierPassword(), c) || c.password() != task.cashierPassword())
    {
        result.setCode(CoreApiConst::ErrorCode::AccessDenied)
                .setDescr(tr("Доступ запрещен. Неверный номер кассира или пароль"));
        return false;
    }
    cashier = c;

    if(ANY_PERMISSION.contains(task.operation())) return true;

    if(task.operation() == CoreTransaction::Operation::LOGIN) return true;

    if(task.operation() == CoreTransaction::Operation::RECEIPT )
    {
        if(c.hasOneReceiptPermission()) return true;
        result.setCode(CoreApiConst::ErrorCode::AccessDenied)
                .setDescr(tr("Доступ запрещен. Не достаточно прав"));
        return false;

    }
    if(task.operation() == CoreTransaction::Operation::CORRECTION)
    {
        if(c.hasOneCorrectionPermission()) return true;
        result.setCode(CoreApiConst::ErrorCode::AccessDenied)
                .setDescr(tr("Доступ запрещен. Не достаточно прав"));
        return false;

    }

    if(task.operation() == CoreTransaction::Operation::CHECK_LABEL ||
        task.operation() == CoreTransaction::Operation::CLEAN_LABELS)
    {
        if(c.hasOneCorrectionPermission() || c.hasOneReceiptPermission()) return true;
        result.setCode(CoreApiConst::ErrorCode::AccessDenied)
                .setDescr(tr("Доступ запрещен. Не достаточно прав"));
        return false;
    }

    if(!PERMISSIONS.contains(task.operation()) ||
            !c.hasPermission(PERMISSIONS[task.operation()]))
    {
        result.setCode(CoreApiConst::ErrorCode::AccessDenied)
                .setDescr(tr("Доступ запрещен. Не достаточно прав"));
        return false;
    }
    return true;
}

bool FrWorkingThread::getCashier(qint32 id, Cashier &cashier)
{
    return pdw_? pdw_->getCashier(id, cashier) : false;
}

bool FrWorkingThread::getCashier(const QString &login, const QString &password, Cashier &cashier)
{
    return pdw_? pdw_->getCashier(login, password, cashier) : false;
}

Cashier FrWorkingThread::getCashier(const QString &login, const QString &password)
{
    Cashier c;
    if(pdw_) pdw_->getCashier(login, password, c);
    return c;
}



void FrWorkingThread::execCmd()
{
    procesFrStart();

    CoreTransaction task;

    loopTimer_->stop();
    if(!queue_->dequeue(task))
    {
//        checkNotSendedZ();
//        if(fmDC_)
//        {
//            QElapsedTimer t;
//            t.start();
//            fmDC_->exec();
//            if(t.elapsed() > 1000) lmWarning() << t.elapsed();
//        }
        if(loopTimer_)
        {
            loopTimer_->setSingleShot(true);
            loopTimer_->start(MAX_LOOP_INTERVAL);
        }
    }
    else
    {
        CoreApiResult result;
        processTask(task, result);
        result.setUid(task.uid());
        queue_->storeResult(result);

        if(task.operation() == CoreTransaction::Operation::FULL_CLEAR && result.isOk())
        {
            doStop();
            doStart();
        }
        if(loopTimer_)
        {
            loopTimer_->setSingleShot(true);
            loopTimer_->start(MIN_LOOP_INTERVAL);
        }
    }
}

bool FrWorkingThread::procesFrStart()
{
    if(!started_)
    {
        started_ = true;

        FiscalizationController(fs_, pdw_, this).syncRegData();
        RegData rd = pdw_->getRegData();
        pdw_->getFsFullStatus(FsFullStatus::CLEAN_ALL);
        CashboxStatus st = pdw_->getCashboxStatus();

        pdw_->reloadOfd();
        if(st.fs().error() == CoreApiConst::ErrorCode::Ok)
        {
            std::system("echo 1 > /sys/class/leds/red/brightness");
        }
        else
        {
            std::system("echo 1 > /sys/class/leds/red/brightness");
        }

    }
    return true;
}


bool FrWorkingThread::checkModelData(const CoreTransaction &task, CoreApiResult &result)
{
    static const QSet<CoreApiConst::RemoteCommand> SKIP_ALL_CHECK = {
        CoreTransaction::Operation::GET_CASHIER_NAMES,
        CoreTransaction::Operation::LOGIN,
        CoreTransaction::Operation::CASHBOX_STATUS,
        CoreTransaction::Operation::FS_CLEAN,
        CoreTransaction::Operation::FS_COUNTERS,
        CoreTransaction::Operation::OFD_2_FS_CMD,
        CoreTransaction::Operation::LOAD_OFD_SETTINGS,
        CoreTransaction::Operation::SAVE_OFD_SETTINGS,
        CoreTransaction::Operation::DOWNLOAD_CASHIERS,
        CoreTransaction::Operation::SAVE_CASHIER,
        CoreTransaction::Operation::REMOVE_CASHIER,
        CoreTransaction::Operation::GET_CASHBOX_MODELDATA,
        CoreTransaction::Operation::LOAD_CORE_SETTINGS,
        CoreTransaction::Operation::SAVE_CORE_SETTINGS,
        CoreTransaction::Operation::SET_DATE_TIME
    };
    if(!pdw_ || !started_)
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::ServiceDoesNotStarted, tr("Сервис еще не запущен"));
        return false;
    }

    if(SKIP_ALL_CHECK.contains(task.operation())) return true;

    if(QDateTime::currentDateTime() < CASHBOX_BUILD_DT)
    {
        //TODO: Другие проверки времени
        result = CoreApiResult(CoreApiConst::ErrorCode::InvalidCashboxDt, tr("В ККТ установлено некорректное время"));
        return false;
    }

    if(task.operation() == CoreTransaction::Operation::WRITE_SERIAL)
    {
        DeviceHelper h;
        IOSG::cleanModelData();
        if(pdw_ && pdw_->getModelData().checkSign(h.getHardwareHash()))
        {
            result.setCode(CoreApiConst::ErrorCode::SerialExists)
                    .setDescr(tr("Касса уже имеет заводской номер: ") + IOSG::getModelData().serial());
            return false;
        }
        return true;
    }

    ModelData md = IOSG::getModelData();
    FsFullStatus fs = pdw_->getFsFullStatus();


    if(fs.isFiscalized() && !md.serial().isEmpty() &&
            !fs.cashboxSerial().isEmpty() && md.serial() != fs.cashboxSerial())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::AlienFs,
                               tr("Подключенный ФН зарегистрирован с ККТ %1. А ЗН этой ККТ - %2")
                               .arg(fs.cashboxSerial(),md.serial()));
        return false;
    }
    if(fs.supportedFfd() != fs::FFD::FFD1_2 && task.operation() != CoreTransaction::Operation::WRITE_SERIAL)
    {
        if(fs.error() == CoreApiConst::ErrorCode::Ok)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::UnsupportedFsFFDVersion,
                                   tr("Данная версия ККТ работает только с фн, поддерживающим ФФД 1.2"));
        }
        else
        {
            result = CoreApiResult(fs.error(), CoreApiConst::defaultErrorMsg(fs.error()));
        }
        return false;

    }
    if(DEBUG_BUILD_VERSION && fs.isRelease() && !fs.fsNumber().trimmed().isEmpty())
    {
        if(fs.error() == CoreApiConst::ErrorCode::Ok)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::AlienFs,
                                   tr("Настоящая версия ККТ может работать только с МГМ. Подключен боевой ФН %1")
                                       .arg(fs.fsNumber()));
        }
        else
        {
            result = CoreApiResult(fs.error(), CoreApiConst::defaultErrorMsg(fs.error()));
        }
        return false;

    }

    if(fs.isFiscalized() && fs.currentFfd() != fs::FFD::FFD1_2 &&
            task.operation() != CoreTransaction::Operation::FISCALIZATION &&
            task.operation() != CoreTransaction::Operation::FS_CLOSE &&
            task.operation() != CoreTransaction::Operation::CLOSE_CYCLE &&
            task.operation() != CoreTransaction::Operation::WRITE_SERIAL)
    {
        if(fs.error() == CoreApiConst::ErrorCode::Ok)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::UnsupportedFFDVersion,
                                   tr("Данная версия ККТ работает только ФФД 1.2. Необходимо выполнить перерегистрацию ККТ для поддержки ФФД 1.2"));
        }
        else
        {
            result = CoreApiResult(fs.error(), CoreApiConst::defaultErrorMsg(fs.error()));
        }
        return false;
    }

    if(task.operation() != CoreTransaction::Operation::WRITE_SERIAL)
    {
        DeviceHelper h;
        IOSG::cleanModelData();
        if(pdw_ && !pdw_->getModelData().checkSign(h.getHardwareHash()))
        {
            result.setCode(CoreApiConst::ErrorCode::NoCashboxSerial)
                    .setDescr(tr("Не задан ЗН ККТ"));
            return false;
        }
    }

    switch (task.operation()) {
    case CoreTransaction::Operation::WRITE_SERIAL:
    {
        DeviceHelper h;
        IOSG::cleanModelData();
        if(pdw_ && pdw_->getModelData().checkSign(h.getHardwareHash()))
        {
            result.setCode(CoreApiConst::ErrorCode::SerialExists)
                    .setDescr(tr("Касса уже имеет заводской номер: ") + IOSG::getModelData().serial());
            return false;
        }
        return true;
    }break;
    case CoreTransaction::Operation::FISCALIZATION:
        case CoreTransaction::Operation::FS_CLOSE:
    {
        CashboxStatus cs = pdw_->getCashboxStatus();
        if(cs.fs().isFiscalized() && cs.fs().cycleIsOpen())
        {
            result.setCode(CoreApiConst::ErrorCode::CycleIsOpened)
                    .setDescr(tr("Перед перерегистрации ККТ необходимо закрыть смену"));
            return false;
        }
        return true;
    }break;
    case CoreTransaction::Operation::GET_OFFLINE_NOTIFICATIONS:
    {
        RegData rd = pdw_->getRegData();
        if(!rd.getOfflineMode())
        {
            result.setCode(CoreApiConst::ErrorCode::OfflineModeOnly)
                    .setDescr(tr("Команда выполняется только в автономном режиме"));
            return false;
        }
        return true;
    }break;
    case CoreTransaction::Operation::OPEN_CYCLE:
    case CoreTransaction::Operation::CALC_REPORT:
    case CoreTransaction::Operation::CLOSE_CYCLE:
    case CoreTransaction::Operation::RECEIPT:
    case CoreTransaction::Operation::CORRECTION:
    case CoreTransaction::Operation::CHECK_LABEL:
    case CoreTransaction::Operation::CLEAN_LABELS:
    case CoreTransaction::Operation::FIND_FS_DOC:
    case CoreTransaction::Operation::FIND_OFD_TICKET:
    case CoreTransaction::Operation::GET_TLV_DOC:
    case CoreTransaction::Operation::GET_REG_ARCHIVE_DOC:
    case CoreTransaction::Operation::GET_REG_TLV_DOC:
    case CoreTransaction::Operation::SELF_TEST:
    {
        return true;
    }break;
    default:
        break;
    }
    result.setCode(CoreApiConst::ErrorCode::UnsupportedApiCommand)
            .setDescr(tr("Обработчик команды не реализован"));
    return false;
}

CoreApiResult FrWorkingThread::getCashierNames() const
{
    CashiersStore store;
    QVariantList names = store.getCashierNames();

    QVariantMap res;
    res.insert("cashiers", names);
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), res);
}

CoreApiResult FrWorkingThread::getLogin(const CoreTransaction &task)
{
    Cashier cashier;
    if(getCashier(task.cashierLogin(), task.cashierPassword(), cashier))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(),
                             task.formatToExternal()?cashier.toExternalMap(false) : cashier.toMap());
    }
    return CoreApiResult(CoreApiConst::ErrorCode::AccessDenied, tr("Неверный пароль"));
}

CoreApiResult FrWorkingThread::getCashboxStatus(const CoreTransaction &task)
{
    Q_UNUSED(task)


    CashboxStatus status = pdw_->getCashboxStatus();

    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(),
                         task.formatToExternal() ? status.toExternalMap() : status.toMap());
}

CoreApiResult FrWorkingThread::getCashboxModelData(const CoreTransaction &task)
{
    Q_UNUSED(task)

    ModelData md = pdw_->getModelData();

    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(),
                         task.formatToExternal() ? md.toExternalMap() : md.toMap());
}

CoreApiResult FrWorkingThread::writeSerial(const CoreTransaction &task)
{
    ModelData md = pdw_->getModelData();
    QString serial = task.params().value("serial").toString().trimmed();
    QString sign = task.params().value("sign").toString().trimmed();
    bool ok = false;
    if(serial.length() != md.SERIAL_LENGTH || serial.toLongLong(&ok) <= 0 || !ok)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::SerialWriteError,
                             tr("Некорректный заводской номер"));
    }
    md.setSerial(serial);
    md.setSign(sign);
    if(!md.checkSign(DeviceHelper().getHardwareHash()))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::SerialWriteError,
                             tr("Ошибка проверки подписи"));
    }
    ModelDataStorage mds;
    mds.setModelData(md);
    mds.setRegData(RegData());
    IOSG::cleanRegData();
    IOSG::cleanModelData();
    md = pdw_->getModelData();
    if(!md.checkSign(DeviceHelper().getHardwareHash()))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::SerialWriteError,
                             tr("Ошибка записи заводского номера"));
    }

    FiscalizationController(fs_, pdw_, this).syncRegData();
    return CoreApiResult(CoreApiConst::ErrorCode::Ok,
                         QString(), QVariantMap());
}

CoreApiResult FrWorkingThread::fiscalize(const CoreTransaction &task)
{

    return FiscalizationController(fs_, pdw_, this).fiscalize(task);
}


CoreApiResult FrWorkingThread::closefs(const CoreTransaction &task)
{
    return FiscalizationController(fs_, pdw_, this).closefs(task);
}

CoreApiResult FrWorkingThread::fsCounters(const CoreTransaction &task)
{
    FsTotalType type = static_cast<FsTotalType>(task.params()["type"].toInt());
    if(type != FsTotalType::CycleTotal) type = FsTotalType::FsTotalType;
    FsCounters counters;
    CoreApiConst::ErrorCode err = fs_->readFsCounters(type, counters);
    if(err != CoreApiConst::ErrorCode::Ok) return CoreApiResult(err);
    return CoreApiResult(err, QString(), task.formatToExternal() ? counters.toExternalMap() : counters.toMap());
}

CoreApiResult FrWorkingThread::fsClean(const CoreTransaction &task)
{
    Q_UNUSED(task)
    QString version;
    bool isRelease = false;
    CoreApiConst::ErrorCode err = fs_->getFsVersion(version, isRelease);
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        if(isRelease) err= CoreApiConst::ErrorCode::UnknownFsError;
        else err  = fs_->cleanDebugFs();
    }
    IOSG::cleanFs();


    pdw_->getCashboxStatus();
    return CoreApiResult(err);
}

CoreApiResult FrWorkingThread::downloadCashiers(const CoreTransaction &task)
{
    CashiersStore store;
    QVariantList cashiers;
    if(!task.params().contains("cashierId"))
    {
        for(int i = Cashier::SUPER_USER_ID; i <= Cashier::MAX_CASHIERS_ID; ++i)
        {
            Cashier c = store.getCashier(i);
            if(!c.login().trimmed().isEmpty()) cashiers << (task.formatToExternal() ?c.toExternalMap(true): c.toMap());
        }
    }
    else
    {
        int id = task.params().value("cashierId").toInt();
        if(id < Cashier::SUPER_USER_ID || id > Cashier::MAX_CASHIERS_ID)
        {
            return CoreApiResult(CoreApiConst::ErrorCode::DatabaseLoadingError,
                                 tr("Ошибка загрузки данных"));
        }
        Cashier c = store.getCashier(id);
        cashiers << (task.formatToExternal() ?c.toExternalMap(true): c.toMap());

    }
    QVariantMap res;
    res.insert("cashiers", cashiers);
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), res);

}

CoreApiResult FrWorkingThread::saveCashier(const CoreTransaction &task)
{
    CashiersStore store;
    Cashier c (task.params());
    IOSG::cleanActiveCashiers();
    if(!store.saveCashier(c))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::DatabaseSavingError,
                             tr("Ошибка сохранения данных кассира"));

    }
    c = store.getCashier(c.id());
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(),
                         task.formatToExternal() ? c.toExternalMap(true) : c.toMap());
}

CoreApiResult FrWorkingThread::removeCashier(const CoreTransaction &task)
{
    CashiersStore store;
    IOSG::cleanActiveCashiers();

    qint32 id =  task.params().value("cashierId").toInt();

    if(!store.removeCashier(id))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::DatabaseSavingError,
                             tr("Ошибка удаления данных кассира"));

    }

    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), task.params());
}


CoreApiResult FrWorkingThread::downloadOfdSettings(const CoreTransaction &task)
{
    Q_UNUSED(task)
    OfdSettings ofd = pdw_->reloadOfd();
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), ofd.toMap());

}

CoreApiResult FrWorkingThread::saveOfdSettings(const CoreTransaction &task)
{
    OfdSettings ofd (task.params());
    CoreConfigStore store;
    store.setOfdConfig(ofd);
    ofd = pdw_->reloadOfd();
    pdw_->getFsFullStatus(FsFullStatus::CLEAN_CYCLE);
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), ofd.toMap());
}

CoreApiResult FrWorkingThread::setDateTime(const CoreTransaction &task)
{
    QDateTime dt = STR2DT_(task.params().value("dt").toString());
    if(!dt.isValid() || dt < CASHBOX_BUILD_DT)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::InvalidDt, tr("Передана некорректная дата для установки в ККТ"));
    }
    FsFullStatus st = pdw_->getFsFullStatus();
    if(st.isFiscalized() && dt < st.status().lastDocDt())
    {
        return CoreApiResult(CoreApiConst::ErrorCode::InvalidDt,
                             tr("Переданаzдата для установки в ККТ меньше даты последнего документа в ФН"));
    }
    DeviceHelper().setDateTime(dt);
    QThread::msleep(500);
    if(dt.secsTo(QDateTime::currentDateTime()) < 0)
    {

        return CoreApiResult(CoreApiConst::ErrorCode::DtSettingError,
                             tr("Ошибка устанвоки даты"));
    }
    QVariantMap res = {{"dt", DT2STR_(QDateTime::currentDateTime())}};
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), res);

}

CoreApiResult FrWorkingThread::ofdFsCommand(const CoreTransaction &task)
{
    FsFullStatus st = pdw_->getFsFullStatus();
//    lmWarning() << "OFD_CMD_START" << logtab << logvariant(task.toMap())
//                << logtab << logvariant(st.toMap());
    OfdFsCmd cmd(task.params());
    if(!cmd.isValid()) return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsCmd);
    FsOfdResponse resp = fs_->ofdCmd(cmd);
    if(cmd.cmdIsTransport())
    {
        st = pdw_->getFsFullStatus(FsFullStatus::CLEAN_TRANSPORT);
    }
    if(cmd.cmdIsLabelTransport())
    {
        st = pdw_->getFsFullStatus(FsFullStatus::CLEAN_LABELS_TRANSPORT);
    }

//    lmWarning() << "OFD_CMD_STOP"<< logtab<< logvariant(st.toMap()) << logtab
//                << logvariant(task.toMap()) << logtab
//                << logvariant(resp.toMap());
    return resp.toResult();
}

CoreApiResult FrWorkingThread::getOfflineNotifications(const CoreTransaction &task)
{
    OfflineDSessionStatus status;
    CoreApiConst::ErrorCode err = fs_->d3(true, status);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(status.notSendedCount() == 0u)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::NoDataRequested,
                             tr("В ФН нет уведомлений для выгрузки"));
    }
    QMap<quint32, quint16> crcMap;
    QString fn = APATH.offlineNotifications() + "/" + task.externalId();
    QFile f(fn);
    if(!f.open(QIODevice::WriteOnly))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::OfflineFileCreationError,
                             tr("Ошибка создания файла: ") + f.errorString());
    }
    QByteArray header;
    QDataStream ds(&header, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    QString sbuf = "Отчет о реализации маркированного товара";
    QTextEncoder cp866(QTextCodec::codecForName("CP866"));
    QByteArray buf = cp866.fromUnicode(sbuf);
    if(buf.size() < 66) buf.append(QByteArray(66 - buf.size(), ' '));
    ds.writeRawData(buf.constData(), buf.size());
    {
        ModelData md = pdw_->getModelData();
        buf = cp866.fromUnicode(md.modelName() + ", версия " + md.version());
        if(buf.size() < 256) buf.append(QByteArray(256 - buf.size(), ' '));
        ds.writeRawData(buf.constData(), buf.size());
    }
    {
        RegData rd = pdw_->getRegData();
        buf = cp866.fromUnicode(rd.regNumber());
        if(buf.size() < 20) buf.append(QByteArray(20 - buf.size(), ' '));
        ds.writeRawData(buf.constData(), buf.size());
    }
    FsFullStatus fs = pdw_->getFsFullStatus();
    {
        buf = cp866.fromUnicode(fs.fsNumber().trimmed());
        if(buf.size() < 16) buf.append(QByteArray(16 - buf.size(), ' '));
        ds.writeRawData(buf.constData(), buf.size());
    }
    quint32 docsCount = status.sessionNotSendedCount();
    ds << static_cast<quint8>(4) << status.fistSessionNotSended() << static_cast<quint32>(0)
       << static_cast<quint32>(status.sessionNotSendedCount()) << static_cast<quint32>(0);

    f.write(header);
    Crc16CITT crc;
    Crc32 crc32;
    crc32.startCalculating();
    bool first = true;
    quint32 last = status.fistSessionNotSended();
    quint32 firstDoc = status.fistSessionNotSended();
    while(status.sessionNotSendedCount() > 0u)
    {
        QByteArray data;
        quint16 len = 0;
        quint32 num = 0;
        err = fs_->d4(!first, len, num);
        if(num > 0 ) last = num;

        if(err != CoreApiConst::ErrorCode::Ok)
        {
            if(err == CoreApiConst::ErrorCode::NoDataRequested) break;
            f.close();
            QFile::remove(fn);
            return CoreApiResult(err);
        }
        for(quint16 offcet = 0; offcet < len; offcet += 1024)
        {
            quint16 sz = qMin(1024, len - offcet);
            QByteArray buf;
            quint16 invalidTag = 0;
            err = fs_->d5(offcet, sz, buf, invalidTag);
            if(err != CoreApiConst::ErrorCode::Ok)
            {
                QString msg = CoreApiConst::defaultErrorMsg(err);
                if(invalidTag > 0) msg += QString(" [ТЭГ: %1]").arg(invalidTag);
                return CoreApiResult(err);
            }            
            data.append(buf);
        }

        crcMap.insert(num, crc(data.mid(0, 2) + data.mid(4)));

        QByteArray hdata;
        QDataStream hdds(&hdata, QIODevice::WriteOnly);
        hdds.setByteOrder(QDataStream::LittleEndian);
        hdds << static_cast<quint16>(201) << static_cast<quint16>(data.size());
        data = hdata + data;

        f.write(data);
        crc32.addData(data);
        err = fs_->d3(false, status);
        first = false;
    }
    f.flush();
    header.clear();
    QDataStream sds(&header, QIODevice::WriteOnly);
    sds.setByteOrder(QDataStream::LittleEndian);
    sds << last << crc32.finishCalculating();
    f.seek(363);
    f.write(header.mid(0, 4));
    f.seek(371);
    f.write(header.mid(4, 4));
    f.close();

    quint32 num = 0;
    quint16 notConfirmed = 0;
    err = fs_->d6_0(notConfirmed, num);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        QFile::remove(fn);
        return CoreApiResult(err);
    }
    while(notConfirmed > 0)
    {
        err = fs_->d6_1(num, crcMap[num]);
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            if(err == CoreApiConst::ErrorCode::InvalidCmdParams) continue;
            QFile::remove(fn);
            return CoreApiResult(err);
        }
        err = fs_->d6_0(notConfirmed, num);
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            QFile::remove(fn);
            return CoreApiResult(err);
        }
    }


    QVariantMap res;
    res["fileName"] = fn;
//    QString baseName = QStringLiteral("Arc_%1_%2_%3.crpt").arg(fs.fsNumber()).arg(firstDoc).arg(last);
    QString baseName = QStringLiteral("%1_%2%3.fnm").arg(fs.fsNumber()).arg(firstDoc).arg(docsCount);
    res["baseName"] = baseName;
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), res);
}

CoreApiResult FrWorkingThread::selfTest(const CoreTransaction &task)
{
    Q_UNUSED(task)
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    FsFullStatus fs = pdw_->getFsFullStatus(FsFullStatus::CLEAN_ALL);
    if(fs.error() != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(fs.error());
    }
    return CoreApiResult(CoreApiConst::ErrorCode::Ok);
}




CoreApiResult FrWorkingThread::fullClear(const CoreTransaction &task)
{
    CoreApiResult result;
    return result;
    //TODO: Надо? не?
    // UmkaAuth *auth = new UmkaAuth(this);
    // if(!auth->syncLogin(task.cashierLogin(), task.cashierPassword()))
    // {
    //     result = CoreApiResult(CoreApiConst::ErrorCode::AccessDenied, auth->lastError());
    // }
    // else
    // {
    //     umka365::SessionData sd = auth->lastData();
    //     if(sd.dealerId() != FULL_CLEAR_AUTH_DEALER_ID)
    //     {
    //         result = CoreApiResult(CoreApiConst::ErrorCode::AccessDenied, tr("Доступ разрешен только для пользователей кабинета разработчика"));
    //     }
    //     else
    //     {
    //         //FIXME: сюда надо впихнуть очистку нварм, если есть, и баз
    //         ModelDataStorage mds;
    //         mds.cleanAll();
    //         IOSG::cleanRegData();
    //         IOSG::cleanModelData();
    //         mds.cleanAll();
    //         WhatcherDb wdb(WhatcherDb::CM_READ_WRITE);
    //         wdb.resetDb();
    //         result = CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), auth->lastData().toMap());
    //     }
    // }
    // auth->deleteLater();
    // return result;
}

CoreApiResult FrWorkingThread::loadConfig(const CoreTransaction &task)
{
    return pdw_->coreConfig()->processTransaction(task);
}

CoreApiResult FrWorkingThread::saveConfig(const CoreTransaction &task)
{
    return pdw_->coreConfig()->processTransaction(task);
}





