#include "keysupdater.h"
#include "internalobjectsorage.h"

#include "ofdmsgcontainer.h"
#include "fsfullstatus.h"
#include "coreruntimestorage.h"


KeysUpdater::KeysUpdater(FsWrapper *fs, ProcessingDataWrapper *pdw, QObject *parent)
    : QObject(parent)
    , fs_(fs)
    , pdw_(pdw)
    , socket_(new SyncSocketWrapper(this))
    , lastErroCode_(CoreApiConst::ErrorCode::Ok)
    , lastErrorMsg_()
{

}

KeysUpdater::~KeysUpdater()
{

}

CoreApiConst::ErrorCode KeysUpdater::lastErroCode() const
{
    return lastErroCode_;
}

const QString &KeysUpdater::lastErrorMsg() const
{
    return lastErrorMsg_;
}

CoreApiConst::ErrorCode KeysUpdater::exec(QString &printableText)
{
    lastErroCode_ = CoreApiConst::ErrorCode::Ok;
    lastErrorMsg_.clear();

    fs::KeysStatus ks = fs::KeysStatus::KeysUpdated, buf = fs::KeysStatus::KeysUpdated;
    QString addr;
    CoreApiConst::ErrorCode err = fs_->getKeysAddress(true, ks, addr);
//Это для отладки вместо МГМ без d7
//    err = CoreApiConst::ErrorCode::UnknownFsCmd;
//TEST
    if(err == CoreApiConst::ErrorCode::UnknownFsCmd)
    {
        CoreRuntimeStorage rstore;
        FsFullStatus st = pdw_->getFsFullStatus();
        if(!rstore.getKeysStatus(st.fsNumber().trimmed(), ks))
        {
            return CoreApiConst::ErrorCode::KeysStatusLoadingError;
        }

        if(st.cycle().cycle() > 0 && ks == fs::KeysStatus::KeysUpdated) return CoreApiConst::ErrorCode::Ok;
        OfdSettings ofdsts = pdw_->getOfd();
        addr = ofdsts.keysAddress().trimmed() + ":" + QString::number(ofdsts.keysPort());
    }
    else
    {
//        ks = fs::KeysStatus::Keys15; //Для тестов
        if(ks == fs::KeysStatus::KeysUpdated) return CoreApiConst::ErrorCode::Ok;
        err = fs_->getKeysAddress(false, buf, addr);
        if(err != CoreApiConst::ErrorCode::Ok) return err;
    }
    quint8 a20 = 0;
    quint16 srvErr = 0;
    QString srvMsg;
    err = update(addr, a20, srvErr, srvMsg);
    //TEST
//    err = CoreApiConst::ErrorCode::LabledProductsNotSupported;
    //TEST
    QStringList txt;
    if(err == CoreApiConst::ErrorCode::Ok)
    {
        //NOTE: Павлов сказал добавить
        //TODO: Возможно нужно возвращать наружу причину отказа в обновлении ключей при открытии смены
        txt << "Обновление ключей проверки КМ прошло успешно!";
        CoreRuntimeStorage rstore;
        rstore.updateKeysStatus(IOSG::getFs().fsNumber().trimmed());
        printableText = txt.join("\n");
        return err;
    }
    switch (ks) {
    case fs::KeysStatus::KeysUpdated:
        txt << "Нет необходимости в проведении процедуры обновления ключей проверки КМ";
        break;
    case fs::KeysStatus::Keys15:
        txt << "Обновление ключей проверки не выполнено.";
        break;
    case fs::KeysStatus::Keys60:
        txt << "Обновление ключей проверки не выполнено. Обратитесь в службу технической поддержки.";
        break;
    default:
        break;
    }
    if(err == CoreApiConst::ErrorCode::KeysServerError)
    {
        txt << tr("ОШИБКА СВЯЗИ С СЕРВЕРОМ ОКП");
    }
    else if(err == CoreApiConst::ErrorCode::OfdMsgError)
    {
        switch (a20) {
        case 1: txt << "Неверный фискальныйпризнак";break;
        case 2: txt << "Неверный формат ответа";break;
        case 3: txt << "Неверный номер запроса в ответе";break;
        case 4: txt << "Неверный номер ФН";break;
        case 5: txt << "Неверный CRC блока данных";break;
        case 7: txt << "Неверная длина ответа";break;
        default:
            break;
        }
    }
    else if(err == CoreApiConst::ErrorCode::KeysServerFailure)
    {
        txt << tr("Ошибка сервиса обновления ключей проверки КМ")
            << tr("Код ответа сервиса\t%1").arg(srvErr)
            << srvMsg;
    }
    else if(err == CoreApiConst::ErrorCode::KeysServerUnknownError)
    {
        txt << "Неизвестный ответ сервиса обновления ключей проверки";
    }
    else
    {
        txt << CoreApiConst::defaultErrorMsg(err);
    }
    printableText = txt.join("\n");
    return err;
}

CoreApiConst::ErrorCode KeysUpdater::update(const QString &addr, quint8 &a20,
                                            quint16 &srvAns, QString &srvMsg)
{
    QString host = addr;
    if(host.contains("//")) host = host.mid(host.indexOf("//") + 2);
    quint16 port = host.mid(host.lastIndexOf(":") + 1).trimmed().toUInt();
    host = host.mid(0, host.lastIndexOf(":"));

    QByteArray data;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    bool needUpdate = true;
    ReadFunc func = [this](QTcpSocket *socket) -> QByteArray {
        return readMsg(socket);
    };
    while(needUpdate)
    {
        err = getKeysData(data);
        if(err != CoreApiConst::ErrorCode::Ok) return err;
        if(!socket_->connectedToHost())
        {
            if(!socket_->connectToHost(host, port)) return CoreApiConst::ErrorCode::KeysServerError;
        }
        data = socket_->sendMsg(data, func);
        if(data.isEmpty()) return CoreApiConst::ErrorCode::KeysServerError;
        err = writeKeysData(data, a20, srvAns, srvMsg);
        needUpdate = (err == CoreApiConst::ErrorCode::RetryKeysUpdate);
    }
    socket_->setExchangeTimeout(1000);
    socket_->disconnectFromHost();
    return err;
}

QByteArray KeysUpdater::readMsg(QTcpSocket *socket) const
{
    if(!socket) return QByteArray();
    QByteArray res;
    qint32 len = 30;
    QElapsedTimer readTimer;
    readTimer.start();
    while(socket && res.size() < len && readTimer.elapsed() < 10000)
    {
        res.append(socket->read(len - res.size()));
        if(len == 30 && res.size() >= 30)
        {
            QDataStream ds(res.mid(24, 2));
            ds.setByteOrder(QDataStream::LittleEndian);
            quint16 buf = 0;
            ds >> buf;
            len += buf;
        }
    }
    return res;
}

CoreApiConst::ErrorCode KeysUpdater::getKeysData(QByteArray &data)
{
    CoreApiConst::ErrorCode err = fs_->getKeysData(QDateTime::currentDateTime(), data);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    FsFullStatus fs = IOSG::getFs();
    KeysMsgContainer c(0x0016u, fs.fsNumber().trimmed(), data);
    data = c.serialize();
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode KeysUpdater::writeKeysData(const QByteArray &data, quint8 &fsAnswer,
                                                   quint16 &srvAns, QString &srvMsg)
{
    FsFullStatus fs = IOSG::getFs();
    KeysMsgContainer c(fs.fsNumber(), data);
    return  fs_->writeKeysData(c.container(), fsAnswer, srvAns, srvMsg);
}

