#include "labelschecker.h"
#include "internalobjectsorage.h"

#include "ofdmsgcontainer.h"
#include "fsfullstatus.h"
#include "labelcheckdata.h"
#include "coreruntimestorage.h"


#include <QCoreApplication>

LabelsChecker::LabelsChecker(FsWrapper *fs, ProcessingDataWrapper *pdw, QObject *parent)
    : QObject(parent)
    , fs_(fs)
    , pdw_(pdw)
    , socket_(new SyncSocketWrapper(this))
    , lastErroCode_(CoreApiConst::ErrorCode::Ok)
    , lastErrorMsg_()
    , alternatives_()
{

}

LabelsChecker::~LabelsChecker()
{

}

CoreApiConst::ErrorCode LabelsChecker::lastErroCode() const
{
    return lastErroCode_;
}

const QString &LabelsChecker::lastErrorMsg() const
{
    return lastErrorMsg_;
}

/*
 *        SkipFsResult                    = 0x01,
        SkipFsWrong                     = 0x02,
        SkipNetworkError                = 0x04,
        SkipIfRequestError              = 0x08,
        SkipIsStatusError               = 0x10
*/

CoreApiConst::ErrorCode LabelsChecker::execute(const LabelCheckData &op, LabelCheckResult &result,
                                               QString &msg)
{
    alternatives_ = op.withAlternativeStatuses();
    return exec(op, result, msg);
}

CoreApiConst::ErrorCode LabelsChecker::exec(const LabelCheckData &op, LabelCheckResult &result, QString &msg)
{
    if(pdw_->getRegData().getOfflineMode()) return execOffline(op, result, msg);
    fdf::LabelCodeCheckFsResult r;
    fdf::LabelCodeCheckFsStatus s;
    CoreApiConst::ErrorCode err = fs_->b1(op.labelCode(), r, s);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(!checkB1Result(op, r, s, err, msg))
    {
        fs_->b2Cancel();
        return err;
    }
    QByteArray data;
    bool skipB5 = false;
    err = b5(op.checkFlags().testFlag(LabelCheckData::LabelCheckFlag::SkipB5), op.forB5(), skipB5, data);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->b2Cancel();
        return err;
    }
    if(skipB5)
    {
        fdf::CheckLabelFlags fl;
        err = fs_->b2(true, fl);
        if(err == CoreApiConst::ErrorCode::Ok) result = prepareFailData(op, r, s, fl);
        else fs_->b2Cancel();
        return err;
    }
    OfdSettings ofdsts = pdw_->getOfd();
    ReadFunc func = [this, ofdsts](QTcpSocket *socket) -> QByteArray {
        return readMsg(socket, ofdsts.testTimeout());
    };
    socket_->setExchangeTimeout(ofdsts.testTimeout());
    //TODO: генерация говна, если нет связи с сервером
    if(!socket_->connectedToHost())
    {
        if(!socket_->connectToHost(ofdsts.labelsAddress(), ofdsts.labelsPort()))
        {
            if(op.checkFlags().testFlag(LabelCheckData::SkipNetworkError))
            {
                fdf::CheckLabelFlags fl;
                err = fs_->b2(true, fl);
                if(err == CoreApiConst::ErrorCode::Ok) result = prepareFailData(op, r, s, fl);
                return err;
            }
            fs_->b2Cancel();
            return CoreApiConst::ErrorCode::LavbelsServerError;
        }
    }
    data = socket_->sendMsg(data, func);
    if(data.isEmpty())
    {
        if(op.checkFlags().testFlag(LabelCheckData::SkipNetworkError))
        {
            fdf::CheckLabelFlags fl;
            err = fs_->b2(true, fl);
            if(err == CoreApiConst::ErrorCode::Ok) result = prepareFailData(op, r, s, fl);
            return err;
        }
        fs_->b2Cancel();
        return CoreApiConst::ErrorCode::LavbelsServerError;
    }
    fdf::CheckLabelFlags flags;
    Tlv::Stlv out;
    err = b6(data, flags, out);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->b2Cancel();
        return err;
    }
    if(flags.testFlag(fdf::CheckLabelFlag::LabelChecked) &&
            !flags.testFlag(fdf::CheckLabelFlag::LabelValid) &&
            !op.checkFlags().testFlag(LabelCheckData::SkipIfRequestError))
    {
        msg = "Сервер ОИСМ: Некорректная марка";
        fs_->b2Cancel();
        return CoreApiConst::ErrorCode::LabelCheckOismError;
    }
    if(flags.testFlag(fdf::CheckLabelFlag::LabelChecked) &&
       flags.testFlag(fdf::CheckLabelFlag::LabelValid) &&
       flags.testFlag(fdf::CheckLabelFlag::StatusChecked) &&
       !flags.testFlag(fdf::CheckLabelFlag::StatusValid) &&
       !op.checkFlags().testFlag(LabelCheckData::SkipIsStatusError))
    {
        if(alternatives_.isEmpty())
        {
            msg = "Сервер ОИСМ: Некорректный статус товара";
            fs_->b2Cancel();
            return CoreApiConst::ErrorCode::LabelCheckOismError;
        }
        LabelCheckData d = alternatives_.first();
        alternatives_.pop_front();
        return exec(d, result, msg);
    }

    err = fs_->b2(true, flags);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->b2Cancel();
        return err;
    }

    result = LabelCheckResult(op);
    result.parseB6(flags, out);
    return err;
}

CoreApiConst::ErrorCode LabelsChecker::execOffline(const LabelCheckData &op, LabelCheckResult &result, QString &msg)
{
    fdf::LabelCodeCheckFsResult r;
    fdf::LabelCodeCheckFsStatus s;
    CoreApiConst::ErrorCode err = fs_->b1(op.labelCode(), r, s);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    if(!checkB1Result(op, r, s, err, msg))
    {
        fs_->b2Cancel();
        return err;
    }

    fdf::CheckLabelFlags flags;
    err = fs_->b2(true, flags);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->b2Cancel();
        return err;
    }

    result = LabelCheckResult(op);

    return err;
}




QByteArray LabelsChecker::readMsg(QTcpSocket *socket, qint32 tout) const
{
    if(!socket)
    {
        return QByteArray();
    }
    QByteArray res;
    qint32 len = 30;
    QElapsedTimer readTimer;
    readTimer.start();
    while(socket && res.size() < len && readTimer.elapsed() < tout)
    {
        res.append(socket->read(len - res.size()));
        if(len == 30 && res.size() >= 30)
        {
            QDataStream ds(res.mid(24, 2));
            ds.setByteOrder(QDataStream::LittleEndian);
            quint16 buf = 0;
            ds >> buf;
            len += buf;
        }
        QCoreApplication::processEvents();
    }
    return res;
}

CoreApiConst::ErrorCode LabelsChecker::b5(bool canSkipB5, const Tlv::Stlv &in, bool &skipB5, QByteArray &data)
{

    skipB5 = false;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    if(canSkipB5)
    {
        err = fs_->readBStatus2(skipB5);
        if(err != CoreApiConst::ErrorCode::Ok || skipB5)
        {
            return err;
        }
    }
    err = fs_->b5(QDateTime::currentDateTime(), in, data);
    if(err != CoreApiConst::ErrorCode::Ok) return err;
    FsFullStatus fs = IOSG::getFs();
    LabelTestContainer c(0x0016u, fs.fsNumber().trimmed(), data);
    data = c.serialize();
    return CoreApiConst::ErrorCode::Ok;
}

CoreApiConst::ErrorCode LabelsChecker::b6(const QByteArray &data, fdf::CheckLabelFlags &flags,
                                                   Tlv::Stlv &out)
{
    FsFullStatus fs = IOSG::getFs();
    LabelTestContainer c(fs.fsNumber(), data);
    return  fs_->b6(c.container(), flags, out);
}

bool LabelsChecker::checkB1Result(const LabelCheckData &op, fdf::LabelCodeCheckFsResult r,
                                  fdf::LabelCodeCheckFsStatus s,
                                  CoreApiConst::ErrorCode &err, QString &msg)
{
    if(r == fdf::LabelCodeCheckFsResult::NotChecked &&
            s!= fdf::LabelCodeCheckFsStatus::NotNeedCheck &&
            !op.checkFlags().testFlag(LabelCheckData::SkipFsResult))
    {
        msg = tr("Марка не проверена в ФН");
        err = CoreApiConst::ErrorCode::LabelCheckFsError;
        return false;
    }
    if(r == fdf::LabelCodeCheckFsResult::CheckedWrong)
    {
        CoreRuntimeStorage store;
        FsFullStatus st = IOSG::getFs();
        qint32 cycle = st.cycleIsOpen() ? st.cycle().cycle() : st.cycle().cycle() + 1;
        store.update2112(st.fsNumber().trimmed(), fdf::IncorrectLabelsFlag::IncorrectLabelResponse, cycle);
    }
    if(r == fdf::LabelCodeCheckFsResult::CheckedWrong &&
            !op.checkFlags().testFlag(LabelCheckData::SkipFsWrong))
    {
        msg = tr("Ошибка проверки марки в фн");
        if(s == fdf::LabelCodeCheckFsStatus::FsHasNoValidKey)
            msg += " (В ФН нет валидных рабочих для проверки)";
        if(s == fdf::LabelCodeCheckFsStatus::No91And92)
            msg += " (Марка не содержит полей 91 и 92)";
        err = CoreApiConst::ErrorCode::LabelCheckFsError;
        return false;
    }
    return true;
}

LabelCheckResult LabelsChecker::prepareFailData(const LabelCheckData &from,
                                                fdf::LabelCodeCheckFsResult r,
                                                fdf::LabelCodeCheckFsStatus s,
                                                const fdf::CheckLabelFlags &fl)
{
    Q_UNUSED(r)
    Q_UNUSED(s)
    LabelCheckResult result(from);
    result.setClFlags(fl);
    return result;
}


