#include "ofdworker.h"


#include "internalobjectsorage.h"
#include "ofdmsgcontainer.h"

#include <QEventLoop>

#include <QThread>

OfdWorker::OfdWorker()
    : AbstractWorker()
    , queue_(new ProcessingQueue())
    , loopTimer_(nullptr)
    , socket_(nullptr)
{

}

OfdWorker::~OfdWorker()
{
    delete queue_;
}

void OfdWorker::send()
{
    if(loopTimer_) loopTimer_->stop();
    loop();
}

bool OfdWorker::doStart()
{
    doStop();
    fsTimer_.invalidate();
    timerS_.invalidate();
    loopTimer_ = new QTimer(this);
    loopTimer_->setSingleShot(true);
    connect(loopTimer_, &QTimer::timeout, this, &OfdWorker::loop);
    loopTimer_->setInterval(LOOP_INTERVAL);
    loopTimer_->start();
    socket_ = new SyncSocketWrapper(this);
    return true;
}

bool OfdWorker::doStop()
{
    if(loopTimer_)
    {
        loopTimer_->stop();
        loopTimer_->deleteLater();
        loopTimer_ = nullptr;
    }
    if(socket_)
    {
        socket_->disconnectFromHost();
        socket_->deleteLater();
        socket_ = nullptr;
    }
    return true;
}

FsOfdResponse OfdWorker::exec(const OfdFsCmd &cmd)
{
    if(!cmd.isValid())
    {
        FsOfdResponse res(cmd);
        res.setError(CoreApiConst::ErrorCode::UnknownFsCmd);
        return res;
    }
    CoreTransaction transaction = cmd.transaction();
    QEventLoop loop;
    auto resultWaiter = [&loop] () ->void {
        QTimer::singleShot(5, &loop, SLOT(quit()));
    };
    bool savedToQueue =  queue_->enqueue(transaction, resultWaiter);

    CoreApiResult res;
    if(savedToQueue)
    {
        loop.exec();
        res = queue_->takeResult(transaction.uid());
    }
    else
    {
        FsOfdResponse resp(cmd);
        resp.setError(CoreApiConst::ErrorCode::InvalidTransactionType);
        return resp;
    }
    FsOfdResponse resp(res.data());
    return resp;
}

bool OfdWorker::syncConnectedState(bool connected, const FsTransportState &st)
{
    if(connected || socket_)
    {
        // lmCWarning(OFD) << connected << st.ofdConnected() << socket_;
    }
    if(!connected)
    {
        if(socket_)socket_->disconnectFromHost();
        if(st.ofdConnected())sendConnectionStatus(false);
        return true;
    }
    if(!socket_) return false;
    OfdSettings ofd = IOSG::getOfd();
    if(!ofd.isValid()) return false;
    socket_->setExchangeTimeout(EXCHANGE_INTERVAL);
    return (socket_->connectedToHost() || socket_->connectToHost(ofd.address(), ofd.port())) &&
            (st.ofdConnected() || sendConnectionStatus(true));
}

QByteArray OfdWorker::readMsg(QTcpSocket *socket) const
{
    if(!socket) return QByteArray();
    QByteArray res;
    qint32 len = 30;
    QElapsedTimer readTimer;
    readTimer.start();
    while(socket && res.size() < len && readTimer.elapsed() < 10000)
    {
        res.append(socket->read(len - res.size()));
        if(len == 30 && res.size() >= 30)
        {
            QDataStream ds(res.mid(24, 2));
            ds.setByteOrder(QDataStream::LittleEndian);
            quint16 buf = 0;
            ds >> buf;
            len += buf;
        }
    }
    return res;
}

void OfdWorker::loop()
{
    if(loopTimer_) loopTimer_->stop();
    OfdSettings ofd = IOSG::getOfd();
    if((timerS_.isValid() && timerS_.elapsed() < ofd.timerS()) || (fsTimer_.isValid() && fsTimer_.elapsed() < ofd.queryInterval()))
    {
        qint64 timeout1 = timerS_.isValid() ? ofd.timerS() - timerS_.elapsed() : 0;
        qint64 timeout2 = fsTimer_.isValid() ? ofd.queryInterval() - fsTimer_.elapsed() : 0;
//        lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval()
//                                               << logtab << loopTimer_->isActive() << timeout1 << timeout2;
        timeout1 = std::max(timeout1, timeout2);
        if(timeout1 > 0) QThread::msleep(timeout1);
        if(loopTimer_)loopTimer_->start();
        return;
    }
//    lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
    timerS_.invalidate();
    fsTimer_.invalidate();
    ReadFunc func = [this](QTcpSocket *socket) -> QByteArray {
        return readMsg(socket);
    };
    FsFullStatus st = IOSG::getFs();

    if(st.hasTranspot() && st.transport().hasMessageForOfd())
    {
//        lmWarning() << st.hasTranspot() << st.transport().hasMessageForOfd();
        QByteArray msg;
        if(!syncConnectedState(true, st.transport()) || !readOfdMessage(msg, st.fsNumber()))
        {
            timerS_.start();
//            lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
        }
        else
        {
            msg = socket_->sendMsg(msg, func);
            if(msg.isEmpty() || !writeOfdTicket(msg, st.fsNumber()))
            {
                timerS_.start();
//                lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
            }
            else if(ofd.queryInterval() > 0)
            {
                fsTimer_.start();
//                lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
            }
            st = IOSG::getFs();
        }
    }
    else if(ofd.queryInterval() > 0)
    {
        fsTimer_.start();
//        lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
    }
    st = IOSG::getFs();
    syncConnectedState(false, st.transport());
//    lmCWarning(OFD) << "BROKE " << loopTimer_;
    if(loopTimer_)loopTimer_->start();
//    lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
}

bool OfdWorker::sendConnectionStatus(bool connected)
{
    OfdFsCmd cmd(OfdFsCmd::Code::SendOfdConnectionState);
    cmd.setOfdConnected(connected);
    FsOfdResponse resp = exec(cmd);
    return resp.error() == CoreApiConst::ErrorCode::Ok;
}

bool OfdWorker::readOfdMessage(QByteArray &msg, const QString &fsNumber)
{
    OfdFsCmd cmd(OfdFsCmd::Code::ReadOfdMsg);
    FsOfdResponse resp = exec(cmd);
    if(resp.error() == CoreApiConst::ErrorCode::Ok)
    {
        OfdMsgContainer c(0x0016u, fsNumber.trimmed(), resp.ofdMessage());
        msg = c.serialize();
    }
    return !msg.isEmpty();
}

bool OfdWorker::cancelOfdMessage()
{
    OfdFsCmd cmd(OfdFsCmd::Code::CancelOfdMsgReading);
    FsOfdResponse resp = exec(cmd);
    return resp.error() == CoreApiConst::ErrorCode::Ok;
}

bool OfdWorker::writeOfdTicket(const QByteArray &msg, const QString &fsNumber)
{
    OfdMsgContainer c(fsNumber, msg);
    if(c.container().isEmpty()) return false;
    OfdFsCmd cmd(OfdFsCmd::Code::SendOfdReceipt);
    cmd.setOfdTicket(c.container());
    FsOfdResponse resp = exec(cmd);
    return resp.error() == CoreApiConst::ErrorCode::Ok;
}
