#include "oismworker.h"


#include "internalobjectsorage.h"
#include "ofdmsgcontainer.h"

#include <QEventLoop>

#include <QThread>


OismWorker::OismWorker()
    : AbstractWorker()
    , queue_(new ProcessingQueue())
    , loopTimer_(nullptr)
    , socket_(nullptr)
{

}

OismWorker::~OismWorker()
{
    delete queue_;
}

void OismWorker::send()
{
    if(loopTimer_) loopTimer_->stop();
    loop();
}

bool OismWorker::doStart()
{
    doStop();
    fsTimer_.invalidate();
    timerS_.invalidate();
    loopTimer_ = new QTimer(this);
    loopTimer_->setSingleShot(true);
    connect(loopTimer_, &QTimer::timeout, this, &OismWorker::loop);
    loopTimer_->setInterval(LOOP_INTERVAL);
    loopTimer_->start();
    socket_ = new SyncSocketWrapper(this);
    return true;
}

bool OismWorker::doStop()
{
    if(loopTimer_)
    {
        loopTimer_->stop();
        loopTimer_->deleteLater();
        loopTimer_ = nullptr;
    }
    if(socket_)
    {
        socket_->disconnectFromHost();
        socket_->deleteLater();
        socket_ = nullptr;
    }
    return true;
}

FsOfdResponse OismWorker::exec(const OfdFsCmd &cmd)
{
    if(!cmd.isValid())
    {
        FsOfdResponse res(cmd);
        res.setError(CoreApiConst::ErrorCode::UnknownFsCmd);
        return res;
    }
    CoreTransaction transaction = cmd.transaction();
    QEventLoop loop;
    auto resultWaiter = [&loop] () ->void {
        QTimer::singleShot(5, &loop, SLOT(quit()));
    };
    bool savedToQueue =  queue_->enqueue(transaction, resultWaiter);

    CoreApiResult res;
    if(savedToQueue)
    {
        loop.exec();
        res = queue_->takeResult(transaction.uid());
    }
    else
    {
        FsOfdResponse resp(cmd);
        resp.setError(CoreApiConst::ErrorCode::InvalidTransactionType);
        return resp;
    }
    FsOfdResponse resp(res.data());
    return resp;
}

bool OismWorker::syncConnectedState(bool connectToHost)
{

    if(connectToHost || socket_)
    {
        // lmCWarning(OISM) << connectToHost  << socket_;
    }
    if(!connectToHost)
    {
        if(socket_)socket_->disconnectFromHost();
        return true;
    }
    if(!socket_) return false;
    OfdSettings ofd = IOSG::getOfd();
    if(!ofd.isValid()) return false;
    socket_->setExchangeTimeout(EXCHANGE_INTERVAL);
    return (socket_->connectedToHost() || socket_->connectToHost(ofd.labelsAddress(), ofd.labelsPort()));
}

QByteArray OismWorker::readMsg(QTcpSocket *socket) const
{
    if(!socket) return QByteArray();
    QByteArray res;
    qint32 len = 30;
    QElapsedTimer readTimer;
    readTimer.start();
    while(socket && res.size() < len && readTimer.elapsed() < 10000)
    {
        res.append(socket->read(len - res.size()));
        if(len == 30 && res.size() >= 30)
        {
            QDataStream ds(res.mid(24, 2));
            ds.setByteOrder(QDataStream::LittleEndian);
            quint16 buf = 0;
            ds >> buf;
            len += buf;
        }
    }
    return res;
}

void OismWorker::loop()
{
    if(loopTimer_) loopTimer_->stop();
    OfdSettings ofd = IOSG::getOfd();
    if((timerS_.isValid() && timerS_.elapsed() < ofd.timerS()) || (fsTimer_.isValid() && fsTimer_.elapsed() < ofd.queryInterval()))
    {
        qint64 timeout1 = timerS_.isValid() ? ofd.timerS() - timerS_.elapsed() : 0;
        qint64 timeout2 = fsTimer_.isValid() ? ofd.queryInterval() - fsTimer_.elapsed() : 0;
        //        lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval()
//        << logtab << loopTimer_->isActive() << timeout1 << timeout2;
        timeout1 = std::max(timeout1, timeout2);
        if(timeout1 > 0) QThread::msleep(timeout1);
        if(loopTimer_)loopTimer_->start();
        return;
    }
    //    lmWarning() << timerS_.isValid() << timerS_.elapsed() << ofd.timerS() << logtab << fsTimer_.isValid() << fsTimer_.elapsed() << ofd.queryInterval() << logtab << loopTimer_->isActive();
    timerS_.invalidate();
    fsTimer_.invalidate();

    ReadFunc func = [this](QTcpSocket *socket) -> QByteArray {
        return readMsg(socket);
    };
    FsFullStatus st = IOSG::getFs();
//    lmCWarning(OISM) << st.hasLabelsTranspot() << st.labelsTransport().notificationCount();
    if (st.hasLabelsTranspot() && st.labelsTransport().notificationCount() > 0)
    {
        QByteArray msg;
        if(!syncConnectedState(true) || !readOismMessage(msg, st.fsNumber()))
        {
            timerS_.start();
        }
        else
        {
            msg = socket_->sendMsg(msg, func);
            if(msg.isEmpty() || !writeOismTicket(msg, st.fsNumber()))
            {
                timerS_.start();
            }
            else if(ofd.queryInterval() > 0)
            {
                fsTimer_.start();
            }
            st = IOSG::getFs();
        }
    }
    else if(ofd.queryInterval() > 0)
    {
        fsTimer_.start();
    }
    st = IOSG::getFs();
    syncConnectedState(false);
//    lmCWarning(OISM) << "BROKE " << loopTimer_;
    if(loopTimer_)loopTimer_->start();
}


bool OismWorker::readOismMessage(QByteArray &msg, const QString &fsNumber)
{
    OfdFsCmd cmd(OfdFsCmd::Code::ReadNotification);
    FsOfdResponse resp = exec(cmd);
    if(resp.error() == CoreApiConst::ErrorCode::Ok)
    {
        LabelNotifyContainer c(0x0016u, fsNumber.trimmed(), resp.ofdMessage());
        msg = c.serialize();
    }
    return !msg.isEmpty();
}

bool OismWorker::cancelOismMessage()
{
    OfdFsCmd cmd(OfdFsCmd::Code::CancelNofification);
    FsOfdResponse resp = exec(cmd);
    return resp.error() == CoreApiConst::ErrorCode::Ok;
}

bool OismWorker::writeOismTicket(const QByteArray &msg, const QString &fsNumber)
{
    LabelNotifyContainer c(fsNumber, msg);
    if(c.container().isEmpty()) return false;
    OfdFsCmd cmd(OfdFsCmd::Code::WriteNotificationTicket);
    cmd.setOfdTicket(c.container());
    FsOfdResponse resp = exec(cmd);
    return resp.error() == CoreApiConst::ErrorCode::Ok;
}
