#include "receiptscontroller.h"



#include "internalobjectsorage.h"
#include "cycleopendoc.h"
#include "calcreportdoc.h"
#include "cycleclosedoc.h"
#include "fsmsg.h"
#include "receipt.h"
#include "receiptdoc.h"
#include "keysupdater.h"
#include "labelcheckdata.h"
#include "labelschecker.h"
#include "coreruntimestorage.h"


#include "formatutils.h"



ReceiptsController::ReceiptsController(FsWrapper *fs,  ProcessingDataWrapper *dw, QObject *parent)
    : BaseTaskProcessor(fs, dw, parent)
{

}

ReceiptsController::~ReceiptsController()
{

}

bool ReceiptsController::checkState(const CoreTransaction &task, CoreApiResult &result)
{
    if(!BaseTaskProcessor::checkState(task, result)) return false;

    CashboxStatus st = pdw_->getCashboxStatus();



    if(!st.isFiscalized())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::NotFiscalized);
        return false;
    }
    switch (task.operation()) {
    case CoreTransaction::Operation::OPEN_CYCLE: return canOpenCycle(result);
    case CoreTransaction::Operation::CLOSE_CYCLE: return canCloseCycle(result);
    case CoreTransaction::Operation::RECEIPT:
    case CoreTransaction::Operation::CORRECTION:return canMakeReceipt(result);
    case CoreTransaction::Operation::CHECK_LABEL: return canCheckLabel(result);
    case CoreTransaction::Operation::CLEAN_LABELS: return canCheckLabel(result);
    default:
        return true;
    }
    return true;
}

CoreApiResult ReceiptsController::openCycle(const CoreTransaction &task)
{
    CycleOpenIncoming incoming (task.params());

    RegData rd = pdw_->getRegData();
    FsFullStatus fs = pdw_->getFsFullStatus();
    if(fs.error() != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(fs.error());
    }
    //TODO: Возвращать результат обновления ключей в докумен открытия смены
    QDateTime dt = QDateTime::currentDateTime();
    QString updateKeysText;
    updateKeys(updateKeysText);
//    return CoreApiResult(CoreApiConst::ErrorCode::UnknownError, tr("ТЕСТ ОБНОВЛЕНИЯ КЛЮЧЕЙ"));

    CoreApiConst::ErrorCode err = fs_->openCycleStart(dt);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    err = fs_->sendDocData(incoming.tlv(rd));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    quint16 cycle = 0;
    qint32 fd = 0;
    quint32 fiscCode = 0;
    err = fs_->openCycleCommit(cycle, fd, fiscCode);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        std::optional<FsStatus> st;
        return CoreApiResult(err);
    }


    quint16 docType = 0;
    Tlv::Stlv list;
    err = fs_->readTlvDoc(fd, docType, list);
    CycleOpenDoc doc;
    doc.setDataFromTask(task);

    if(err != CoreApiConst::ErrorCode::Ok)
    {


        doc.setIncoming(incoming, true);
        doc.setDt(dt);
        doc.setCycle(cycle);
        doc.setFd(fd);
        doc.setFiscalCode(fiscCode);
        doc.setFsNumber(fs.fsNumber());
        doc.setRegData(rd);
    }
    else
    {

        doc.parseFromTlvList(list);
        doc.setFiscalCode(fiscCode);
        doc.setDt(dt);
        doc.setFd(fd);
        doc.setCycle(cycle);
        doc.setIncoming(incoming, true);
        doc.setRegData(rd, false);
    }
    QVariantMap resdata = task.formatToExternal() ? doc.toExternalMap() : doc.toMap();


    pdw_->getFsFullStatus(FsFullStatus::CLEAN_CYCLE);
    return CoreApiResult(err, QString(), resdata);
}

CoreApiResult ReceiptsController::closeCycle(const CoreTransaction &task)
{

    CycleCloseIncoming incoming (task.params());

    RegData rd = pdw_->getRegData();
    FsFullStatus fs = pdw_->getFsFullStatus();

    QDateTime dt = QDateTime::currentDateTime();

    if(fs.lifeTime().expiredDt() <= QDate::currentDate())
    {
        dt = fs.status().lastDocDt().addSecs(1);
    }

    CoreApiConst::ErrorCode err = fs_->closeCycleStart(dt);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    fdf::IncorrectLabelsFlags ilf;
    fdf::IncorrectNotificationsFlags inf;

    CoreRuntimeStorage rstore;
    rstore.get2112And2113(fs.fsNumber().trimmed(), fs.cycle().cycle(), ilf, inf);

    err = fs_->sendDocData(incoming.tlv(rd, ilf, inf));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    quint16 cycle = 0;
    qint32 fd = 0;
    quint32 fiscCode = 0;
    fs::Warnings warnings;
    err = fs_->closeCycleCommit(cycle, fd, fiscCode, warnings);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        std::optional<FsStatus> st;
        return CoreApiResult(err);
    }


    quint16 docType = 0;
    Tlv::Stlv list;
    err = fs_->readTlvDoc(fd, docType, list);
    CycleCloseDoc doc;
    doc.setDataFromTask(task);

    if(err != CoreApiConst::ErrorCode::Ok)
    {
        doc.setIncoming(incoming, true);
        doc.setDt(dt);
        doc.setCycle(cycle);
        doc.setFd(fd);
        doc.setFiscalCode(fiscCode);
        doc.setFsNumber(fs.fsNumber());
        doc.setRegData(rd);
        doc.setWarnings(warnings);
    }
    else
    {

        doc.parseFromTlvList(list);
        doc.setFiscalCode(fiscCode);
        doc.setDt(dt);
        doc.setFd(fd);
        doc.setCycle(cycle);
        doc.setIncoming(incoming, true);
        doc.setRegData(rd, false);
    }
    QVariantMap resdata = task.formatToExternal() ? doc.toExternalMap() : doc.toMap();


    pdw_->getFsFullStatus(FsFullStatus::CLEAN_CYCLE);
    return CoreApiResult(err, QString(), resdata);
}

CoreApiResult ReceiptsController::calcReport(const CoreTransaction &task)
{
    CalcReportIncoming incoming (task.params());

    RegData rd = pdw_->getRegData();
    FsFullStatus fs = pdw_->getFsFullStatus();
    QDateTime dt = QDateTime::currentDateTime();
    CoreApiConst::ErrorCode err = fs_->calcReportStart(dt);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    err = fs_->sendDocData(incoming.tlv(rd));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        return CoreApiResult(err);
    }
    qint32 fd = 0;
    quint32 fiscCode = 0, notSended = 0;
    QDate notSendedDt;

    err = fs_->calcReportCommit(fd, fiscCode, notSended, notSendedDt);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        std::optional<FsStatus> st;
        return CoreApiResult(err);
    }

    quint16 docType = 0;
    Tlv::Stlv list;
    err = fs_->readTlvDoc(fd, docType, list);
    CalcReportDoc doc;
    doc.setDataFromTask(task);

    if(err != CoreApiConst::ErrorCode::Ok)
    {
        doc.setIncoming(incoming, true);
        doc.setDt(dt);
        doc.setCycle(fs.cycle().cycle());
        doc.setFd(fd);
        doc.setFiscalCode(fiscCode);
        doc.setFsNumber(fs.fsNumber());
        doc.setRegData(rd);
        doc.setNotSendedDocs(notSended);
        doc.setFirstNotSendedDocDt(notSendedDt);
    }
    else
    {

        doc.parseFromTlvList(list);
        doc.setFiscalCode(fiscCode);
        doc.setDt(dt);
        doc.setFd(fd);
        doc.setNotSendedDocs(notSended);
        doc.setFirstNotSendedDocDt(notSendedDt);
        doc.setIncoming(incoming, true);
        doc.setRegData(rd, false);
    }
    doc.setOfflineMode(rd.getOfflineMode());

    QVariantMap resdata = task.formatToExternal() ? doc.toExternalMap() : doc.toMap();
    pdw_->getFsFullStatus(FsFullStatus::CleanFlag::Status | FsFullStatus::CleanFlag::Resource | FsFullStatus::CleanFlag::Transport);
    return CoreApiResult(err, QString(), resdata);
}

CoreApiResult ReceiptsController::receipt(const CoreTransaction &task, const Cashier &cashier)
{
    Receipt r(task.params());
    FixNumber maxAmount = pdw_->coreConfig()->receiptConfig().maxReceiptAmount();
    if(maxAmount.value() > 0 && r.amount().value() > maxAmount.value())
    {
        return CoreApiResult(CoreApiConst::ErrorCode::ReceiptAmountToLarge,
                             tr("Превышена максимальная сумма по чеку (разрешено %1)").arg(maxAmount.toString()));
    }
    // FixNumber cashBoxCash = pdw_->getCashStatus().cash();
    // if(r.isCredit() && r.cash().has_value() && r.cash().value().value() > cashBoxCash.value())
    // {
    //     return CoreApiResult(CoreApiConst::ErrorCode::NoCashMoney,
    //                          tr("В кассе не достаточно наличных: сумма наличными по чеку %1; в кассе - %2)")
    //                              .arg(r.cash().value().toString(), cashBoxCash.toString()));
    // }

    if((r.isRefund() && !cashier.hasPermission(Cashier::Permission::ReturnReceipts)) ||
            (!r.isRefund() && !cashier.hasPermission(Cashier::Permission::Receipts)))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::AccessDenied, tr("Доступ запрещен. Не достаточно прав"));
    }

    RegData rd = pdw_->getRegData();
    r.setCorrection(false);
    r.setFosa(rd.extFlags().getFosaCashbox());
    r.setInternetCashbox(rd.extFlags().getInternetOnly());
    r.setSupportLabledTrandes(rd.extFlags().getLabledProducts());
    r.setSupportExciseTrandes(rd.extFlags().getExcisableProducts());

    return makeReceipt(task, r, rd);

}

CoreApiResult ReceiptsController::correction(const CoreTransaction &task, const Cashier &cashier)
{
    Receipt r(task.params());

    if((r.isRefund() && !cashier.hasPermission(Cashier::Permission::ReturnCorrections)) ||
            (!r.isRefund() && !cashier.hasPermission(Cashier::Permission::Corrections)))
    {
        return CoreApiResult(CoreApiConst::ErrorCode::AccessDenied, tr("Доступ запрещен. Не достаточно прав"));
    }

    RegData rd = pdw_->getRegData();
    r.setCorrection(true);
    r.setFosa(rd.extFlags().getFosaCashbox());
    r.setInternetCashbox(rd.extFlags().getInternetOnly());
    r.setSupportLabledTrandes(rd.extFlags().getLabledProducts());
    r.setSupportExciseTrandes(rd.extFlags().getExcisableProducts());

    return makeReceipt(task, r, rd);

}

CoreApiResult ReceiptsController::checkLabel(const CoreTransaction &task)
{    
    LabelCheckData lcd(task.params());
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::UnknownError;
    QString msg;
    if(!lcd.isValid(err, msg))
    {
        return CoreApiResult(err, msg);
    }

    LabelsChecker c(fs_, pdw_, this);
    LabelCheckResult result;
    err = c.execute(lcd, result, msg);

    if(err != CoreApiConst::ErrorCode ::Ok) return CoreApiResult(err, msg);

    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(),
                         task.formatToExternal() ? result.toExternalMap() : result.toMap());
}

CoreApiResult ReceiptsController::cleanLabels(const CoreTransaction &task)
{
    Q_UNUSED(task)
    CoreApiConst::ErrorCode err = fs_->b3();
    return CoreApiResult(err);
}



bool ReceiptsController::canOpenCycle(CoreApiResult &result)
{
    CashboxStatus st = pdw_->getCashboxStatus();
    if(st.fs().cycleIsOpen())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::CycleIsOpened);
        return false;
    }
    return true;
}

bool ReceiptsController::canCloseCycle(CoreApiResult &result)
{
    CashboxStatus st = pdw_->getCashboxStatus();
    if(!st.fs().cycleIsOpen())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::CycleIsClosed);
        return false;
    }
    return true;
}

bool ReceiptsController::canMakeReceipt(CoreApiResult &result)
{
    CashboxStatus st = pdw_->getCashboxStatus();
    if(!st.fs().cycleIsOpen())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::CycleIsClosed);
        return false;
    }
    return true;
}

bool ReceiptsController::canCheckLabel(CoreApiResult &result)
{
    if(!canMakeReceipt(result)) return false;
    RegData rd = pdw_->getRegData();
    if(!rd.extFlags().getLabledProducts())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::LabelsNotSupported);
        return false;
    }
    return true;
}

CoreApiResult ReceiptsController::makeReceipt(const CoreTransaction &task, Receipt &receipt, const RegData &rd)
{
    Q_UNUSED(task)


    FsFullStatus fs = pdw_->getFsFullStatus();


    QDateTime dt = QDateTime::currentDateTime();

    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    QString errMsg;
    if(!receipt.isValid(rd, err, errMsg))
    {
        return CoreApiResult(err, errMsg);
    }

    err = receipt.isCorrection() ? fs_->correctionStart(dt) : fs_->receiptStart(dt);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        if(receipt.mayBeLabled()) fs_->b3();
        return CoreApiResult(err);
    }
    if(receipt.containsLabels())
    {
        err = processLabledOperations(receipt, errMsg, fs.status().lastFd() + 1);
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            fs_->cancelDoc();
            fs_->b3();
            return CoreApiResult(err, errMsg);
        }
        ReceiptOperations ops = receipt.labledOperattions(false, true);
        Tlv::Stlv tops;
        std::for_each(ops.constBegin(), ops.constEnd(),
                      [&tops](const ReceiptOperation &o){tops << o.to1059();});
        if(!tops.isEmpty())
        {
            err = fs_->sendDocData(tops);
            if(err != CoreApiConst::ErrorCode::Ok)
            {
                fs_->cancelDoc();
                fs_->b3();
                return CoreApiResult(err);
            }
        }
    }
    err = fs_->sendDocData(receipt.toTlv(rd));
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        if(receipt.mayBeLabled()) fs_->b3();
        return CoreApiResult(err);
    }
    quint16 rec = 0;
    qint32 fd = 0;
    quint32 fiscalCode = 0;
    err = fs_->receiptCommit(dt, receipt.paymentAttr(), receipt.amount(), rec, fd, fiscalCode);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        fs_->cancelDoc();
        if(receipt.mayBeLabled()) fs_->b3();
        std::optional<FsStatus> st;
        return CoreApiResult(err);
    }

    BaseReceiptDoc *doc = BaseReceiptDoc::createReceiptDoc(receipt, this);    
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownError);
    }
    doc->setDataFromTask(task);
    doc->setRegData(rd);
    doc->setDt(dt);
    doc->setFd(fd);
    doc->setCycle(fs.cycle().cycle());
    doc->setFiscalCode(fiscalCode);
    doc->setFsNumber(fs.fsNumber());
    doc->setReceiptNumber(rec);

    QVariantMap resdata = task.formatToExternal() ? doc->toExternalMap() : doc->toMap();
    doc->deleteLater();
    pdw_->getFsFullStatus(FsFullStatus::CLEAN_CYCLE);
    return CoreApiResult(err, QString(), resdata);
}

CoreApiConst::ErrorCode ReceiptsController::processLabledOperations(Receipt &r, QString &msg, quint32 fd)
{
    r.setT2107(255);
    ReceiptOperations ops = r.labledOperattions(true, false);
    if(ops.isEmpty()) return CoreApiConst::ErrorCode::Ok;
    quint8 t2107 = 0;
    CoreApiConst::ErrorCode err = CoreApiConst::ErrorCode::Ok;
    for(const ReceiptOperation &o: ops)
    {
        quint16 invalidTag = 0;
        if(!o.isValid(err, msg)) return err;
        err = fs_->b7_1(o.to2007(), invalidTag);
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            msg = CoreApiConst::defaultErrorMsg(err);
            if(invalidTag > 0)
            {
                msg += QString(" [ТЭГ: %1]").arg(invalidTag);
            }
            return err;
        }
        err = fs_->b7_2(o.to1059(), invalidTag);
        if(err != CoreApiConst::ErrorCode::Ok)
        {
            msg = CoreApiConst::defaultErrorMsg(err);
            if(invalidTag > 0)
            {
                msg += QString(" [ТЭГ: %1]").arg(invalidTag);
            }
            return err;
        }
        if(o.checkLabelFlags() !=  fdf::CheckLabelFlags(0x0f) &&
                o.checkLabelFlags() != fdf::CheckLabelFlags(0x13))
        {
            t2107 = 1;
        }
    }
    quint16 invalidTag = 0;
    r.setT2107(t2107);
    err = fs_->b7_3(r.toTlvB7_3(pdw_->getRegData(), fd), invalidTag);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        msg = CoreApiConst::defaultErrorMsg(err);
        if(invalidTag > 0)
        {
            msg += QString(" [ТЭГ: %1]").arg(invalidTag);
        }
        return err;
    }
    return err;

}

CoreApiConst::ErrorCode ReceiptsController::updateKeys(QString &printableText)
{
    KeysUpdater u(fs_, pdw_, this);
    CoreApiConst::ErrorCode err = u.exec(printableText);
    return err;
}



