#include "reportscontroller.h"


#include "regdoc.h"
#include "cycleclosedoc.h"
#include "cycleopendoc.h"
#include "calcreportdoc.h"
#include "receiptdoc.h"
#include "fsclosedoc.h"


ReportsController::ReportsController(FsWrapper *fs, ProcessingDataWrapper *dw, QObject *parent)
    : BaseTaskProcessor(fs, dw, parent)
{

}

ReportsController::~ReportsController()
{

}

CoreApiResult ReportsController::findFsDocument(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    qint32 fd = 0;
    if(task.params().contains("fd")) fd = task.params()["fd"].toInt();
    else if(task.params().contains("number")) fd = task.params()["number"].toInt();
    if(fd == 0)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::InvalidFd, tr("Не указан номер документа"));
    }

    PFsArchiveReportBase doc;
    CoreApiConst::ErrorCode err = fs_->findDocument(fd, doc, createReportHeader());
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    if(doc->hasTicket())
    {
        PFsArchiveOfdTicket tdoc;
        err = fs_->getOfdTicketInfo(fd, tdoc);
        if(err == CoreApiConst::ErrorCode::Ok)doc->setTicket(tdoc->sign());
    }
    return CoreApiResult(err, QString(), task.formatToExternal() ? doc->toExternalMap() : doc->toMap());
}

CoreApiResult ReportsController::findOfdTicket(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    qint32 fd = 0;
    if(task.params().contains("fd")) fd = task.params()["fd"].toInt();
    else if(task.params().contains("number")) fd = task.params()["number"].toInt();
    if(fd == 0)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::InvalidFd, tr("Не указан номер документа"));
    }

    PFsArchiveOfdTicket doc;
    CoreApiConst::ErrorCode err = fs_->getOfdTicketInfo(fd, doc);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    return CoreApiResult(err, QString(), task.formatToExternal() ? doc->toExternalMap() : doc->toMap());
}

CoreApiResult ReportsController::tlvDocument(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    qint32 fd = 0;
    if(task.params().contains("fd")) fd = task.params()["fd"].toInt();
    else if(task.params().contains("number")) fd = task.params()["number"].toInt();
    if(fd == 0)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::InvalidFd, tr("Не указан номер документа"));
    }

    BaseDoc *doc = nullptr;
    quint16 docType = 0;
    Tlv::Stlv list;
    CoreApiConst::ErrorCode err = fs_->readTlvDoc(fd, docType, list);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    switch (static_cast<fdf::DocType>(docType)) {
    case fdf::DocType::Registration: doc = new RegDoc(this);break;
    case fdf::DocType::ReRegistration: doc = new ReRegDoc(this);break;
    case fdf::DocType::CycleOpen: doc = new CycleOpenDoc(this);break;
    case fdf::DocType::CycleClose: doc = new CycleCloseDoc(this);break;
    case fdf::DocType::CalcReport: doc = new CalcReportDoc(this);break;
    case fdf::DocType::FsClose: doc = new FsCloseDoc(this);break;
    case fdf::DocType::Receipt: doc = new ReceiptDoc(this);break;
    case fdf::DocType::Fosa: doc = new FosaDoc(this);break;
    case fdf::DocType::Correction: doc = new CorrectionDoc(this);break;
    case fdf::DocType::FosaCorrection: doc = new CorrectionFosaDoc(this);break;
    default:
        break;
    }
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    doc->parseFromTlvList(list);
    doc->setIsCopy(true);
    if(doc->fsNumber().trimmed().isEmpty())doc->setFsNumber(pdw_->getFsFullStatus().fsNumber());

    BaseWorkDoc *wdoc = dynamic_cast<BaseWorkDoc *>(doc);

    if(wdoc)
    {
        wdoc->setRegData(pdw_->getRegData());
    }

    CoreApiResult res =  CoreApiResult(err, QString(), task.formatToExternal() ? doc->toExternalMap() : doc->toMap());
    doc->deleteLater();
    return res;
}


CoreApiResult ReportsController::regArchiveDocument(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    qint32 num = 0;
    if(task.params().contains("number")) num = task.params()["number"].toInt();

    PFsArchiveReportBase doc;
    CoreApiConst::ErrorCode err = fs_->findRegDoc(num, doc, createReportHeader());
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    return CoreApiResult(err, QString(), task.formatToExternal() ? doc->toExternalMap() : doc->toMap());
}

CoreApiResult ReportsController::regArchiveTlvDocument(const CoreTransaction &task)
{
    CoreApiResult result;
    if(!checkState(task, result)) return result;
    qint32 num = 0;
    if(task.params().contains("number")) num = task.params()["number"].toInt();

    PFsArchiveReportBase doc;
    CoreApiConst::ErrorCode err = fs_->findRegDoc(num, doc, QStringList());
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(!doc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    if(doc->dt().daysTo(QDateTime::currentDateTime()) < 30)
    {
        QVariantMap params;
        params["fd"] = doc->docNumber();
        CoreTransaction t = task;
        t.setOperation(CoreApiConst::RemoteCommand::GET_TLV_DOC);
        t.setParams(params);
        t.genUid();
        result = tlvDocument(t);
        if(result.isOk())
        {
            return result;
        }
    }
    BaseRegDoc *regdoc = nullptr;
    err = fs_->readRegDoc(num, regdoc);
    if(err != CoreApiConst::ErrorCode::Ok)
    {
        return CoreApiResult(err);
    }
    if(!regdoc)
    {
        return CoreApiResult(CoreApiConst::ErrorCode::UnknownFsError);
    }
    if(regdoc->fsNumber().trimmed().isEmpty())regdoc->setFsNumber(pdw_->getFsFullStatus().fsNumber());
    QVariantMap data = task.formatToExternal() ? regdoc->toExternalMap() : regdoc->toMap();
    delete regdoc;
    return CoreApiResult(CoreApiConst::ErrorCode::Ok, QString(), data);
}



bool ReportsController::checkState(const CoreTransaction &task, CoreApiResult &result)
{
    if(!BaseTaskProcessor::checkState(task, result)) return false;
    FsFullStatus st = pdw_->getFsFullStatus();
    if(!st.isFiscalized())
    {
        result = CoreApiResult(CoreApiConst::ErrorCode::NotFiscalized);
        return false;
    }
    switch (task.operation()) {
    case CoreApiConst::RemoteCommand::FIND_FS_DOC:
    case CoreApiConst::RemoteCommand::FIND_OFD_TICKET:
    case CoreApiConst::RemoteCommand::GET_TLV_DOC    :
    {
        quint32 fd = 0;
        if(task.params().contains("fd")) fd = task.params()["fd"].toUInt();
        else if(task.params().contains("number")) fd = task.params()["number"].toUInt();
        if(fd == 0)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::InvalidFd, tr("Не указан номер документа"));
            return false;
        }
        if(st.status().lastFd() < fd)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::InvalidFd, tr("В ФН нет такого документа"));
            return false;
        }
    }break;
    case CoreApiConst::RemoteCommand::GET_REG_ARCHIVE_DOC:
    case CoreApiConst::RemoteCommand::GET_REG_TLV_DOC:
    {
        qint32 num = 0;
        if(task.params().contains("number")) num = task.params()["number"].toInt();
        if(st.lifeTime().regsCount() < num)
        {
            result = CoreApiResult(CoreApiConst::ErrorCode::InvalidParameter, tr("В ФН не было столько перерегистраций"));
            return false;
        }
        return true;
    }break;
    default:
        break;
    }
    return true;
}

QStringList ReportsController::createReportHeader() const
{
    RegData reg = pdw_->getRegData();
    QStringList text;
    text << reg.ownerName() << reg.address();
    if(!reg.place().isEmpty()) text << tr("МЕСТО РАСЧЕТОВ %1").arg(reg.place());
    text << tr("ЗН ККТ\t%1").arg(reg.serial());
    text << tr("РНМ ККТ\t%2").arg(reg.regNumber());
    text << tr("ФН\t%3").arg(pdw_->getFsFullStatus().fsNumber());
    text << tr("ИНН\t%1").arg(reg.ownerInn());
    return text;
}
