#ifndef CRC_H
#define CRC_H

#include <QtGlobal>
#include <QByteArray>

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
class Crc {
public:
    Crc();
    Crc(const Crc &other);
    Crc(Crc &&other);
    ~Crc();

    T operator ()(const quint8 *const data, quint32 size, bool fast = true) const;
    T operator ()(const char *const data, quint32 size, bool fast = true) const;
    T operator ()(const QByteArray &data, bool fast = true) const;

    void startCalculating();
    void addData(const quint8 *const data, quint32 size);
    void addData(const char *const data, quint32 size);
    void addData(const QByteArray &data);
    T finishCalculating();

    Crc &operator=(const Crc &other);
    Crc &operator=(Crc &&other);

private:
    quint64 topBit_;
    T crcTable_[256];
    T hash_;

    T reflect(T data, quint8 nBits) const;
    T crcSlow(const quint8 *const data, quint32 size) const;
    T crcFast(const quint8 *const data, quint32 size) const;
    quint8 reflectData(quint8 byte) const  {return refin ?(quint8) reflect(byte, 8) : byte;}
    T reflectReminder(T data) const {return refout ?(T) reflect(data, width) : data;}
};

typedef Crc<quint32, 32, 0xFFFFFFFFul, 0x04C11DB7ul, 0xFFFFFFFFul, true, true> Crc32;
typedef Crc<quint16, 16, 0x0000u, 0x8005u, 0x0000u, true, true> Crc16;
typedef Crc<quint16, 16, 0xFFFFu, 0x8005u, 0x0000u, true, true> Crc16IBM;
typedef Crc<quint16, 16, 0xFFFFu, 0x1021u, 0x0000u, false, false> Crc16CITT;
typedef Crc<quint16, 16, 0x0000u, 0x8408u, 0x0000u, true, true> CrcXMODEM;
typedef Crc<quint16, 16, 0xFFFFu, 0x1021, 0xFFFF, true, true> Crc16X25;
typedef Crc<quint8, 8, 0xFFu, 0x31u, 0x00u, false, false> Crc8Dallas;

//--------------------------------------------------------------------------------------------------
template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout>::Crc() :
    topBit_(1 << (width - 1)),
    crcTable_{0},
    hash_(init)
{
    T remainder;
    qint32 dividend;
    quint8  bit;
    for (dividend = 0; dividend < 256; ++dividend)
    {
        remainder = dividend << (width - 8);
        for (bit = 8; bit > 0; --bit)
        {
            if (remainder & topBit_)
            {
                remainder = (remainder << 1) ^ polynom;
            }
            else
            {
                remainder = (remainder << 1);
            }
        }
        crcTable_[dividend] = remainder;
    }
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout>::Crc(const Crc &other) :
    topBit_(1 << (width - 1)),
    crcTable_{0},
    hash_(other.hash_)

{
    memcpy(crcTable_, other.crcTable_, 256 * sizeof(T));
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout>::Crc(Crc &&other):
    topBit_(1 << (width - 1)),
    crcTable_{0},
    hash_(other.hash_)
{
    memcpy(crcTable_, other.crcTable_, 256 * sizeof(T));
}
template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout>::~Crc()
{

}
template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::operator ()(const quint8 * const data, quint32 size, bool fast) const
{
    return fast ? crcFast(data, size) : crcSlow(data, size);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::operator ()(const char * const data, quint32 size, bool fast) const
{
    return operator ()((quint8*)data, size, fast);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::operator ()(const QByteArray &data, bool fast) const
{
    return operator ()((quint8*)data.data(), data.size(), fast);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout> &
Crc<T, width, init, polynom, xorout, refin, refout>::operator=(const Crc &other)
{
    topBit_ = other.topBit_;
    memcpy(crcTable_, other.crcTable_, 256 * sizeof(T));
    return *this;
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
Crc<T, width, init, polynom, xorout, refin, refout>
&Crc<T, width, init, polynom, xorout, refin, refout>::operator=(Crc &&other)
{
    topBit_ = other.topBit_;
    memcpy(crcTable_, other.crcTable_, 256 * sizeof(T));
    return *this;
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::reflect(T data, quint8 nBits) const
{
    T  reflection = 0;
    quint8 bit;
    for (bit = 0; bit < nBits; ++bit)
    {
        if (data & 0x01)
        {
            reflection |= (1ULL << ((nBits - 1) - bit));
        }
        data = (data >> 1);
    }
    return (reflection);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::crcSlow(const quint8 * const data, quint32 size) const
{
    T remainder = init;
    for (quint32 byte = 0; byte < size; ++byte)
    {
        quint8 bt = reflectData(data[byte]);
        remainder ^= (T(bt) << (width - 8));
        for (quint8 bit = 8; bit > 0; --bit)
        {
            if (remainder & topBit_)
            {
                remainder = (remainder << 1) ^ polynom;
            }
            else
            {
                remainder = (remainder << 1);
            }
        }
    }
    return (reflectReminder(remainder) ^ xorout);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::crcFast(const quint8 * const data, quint32 size) const
{
    T remainder = init;

    for (quint32 byte = 0; byte < size; ++byte)
    {
        quint8 dat = reflectData(data[byte]) ^ (remainder >> (width - 8));
        remainder = crcTable_[dat] ^ (remainder << 8);
    }
    return (reflectReminder(remainder) ^ xorout);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
void Crc<T, width, init, polynom, xorout, refin, refout>::startCalculating()
{
    hash_ = init;
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
void Crc<T, width, init, polynom, xorout, refin, refout>::addData(const quint8 *const data, quint32 size)
{
    for (quint32 byte = 0; byte < size; ++byte)
    {
        quint8 dat = reflectData(data[byte]) ^ (hash_ >> (width - 8));
        hash_ = crcTable_[dat] ^ (hash_ << 8);
    }
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
void Crc<T, width, init, polynom, xorout, refin, refout>::addData(const char *const data, quint32 size)
{
    addData((quint8 *) data, size);
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
void Crc<T, width, init, polynom, xorout, refin, refout>::addData(const QByteArray &data)
{
    addData((quint8*)data.data(), data.size());
}

template<typename T, quint8 width, T init, T polynom, T xorout, bool refin, bool refout>
T Crc<T, width, init, polynom, xorout, refin, refout>::finishCalculating()
{
    return (reflectReminder(hash_) ^ xorout);
}


#endif // CRC_H

