#include "devicehelper.h"

#include "cashboxbuildconfig.h"

#include <QMutexLocker>
#include <QFile>
#include <QNetworkInterface>

#include <unistd.h>
#include <sys/time.h>
#include <cstdlib>
#include <optional>

#include <QCoreApplication>
#include <QProcess>
#include <QStorageInfo>


QString DeviceHelper::_dsn;
QMutex DeviceHelper::_mutex;


DeviceHelper::DeviceHelper()
{

}

QString DeviceHelper::getHardwareHash() const
{
    return getDsn();
}

QString DeviceHelper::getDsn() const
{
#ifdef RELEASE_UNIT
    QMutexLocker locker(&_mutex);
    Q_UNUSED(locker)
    if(!_dsn.isEmpty()) return _dsn;
    QFile f("/proc/cpuinfo");
    if(f.open(QIODevice::ReadOnly))
    {
        QString s = QString::fromUtf8(f.readAll());
        if(s.contains("Serial"))
        {
            s = s.mid(s.indexOf("Serial"));
            s = s.mid(0, s.indexOf("\n"));
            s = s.mid(0, s.indexOf("\r"));
            _dsn = s.mid(s.lastIndexOf(":") + 1).trimmed().toUpper();
        }
        f.close();
    }
    return _dsn;
#else
    auto itfs = QNetworkInterface::allInterfaces();
    std::optional<QNetworkInterface> eth, wifi;
    for(auto i: itfs)
    {
        if(i.type() == QNetworkInterface::Ethernet)eth = i;
        else if(i.type() == QNetworkInterface::Wifi)wifi = i;
    }
    QMutexLocker locker(&_mutex);
    Q_UNUSED(locker)
    if(eth.has_value() && !eth->hardwareAddress().isEmpty())
    {
        _dsn = eth->hardwareAddress().remove(":");
    }
    else if(wifi.has_value() && !wifi->hardwareAddress().isEmpty())
    {
        _dsn = wifi->hardwareAddress().remove(":");
    }
    else
    {
        _dsn = "unknown";
    }
    return _dsn;
#endif
}

qint32 DeviceHelper::versionCode() const
{
    return CASHBOX_APP_VERSION_CODE;
}

QString DeviceHelper::versionName() const
{
    return CASHBOX_APP_VERSION;
}

QString DeviceHelper::getIpAddress() const
{
    QList<QNetworkInterface> interfaces = QNetworkInterface::allInterfaces();
    QStringList outRes;
    for(const QNetworkInterface &i :interfaces)
    {
        if(i.flags().testFlag(QNetworkInterface::IsLoopBack) ||
            !i.flags().testFlag(QNetworkInterface::IsUp)) continue;
        QList<QNetworkAddressEntry> entries = i.addressEntries();
        for(const QNetworkAddressEntry &ae: entries)
        {
            if(!ae.ip().isNull() && !ae.ip().isLoopback())
            {
                bool ok = false;
                ae.ip().toIPv4Address(&ok);
                if(ok && ae.ip().toString().contains(QRegExp("(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}"
                                                              "(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)")) &&
                    !ae.ip().toString().startsWith("169.254."))
                {
                    outRes << ae.ip().toString();
                }
            }
        }
    }
    return outRes.isEmpty() ? QString() : outRes.join(",");
}

QString DeviceHelper::networkInterfaces() const
{
    QList<QNetworkInterface> interfaces = QNetworkInterface::allInterfaces();
    QStringList outRes;
    for(const QNetworkInterface &i :interfaces)
    {
        if(i.flags().testFlag(QNetworkInterface::IsLoopBack) ||
            !i.flags().testFlag(QNetworkInterface::IsUp)) continue;
        QList<QNetworkAddressEntry> entries = i.addressEntries();
        for(const QNetworkAddressEntry &ae: entries)
        {
            if(!ae.ip().isNull() && !ae.ip().isLoopback())
            {
                bool ok = false;
                ae.ip().toIPv4Address(&ok);
                if(ok && ae.ip().toString().contains(QRegExp("(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}"
                                                              "(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)")) &&
                    !ae.ip().toString().startsWith("169.254."))
                {
                    outRes << i.humanReadableName();
                }
            }
        }
    }
    return outRes.isEmpty() ? QString() : outRes.join(",");
}

QVariantMap DeviceHelper::getMemInfo() const
{
    QMutexLocker locker(&_mutex);
    Q_UNUSED(locker)
    QFile f("/proc/meminfo");
    QVariantMap res;
    if(f.open(QIODevice::ReadOnly))
    {
        QStringList sl = QString::fromUtf8(f.readAll()).split("\n");
        for(const QString &s: sl)
        {
            if(s.startsWith("MemTotal:"))
            {
                res.insert("memTotal", s.mid(s.indexOf(":")+ 1).trimmed());
            }
            else if(s.startsWith("MemFree:"))
            {
                res.insert("memFree", s.mid(s.indexOf(":")+ 1).trimmed());
            }
            else if(s.startsWith("MemAvailable:"))
            {
                res.insert("memAvailable", s.mid(s.indexOf(":")+ 1).trimmed());
            }
            if(res.size() >= 3) break;
        }
    }
    return res;
}

QVariantMap DeviceHelper::getStorageInfo() const
{
    QStorageInfo si("/");
    return {
        {"bytesTotal", si.bytesTotal()},
        {"bytesFree", si.bytesFree()},
        {"bytesAvailable", si.bytesAvailable()}
    };
}

void DeviceHelper::setDateTime(const QDateTime &dt)
{
    if(dt.isValid())
    {
        if (dt.timeSpec() == Qt::OffsetFromUTC)
        {
            int offsetSeconds = dt.offsetFromUtc();
            int offsetHours = offsetSeconds / 3600;
            int offsetMinutes = (offsetSeconds % 3600) / 60;
            if (offsetMinutes == 0)
            {
                QString tzName;
                if (offsetHours == 0)
                {
                    tzName = "Etc/UTC";
                }
                else
                {
                    tzName = QString("Etc/GMT%1%2")
                    .arg(offsetHours > 0 ? '-' : '+')
                        .arg(qAbs(offsetHours));
                }
                QString zonePath = "/usr/share/zoneinfo/uclibc/" + tzName;
                if (QFile::exists(zonePath))
                {
                    //TODO: Что-то сделать?
                    if (::unlink("/etc/TZ") != 0 && errno != ENOENT)
                    {
                        Q_UNUSED(dt)
                    }
                    else if (::symlink(zonePath.toUtf8().constData(), "/etc/TZ") != 0)
                    {
                        Q_UNUSED(dt)
                    }

                }
            }
        }
        QDateTime utcDateTime = dt.toUTC();
        qWarning() << utcDateTime <<dt;
        struct timeval tv;
        tv.tv_sec = utcDateTime.toSecsSinceEpoch();
        tv.tv_usec = 0;

        if (settimeofday(&tv, nullptr) != 0) {
            qWarning() << "Ошибка установки времени";
        }
        // 3. Принудительная синхронизация часов
        std::system("hwclock --systohc >/dev/null 2>&1");
    }
}

void DeviceHelper::updateTo(const QString &fileName)
{
    QFileInfo fi(QCoreApplication::applicationDirPath() + "/updater.sh");
    if(!fileName.isEmpty() && QFileInfo::exists(fileName))
    {
        QByteArray cmd = "chmod ugo+x " + fileName.toUtf8();
        cmd.push_back('\x00');
        std::system(cmd.constData());
        QProcess::startDetached(fi.absoluteFilePath(), QStringList() << "--new_file" << fileName);
    }
}

