#include "firmwaremanipulator.h"

#include <QDataStream>
#include <QFileInfo>
#include <QFile>
#include <QCryptographicHash>
#include <numeric>

FirmwareManipulator::FirmwareManipulator()
{

}

FirmwareManipulator::~FirmwareManipulator()
{

}

QByteArray FirmwareManipulator::encrypt(QByteArray &message, quint64 e, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1) {
        throw std::runtime_error("Modulus too small for encryption");
    }

    size_t plain_block_size = n_size - 1;

    QByteArray tmp;
    QDataStream ds(&tmp, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    ds << static_cast<quint32>(message.size());
    message = tmp + message;

    quint8 offcet = message.size() % plain_block_size;
    if (offcet != 0)
    {
        offcet = plain_block_size - offcet;
        message = message + QByteArray(offcet, '\xFF');
    }

    return encryptAligned(message, e, n);

}

QByteArray FirmwareManipulator::decrypt(const QByteArray &message, quint64 d, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1)
    {
        return QByteArray();
    }
    if (message.size() % n_size != 0)
    {
        return QByteArray();
    }

    QByteArray res = decryptAligned(message, d, n);
    QDataStream ds(res);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint32 len = 0;
    ds >> len;
    res = res.mid(4, len);
    return res;
}

bool FirmwareManipulator::encryptFile(const QString &srcFile, const QString &destFile, quint32 versionCode, const QString &version, quint64 e, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1)
    {
        return false;
    }

    size_t plain_block_size = n_size - 1;
    QFileInfo srcFi (srcFile);
    if(!srcFi.exists() || srcFi.size() == 0 || srcFi.size() > 15 *1024 * 1024) return false;
    QByteArray prefix;
    QDataStream ds(&prefix, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint8 prefixSize = 4 + 4 + 1 + version.toUtf8().size();
    if(prefixSize % plain_block_size != 0)
    {
        prefixSize += plain_block_size - (prefixSize % plain_block_size);
    }
    ds << static_cast<quint32>(srcFi.size()) << versionCode << static_cast<quint8>(prefixSize - 9);
    prefix += version.toUtf8();
    if(prefix.size() < prefixSize) prefix += QByteArray(prefixSize - prefix.size(), ' ');
    QFile src(srcFile), dest(destFile);
    QCryptographicHash hash(QCryptographicHash::Md5);
    if(!src.open(QIODevice::ReadOnly))
    {
        return false;
    }
    if(!dest.open(QIODevice::WriteOnly))
    {
        src.close();
        return false;
    }
    dest.write(encryptAligned(prefix, e, n));
    const qint32 blockSize = plain_block_size * 256;
    for(int i = 0; i < srcFi.size(); i += blockSize)
    {
        QByteArray buf = src.read(blockSize);
        hash.addData(buf);
        if(buf.size() % plain_block_size != 0)
        {
            buf += QByteArray( plain_block_size - (buf.size() % plain_block_size), '\x00');
        }
        dest.write(encryptAligned(buf, e, n));
    }
    dest.write(hash.result());
    src.close();
    dest.close();
    return true;
}

bool FirmwareManipulator::encryptFile(const QString &srcFile, QIODevice *dest, quint32 versionCode, const QString &version, quint64 e, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1 || !dest)
    {
        return false;
    }

    size_t plain_block_size = n_size - 1;
    QFileInfo srcFi (srcFile);
    if(!srcFi.exists() || srcFi.size() == 0 || srcFi.size() > 15 *1024 * 1024) return false;
    QByteArray prefix;
    QDataStream ds(&prefix, QIODevice::WriteOnly);
    ds.setByteOrder(QDataStream::LittleEndian);
    quint8 prefixSize = 4 + 4 + 1 + version.toUtf8().size();
    if(prefixSize % plain_block_size != 0)
    {
        prefixSize += plain_block_size - (prefixSize % plain_block_size);
    }
    ds << static_cast<quint32>(srcFi.size()) << versionCode << static_cast<quint8>(prefixSize - 9);
    prefix += version.toUtf8();
    if(prefix.size() < prefixSize) prefix += QByteArray(prefixSize - prefix.size(), ' ');
    QFile src(srcFile);
    QCryptographicHash hash(QCryptographicHash::Md5);
    if(!src.open(QIODevice::ReadOnly))
    {
        return false;
    }
    if(!dest->isOpen() || !dest->open(QIODevice::WriteOnly))
    {
        src.close();
        return false;
    }
    dest->write(encryptAligned(prefix, e, n));
    const qint32 blockSize = plain_block_size * 256;
    for(int i = 0; i < srcFi.size(); i += blockSize)
    {
        QByteArray buf = src.read(blockSize);
        hash.addData(buf);
        if(buf.size() % plain_block_size != 0)
        {
            buf += QByteArray( plain_block_size - (buf.size() % plain_block_size), '\x00');
        }
        dest->write(encryptAligned(buf, e, n));
    }
    dest->write(hash.result());
    src.close();
    dest->close();
    return true;
}

bool FirmwareManipulator::decryptFile(const QString &srcFile, const QString &destFile, quint64 d, quint64 n, quint32 &versionCode, QString &version)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1)
    {
        return false;
    }
    QFileInfo srcFi (srcFile);
    if(!srcFi.exists() || srcFi.size() == 0 || srcFi.size() > 15 *1024 * 1024 || (srcFi.size() - 16) % n_size != 0) return false;
    QFile src(srcFile), dest(destFile);
    if(!src.open(QIODevice::ReadOnly))
    {
        return false;
    }
    if(!dest.open(QIODevice::WriteOnly))
    {
        src.close();
        return false;
    }

    // int plain_block_size = n_size - 1;
    int cipher_block_size = n_size;
    const qint32 blockSize = cipher_block_size * 256;
    QCryptographicHash hash(QCryptographicHash::Md5);

    int fsz = srcFi.size() - 16;
    quint32 sourceFileSize = 0;
    quint32 storedFileSize = 0;
    for(int i = 0; i < fsz; i+= blockSize)
    {
        QByteArray buf = src.read(i + blockSize <= fsz ? blockSize : fsz - i);
        buf = decryptAligned(buf, d, n);
        if(i == 0)
        {
            QDataStream ds (buf);
            ds.setByteOrder(QDataStream::LittleEndian);
            quint8 vs = 0;
            ds >> sourceFileSize >> versionCode >> vs;
            version = QString::fromUtf8(buf.mid(9, static_cast<int>(vs))).trimmed();
            buf = buf.mid(static_cast<int>(vs) + 9);
        }
        storedFileSize += buf.size();
        hash.addData(buf);
        dest.write(buf);
    }
    QByteArray hash1 = hash.result(), hash2 = src.read(16);
    dest.close();
    src.close();    
    return hash1 == hash2;
}

bool FirmwareManipulator::decryptFile(QIODevice *src, const QString &destFile, quint64 d, quint64 n, quint32 &versionCode, QString &version)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1 || !src)
    {
        return false;
    }
    if(src->size() == 0 || src->size() > 15 *1024 * 1024 || (src->size() - 16) % n_size != 0) return false;
    QFile dest(destFile);
    if(!src->isOpen() && !src->open(QIODevice::ReadOnly))
    {
        return false;
    }
    if(!dest.open(QIODevice::WriteOnly))
    {
        src->close();
        return false;
    }

    // int plain_block_size = n_size - 1;
    int cipher_block_size = n_size;
    const qint32 blockSize = cipher_block_size * 256;
    QCryptographicHash hash(QCryptographicHash::Md5);

    int fsz = src->size() - 16;
    quint32 sourceFileSize = 0;
    quint32 storedFileSize = 0;
    for(int i = 0; i < fsz; i+= blockSize)
    {
        QByteArray buf = src->read(i + blockSize <= fsz ? blockSize : fsz - i);
        buf = decryptAligned(buf, d, n);
        if(i == 0)
        {
            QDataStream ds (buf);
            ds.setByteOrder(QDataStream::LittleEndian);
            quint8 vs = 0;
            ds >> sourceFileSize >> versionCode >> vs;
            version = QString::fromUtf8(buf.mid(9, static_cast<int>(vs))).trimmed();
            buf = buf.mid(static_cast<int>(vs) + 9);
        }
        storedFileSize += buf.size();
        hash.addData(buf);
        dest.write(buf);
    }
    QByteArray hash1 = hash.result(), hash2 = src->read(16);
    dest.close();
    src->close();
    return hash1 == hash2;
}

void FirmwareManipulator::generateRsaKeys(quint64 p, quint64 q, quint64 &e, quint64 &d, quint64 &n)
{
    n = p * q;
    quint64 phi = (p - 1) * (q - 1);

    // Выбор открытой экспоненты
    e = 65537;
    if (e >= phi)
    {
        e = 3; // Используем меньшую экспоненту
    }

    // Ищем подходящую экспоненту
    while (e < phi)
    {
        if (std::gcd(e, phi) == 1) break;
        e++;
    }

    // Вычисление секретной экспоненты
    auto [g, x, y] = extendedGcd(e, phi);
    d = (x % static_cast<qint64>(phi) + phi) % phi;
}

size_t FirmwareManipulator::numBytes(quint64 n)
{
    if (n == 0) return 1;
    size_t count = 0;
    while (n)
    {
        count++;
        n >>= 8;
    }
    return count;
}

quint64 FirmwareManipulator::modExp(quint64 base, quint64 exp, quint64 mod)
{
    quint64 result = 1;
    base %= mod;

    while (exp > 0)
    {
        if (exp & 1)
        {
            result = (result * base) % mod;
        }
        exp >>= 1;
        base = (base * base) % mod;
    }
    return result;
}

std::tuple<qint64, qint64, qint64> FirmwareManipulator::extendedGcd(qint64 a, qint64 b)
{
    if (b == 0)
    {
        return {a, 1, 0};
    }

    auto [gcd, x1, y1] = extendedGcd(b, a % b);
    qint64 x = y1;
    qint64 y = x1 - (a / b) * y1;

    return {gcd, x, y};
}

// quint64 FirmwareManipulator::gcd(quint64 a, quint64 b)
// {
//     while (b != 0)
//     {
//         quint64 temp = b;
//         b = a % b;
//         a = temp;
//     }
//     return a;
// }

QByteArray FirmwareManipulator::numToBytes(quint64 num, size_t len)
{
    QByteArray result(len, '\x00');
    for (int i = len; i > 0; i--)
    {
        result[i-1] = static_cast<char>(static_cast<quint8>(num & 0xFF));
        num >>= 8;
    }
    return result;
}

quint64 FirmwareManipulator::bytesToNum(const uint8_t *bytes, size_t len)
{
    quint64 result = 0;
    for (size_t i = 0; i < len; i++)
    {
        result = (result << 8) | bytes[i];
    }
    return result;
}


QByteArray FirmwareManipulator::encryptAligned(const QByteArray &message, quint64 e, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1)
    {
        return QByteArray();
    }

    size_t plain_block_size = n_size - 1;
    size_t cipher_block_size = n_size;

    // Проверка кратности длины данных размеру блока
    if (message.size() % plain_block_size != 0)
    {
        return QByteArray();
    }

    size_t block_count = message.size() / plain_block_size;
    QByteArray ciphertext(block_count * cipher_block_size, '\x00');

    for (size_t i = 0; i < block_count; i++)
    {
        const uint8_t* block = reinterpret_cast<const uint8_t*>(message.constData()) + i * plain_block_size;
        uint64_t msg = bytesToNum(block, plain_block_size);

        if (msg >= n)
        {
            return QByteArray();
        }

        uint64_t encrypted = modExp(msg, e, n);
        QByteArray encrypted_bytes = numToBytes(encrypted, cipher_block_size);

        std::copy(encrypted_bytes.begin(), encrypted_bytes.end(),
                  ciphertext.begin() + i * cipher_block_size);
    }

    return ciphertext;
}

QByteArray FirmwareManipulator::decryptAligned(const QByteArray &ciphertext, quint64 d, quint64 n)
{
    size_t n_size = numBytes(n);
    if (n_size <= 1)
    {
        return QByteArray();
    }

    int plain_block_size = n_size - 1;
    int cipher_block_size = n_size;

    // Проверка кратности длины данных размеру блока
    if (ciphertext.size() % cipher_block_size != 0)
    {
        return QByteArray();
    }

    size_t block_count = ciphertext.size() / cipher_block_size;
    QByteArray plaintext(block_count * plain_block_size, '\x00');

    for (size_t i = 0; i < block_count; i++)
    {
        const uint8_t* block = reinterpret_cast<const uint8_t*>(ciphertext.constData()) + i * cipher_block_size;
        uint64_t encrypted = bytesToNum(block, cipher_block_size);

        if (encrypted >= n)
        {
            return QByteArray();
        }

        uint64_t decrypted = modExp(encrypted, d, n);
        QByteArray decrypted_bytes = numToBytes(decrypted, plain_block_size);
        std::copy(decrypted_bytes.begin(), decrypted_bytes.end(),
                  plaintext.begin() + i * plain_block_size);
    }

    return plaintext;
}
