#include "fixnumber.h"

#include <QLocale>
#include <QStringList>
#include <QRegExp>


FixNumber::FixNumber() noexcept
    : realMultiplier_{DEFAULT_MULTIPLIER}
    , value_{0}
{
}

FixNumber::FixNumber(quint32 realMultiplier) noexcept
    : realMultiplier_{realMultiplier}
    , value_{0}
{
}

FixNumber::FixNumber(const QVariantMap &map)
    : realMultiplier_{DEFAULT_MULTIPLIER}
    , value_{0}
{
    setMap(map);
}

FixNumber::FixNumber(quint32 realMultiplier, qint64 value) noexcept
    : realMultiplier_{realMultiplier}
    , value_{value}
{
}

FixNumber::FixNumber(quint32 realMultiplier, double value) noexcept
    : realMultiplier_{realMultiplier}
    , value_{0ll}
{
    setDouble(value);
}

FixNumber::FixNumber(quint32 realMultiplier, const QString &value) noexcept
    : realMultiplier_{realMultiplier}
    , value_{0ll}
{
    setString(value);
}

FixNumber::FixNumber(const FixNumber &other) noexcept
    : realMultiplier_{other.realMultiplier_}
    , value_{other.value_}
{
}

FixNumber::FixNumber(FixNumber &&other) noexcept
    : realMultiplier_{other.realMultiplier_}
    , value_{other.value_}
{

}

FixNumber::~FixNumber()
{
}

quint32 FixNumber::realMultiplier() const
{
    return realMultiplier_;
}

void FixNumber::setRealMultiplier(const quint32 &realMultiplier)
{
    realMultiplier_ = realMultiplier;
}

qint64 FixNumber::value() const
{
    return value_;
}

void FixNumber::setValue(const qint64 &value)
{
    value_ = value;
}


double FixNumber::toDouble() const
{
    double res = value_;
    if(realMultiplier_)res /= realMultiplier_;
    return res;
}

QString FixNumber::toString() const
{
    return toString(QLocale().decimalPoint());
}

QString FixNumber::toString(const QChar &dp, const QChar &intDelim) const
{
    if(realMultiplier_ == 0) return QString::number(value_);
    qint64 val = qAbs(value_);
    int sign = static_cast<qint32>(val ? value_ / qAbs(val) : 1ll);
    if(realMultiplier_ <= 1)
    {
        return QString("%1").arg(value_);
    }

    qint64 decimal = val % realMultiplier_;
    qint64 intpart =  val / realMultiplier_;
    QStringList sint;
    do
    {
        if(intpart < 1000)
        {
            sint.push_front(QString::number(intpart));
            break;
        }
        else
        {
            sint.push_front(QString("%1").arg(intpart % 1000, 3, 10, QLatin1Char('0')));
            intpart = intpart / 1000;
        }
    }
    while(intpart > 0);
    QString sdec = QStringLiteral("%1").arg(decimal, decimals(), 10, QChar('0'));
    int dec = decimals();
    if(sdec.size() > dec)sdec = sdec.mid(0, decimals());
    else if (sdec.size() < dec) sdec += QString(dec - sdec.size(), QLatin1Char('0'));

    return QString("%1%2%3%4")
            .arg(sign < 0 ? QString("-") : QString())
            .arg(sint.join(intDelim.isNull() ? QStringLiteral("") : QString(intDelim)))
            .arg(dp).arg(sdec);
}

QString FixNumber::toShortString(const QChar &dp) const
{
    QString end = dp + QString(decimals(), QLatin1Char('0'));
    QString res = toString(dp);
    if(res.endsWith(end)) res = res.remove(end);
    return res;
}

FixNumber FixNumber::round(quint32 realMultiplier) const
{

    FixNumber res (realMultiplier);
    if(realMultiplier_ == realMultiplier)
    {
        res.setValue(value_);
        return res;
    }
    if(realMultiplier > realMultiplier_)
    {
        res.setValue(value_ * realMultiplier / realMultiplier_);
        return res;
    }
    qint64 val = value_ /(realMultiplier_ / realMultiplier);
    qint32 x = value_ % (realMultiplier_ / realMultiplier);
    x = x /(realMultiplier_ / realMultiplier / 10);
    qint32 y = val % 2;

    if(x > 5 || (x == 5 && y)) val++;
    res.setValue(val);
    return res;
}

void FixNumber::setDouble(double val)
{
    //Исправить округление периодических дробей с 9 в периоде
    val *= realMultiplier_;
    qint64 res = qRound64(val * 10) / 10;
    qint32 x = qRound((val - res) * 10);
    qint32 y = res % 2;
            //1 - static_cast<qint32>((val - res) * 100 ) % 2 ;
    if(x > 5 || (x == 5 && y)) res++;
    value_ = res;
}

void FixNumber::setString(const QString &val)
{
    QRegExp re ("(\\-?[0-9]+)([\\.,\\,]?)([0-9]*)");
    if(re.exactMatch(val))
    {
        QString s = val;
        s.replace(QString(","), QString("."));
        setDouble(s.toDouble());
    }
}

bool FixNumber::setAllFromString(const QString &val)
{
    QString s = val.trimmed();
    s = s.replace(QString(","), QString("."));
    bool res = true;
    if(val.trimmed().isEmpty())
    {
        realMultiplier_ = 1;
        value_ = 0ll;
    }
    else if(s.indexOf(".") < 0)
    {
        realMultiplier_ = 1;
        value_ = s.toLongLong(&res);
    }
    else
    {
        QStringList sl = s.split(".");
        if(sl.size() != 2) res = false;
        else
        {
            sl[0] = sl[0].trimmed();
            sl[1] = sl[1].trimmed();
            realMultiplier_ = static_cast<quint32>(10 * sl[1].size());
            if (realMultiplier_ == 0) realMultiplier_ = 1;
            value_ = (sl[0] + sl[1]).toLongLong(&res);
        }
    }
    return res;
}

qint32 FixNumber::decimals() const
{
    quint32 dec = realMultiplier_;
    qint32 res = 0;
    while(dec > 1)
    {
        ++res;
        dec /= 10;
    }
    return res;
}

void FixNumber::setDecimal(qint32 d)
{
    if(d >= 0)
    {
        realMultiplier_ = 1;
        for(int i = 0; i < d; ++i) realMultiplier_ *= 10;
    }
}

void FixNumber::setMap(const QVariantMap &map)
{
    realMultiplier_ = map["rm"].toString().toUInt();
    value_ = map["value"].toString().toLongLong();
}

QVariantMap FixNumber::toMap() const
{
    QVariantMap res;
    if(realMultiplier_ > 0)
    {
        res.insert("rm", QString::number(realMultiplier_));
        res.insert("value", QString::number(value_));
    }
    return res;
}

FixNumber &FixNumber::operator =(const FixNumber &other) noexcept
{
    realMultiplier_ = other.realMultiplier_;
    value_ = other.value_;
    return *this;
}

FixNumber &FixNumber::operator=(FixNumber &&other) noexcept
{
    realMultiplier_ = other.realMultiplier_;
    value_ = other.value_;
    return *this;
}

bool FixNumber::operator ==(const FixNumber &other) const noexcept
{
    return (realMultiplier_ == other.realMultiplier_ &&
            value_ == other.value_) || qFuzzyCompare(toDouble(), other.toDouble());
}

bool FixNumber::operator !=(const FixNumber &other) const noexcept
{
    return !(*this == other);
}

bool FixNumber::operator >(const FixNumber &other) const noexcept
{
    if(realMultiplier_ == other.realMultiplier_) return value_ > other.value_;
    if(realMultiplier_ == 0) return false;
    if(other.realMultiplier_ == 0) return  true;
    quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
    qint64 v1 = (rm / realMultiplier_) * value_, v2 = (rm / other.realMultiplier_) * other.value_;
    return v1 > v2;
}

bool FixNumber::operator >=(const FixNumber &other) const noexcept
{
    return (*this == other) || (*this > other);
}

bool FixNumber::operator <(const FixNumber &other) const noexcept
{
    if(realMultiplier_ == other.realMultiplier_) return value_ < other.value_;
    if(realMultiplier_ == 0) return true;
    if(other.realMultiplier_ == 0) return false;
    quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
    qint64 v1 = (rm / realMultiplier_) * value_, v2 = (rm / other.realMultiplier_) * other.value_;
    return v1 < v2;
}

bool FixNumber::operator <=(const FixNumber &other) const noexcept
{
    return (*this == other) || (*this < other);
}

FixNumber FixNumber::operator+(const FixNumber &other) const noexcept
{
    if(realMultiplier_ == other.realMultiplier_)
        return FixNumber(realMultiplier_, value_ + other.value_);
    quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
    qint64 m1 = realMultiplier_ ? rm/realMultiplier_ : 0;
    qint64 m2 = other.realMultiplier_ ? rm/other.realMultiplier_ : 0;
    return FixNumber(rm, value_ * m1 + other.value_ * m2);
}

FixNumber FixNumber::operator-(const FixNumber &other) const noexcept
{
    if(realMultiplier_ == other.realMultiplier_)
        return FixNumber(realMultiplier_, value_ - other.value_);
    quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
    qint64 m1 = realMultiplier_ ? rm/realMultiplier_ : 0;
    qint64 m2 = other.realMultiplier_ ? rm/other.realMultiplier_ : 0;
    return FixNumber(rm, value_ * m1 - other.value_ * m2);

}

FixNumber &FixNumber::operator+=(const FixNumber &other) noexcept
{
    if(0 == other.realMultiplier_) return *this;
    if(0 == realMultiplier_)
    {
        realMultiplier_ = other.realMultiplier_;
        value_ = other.value_;
        return *this;
    }
    if(realMultiplier_ == other.realMultiplier_)
    {
        value_ += other.value_;
    }
    else
    {
        quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
        qint64 m1 = rm/realMultiplier_;
        qint64 m2 = rm/other.realMultiplier_;
        realMultiplier_ = rm;
        value_ = value_ * m1 + other.value_ * m2;
    }
    return *this;
}

FixNumber &FixNumber::operator-=(const FixNumber &other) noexcept
{
    if(0 == other.realMultiplier_) return *this;
    if(0 == realMultiplier_)
    {
        realMultiplier_ = other.realMultiplier_;
        value_ = -other.value_;
        return *this;
    }
    if(realMultiplier_ == other.realMultiplier_)
    {
        value_ -= other.value_;
    }
    else
    {
        quint32 rm = qMax(realMultiplier_, other.realMultiplier_);
        qint64 m1 = rm/realMultiplier_;
        qint64 m2 = rm/other.realMultiplier_;
        realMultiplier_ = rm;
        value_ = value_ * m1 - other.value_ * m2;
    }
    return *this;
}

